/*
 * Copyright (C) 2000 Mdate Development Team, Project Admin:
 * Sean Dwyer <ewe2@users.sourceforge.net> . All Rights Reserved.
 * This code is distributed under the terms of the GPL.
 */

#ifdef HAVE_CONFIG_H
#  include "../config.h"
#endif

#include <mdate.h>

#if HAVE_LOCALE_H
# include <locale.h>
#endif
#if !HAVE_SETLOCALE
# define setlocale(Category, Locale)    /* empty */
#endif

#if ENABLE_NLS
#include <libintl.h>
#define _(x) gettext (x)
#else
# undef bindtextdomain
# define bindtextdomain(Domain, Directory)  /* empty */
# undef textdomain
# define textdmain(Domain)  /* empty */
# define _(Text) Text
#endif

#ifdef gettext_noop
# define N_(x) gettext_noop(x)
#else
# define N_(x) (x)
#endif

typedef enum
{ pretty_out, parseable_out }
output_types;

/* *INDENT-OFF* */
__BEGIN_DECLS 
void display_usage __P((void));
void display_version __P((void));
BOOL parse_args __P((int argc, char **argv, julian_date * thedate,
		     output_types * curr_out_type));
void display_cmd_results __P((julian_date thedate));
void display_prog_results __P((julian_date thedate));
__END_DECLS
/* *INDENT-ON* */

int main(int argc, char **argv)
{
  julian_date thedate;
  output_types curr_out_type;

  setlocale(LC_ALL, "");
  /* LOCALEDIR is configured in */
  bindtextdomain(PACKAGE, LOCALEDIR);
  textdomain(PACKAGE);

  if (parse_args(argc, argv, &thedate, &curr_out_type) == FALSE)
    return -1;

  switch (curr_out_type) {
    case parseable_out:
    display_prog_results(thedate);
    break;
    case pretty_out:
    display_cmd_results(thedate);
    break;
    default:
    fprintf(stderr, _("Invalid output type - please report this bug "
		      "to the Mdate team. Thanks.\n"));
    return 1;
  };
  return 0;
}

void display_usage(void)
{
  /* *INDENT-OFF* */
  printf("Mdate %s (%s-%s-%s) \n", VERSION, T_CPU, T_VENDOR, T_OS);
  printf(_("\n  usage: mdate [flags] [input]\n\n"));
  printf(_("    -h                 this help message.\n"));
  printf(_("    -v                 prints version and exits.\n"));
  printf(_("    -p                 program-parseable output (default=no).\n"));
  printf(_("    -d DD MM [-]YYYY   day month year (AD or BC Gregorian).\n"));
  printf(_("    -j NNNNNNNNN       julian day number [up to 9 digits].\n"));
  printf(_("    -l NN NN NN NN NN  long count.\n"));
  printf(_("\nReport bugs to ewe2@users.sourceforge.net\n\n"));
  /* *INDENT-ON* */
}

void display_version(void)
{
  printf("Mdate %s (%s-%s-%s) ", VERSION, T_CPU, T_VENDOR, T_OS);
  printf("Copyright (C) 2000 Mdate Development Team\n");
  printf("Project Admin: Sean Dwyer <ewe2@users.sourceforge.net>\n");
  exit(0);
}

/* add a set for the type of output we want: 
 * arguments after -p are set  cargz->arg = COM_PROG
 * arguments otherwise are set cargz->arg = COM_PRETTY
 */

BOOL
parse_args(int argc, char **argv, julian_date * thedate,
	   output_types * curr_out_type)
{
  int curr;
  long t;
  struct tm *today;

  *curr_out_type = pretty_out;
  if ((argc == 1)) {		/* No args given, use today's date */
    t = time(NULL);
    today = localtime(&t);
    if (!(jdate_from_tm(*today, thedate))) {
      return FALSE;
    }
    return TRUE;
  }

/* XXX: dirty hack, but it works :) -- ewe2 */
  if ((argc < 3) && (argv[1][1] == 'p')) {
    *curr_out_type = parseable_out;
    t = time(NULL);
    today = localtime(&t);
    if (!(jdate_from_tm(*today, thedate))) {
      return FALSE;
    }
    return TRUE;
  }

  /* Well, we have args ... parse them */
  for (curr = 1; curr < argc; curr++) {
    switch (argv[curr][0]) {
      case ' ':
      case '+':
      break;
      case '-':
      switch (argv[curr][1]) {
	case 'h':
	case '?':
	display_usage();
	return FALSE;
	case 'p':
	*curr_out_type = parseable_out;
	break;
	case 'j':
	if (curr == argc - 1) {
	  display_usage();
	  return FALSE;
	}

	if (!jdate_from_double_jd(strtod(argv[curr + 1], NULL), thedate)) {
	  fprintf(stderr, _("Bounds error for epoch, invalid JDN.\n"));
	  return FALSE;
	}
	return TRUE;
	case 'l':
	{
	  long_count from_ld;

	  if (argc < curr + 6) {
	    display_usage();
	    return FALSE;
	  }
	  from_ld.bak = atoi(argv[curr + 1]);
	  from_ld.kat = atoi(argv[curr + 2]);
	  from_ld.tun = atoi(argv[curr + 3]);
	  from_ld.uin = atoi(argv[curr + 4]);
	  from_ld.kin = atoi(argv[curr + 5]);
	  if (!(jdate_from_long_count(from_ld, thedate))) {
	    fprintf(stderr, _("Bad long date on command line.\n"));
	    return FALSE;
	  }
	  return TRUE;
	}
	case 'd':
	{
	  greg_date from_gd;

	  if (argc < curr + 4) {
	    display_usage();
	    return FALSE;
	  }
	  from_gd.day = atoi(argv[curr + 1]);
	  from_gd.month = atoi(argv[curr + 2]);
	  from_gd.year = atoi(argv[curr + 3]);
	  if (!jdate_from_greg_date(from_gd, thedate)) {
	    fprintf(stderr, _("Invalid date specified\n"));
	    return FALSE;
	  }
	  return TRUE;
	}
	case 'v':
	display_version();
	return FALSE;
	default:
	display_usage();
	return FALSE;
      }
    }
  }
  display_usage();
  return FALSE;
}

void display_prog_results(julian_date thedate)
{
  greg_date gd;
  haab_date hd;
  tzolkin_date td;
  long_count ld;
  char month_name[256];

  haab_date_from_jdate(thedate, &hd);
  tzolkin_date_from_jdate(thedate, &td);
  
  /* XXX: we now expect the frontends to handle error messages that
   * may require NLS support -- ewe2
   */
  if (!long_count_from_jdate(thedate, &ld))
	  fprintf(stderr, _("Bounds error for epoch, invalid data.\n"));
  if (!greg_date_from_jdate(thedate, &gd))
	  fprintf(stderr, _("Bounds error for epoch, invalid data."));

  /* no translation unless you want to alter tkmdate's parsing also! */
  printf("JDN: %9.1f ", thedate);
  printf("date: % .2d % .2d % d ", gd.day, gd.month, gd.year);
  printf("%.2d.%.2d.%.2d.%.2d.%.2d ", ld.bak, ld.kat, ld.tun, ld.uin, ld.kin);
  tzolkin_month_to_str(td.month, month_name);
  printf(" %.2d %-6.8s ", td.day, month_name);
  haab_month_to_str(hd.month, month_name);
  printf(" %.2d %-6.6s ", hd.day, month_name);
  printf("\n");
}

/* XXX: note the new data checks in here also -- ewe2 */

void display_cmd_results(julian_date thedate)
{
  greg_date gd;
  haab_date hd;
  tzolkin_date td;
  long_count ld;
  char date_buffer[256];

  tzolkin_date_from_jdate(thedate, &td);
  
  if (!long_count_from_jdate(thedate, &ld))
	  fprintf(stderr, _("Bounds error for epoch, invalid data.\n"));

  puts("\n");
  
  if (!greg_date_from_jdate(thedate, &gd))
	  fprintf(stderr, _("Bounds error for epoch, invalid data.\n"));
  
  greg_date_to_str(gd, date_buffer);
  printf(_("Gregorian Date          : %s\n"), date_buffer);
  jdate_to_str(thedate, date_buffer);
  printf(_("Julian Day Number       : %s\n"), date_buffer);
  
  if(!long_count_from_jdate(thedate, &ld))
	  fprintf(stderr, _("Bounds error for epoch, invalid data.\n"));
  
  long_count_to_str(ld, date_buffer);
  printf(_("Long Count              : %s\n"), date_buffer);
  tzolkin_date_from_jdate(thedate, &td);
  tzolkin_date_to_str(td, date_buffer);
  printf(_("Tzolkin Date            : %s\n"), date_buffer);
  haab_date_from_jdate(thedate, &hd);
  haab_date_to_str(hd, date_buffer);
  printf(_("Haab Date               : %s\n"), date_buffer);
  printf("\n");
}

/*
 * $Id: console.c,v 1.1.1.1 2000/04/06 15:27:57 ewe2 Exp $
 * $Log: console.c,v $
 * Revision 1.1.1.1  2000/04/06 15:27:57  ewe2
 * Mdate 1.2.0 release
 *
 * Revision 1.1.1.2  2000/03/27 00:58:39  ewe2
 * Mdate 1.1.0 tentative version
 *
 * Revision 1.2  2000/03/26 06:38:47  ewe2
 * Mdate 1.1.0 tentative version
 *
 * Revision 1.1.1.1  2000/03/16 03:39:17  ewe2
 * Initial import
 *
 * $State: Exp $
 */
