// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: gxsocket.cpp
// C++ Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 09/20/1999
// Date Last Modified: 06/27/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.
 
You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

The gxSocket class is an object-oriented Winsock/BSD wrapper 
used to create TCP/IP sockets on WIN32 and UNIX platforms. The 
gxSocket class supports stream sockets and datagram sockets and 
includes several low-level functions needed by derived classes to 
establish communication end-points and transfer data.
*/
// ----------------------------------------------------------- // 
#include<string.h>
#include <fcntl.h>
#include "gxsocket.h"

#if defined(__UNIX__) && !defined(__BSD_UNIX__)  
#include <sys/utsname.h> // Used for host name information
#endif // __UNIX__

// NOTE: This array must contain the same number of exceptions as the
// gxSocketError enumeration. 
const int gxsMaxSocketExceptionMessages = 29;
const char *gxsSocketExceptionMessages[gxsMaxSocketExceptionMessages] = {
  "Socket exception: No exception reported",             // NO_ERROR
  "Socket exception: Invalid exception code",            // INVALID_ERROR_CODE 
  "Socket exception: Error accepting remote socket",     // ACCEPT_ERROR
  "Socket exception: Could not bind socket",             // BIND_ERROR
  "Socket exception: Buffer overflow error",             // BUFOVER_ERROR
  "Socket exception: Could not connect socket",          // CONNECT_ERROR
  "Socket exception: A file system error occurred",      // FILESYSTEM_ERROR
  "Socket exception: Error getting socket option",       // GETOPTION_ERROR
  "Socket exception: Could not resolve hostname",        // HOSTNAME_ERROR
  "Socket exception: Initialization error",              // INIT_ERROR
  "Socket exception: Listen error",                      // LISTEN_ERROR
  "Socket exception: Get peer name error",               // PEER_ERROR
  "Socket exception: Unsupported protocol requested",    // PROTOCOL_ERROR
  "Socket exception: Receive error",                     // RECEIVE_ERROR
  "Socket exception: Request timed out",                 // REQUEST_TIMEOUT
  "Socket exception: Unsupported service requested",     // SERVICE_ERROR
  "Socket exception: Error setting socket option",       // SETOPTION_ERROR
  "Socket exception: Get socket name error",             // SOCKNAME_ERROR
  "Socket exception: Unsupported socket type requested", // SOCKETTYPE
  "Socket exception: Transmit error",                    // TRANSMIT_ERROR

  // WIN32 exception messages (Transparent to UNIX applications)
  "WinSock exception: WSACleanup operation failed",        // WINSOCKCLEANUP
  "WinSock exception: Version specified is not supported by this DLL",
                                                           // WINSOCKEINVAL
  "WinSock exception: Could not initialize Winsock",       // WINSOCKINIT
  "WinSock exception: Network subsystem is not ready",     // WINSOCKNETWORK
  "WinSock exception: Requested version in not supported", // WINSOCKVERSION
  
  // Variable block exception messages
  "Socket exception: Variable block acknowledgment error", // BLOCKACK
  "Socket exception: Bad database block header",           // BLOCKHEADER
  "Socket exception: Bad database block size",             // BLOCKSIZE
  "Socket exception: Variable block synchronization error" // BLOCKSYNC
};

gxSocket::gxSocket()
// Socket constructor that performs no initialization other then
// setting default values for the socket data members.
{
  address_family = AF_INET;      // Default address family
  socket_type = SOCK_STREAM;     // Default socket type
  protocol_family = IPPROTO_TCP; // Default protocol family
  port_number = gxSOCKET_DEFAULT_PORT; // Default port number
  gxsocket = -1;
  remote_socket = -1;
  bytes_read = bytes_moved = 0;
  is_connected = 0;
  is_bound = 0;
  socket_error = gxSOCKET_NO_ERROR;
  socket_version = gxSOCKET_WSAVER_ONEONE;
}

gxSocket::gxSocket(gxsAddressFamily af, gxsSocket_t st, gxsProtocolFamily pf,
		   int port, char *hostname)
// Socket constructor used to initialize the socket according to the
// address family, socket type, and protocol family. A hostname name should
// only be specified for client sockets.
{
  gxsocket = -1;
  remote_socket = -1;
  bytes_read = bytes_moved = 0;
  is_connected = 0;
  is_bound = 0;
  socket_error = gxSOCKET_NO_ERROR;
  socket_version = gxSOCKET_WSAVER_ONEONE;

  // Initialize the socket. NOTE: Any errors detected during initialization
  // will be recorded in the socket_error member.
  if(InitSocketLibrary() == 0) InitSocket(af, st, pf, port, hostname);
}

gxSocket::gxSocket(gxsSocket_t st, gxsPort_t port, char *hostname) 
// Socket constructor used to initialize the socket according to the
// socket type. A hostname name should only be specified for client
// sockets.
{
  gxsocket = -1;
  remote_socket = -1;
  bytes_read = bytes_moved = 0;
  is_connected = 0;
  is_bound = 0;
  socket_error = gxSOCKET_NO_ERROR;
  socket_version = gxSOCKET_WSAVER_ONEONE;

  // Initialize the socket. NOTE: Any errors detected during initialization
  // will be recorded in the socket_error member.
  if(InitSocketLibrary() == 0) InitSocket(st, port, hostname);
}

gxSocket::~gxSocket()
{
  Close();
}

gxsSocket_t gxSocket::Socket()
// Create a socket. Returns a valid socket descriptor or
// -1 if the socket cannot be initialized.
{
  gxsocket = socket(address_family, socket_type, protocol_family);
#if defined (__WIN32__)
  // The SOCKET type is unsigned in the WinSock library
  if(gxsocket == INVALID_SOCKET) // Defined as (SOCKET)(~0)
#elif defined (__UNIX__)
  if(gxsocket < 0)
#else
#error You must define a target platform:\n  __WIN32__ or __UNIX__
#endif
    {
      socket_error = gxSOCKET_INIT_ERROR;
      return -1;
    }

  return gxsocket;
}

gxsSocket_t gxSocket::InitSocket(gxsAddressFamily af,
				 gxsSocket_t st,
				 gxsProtocolFamily pf,
				 int port, char *hostname)
// Create and initialize a socket according to the address family,
// socket type, and protocol family. The "hostname" variable is an
// optional parameter that allows clients to specify a server name.
// Returns a valid socket descriptor or -1 if the socket cannot be
// initialized. NOTE: Under all UNIX platforms you must have root 
// privileges to use raw sockets. Windows 95/98/ME/NT only works 
// with the IPPROTO_ICMP protocol when using raw sockets and does
// not support the IPPROTO_RAW protocol.
{
#if defined (__WIN32__) && !defined (__WIN2000__)
  if((st == SOCK_RAW) && (pf != IPPROTO_ICMP)) {
    socket_error = gxSOCKET_SOCKETTYPE_ERROR;
    return -1;
  }
#endif

  address_family = af;
  socket_type = st;
  protocol_family = pf;
  port_number = port;

  // Put the server information into the server structure.
  sin.sin_family = address_family;

  if(hostname) {
    // Get the server's Internet address
    gxsHostNameInfo *hostnm = gethostbyname(hostname); 
    if(hostnm == (struct hostent *) 0) {
      socket_error = gxSOCKET_HOSTNAME_ERROR;
      return -1;
    }
    // Put the server information into the client structure.
    sin.sin_addr.s_addr = *((unsigned long *)hostnm->h_addr);
  }
  else   
    sin.sin_addr.s_addr = INADDR_ANY; // Use my IP address

  // The port must be put into network byte order.
  // htons()--"Host to Network Short" 
  // htonl()--"Host to Network Long" 
  // ntohs()--"Network to Host Short" 
  // ntohl()--"Network to Host Long" 
  sin.sin_port = htons(port_number);

  // Create a TCP/IP
  if(Socket() < 0) {
    socket_error = gxSOCKET_INIT_ERROR;
    return -1;
  }

  return gxsocket;
}

gxsSocket_t gxSocket::InitSocket(gxsSocket_t st, gxsPort_t port,
				 char *hostname)
// Create and initialize a socket according to the socket type. This 
// cross-platform fucntion will only accept SOCK_STREAM and SOCK_DGRAM
// socket types. The "hostname" variable is an optional parameter that 
// allows clients to specify a server name. Returns a valid socket 
// descriptor or -1 if the socket cannot be initialized. NOTE: Port 
// numbers are divided into three ranges: the Well Known Ports, the 
// Registered Ports, and the Dynamic and/or Private Ports. A complete 
// list of all assigned port numbers is maintained by IANA: 
//
// http://www.iana.org
//
// The Well Known Ports are those from 0 through 1023.
// The Registered Ports are those from 1024 through 49151
// The Dynamic and/or Private Ports are those from 49152 through 65535
{
  address_family = AF_INET;
  port_number = port; 

  if(st == SOCK_STREAM) {
    socket_type = SOCK_STREAM;
    protocol_family = IPPROTO_TCP;
  }
  else if(st == SOCK_DGRAM) {
    socket_type = SOCK_DGRAM;
    protocol_family = IPPROTO_UDP;
  }
  else {
    socket_error = gxSOCKET_SOCKETTYPE_ERROR;
    return -1;
  }

  // Put the server information into the server structure.
  sin.sin_family = address_family;

  if(hostname) {
    // Get the server's Internet address
    gxsHostNameInfo *hostnm = gethostbyname(hostname); 
    if(hostnm == (struct hostent *) 0) {
      socket_error = gxSOCKET_HOSTNAME_ERROR;
      return -1;
    }

    // Put the server information into the client structure.
    sin.sin_addr.s_addr = *((unsigned long *)hostnm->h_addr);
  }
  else   
    sin.sin_addr.s_addr = INADDR_ANY; // Use my IP address

  // The port must be put into network byte order.
  // htons()--"Host to Network Short" 
  // htonl()--"Host to Network Long" 
  // ntohs()--"Network to Host Short" 
  // ntohl()--"Network to Host Long" 
  sin.sin_port = htons(port_number);

  // Create a TCP/IP socket
  if(Socket() < 0) {
    socket_error = gxSOCKET_INIT_ERROR;
    return -1;
  }

  return gxsocket;
}

int gxSocket::Bind()
// Bind the socket to a name so that other processes can
// reference it and allow this socket to receive messages.
// Returns -1 if an error occurs.
{
  int rv = bind(gxsocket, (struct sockaddr *)&sin, sizeof(sin));
  if(rv >= 0) {
    is_bound = 1;
  } 
  else {
    socket_error = gxSOCKET_BIND_ERROR;
    is_bound = 0;
  }
  return rv;
}

int gxSocket::Connect()
// Connect the socket to a client or server. On the client side
// a connect call is used to initiate a connection.
// Returns -1 if an error occurs.
{
  int rv = connect(gxsocket, (struct sockaddr *)&sin, sizeof(sin));
  if(rv >= 0) {
    is_connected = 1; 
  }
  else {
    socket_error = gxSOCKET_CONNECT_ERROR;
    is_connected = 0;
  }
  return rv;
}

int gxSocket::ReadSelect(gxsSocket_t s, int seconds, int useconds)
// Function used to multiplex reads without polling. Returns false if a 
// reply time is longer then the timeout values. 
{
  struct timeval timeout;
  fd_set fds;
  FD_ZERO(&fds);
  FD_SET(s, &fds);

  timeout.tv_sec = seconds;
  timeout.tv_usec = useconds;

  // This function calls select() giving it the file descriptor of
  // the socket. The kernel reports back to this function when the file
  // descriptor has woken it up.
  return select(s+1, &fds, 0, 0, &timeout);
}

int gxSocket::Recv(void *buf, int bytes, int flags)
// Receive a block of data from the bound socket and do not return
// until all the bytes have been read. Returns the total number of 
// bytes received or -1 if an error occurs.
{
  return Recv(gxsocket, buf, bytes, flags);
}

int gxSocket::Recv(void *buf, int bytes, int seconds, int useconds, int flags)
// Receive a block of data from the bound socket and do not return
// until all the bytes have been read or the timeout value has been 
// exceeded. Returns the total number of bytes received or -1 if an 
// error occurs.
{
  return Recv(gxsocket, buf, bytes, seconds, useconds, flags);
}

int gxSocket::Send(const void *buf, int bytes, int flags)
// Send a block of data to the bound socket and do not return
// until all the bytes have been written. Returns the total number 
// of bytes sent or -1 if an error occurs.
{
  return Send(gxsocket, buf, bytes, flags);
}

int gxSocket::Recv(gxsSocket_t s, void *buf, int bytes, int flags)
// Receive a block of data from a specified socket and do not return
// until all the bytes have been read. Returns the total number of
// bytes received or -1 if an error occurs.
{
  bytes_read = 0;           // Reset the byte counter
  int num_read = 0;         // Actual number of bytes read
  int num_req = (int)bytes; // Number of bytes requested 
  char *p = (char *)buf;    // Pointer to the buffer

  while(bytes_read < bytes) { // Loop until the buffer is full
    if((num_read = recv(s, p, num_req-bytes_read, flags)) > 0) {
      bytes_read += num_read;   // Increment the byte counter
      p += num_read;            // Move the buffer pointer for the next read
    }
    if(num_read < 0) {
      socket_error = gxSOCKET_RECEIVE_ERROR;
      return -1; // An error occurred during the read
    }
  }
  
  return bytes_read;
}

int gxSocket::Recv(gxsSocket_t s, void *buf, int bytes, 
		   int seconds, int useconds, int flags)
// Receive a block of data from a specified socket and do not return
// until all the bytes have been read or the timeout value has been 
// exceeded. Returns the total number of bytes received or -1 if an 
// error occurs.
{
  bytes_read = 0;           // Reset the byte counter
  int num_read = 0;         // Actual number of bytes read
  int num_req = (int)bytes; // Number of bytes requested 
  char *p = (char *)buf;    // Pointer to the buffer

  while(bytes_read < bytes) { // Loop until the buffer is full
    if(!ReadSelect(s, seconds, useconds)) {
      socket_error = gxSOCKET_REQUEST_TIMEOUT;
      return -1; // Exceeded the timeout value
    }
    if((num_read = recv(s, p, num_req-bytes_read, flags)) > 0) {
      bytes_read += num_read;   // Increment the byte counter
      p += num_read;            // Move the buffer pointer for the next read
    }
    if(num_read < 0) {
      socket_error = gxSOCKET_RECEIVE_ERROR;
      return -1; // An error occurred during the read
    }
  }
  
  return bytes_read;
}

int gxSocket::Send(gxsSocket_t s, const void *buf, int bytes, int flags)
// Send a block of data to a specified socket and do not return
// until all the bytes have been written. Returns the total number of
// bytes sent or -1 if an error occurs.
{
  bytes_moved = 0;           // Reset the byte counter
  int num_moved = 0;         // Actual number of bytes written
  int num_req = (int)bytes;  // Number of bytes requested 
  char *p = (char *)buf;     // Pointer to the buffer

  while(bytes_moved < bytes) { // Loop until the buffer is empty
    if((num_moved = send(s, p, num_req-bytes_moved, flags)) > 0) {
      bytes_moved += num_moved;  // Increment the byte counter
      p += num_moved;            // Move the buffer pointer for the next read
    }
    if(num_moved < 0) {
      socket_error = gxSOCKET_TRANSMIT_ERROR;
      return -1; // An error occurred during the read
    }
  }

  return bytes_moved;
}

int gxSocket::RemoteRecv(void *buf, int bytes, int seconds, int useconds, 
			 int flags)
// Receive a block of data from a remote socket in blocking mode with a
// specified timeout value. Returns the total number of bytes received or 
// -1 if an error occurs.
{
  return Recv(remote_socket, buf, bytes, seconds, useconds, flags);
}

int gxSocket::RemoteRecv(void *buf, int bytes, int flags)
// Receive a block of data from a remote socket in blocking mode.
// Returns the total number of bytes received or -1 if an error occurs.
{
  return Recv(remote_socket, buf, bytes, flags);
}

int gxSocket::RemoteSend(const void *buf, int bytes, int flags)
// Send a block of data to a remote socket and do not return
// until all the bytes have been written. Returns the total 
// number of bytes received or -1 if an error occurs.
{
  return Send(remote_socket, buf, bytes, flags);
}

void gxSocket::ShutDown(int how)
// Used to close and un-initialize a full-duplex socket.
{
  bytes_moved = 0;
  bytes_read = 0;
  is_connected = 0;
  is_bound = 0;

  // NOTE 05/25/2000: The last error is no longer reset in following
  // a close call to allow the that appllication can close the socket
  // and catch the last exception reported if the socket has to be
  // closed after a Bind(), Listen(), or any other initialization
  // failure.
  // socket_error = gxSOCKET_NO_ERROR;
 
  // 0 = User is no longer interested in reading data
  // 1 = No more data will be sent
  // 2 = No data is to be sent or received
  if(gxsocket != -1) shutdown(gxsocket, how);
  if(remote_socket != -1) shutdown(remote_socket, how);

  gxsocket = -1;
  remote_socket = -1;
}

void gxSocket::ShutDown(gxsSocket_t &s, int how)
// Used to close and un-initialize the specified full-duplex socket.
{
  if(s != -1) shutdown(s, how);
  s = -1;
}

void gxSocket::ShutDownSocket(int how)
// Used to close a full-duplex server side socket.
{
  if(gxsocket != -1) shutdown(gxsocket, how);
  gxsocket = -1;
}

void gxSocket::ShutDownRemoteSocket(int how)
// Used to close a full-duplex client side socket.
{
  if(remote_socket != -1) shutdown(remote_socket, how);
  remote_socket = -1;
}

void gxSocket::Close()
// Close any and un-initialize any bound sockets.
{
  bytes_moved = 0;
  bytes_read = 0;
  is_connected = 0;
  is_bound = 0;

  // NOTE 05/25/2000: The last error is no longer reset in following
  // a close call to allow the that appllication can close the socket
  // and catch the last exception reported if the socket has to be
  // closed after a Bind(), Listen(), or any other initialization
  // failure.
  // socket_error = gxSOCKET_NO_ERROR;
#if defined (__WIN32__)
  if(gxsocket != -1) closesocket(gxsocket);
  if(remote_socket != -1) closesocket(remote_socket);
#elif defined (__UNIX__)
  if(gxsocket != -1) close(gxsocket);
  if(remote_socket != -1) close(remote_socket);
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
}

void gxSocket::Close(gxsSocket_t &s)
// Close the specified socket
{
#if defined (__WIN32__)
  if(s != -1) closesocket(s);
#elif defined (__UNIX__)
  if(s != -1) close(s);
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
  s = -1;
}

void gxSocket::CloseSocket()
// Close the server side socket
{
#if defined (__WIN32__)
  if(gxsocket != -1) closesocket(gxsocket);
#elif defined (__UNIX__)
  if(gxsocket != -1) close(gxsocket);
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
  gxsocket = -1;
}

void gxSocket::CloseRemoteSocket()
// Close the client socket
{
#if defined (__WIN32__)
  if(remote_socket != -1) closesocket(remote_socket);
#elif defined (__UNIX__)
  if(remote_socket != -1) close(remote_socket);
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
  remote_socket = -1;
}

int gxSocket::Listen(int max_connections)
// Listen for connections if configured as a server.
// The "max_connections" variable determines how many
// pending connections the queue will hold. Returns -1
// if an error occurs.
{
  int rv = listen(gxsocket,         // Bound socket
		  max_connections); // Number of connection request queue
  if(rv < 0) socket_error = gxSOCKET_LISTEN_ERROR;
  return rv;
}

gxsSocket_t gxSocket::Accept()
// Accept a connect from a remote socket. An Accept() 
// call blocks the server until the a client requests 
// service. Returns a valid socket descriptor or -1 
// if an error occurs.
{
  // Length of client address
  gxsSocketLength_t addr_size = (gxsSocketLength_t)sizeof(remote_sin); 

  remote_socket = accept(gxsocket, (struct sockaddr *)&remote_sin, &addr_size);

#if defined (__WIN32__)
  // The SOCKET type is unsigned in the WinSock library
  if(remote_socket == INVALID_SOCKET) // Defined as (SOCKET)(~0)
#elif defined (__UNIX__)
  if(remote_socket < 0)
#else
#error You must define a target platform: __WIN32__ or __UNIX__
#endif
    {
      socket_error = gxSOCKET_ACCEPT_ERROR;
      return -1;
    }

  return remote_socket;
}

int gxSocket::GetSockName(gxsSocket_t s, gxsSocketAddress *sa)
// Retrieves the current name for the specified socket descriptor.
// It is used on a bound and/or connected socket and returns the
// local association. This function is especially useful when a
// connect call has been made without doing a bind first in which
// case this function provides the only means by which you can
// determine the local association which has been set by the system.
// Returns -1 if an error occurs.
{
  gxsSocketLength_t namelen = (gxsSocketLength_t)sizeof(gxsSocketAddress);
  int rv = getsockname(s, (struct sockaddr *)sa, &namelen);
  if(rv < 0) socket_error = gxSOCKET_SOCKNAME_ERROR;
  return rv;
}

int gxSocket::GetSockName()
// Retrieves the current name for this objects socket descriptor.
// Returns -1 if an error occurs.
{
  return GetSockName(gxsocket, &sin);
}

int gxSocket::GetPeerName(gxsSocket_t s, gxsSocketAddress *sa)
// Retrieves the current name of the specified socket descriptor.
// Returns -1 if an error occurs.
{
  gxsSocketLength_t namelen = (gxsSocketLength_t)sizeof(gxsSocketAddress);
  int rv = getpeername(s, (struct sockaddr *)&sa, &namelen);
  if(rv < 0) socket_error = gxSOCKET_PEERNAME_ERROR;
  return rv;
}

int gxSocket::GetPeerName()
// Retrieves the current name for the remote socket descriptor.
// Returns -1 if an error occurs.
{
  return GetPeerName(remote_socket, &remote_sin);
}

int gxSocket::GetSockOpt(gxsSocket_t s, int level, int optName, 
			 void *optVal, unsigned *optLen)
// Gets the current socket option for the specified option level or name.
// Returns -1 if an error occurs.
{
#if defined (__WIN32__)
  int rv = getsockopt(s, level, optName, (char *)optVal, 
		      (gxsSocketLength_t *)optLen);
#else
  int rv = getsockopt(s, level, optName, optVal, (gxsSocketLength_t *)optLen);
#endif
  if(rv < 0) socket_error = gxSOCKET_SETOPTION_ERROR;
  return rv;
}

int gxSocket::GetSockOpt(int level, int optName, void *optVal, 
			 unsigned *optLen)
// Gets the current socket option for the specified option level or name.
// Returns -1 if an error occurs.
{
  return GetSockOpt(gxsocket, level, optName, optVal, optLen);
}

int gxSocket::SetSockOpt(gxsSocket_t s, int level, int optName,
			 const void *optVal, unsigned optLen)
// Sets the current socket option for the specified option level or name.
// Returns -1 if an error occurs.
{
#if defined (__WIN32__)
  int rv = setsockopt(s, level, optName, (const char *)optVal, 
		      (gxsSocketLength_t)optLen);
#else
  int rv = setsockopt(s, level, optName, optVal, (gxsSocketLength_t)optLen);
#endif
  if(rv < 0) socket_error = gxSOCKET_SETOPTION_ERROR;
  return rv;
}

int gxSocket::SetSockOpt(int level, int optName, const void *optVal, 
			 unsigned optLen)
// Sets the current socket option for the specified option level or name.
// Returns -1 if an error occurs.
{
  return SetSockOpt(gxsocket, level, optName, optVal, optLen);
}

gxsServent *gxSocket::GetServiceInformation(char *name, char *protocol)
// Function used to obtain service information about a specified name. 
// The source of this information is dependent on the calling function's
// platform configuration which should be a local services file or NIS 
// database. Returns a pointer to a gxsServent data structure if service
// information is available or a null value if the service cannot be
// found. NOTE: The calling function must free the memory allocated
// for gxsServent data structure upon each successful return.
{
  // If the "protocol" pointer is NULL, getservbyname returns
  // the first service entry for which the name matches the s_name
  // or one of the s_aliases. Otherwise getservbyname matches both
  // the name and the proto.
  gxsServent *sp = getservbyname(name, protocol);
  if(sp == 0) return 0;

  gxsServent *buf = new gxsServent;
  if(!buf) return 0; // Memory allocation error
  memmove(buf, sp, sizeof(gxsServent));
  return buf;
}

gxsServent *gxSocket::GetServiceInformation(int port, char *protocol)
// Function used to obtain service information about a specified port. 
// The source of this information is dependent on the calling function's
// platform configuration which should be a local services file or NIS 
// database. Returns a pointer to a gxsServent data structure if service
// information is available or a null value if the service cannot be
// found. NOTE: The calling function must free the memory allocated
// for gxsServent data structure upon each successful return.
{
  // If the "protocol" pointer is NULL, getservbyport returns the
  // first service entry for which the port matches the s_port.
  // Otherwise getservbyport matches both the port and the proto.
  gxsServent *sp = getservbyport(port, protocol);
  if(sp == 0) return 0;

  gxsServent *buf = new gxsServent;
  if(!buf) return 0; // Memory allocation error
  memmove(buf, sp, sizeof(gxsServent));
  return buf;
}

int gxSocket::GetServByName(char *name, char *protocol)
// Set service information corresponding to a service name and protocol.
// Returns -1 if an unknown service or protocol is requested. NOTE: This
// information is obtained from this machines local services file or
// from a NIS database.
{
  // If the "protocol" pointer is NULL, getservbyname returns
  // the first service entry for which the name matches the s_name
  // or one of the s_aliases. Otherwise getservbyname matches both
  // the name and the proto.
  gxsServent *sp = getservbyname(name, protocol);
  if(sp == 0) {
    socket_error = gxSOCKET_PROTOCOL_ERROR;
    return -1;
  }
  sin.sin_port = sp->s_port;
  return 0;
}

int gxSocket::GetServByPort(int port, char *protocol)
// Set service information corresponding to a port number and protocol.
// Returns -1 if an unknown service or protocol is requested. NOTE: This
// information is obtained from this machines local services file or
// from a NIS database.
{
  // If the "protocol" pointer is NULL, getservbyport returns the
  // first service entry for which the port matches the s_port.
  // Otherwise getservbyport matches both the port and the proto.
  gxsServent *sp = getservbyport(port, protocol);
  if(sp == 0) {
    socket_error = gxSOCKET_PROTOCOL_ERROR;
    return -1;
  }
  sin.sin_port = sp->s_port;
  return 0;
}

int gxSocket::GetPortNumber()
// Return the port number actually set by the system. Use this function
// after a call to gxSocket::GetSockName();
{
  return ntohs(sin.sin_port);
}

int gxSocket::GetRemotePortNumber()
// Return the port number of the client socket.
{
  return ntohs(remote_sin.sin_port);
}

gxsAddressFamily gxSocket::GetAddressFamily()
// Returns the address family of this socket
{
  return sin.sin_family;
}

gxsAddressFamily gxSocket::GetRemoteAddressFamily()
// Returns the address family of the remote socket.
{
  return remote_sin.sin_family;  
}

gxsHostNameInfo *gxSocket::GetHostInformation(char *hostname)
// Function used to obtain hostname information about a specified host. 
// The source of this information is dependent on the calling function's
// platform configuration which should be a DNS, local host table, and/or
// NIS database. Returns a pointer to a gxsHostNameInfo data structure
// if information is available or a null value if the hostname cannot be
// found. NOTE: The calling function must free the memory allocated
// for gxsHostNameInfo data structure upon each successful return.
{
  gxsInternetAddress hostia;
  gxsHostNameInfo *hostinfo;
  hostia.s_addr = inet_addr(hostname);

#if defined (__SOLARIS__)
#ifndef INADDR_NONE // IPv4 Internet address integer constant not defined
#define INADDR_NONE 0xffffffff
#endif
#endif

  if(hostia.s_addr == INADDR_NONE) { // Look up host by name
    hostinfo = gethostbyname(hostname); 
  }
  else {  // Look up host by IP address
    hostinfo = gethostbyaddr((const char *)&hostia, 
			     sizeof(gxsInternetAddress), AF_INET);
  }
  if(hostinfo == (gxsHostNameInfo *) 0) { // No host name info avialable
    return 0;
  }

  gxsHostNameInfo *buf = new gxsHostNameInfo;
  if(!buf) return 0; // Memory allocation error
  memmove(buf, hostinfo, sizeof(gxsHostNameInfo));
  return buf;
}
  
int gxSocket::GetHostName(char *sbuf)
// Pass back the host name of this machine in the "sbuf" variable.
// A memory buffer equal to "gxsMAX_NAME_LEN" must be pre-allocated
// prior to using this function. Return -1 if an error occurs.
{
  // Prevent crashes if memory has not been allocated
  if(!sbuf) sbuf = new char[gxsMAX_NAME_LEN]; 
  int rv = gethostname(sbuf, gxsMAX_NAME_LEN);
  if(rv < 0) socket_error = gxSOCKET_HOSTNAME_ERROR;
  return rv;
}

int gxSocket::GetIPAddress(char *sbuf)
// Pass back the IP Address of this machine in the "sbuf" variable.
// A memory buffer equal to "gxsMAX_NAME_LEN" must be pre-allocated
// prior to using this function. Return -1 if an error occurs.
{
  char hostname[gxsMAX_NAME_LEN];
  int rv = GetHostName(hostname);
  if(rv < 0) return rv;

  gxsInternetAddress *ialist;
  gxsHostNameInfo *hostinfo = GetHostInformation(hostname);
  if(!hostinfo) {
    socket_error = gxSOCKET_HOSTNAME_ERROR;
    return -1;
  }
  ialist = (gxsInternetAddress *)hostinfo->h_addr_list[0];

  // Prevent crashes if memory has not been allocated
  if(!sbuf) sbuf = new char[gxsMAX_NAME_LEN]; 

  strcpy(sbuf, inet_ntoa(*ialist));
  delete hostinfo;
  return 0;
}

int gxSocket::GetDomainName(char *sbuf)
// Pass back the domain name of this machine in the "sbuf" variable.
// A memory buffer equal to "gxsMAX_NAME_LEN" must be pre-allocated
// prior to using this function. Return -1 if an error occurs.
{
  char hostname[gxsMAX_NAME_LEN];
  int rv = GetHostName(hostname);
  if(rv < 0) return rv;

  gxsHostNameInfo *hostinfo = GetHostInformation(hostname);
  if(!hostinfo) {
    socket_error = gxSOCKET_HOSTNAME_ERROR;
    return -1;
  }
  // Prevent crashes if memory has not been allocated
  if(!sbuf) sbuf = new char[gxsMAX_NAME_LEN]; 

  strcpy(sbuf, hostinfo->h_name);
  int i; int len = strlen(sbuf);
  for(i = 0; i < len; i++) {
    if(sbuf[i] == '.') break;
  }
  if(++i < len) {
    len -= i;
    memmove(sbuf, sbuf+i, len);
    sbuf[len] = 0; // Null terminate the string
  }
  delete hostinfo;
  return 0;
}

int gxSocket::GetBoundIPAddress(char *sbuf)
// Pass back the local or server IP address in the "sbuf" variable.
// A memory buffer equal to "gxsMAX_NAME_LEN" must be pre-allocated
// prior to using this function. Return -1 if an error occurs.
{
  char *s = inet_ntoa(sin.sin_addr);
  if(s == 0) {
    socket_error = gxSOCKET_HOSTNAME_ERROR;
    return -1;
  }

  // Prevent crashes if memory has not been allocated
  if(!sbuf) sbuf = new char[gxsMAX_NAME_LEN]; 

  strcpy(sbuf, s);
  return 0;
}

int gxSocket::GetRemoteHostName(char *sbuf)
// Pass back the client host name client in the "sbuf" variable.
// A memory buffer equal to "gxsMAX_NAME_LEN" must be pre-allocated
// prior to using this function. Return -1 if an error occurs.
{
  char *s = inet_ntoa(remote_sin.sin_addr);
  if(s == 0) {
    socket_error = gxSOCKET_HOSTNAME_ERROR;
    return -1;
  }

  // Prevent crashes if memory has not been allocated
  if(!sbuf) sbuf = new char[gxsMAX_NAME_LEN]; 

  strcpy(sbuf, s);
  return 0;
}

int gxSocket::RawRead(void *buf, int bytes, int flags)
// Receive a block of data from the bound socket. NOTE: A
// raw read may return before all the bytes have been read.
// Returns -1 if an error occurs.
{
  return RawRead(gxsocket, buf, bytes, flags);
}

int gxSocket::RawWrite(const void *buf, int bytes, int flags)
// Send a block of data to the bound socket. NOTE: A raw write
// may return before all the bytes have been written. Returns -1
// if an error occurs.
{
  return RawWrite(gxsocket, buf, bytes, flags);
}

int gxSocket::RawRead(gxsSocket_t s, void *buf, int bytes, int flags)
// Receive a block of data from a specified socket. NOTE: A
// raw read may return before all the bytes have been read.
// Returns -1 if an error occurs.
{
  bytes_read = 0;
  bytes_read = recv(s, (char *)buf, bytes, flags);
  if(bytes_read < 0) socket_error = gxSOCKET_RECEIVE_ERROR;
  return bytes_read;
}

int gxSocket::RawWrite(gxsSocket_t s, const void *buf, int bytes,
		       int flags)
// Send a block of data to a specified socket. NOTE: A raw write
// may return before all the bytes have been written. Returns -1 if an
// error occurs.
{
  bytes_moved = 0;
  bytes_moved = send(s, (char *)buf, bytes, flags);
  if(bytes_moved < 0) socket_error = gxSOCKET_TRANSMIT_ERROR;
  return bytes_moved;
}

int gxSocket::RawRemoteRead(void *buf, int bytes, int flags)
// Receive a block of data from a remote socket.  NOTE: A
// raw read may return before all the bytes have been read.
// Returns -1 if an error occurs.
{
  return RawRead(remote_socket, buf, bytes, flags);
}

int gxSocket::RawRemoteWrite(const void *buf, int bytes, int flags)
// Send a block of data to a remote socket. NOTE: A raw write
// may return before all the bytes have been written. Returns
// -1 if an error occurs.
{
  return RawWrite(remote_socket, buf, bytes, flags);
}

int gxSocket::RawReadFrom(void *buf, int bytes, int flags)
// Receive a block of data from a remote datagram socket. NOTE: 
// A raw read may return before all the bytes have been read.
// Returns -1 if an error occurs.
{
  return RawReadFrom(gxsocket, &remote_sin, buf, bytes, flags);
}

int gxSocket::RawWriteTo(void *buf, int bytes, int flags)
// Send a block of data to a datagram socket. NOTE: A raw write
// may return before all the bytes have been written. Returns -1
// if an error occurs.
{
  return RawWriteTo(gxsocket, &sin, buf, bytes, flags);
}

const char *gxSocket::SocketExceptionMessage()
// Returns a null terminated string that can
// be use to log or print a socket exception.
{
  if((int)socket_error > gxsMaxSocketExceptionMessages)
    socket_error = gxSOCKET_INVALID_ERROR_CODE;
  
  // Find the corresponding message in the exception array
  int error = (int)socket_error;
  return gxsSocketExceptionMessages[error];
}

void gxSocket::GetClientInfo(char *client_name, int &r_port)
// Get the client's host name and port number. NOTE: This
// function assumes that a block of memory equal to the
// gxsMAX_NAME_LEN constant has already been allocated.
{
  int rv = GetRemoteHostName(client_name);
  if(rv < 0) {
    char *unc = "UNKNOWN";
    for(unsigned i = 0; i < gxsMAX_NAME_LEN; i++) client_name[i] = '\0';
    strcpy(client_name, unc);
  }
  r_port = GetRemotePortNumber();
}

int gxSocket::RecvFrom(void *buf, int bytes, int seconds, int useconds, 
		       int flags)
// Receive a block of data from a remote datagram socket 
// and do not return until all the bytes have been read 
// or the timeout value has been exceeded. Returns the total 
// number of bytes received or -1 if an error occurs.
{
  return RecvFrom(gxsocket, &remote_sin, buf, bytes, seconds, useconds, flags);
}

int gxSocket::RecvFrom(void *buf, int bytes, int flags)
// Receive a block of data from a remote datagram socket 
// and do not return until all the bytes have been read. 
// Returns the total number of bytes received or -1 if 
// an error occurs.
{
  return RecvFrom(gxsocket, &remote_sin, buf, bytes, flags);
}

int gxSocket::SendTo(void *buf, int bytes, int flags)
// Send a block of data to a datagram socket and do not return
// until all the bytes have been written. Returns the total number
// of bytes sent or -1 if an error occurs.
{
  return SendTo(gxsocket, &sin, buf, bytes, flags);
}

int gxSocket::RecvFrom(gxsSocket_t s, gxsSocketAddress *sa, void *buf,
		       int bytes, int seconds, int useconds, int flags)
// Receive a block of data from a remote datagram socket 
// and do not return until all the bytes have been read 
// or the timeout value has been exceeded. Returns the total 
// number of bytes received or -1 if an error occurs.
{
  // Length of client address
  gxsSocketLength_t addr_size = (gxsSocketLength_t)sizeof(gxsSocketAddress); 
  bytes_read = 0;           // Reset the byte counter
  int num_read = 0;         // Actual number of bytes read
  int num_req = (int)bytes; // Number of bytes requested 
  char *p = (char *)buf;    // Pointer to the buffer

  while(bytes_read < bytes) { // Loop until the buffer is full
    if(!ReadSelect(s, seconds, useconds)) {
      socket_error = gxSOCKET_REQUEST_TIMEOUT;
      return -1; // Exceeded the timeout value
    }
    if((num_read = recvfrom(s, p, num_req-bytes_read, flags, \
			    (struct sockaddr *)sa, &addr_size)) > 0) {
      bytes_read += num_read;   // Increment the byte counter
      p += num_read;            // Move the buffer pointer for the next read
    }
    if(num_read < 0) {
      socket_error = gxSOCKET_RECEIVE_ERROR;
      return -1; // An error occurred during the read
    }
  }
  return bytes_read;
}

int gxSocket::RecvFrom(gxsSocket_t s, gxsSocketAddress *sa, void *buf,
		       int bytes, int flags)
// Receive a block of data from a remote datagram socket 
// and do not return until all the bytes have been read. 
// Returns the total number of bytes received or -1 if 
// an error occurs.
{
  // Length of client address
  gxsSocketLength_t addr_size = (gxsSocketLength_t)sizeof(gxsSocketAddress); 
  bytes_read = 0;           // Reset the byte counter
  int num_read = 0;         // Actual number of bytes read
  int num_req = (int)bytes; // Number of bytes requested 
  char *p = (char *)buf;    // Pointer to the buffer

  while(bytes_read < bytes) { // Loop until the buffer is full
    if((num_read = recvfrom(s, p, num_req-bytes_read, flags, \
			    (struct sockaddr *)sa, &addr_size)) > 0) {
      bytes_read += num_read;   // Increment the byte counter
      p += num_read;            // Move the buffer pointer for the next read
    }
    if(num_read < 0) {
      socket_error = gxSOCKET_RECEIVE_ERROR;
      return -1; // An error occurred during the read
    }
  }
  return bytes_read;
}

int gxSocket::SendTo(gxsSocket_t s, gxsSocketAddress *sa, void *buf,
		     int bytes, int flags)
// Send a block of data to a datagram socket and do not return
// until all the bytes have been written. Returns the total number
// of bytes sent or -1 if an error occurs.
{
  // Length of address
  gxsSocketLength_t addr_size = (gxsSocketLength_t)sizeof(gxsSocketAddress);
  bytes_moved = 0;             // Reset the byte counter
  int num_moved = 0;           // Actual number of bytes written
  int num_req = (int)bytes;    // Number of bytes requested 
  char *p = (char *)buf;       // Pointer to the buffer

  while(bytes_moved < bytes) { // Loop until the buffer is full
    if((num_moved = sendto(s, p, num_req-bytes_moved, flags, \
			   (const struct sockaddr *)sa, addr_size)) > 0) {
      bytes_moved += num_moved;  // Increment the byte counter
      p += num_moved;            // Move the buffer pointer for the next read
    }
    if(num_moved < 0) {
      socket_error = gxSOCKET_TRANSMIT_ERROR;
      return -1; // An error occurred during the read
    }
  }
  return bytes_moved;
}

int gxSocket::RawReadFrom(gxsSocket_t s, gxsSocketAddress *sa, void *buf,
			  int bytes, int flags)
// Receive a block of data from a remote datagram socket. NOTE: 
// A raw read may return before all the bytes have been read.
// Returns -1 if an error occurs.
{
  // Length of client address
  gxsSocketLength_t addr_size = (gxsSocketLength_t)sizeof(gxsSocketAddress);
  bytes_read = 0;
  bytes_read = recvfrom(s, (char *)buf, bytes, flags,
			(struct sockaddr *)sa, &addr_size);
  if(bytes_read < 0) socket_error = gxSOCKET_RECEIVE_ERROR;
  return bytes_read;
}

int gxSocket::RawWriteTo(gxsSocket_t s, gxsSocketAddress *sa, void *buf,
			 int bytes, int flags)
// Send a block of data to a datagram socket. NOTE: A raw write
// may return before all the bytes have been written. Returns -1
// if an error occurs.
{
  // Length of address
  gxsSocketLength_t addr_size = (gxsSocketLength_t)sizeof(gxsSocketAddress);
  bytes_moved = 0;
  bytes_moved = sendto(s, (char *)buf, bytes, flags,
		       (const struct sockaddr *)sa, addr_size);
  if(bytes_moved < 0) socket_error = gxSOCKET_TRANSMIT_ERROR;
  return bytes_moved;
}

int gxSocket::InitSocketLibrary()
// Perform any platform specific initialization required
// before network communication can be established.
// Returns zero if no errors occur, -1 if an error occurs
// and the error can be mapped internally, or a value greater
// then zero if an error occurs and the error cannot be
// determined.
{
#if defined (__WIN32__)
  // Initialize the WinSock DLL with the specified version.
  WORD wVersionRequested;
  switch(socket_version) { 
    // NOTE: An application can successfully use a Windows Sockets
    // DLL if there is any overlap in the version ranges. 
    case gxSOCKET_WSAVER_ONEZERO:
      // Included for test purposes only
      wVersionRequested = MAKEWORD(1, 0);
      break;
    case gxSOCKET_WSAVER_ONEONE:
      // Windows 95A/B/C, 98, 98SE, NT4.0 compatiable
      wVersionRequested = MAKEWORD(1, 1);
      break;
    case gxSOCKET_WSAVER_TWOZERO:
      // Windows 98, 98SE, 2000 compatiable
      wVersionRequested = MAKEWORD(2, 0);
      break;
    case gxSOCKET_WSAVER_TWOTWO:
      // Windows 98SE, 2000 compatiable
      wVersionRequested = MAKEWORD(2, 2);
      break;
    default:
      // Should work on all WIN32 platforms
      wVersionRequested = MAKEWORD(1, 1);
      break;
  }

  // Initialize the WinSock DLL. Every Windows Sockets application
  // MUST make a WSAStartup call before issuing any other Windows
  // Sockets API calls. 
  int rv = WSAStartup(wVersionRequested, &socket_data);
  if(rv != 0) {
    switch(rv) {
      // NOTE: Application calls to WSAGetLastError to determine the WinSock
      // error code cannot be here used since the Windows Sockets DLL may
      // not have established the client data area where the "last error"
      // information is stored. 
      case WSASYSNOTREADY:
	// Network subsystem is not ready for network communication
	socket_error = gxSOCKET_WINSOCKNETWORK_ERROR;
	return -1;
      case WSAVERNOTSUPPORTED:
	// The requested WinSock version is not supported
	socket_error = gxSOCKET_WINSOCKVERSION_ERROR;
	return -1;
      case WSAEINVAL:
	// The WinSock version specified is not supported by this DLL	
	socket_error = gxSOCKET_WINSOCKEINVAL_ERROR;
	return -1;
      default:
	socket_error = gxSOCKET_WINSOCKINIT_ERROR;
	return rv; // Can't map this error so return the WIN32 error code
    }
  }

  // Confirm that the WinSock DLL supports the requested version
  if(socket_data.wVersion != wVersionRequested) return -1;
  switch(socket_version) { 
    case gxSOCKET_WSAVER_ONEZERO:
      if(LOBYTE(socket_data.wVersion ) != 1 ||
	 HIBYTE(socket_data.wVersion ) != 0) {
	socket_error = gxSOCKET_WINSOCKVERSION_ERROR;
	return -1; 
      }
      break;
    case gxSOCKET_WSAVER_ONEONE:
      if(LOBYTE(socket_data.wVersion ) != 1 ||
	 HIBYTE(socket_data.wVersion ) != 1) {
	socket_error = gxSOCKET_WINSOCKVERSION_ERROR;
	return -1; 
      }
      break;
    case gxSOCKET_WSAVER_TWOZERO:
      if(LOBYTE(socket_data.wVersion ) != 2 ||
	 HIBYTE(socket_data.wVersion ) != 0) {
	socket_error = gxSOCKET_WINSOCKVERSION_ERROR;
	return -1; 
      }
      break;
    case gxSOCKET_WSAVER_TWOTWO:
      if(LOBYTE(socket_data.wVersion ) != 2 ||
	 HIBYTE(socket_data.wVersion ) != 2) {
	socket_error = gxSOCKET_WINSOCKVERSION_ERROR;
	return -1; 
      }
      break;
    default:
      socket_error = gxSOCKET_WINSOCKVERSION_ERROR;
      return -1; 
  }
#endif

  return 0;
}

int gxSocket::ReleaseSocketLibrary()
// Perform any platform specific operations required
// to release the socket library and free any resources
// associated with it. Returns -1 if any errors occur.
{
#if defined (__WIN32__)
  // All WIN32 applications or DLLs are required to perform a successful
  // WSAStartup() call before they can use Windows Sockets services. When
  // finished the application or DLL must call WSACleanup() to deregister
  // itself from a Windows Sockets implementation. Any pending blocking or
  // asynchronous calls issued by any thread in this process are canceled
  // without posting any notification messages, or signaling any event
  // objects. NOTE: There must be a call to WSACleanup() for every successful
  // call to WSAStartup() made by a task. Only the final WSACleanup() for that
  // task does the actual cleanup. The preceding calls decrement an internal
  // reference count in the Windows Sockets DLL. In multithreaded environments
  // WSACleanup() terminates Windows Sockets operations for all threads.

  // Will return zero if the operation was successful. Otherwise the value
  // SOCKET_ERROR is returned, and a specific error number may be retrieved
  // by calling int WSAGetLastError(void);
  if(WSACleanup() != 0) {
    socket_error = gxSOCKET_WINSOCKCLEANUP_ERROR;
    return -1;
  }
#endif
  return 0;
}
// ----------------------------------------------------------- // 
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //

