// ------------------------------- //
// -------- Start of File -------- //
// ------------------------------- //
// ----------------------------------------------------------- // 
// C++ Source Code File Name: db_debug.cpp 
// Compiler Used: MSVC, BCC32, GCC, HPUX aCC, SOLARIS CC
// Produced By: glNET Software
// File Creation Date: 02/04/1997  
// Date Last Modified: 06/12/2001
// Copyright (c) 2001 glNET Software
// ----------------------------------------------------------- // 
// ------------- Program Description and Details ------------- // 
// ----------------------------------------------------------- // 
/*
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.
 
This library is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  
USA

gxDatabase file recovery functions.
*/
// ----------------------------------------------------------- // 
#include <iostream.h>
#include <iomanip.h>
#include <string.h>
#include "gxdbase.h"
#include "gxdstats.h"

// Program name and version number
char *VersionNumber;

#if defined (__64_BIT_DATABASE_ENGINE__)
const char *ProgramName = "db_dbg64";
#else // Use the 32-bit version by default
const char *ProgramName = "db_debug";
#endif

class gxDatabaseDebugManager : public gxDatabase
{
public:
  gxDatabaseDebugManager() { }
  ~gxDatabaseDebugManager() { }

public:
  gxDatabaseError BlindOpen(const char *fname);
  FAU BlockSearch(FAU Offset); 
};

// Function prototypes for display menu
void ClearInputStream(istream &s);
int Quit();
void Menu(gxDatabaseDebugManager *f);
void PausePrg();
void Version();

// Function prototypes for database utilities 
void FindBlock(gxDatabaseDebugManager *f, int verbose = 0, int walk = 0);
void AnalyzeHeader(gxDatabaseDebugManager *f);
void DisplayStats(gxDatabaseDebugManager *f);
void Rebuild(gxDatabaseDebugManager *f, int update_ver = 0);

gxDatabaseError gxDatabaseDebugManager::BlindOpen(const char *fname)
// Open the file in read only mode without throwing a wrong file type
// exception.
{
  // Close any open files
  if(Close() != gxDBASE_NO_ERROR) return gxd_error;
  
  fp = gxdFPTROpen(fname, gxDBASE_READONLY);
    
  if(fp == 0) {
    gxd_error = gxDBASE_FILE_OPEN_ERROR;
#ifdef __CPP_EXCEPTIONS__
    throw CFileOpenError();
#else
    return gxd_error;
#endif
  }
  else {
    ready_for_reading = 1;
    ready_for_writing = 0;
    is_open = 1;
    is_ok = 1;
    strcpy(file_name, fname);
    last_operation = gxDBASE_WRITE;
    
    if(ReadFileHdr()!= gxDBASE_NO_ERROR) return gxd_error;

    // Version 1, 2, and 3 compatibility mode
    if((int)file_header.gxd_ver < 2000) {
      gxDatabase::gxVersion = file_header.gxd_ver;
      gxDatabase::gxInternalCheckWord = 0x0000fefe;
      memmove(gxDatabase::gxSignature, file_header.gxd_sig, 
	      (gxSignatureSize-1)); 
      cout << "Detected version " << (int)file_header.gxd_ver << endl;
      cout << "Signature "; 
      cout.write(file_header.gxd_sig, (gxSignatureSize-1));
      cout << endl;
    }
    else if((int)file_header.gxd_ver < 4000) {
      gxDatabase::gxVersion = file_header.gxd_ver;
      gxDatabase::gxInternalCheckWord = 0xfefefefe;
      memmove(gxDatabase::gxSignature, file_header.gxd_sig, 
	      (gxSignatureSize-1)); 
      cout << "Detected version " << (int)file_header.gxd_ver << endl;
      cout << "Signature "; 
      cout.write(file_header.gxd_sig, (gxSignatureSize-1));
      cout << endl;
    }
    
    // Test file type, checking the revision letter
    if(memcmp(file_header.gxd_sig, gxDatabase::gxSignature,
	      (gxSignatureSize-1)) == 0) {

      // Set the revision letter according to the file header
      char revision[gxSignatureSize];
      memmove(revision, file_header.gxd_sig, gxSignatureSize);
      rev_letter = revision[gxSignatureSize-1];
    }
  }

  // Ensure that true end of file is stored in the file header. 
  FAU filesize;
  filesize = FileSize(fname);
  if(file_header.gxd_eof < filesize) {
    file_header.gxd_eof = filesize;
    if(Flush() != gxDBASE_NO_ERROR) return gxd_error;
  }
  
  return gxd_error = gxDBASE_NO_ERROR;
}

FAU gxDatabaseDebugManager::BlockSearch(FAU Offset)
// Search through the database file until a valid block is found.
// The search starts at the beginning  of the file or the
// offset value. Returns 0 if no valid block is found in the
// file.
{
  gxBlockHeader blk;
  FAU file_address = (FAU)0;

  FAU StaticEOF = FileSize(file_name);

  if(!Offset) // If no Offset, start after database header
    file_address = FileHeaderSize(); 
  else {
    file_address = file_address + Offset; // Offset the starting address 

    if(file_address >= StaticEOF) // Prevent offsetting past EOF
      return 0; // Invalid address
  }
  
  while(1) {
    if((FAU)(file_address + BlockHeaderSize()) >= StaticEOF || !IsOK()) 
      return (FAU)0;

    if(Read(&blk, sizeof(gxBlockHeader), file_address) != gxDBASE_NO_ERROR)
      return (FAU)0;
    if(blk.block_check_word != gxDatabase::gxInternalCheckWord)
      file_address++; // Loop through the file byte by byte
    else
      break; // Found valid block
  }
  return file_address;
}

void DisplayStats(gxDatabaseDebugManager *f)
{
  gxFileHeader fh;

  f->Read(&fh, sizeof(gxFileHeader), (FAU)0);
  if(CheckError((gxDatabase *)f) != 0) return;
  
  if(memcmp(fh.gxd_sig, gxDatabase::gxSignature, (gxSignatureSize-1))) {
    // Test file type
    cout << endl;
    cout << "Database file header is damaged or does not exist" << endl;
    cout << endl;
    return;
  }

  DatabaseStats((gxDatabase *)f);
}

void AnalyzeHeader(gxDatabaseDebugManager *f)
{
  gxFileHeader fh;
  gxBlockHeader blk;
  int errors = 0;
  
  f->Read(&fh, sizeof(gxFileHeader), (FAU)0);
  if(CheckError((gxDatabase *)f) != 0) return;
  cout << endl;
  cout << "Analyzing database file header..." << endl;
  cout << endl;

  cout << "Checking database signature..." << endl;
  if (memcmp(fh.gxd_sig, gxDatabase::gxSignature, (gxSignatureSize-1))) {
    // Test file type
    cout << endl;
    cout << "Database file header is damaged or does not exist" <<endl;
    cout << endl;
    FindBlock(f, 0);
    return;
  }
  else if (memcmp(fh.gxd_sig, gxDatabase::gxSignature, gxSignatureSize)) {
    // Check revision letters
    char revision[gxSignatureSize];
    revision[gxSignatureSize] = 0; // Ensure null termination
    char rev_letter;
    cout << endl;
    cout << "NOTE: The database file revision letters do not match." << endl;
    memmove(revision, gxDatabase::gxSignature, gxSignatureSize);
    rev_letter = revision[gxSignatureSize-1];
    if(rev_letter == 0)
      cout << "Current database file revision = ZERO" << endl;
    else
      cout << "Current database file revision = " << rev_letter << endl;
    memmove(revision, fh.gxd_sig, gxSignatureSize);
    rev_letter = revision[gxSignatureSize-1];
    if(rev_letter == 0)
      cout << "Current database file revision = ZERO" << endl;
    else
      cout << "The file reads: " << rev_letter << endl;
    cout << "Backward compatibility rules will be enforced." << endl;
    cout << endl;
  }

  cout << "Checking for End of File errors..." << endl;
  FAU StaticEOF = f->FileSize(f->DatabaseName());
  if(StaticEOF > fh.gxd_eof) {
    cout << endl;
    cout << "End of file error in file: " << f->DatabaseName() << endl;
    cout << "The actual length is longer then the allocated length!" << endl;
    cout << endl;
    errors++;
  }

  cout << "Checking the heap start value..." << endl;
  f->Read(&blk, sizeof(gxBlockHeader), fh.gxd_hs_fptr);
  if(blk.block_check_word != gxDatabase::gxInternalCheckWord) {
    cout << endl;
    cout << "Bad heap start value in file: " << f->DatabaseName() << endl;
    cout << "No database block found at file address: "
	 << fh.gxd_hs_fptr << endl;
    cout << endl;
    errors++;
  }

  cout << "Checking the free space value..." << endl;
  if(fh.gxd_fs_fptr != (FAU)0) {
    f->Read(&blk, sizeof(gxBlockHeader), fh.gxd_fs_fptr);
    if(blk.block_check_word != gxDatabase::gxInternalCheckWord) {
      cout << endl;
      cout << "Bad free space value in file: " << f->DatabaseName() << endl;
      cout << "No database block found at file address: "
	   << fh.gxd_fs_fptr << endl;
      cout << endl;
      errors++;
    }
  }

  cout << "Checking the highest block value..." << endl;
  f->Read(&blk, sizeof(gxBlockHeader), fh.gxd_hb_fptr);
  if (blk.block_check_word != gxDatabase::gxInternalCheckWord) {
    cout << endl;
    cout << "Bad heap start value in file: " << f->DatabaseName() << endl;
    cout << "No database block found at file address: "
	 << fh.gxd_hb_fptr << endl;
    cout << endl;
    errors++;
  }
  
  cout << "Checking the database version number..." << endl;
  if(fh.gxd_ver != (FAU)gxDatabase::gxVersion) {
    cout << endl;
    cout << "NOTE: The version numbers do not match." << endl;
    cout << "Current database File Version = " << gxDatabase::gxVersion 
	 << endl;
    cout << "The file reads: " << fh.gxd_ver << endl;
    cout << "Backward compatibility rules will be enforced." << endl;
    cout << endl;
  }

  if(errors) {
    cout << endl;
    cout << "Database file header has errors!" << endl;
  }
  else {
    cout << endl;
    cout << "Database file header checks good." << endl;
  }
  cout << endl;
}

void FindBlock(gxDatabaseDebugManager *f, int verbose, int walk)
// Serach the file for all database block.
{
  gxBlockHeader blk;
  int count = 0;
  int badgx = 0;
  
  cout << endl;
  cout << "Searching file for all database blocks..." << endl;
  
  FAU StaticEOF = f->FileSize(f->DatabaseName());
  FAU addr = f->BlockSearch(0); // Search the entire file
  
  if(addr == (FAU)0) {
    cout << endl;
    cout << "No database blocks found in file: "
	 << f->DatabaseName() << endl;
    cout << endl;
    return;
  }

  while(1) { 
    if(addr >= StaticEOF) break;
    f->Read(&blk, sizeof(gxBlockHeader), addr);
    if(CheckError((gxDatabase *)f) != 0) return;
    
    if (blk.block_check_word == gxDatabase::gxInternalCheckWord) {
      count++; // Increment the database block count
      if(verbose) {
	BlockStats((gxDatabase *)f, (addr+f->BlockHeaderSize()));
	if(walk) {
	  char c;
	  cout << endl;
	  cout << "Press enter to continue or enter 'X' to exit >";
	  cin.clear();
	  cin.get(c);
	  switch(c) {
	    case 'x' : case 'X' :
	      cout << endl;
	      return;
	    default:
	      break;
	  }
	}
      }
      addr = addr + blk.block_length; // Goto the next database block
    }
    else {
      badgx++;
      cout << endl;
      cout << "Found bad database block at address " << addr << endl;
      cout << "Searching for next good database block..." << endl;
      addr = f->BlockSearch(addr); // Search for the next good block
      if(!addr) {
	cout << "None found!" << endl;
	cout << endl;
	return;
      }
    }
  }
  cout << endl;
  cout << "Found " << count << " good database blocks." << endl;
  if(badgx) cout << "Found " << badgx << " bad database blocks." << endl;
  cout << endl;
}

void Rebuild(gxDatabaseDebugManager *f, int update_ver)
{
  cout << endl;
  if(update_ver) 
    cout << "Rebuilding database file and updating to latest version..." 
	 << endl;
  else
    cout << "Rebuilding damaged database file..." << endl;
  cout << endl;

  char buf[255];
  int retry = 3;
  
  while(1) { // Loop until a good file name is found
    cout << "Enter new name of file to build >";
    cin.getline(buf, sizeof(buf));
    cout << endl;
    if(!*buf) { // Return if nothing is entered
      Menu(f);
      return;
    }
    if(gxDatabase::Exists(buf)) {
      cout << "File already exists!" << endl << endl;
       retry--;
      if(!retry) {
	Menu(f);
	return;
      }
    }
    else
      break;
  }

  gxFileHeader fh;
  gxBlockHeader blk;
  int errors = 0;
  
  // Analyze the database file header to determine if the file has
  // a pre-allocated static area
  FAU static_area;
  f->Read(&fh, sizeof(gxFileHeader), (FAU)0);
  if(CheckError((gxDatabase *)f) != 0) return;
  char rev_letter = f->GetRevLetter();

  
  // Check the database file header's signature. 06/08/2000: Modified
  // to ignore the revision letter. If the header is damaged the
  // current database revision will be used.
  if (memcmp(fh.gxd_sig, gxDatabase::gxSignature, (gxSignatureSize-1))) {
    rev_letter = gxDatabaseRevisionLetter;
    errors++; // Header is damaged and cannot be read
  }
  
  if(!errors) { // Check the heap start value 
    f->Read(&blk, sizeof(gxBlockHeader), fh.gxd_hs_fptr);
    if(CheckError((gxDatabase *)f) != 0) return;
    if (blk.block_check_word != gxDatabase::gxInternalCheckWord)
      errors++;
  }

  // If no errors, calculate the the size of the static area.
  if(!errors) static_area = fh.gxd_hs_fptr - (FAU_t)f->FileHeaderSize();

  FAU StaticEOF = f->FileSize(f->DatabaseName());
  FAU addr = f->BlockSearch(0); // Search the entire file
  
  if(addr == (FAU)0) {
    cout << endl;
    cout << "No database blocks found in file: "
	 << f->DatabaseName() << endl;
    PausePrg();
    return;
  }

  // Create the new file
  gxDatabaseDebugManager *NewFile = new gxDatabaseDebugManager;
  if(!NewFile) {
    cout << "Memory allocation error!" << endl;
  }
  
  // Get the previous state of all the version specific information
  FAU_t prev_ver = gxDatabase::gxVersion;
  char prev_sig[gxSignatureSize];
  memmove(prev_sig, gxDatabase::gxSignature, gxSignatureSize);
  __LWORD__ prev_check_word = gxDatabase::gxInternalCheckWord;

  if(update_ver) {
    gxDatabase::gxVersion = gxDatabaseVersionNumber;

#if defined (__64_BIT_DATABASE_ENGINE__)
    memmove(gxDatabase::gxSignature, "GXDBASE64", gxSignatureSize);
#else // Use the 32-bit version by default
    memmove(gxDatabase::gxSignature, "GXDBASE", gxSignatureSize);
#endif
    
    gxDatabase::gxInternalCheckWord = gxCheckWord;
    rev_letter = gxDatabaseRevisionLetter;
    NewFile->Create(buf, static_area, rev_letter);
    if(CheckError((gxDatabase *)NewFile) != 0) {
      delete NewFile;
      return;
    }
    gxDatabase::gxVersion = prev_ver;
    memmove(gxDatabase::gxSignature, prev_sig, gxSignatureSize);
    gxDatabase::gxInternalCheckWord = prev_check_word;
  }
  else {
    NewFile->Create(buf, static_area, rev_letter);
    if(CheckError((gxDatabase *)NewFile) != 0) {
      delete NewFile;
      return;
    }
  }
  
  int count = 0;
  int badgx = 0;

  char *v;

  if(static_area) { // Write the static area data
    v = new char[(__ULWORD__)static_area];
    if(!v) {
      cout << "Memory allocation error!" << endl;
      delete NewFile;
      return;
    }
    f->Read(v, (__ULWORD__)static_area, (FAU_t)f->FileHeaderSize());
    if(CheckError((gxDatabase *)f) != 0) {
      delete NewFile;
      return;
    }

    NewFile->Write(v, (__ULWORD__)static_area, (FAU_t)f->FileHeaderSize());
    if(CheckError((gxDatabase *)NewFile) != 0) {
      delete NewFile;
      return;
    }
    delete v;
  }
  
  FAU ObjectLength;
  gxINT32 checksum;
    
  while(1) { 
    if(addr >= StaticEOF) break;

    f->Read(&blk, sizeof(gxBlockHeader), addr);
    if(CheckError((gxDatabase *)f) != 0) {
      delete NewFile;
      return;
    }
    if((blk.block_check_word == gxDatabase::gxInternalCheckWord)) {
      if(blk.block_status == gxNormalBlock) { // Only copy normal blocks
	ObjectLength = f->ObjectLength(addr + f->BlockHeaderSize());
	v = new char[(int)ObjectLength];
	if(!v) {
	  cout << "Memory allocation error!" << endl;
	  delete NewFile;
	  return;
	}
	f->Read(v, (__ULWORD__)ObjectLength);
	if(CheckError((gxDatabase *)f) != 0) {
	  delete NewFile;
	  return;
	}
        if((__SBYTE__)blk.block_status == gxNormalBlock) {
	  count++; // Increment the database block count
	  if(update_ver) {
	    gxDatabase::gxInternalCheckWord = gxCheckWord;
	    NewFile->Write(v, (__ULWORD__)ObjectLength,
			   NewFile->Alloc((__ULWORD__)ObjectLength));
	    if(CheckError((gxDatabase *)NewFile) != 0) {
	      delete NewFile;
	      return;
	    }
	    gxDatabase::gxInternalCheckWord = prev_check_word;
	  }
	  else 
	    NewFile->Write(v, (__ULWORD__)ObjectLength,
			   NewFile->Alloc((__ULWORD__)ObjectLength));
	  if(CheckError((gxDatabase *)NewFile) != 0) {
	    delete NewFile;
	    return;
	  }
	  switch(rev_letter) {
	    case '\0': case ' ': 
	      break;

	    default: // Default to rev A and higher
	      checksum = calcCRC32(v, (__ULWORD__)ObjectLength);
	      NewFile->Write(&checksum, sizeof(checksum));
	      if(CheckError((gxDatabase *)NewFile) != 0) {
		delete NewFile;
		return;
	      }
	      break;
	  }
	}
	delete v;
      }
      addr = addr + blk.block_length; // Goto the next database block
    }
    else {
      badgx++;
      addr = f->BlockSearch(addr); // Search for the next good block
      if(!addr) break; 
    }
    
  }

      cout << endl;

  cout << "Wrote " << count << " good database blocks to file: "
       << NewFile->DatabaseName() << endl;

  if(static_area)
    cout << "Wrote " << static_area << " bytes of Static area data."
	 << endl;
 
  if(badgx) 
    cout << "Did not write " << badgx << " bad blocks found in file: "
	 << f->DatabaseName() << endl;

  if(errors) {
    cout << f->DatabaseName() << " file header is damaged!" << endl;
    cout << "No header information was copied to "
	 << NewFile->DatabaseName() << endl;
  }

  NewFile->Close();
  delete NewFile;
  cout << endl;
}

void Menu(gxDatabaseDebugManager *f)
{
  cout << endl;
  cout << "Analyzing file: " << f->DatabaseName() << endl;
  cout << "Enter the letter of your selection at the prompt." << endl;
  cout << endl;
  cout << "(A, a)    - Analyze the database file header" << endl;
  cout << "(D, d)    - Dump every database block" << endl;
  cout << "(F, f)    - Find every database block in the file" << endl;
  cout << "(H, h, ?) - Displays this menu" << endl;
  cout << "(Q, q)    - Quit this program" << endl;
  cout << "(r)       - Rebuild a damaged database file" << endl;
  cout << "(R)       - Rebuild and update to current version/revision" 
       << endl;
  cout << "(S, s)    - Display database file statistics" << endl;
  cout << "(W, s)    - Walk through every database block" << endl;
  cout << endl;
}

void ClearInputStream(istream &s)
// Used to clear istream
{
  char c;
  s.clear();
  while(s.get(c) && c != '\n') { ; }
}

void PausePrg()
{
  cout << endl;
  cout << "Press enter to continue..." << endl;
  cin.get();
}

int Quit()
{
  // Cleanup and exit the porgram
  cout << "Exiting..." << endl;
  return 0;
}

void Version()
{
  cout << endl;
  cout << ProgramName << " gxDatabase file recovery program." << endl;
  EchoDatabaseVersion();
  cout << endl;
}

int main(int argc, char **argv)
{
  // Display the program version information and exit the program
  if(argc >= 2) {
    if(strcmp(argv[1], "version") == 0) {
      Version();
      return 0;
    }
  }

  if(argc < 2) {
    cout << endl;
    Version();
    cout << "Usage: " << ProgramName << " infile" << endl;
    cout << "Usage: " << ProgramName << " infile (command)" << endl;
    cout << endl;
    return 1;
   }

  gxDatabaseDebugManager *f = new gxDatabaseDebugManager;   
  if(!f) {
    cout << "Memory allocation error!" << endl;
    return 1;
  }

  const char *fname = argv[1];
  if(!gxDatabase::Exists(fname)) {
    cout << "The specified file does not exist!" << endl;
    cout << "Exiting..." << endl;
    delete f;
    return 1;
  }
  else {
    f->BlindOpen(fname);
    if(CheckError((gxDatabase *)f) != 0) return 1;
  }

  char key;
  if(argc <= 2) Menu(f); // Not processing a command
  int rv = 1;

  while(rv) {
    if(argc > 2) { // Process a single command and exit the loop
      key = *(argv[2]);
      rv = 0;
    }
    else {
      if (!cin) { 
	ClearInputStream(cin);
	if (!cin) {
          cout << "Input stream is broken" << endl;
          return 0;
	}
      }
      cout << '>';
      cin >> key;
      if (!cin) continue;
    }
    switch(key) {
      case 'a' : case 'A' :
	if(argc <= 2) ClearInputStream(cin);
	AnalyzeHeader(f);
	break;
      case 'f' : case 'F' :
	if(argc <= 2) ClearInputStream(cin);
	FindBlock(f);
	break;
      case 'd' : case 'D' :
	if(argc <= 2) ClearInputStream(cin);
	FindBlock(f, 1);
	break;
      case 'h' : case 'H' :
	Menu(f);
	break;
      case '?' :
	Menu(f);
	break; 
      case 'q' : case 'Q' :
	rv = Quit();
	break;
      case 'r' : 
	if(argc <= 2) ClearInputStream(cin);
	Rebuild(f);
	break;
      case 'R' :
	if(argc <= 2) ClearInputStream(cin);
	Rebuild(f, 1);
	break;
      case 's' : case 'S' :
	if(argc <= 2) ClearInputStream(cin);
	DisplayStats(f);
	break;
      case 'w' : case 'W' :
	if(argc <= 2) ClearInputStream(cin);
	FindBlock(f, 1, 1);
	break;
      default:
        cout << "Unrecognized command" << endl;
    }
  }

  delete f;
  return 0;
}
// ----------------------------------------------------------- //
// ------------------------------- //
// --------- End of File --------- //
// ------------------------------- //

