/*
 * oracall32.h	1.01 03/15/01
 *
 * Copyright (c) 2001 Sergey Yakovlev. All Rights Reserved.
 *
 *    E-mail:  yakovlev@zdnetonebox.com
 * Voice/Fax:  +1 (617) 250-0001 x1385
 *       ICQ:  86060618
 *
 * Author grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Author.
 *
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. AUTHOR AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL AUTHOR OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF AUTHOR HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 *
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */


// The following ifdef block is the standard way of creating macros which make exporting 
// from a DLL simpler. All files within this DLL are compiled with the ORACALL32_EXPORTS
// symbol defined on the command line. this symbol should not be defined on any project
// that uses this DLL. This way any other project whose source files include this file see 
// ORACALL32_API functions as being imported from a DLL, wheras this DLL sees symbols
// defined with this macro as being exported.

#ifdef ORACALL32_EXPORTS
#define ORACALL32_API __declspec(dllexport)
#else
#define ORACALL32_API __declspec(dllimport)
#endif

#include <time.h>

struct OCIEnv;
struct OCIError;
struct OCISvcCtx;
struct OCIStmt;
struct OCIDescribe;
struct OCILobLocator;
class  CStatement;
class  CResultSet;

/*
 * This class represents a connection to ORACLE database.
 */
class ORACALL32_API CConnection
{
	friend class CStatement;
	friend class CResultSet;
	public:
		CConnection();
		virtual ~CConnection();
		int connect(const char *username, const char *password, const char *dblink);
		int disconnect();
		int commit();
		int rollback();
		CStatement* createStatement();
		CResultSet* getTableComments(); 
		CResultSet* getColumnComments(const char* tab_name = NULL); 
	private:
		OCIEnv*		envhp;	/* the environment handle */
		OCIError*	errhp;	/* the error handle */
		OCISvcCtx*	svchp;	/* the service handle */

		CStatement* stm;	/* the SQL statement*/		
		
		enum conn_state
		{
			not_connected,
			connected
		} state;

		int init(void);
		int deallocate(void);
//		void display_error(FILE* file) const;
};

// the Oracle date structure
struct COCIDate
{
	unsigned char datetime[7];
	COCIDate(tm* time) {
		datetime[0] = 120;				// century
		datetime[1] = time->tm_year;	// year
		datetime[2] = time->tm_mon + 1;	// month
		datetime[3] = time->tm_mday;	// day
		datetime[4] = time->tm_hour + 1;// hour
		datetime[5] = time->tm_min + 1;	// minute
		datetime[6] = time->tm_sec + 1;	// second
	}
};

/*
 * This class represents a Statement object is used for executing 
 * a static SQL statement and obtaining the results produced by it. 
 */
class ORACALL32_API CStatement
{
	friend class CResultSet;
	public:
		CStatement(CConnection*);
		virtual ~CStatement();
		int prepareStatement(const char*);
		CResultSet* executeQuery(const char*);
		CResultSet* executeQuery();
		int executeUpdate(const char*);
		int executeUpdate();
		int setString(const short, const char*);
		int setInt(const short, const int*);
		int setFloat(const short, const float*);
		int setDouble(const short, const double*);
		int setDate(const short, const COCIDate*);
		bool isStmSelect();

	private:
		OCIStmt*		stmthp; /* The statement handle */		
		CConnection*	con;	/* The pointer to connection object */
		CResultSet*		rs;		/* The pointer to resultset object */
		int				type;	/* OCI Statement Type */ 
};

// the field info structure
struct CFieldInfo
{
	// meta data from OCI
	char*	name;
	short	dtype;
	long	dsize;
	long	precision;
	int		scale;
	int		nullability;
};

/*
 * A CResultSet class provides access to a table of data generated 
 * by executing a CStatement. The table rows are retrieved in sequence. 
 * Within a row its column values can be accessed in any order.
 */
 class ORACALL32_API CResultSet
{
	struct CFieldBuffer
	{
		void* pDataBuffer;
		long lDataBuffer;
		short tDataBuffer;
	};
	
	public:
		CResultSet(CStatement*);
		virtual  ~CResultSet();
		void close();
		void getFieldInfo(const short, CFieldInfo&);
		short getFieldInfoCount();
		int next();
		char* getString(const short, char*, const short);
		int getInt(const short);
		float getFloat(const short);
		double getDouble(const short);
		long double getLongDouble(const short);
		int getDate(const short, tm*);
		void setLobOffset(const short index, const unsigned int offset = 1);
		bool readClob(const short, char*, const short);
		int writeClob(const short, const char*, const short);
		unsigned int getLobLenth(const short);
	private:
		CFieldInfo* fieldInfos;	/* array of field info structs with OCI meta-data */
		CFieldBuffer* fieldBuffers; /* array of field buffer structures */
		CStatement* stm;	/* the parent statement pointer */
		short columns;		/* the field counter */
		unsigned int* fieldOffset;	/* the absolute offset from the beginning of the LOB value */

		void describeResult();
//		int allocLobLocator(OCILobLocator* loc);
};

/*
 * Error number macros
 */
#define CONERR_ALRCON -10   /* already connected */
#define CONERR_NOTCON -12   /* not connected */
//#define CURERR_ALROPN -13   /* cursor is already open */
//#define CURERR_NOTOPN -14   /* cursor is not opened */


///////////////////////// Exported function ////////////////////////////
extern "C" {  
ORACALL32_API CConnection*	SYAllocConnect(const char *username, const char *password, const char *dblink);
ORACALL32_API void			SYFreeConnect(CConnection* connection);

ORACALL32_API CStatement*	SYPrepare(CConnection* connection, const char* query);
ORACALL32_API void			SYSetParamString(CStatement* statement, const short index, const char* val);
ORACALL32_API void			SYSetParamInteger(CStatement* statement, const short index, const int* val);
ORACALL32_API void			SYSetParamFloat(CStatement* statement, const short index, const float* val);
ORACALL32_API void			SYSetParamDouble(CStatement* statement, const short index, const double* val);
ORACALL32_API void			SYSetParamDate(CStatement* statement, const short index, const COCIDate* val);
ORACALL32_API CResultSet*	SYExecuteSQL(CConnection* connection, const char* query);
ORACALL32_API int			SYExecuteUpdate(CConnection* connection, const char* query);
ORACALL32_API CResultSet*	SYExecutePrepared(CStatement* statement);
ORACALL32_API int			SYCommit(CConnection* connection);
ORACALL32_API int			SYRollback(CConnection* connection);

ORACALL32_API int			SYNext(CResultSet* rs);
ORACALL32_API void			SYGetFieldInfo(CResultSet* rs, const short index, CFieldInfo& fieldinfo);
ORACALL32_API short			SYGetFieldInfoCount(CResultSet* rs);
ORACALL32_API CResultSet*	SYGetTableComments(CConnection* connection);
ORACALL32_API CResultSet*	SYGetColumnComments(CConnection* connection, const char* tab_name = NULL);
ORACALL32_API char*			SYGetString(CResultSet* rs, const short index, char* dest, const short len);
ORACALL32_API int			SYGetInteger(CResultSet* rs, const short index);
ORACALL32_API float			SYGetFloat(CResultSet* rs, const short index);
ORACALL32_API double		SYGetDouble(CResultSet* rs, const short index);
ORACALL32_API long double	SYGetLongDouble(CResultSet* rs, const short index);
ORACALL32_API int			SYGetDate(CResultSet* rs, const short index, tm* time);
ORACALL32_API void			SYSetLobOffset(CResultSet* rs, const short index, const unsigned int offset = 1);
ORACALL32_API unsigned int	SYGetLobLenth(CResultSet* rs, const short index);
ORACALL32_API bool			SYReadClob(CResultSet* rs, const short index, char* dest, const short len);
ORACALL32_API int			SYWriteClob(CResultSet* rs, const short index, const char* source);
}