unit Scrap;

{  ******
   *
   * Module:    Scrap
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    The TScrap class is instantiate as objects explode into tiny shards.
   *    When this happens, each edge of the original object become an individual
   *    TScrap, complete with is own speed, direction, rotation, etc...
   *
   *    TScrap objects disintegrate into nothing after a few ticks.
   *
   ****** }

interface
uses SoundDev, Visible, Vertex, Edge, General, Forms, WinTypes, Graphics,
     Classes;

type TScrap = class(TVisibleObject)
    public
        { Class constructor. }
        constructor Create(polyParent: TVisibleObject;
                           edgeParent: TEdge;
                           fScale: Real;
                           iLifeSpan: Integer;
                           clrScrap: TColor;
                           listObjects: TList);

        { Overridden movement method. }
        procedure Move; Override;

    protected

        m_iLifeSpan: Integer;           { Lifespan of the scrap in "ticks". }
        m_iLifeUsed: Integer;           { Number of ticks the scrap has existed. }
        m_iLifeFadePoint: Integer;      { Age at which object begins to visually fade. }
        m_edgeScrap: TEdge;             { Edge comprising the scrap. }
end;

implementation

constructor TScrap.Create(polyParent: TVisibleObject;
                          edgeParent: TEdge;
                          fScale: Real;
                          iLifeSpan: Integer;
                          clrScrap: TColor;
                          listObjects: TList);
var
    vtxOne, vtxTwo: TVertex;
    fMidX, fMidY: Real;
    fLength: Real;
    fScreenX1, fScreenY1: Real;
    fScreenX2, fScreenY2: Real;
    fSlopeX, fSlopeY: Real;
begin
    inherited Create(listObjects, -1);

    { Store the amount of movements the missle will last before breaking up. }
    m_iLifeSpan := iLifeSpan;
    m_iLifeUsed := 0;
    m_iLifeFadePoint := Trunc(iLifeSpan * 0.5);

    { Compute the current screen coordinates of the parent edge. }
    polyParent.mtrxTransform.TranslateVertex(edgeParent.m_vtxOne.m_fLocalX,
                                             edgeParent.m_vtxOne.m_fLocalY,
                                             fScreenX1,
                                             fScreenY1);

    polyParent.mtrxTransform.TranslateVertex(edgeParent.m_vtxTwo.m_fLocalX,
                                             edgeParent.m_vtxTwo.m_fLocalY,
                                             fScreenX2,
                                             fScreenY2);

    { Compute the midpoint of the edge. }
    fMidX := (fScreenX1 + fScreenX2) / 2;
    fMidY := (fScreenY1 + fScreenY2) / 2;

    fSlopeX := (fScreenX2 - fScreenX1) /2;
    fSlopeY := (fScreenY2 - fScreenY1) / 2;

    { Define the shape of a scrap object to be a single line. }
    vtxOne := vtxDefineNewVertex(fSlopeX, fSlopeY);
    vtxTwo := vtxDefineNewVertex(-fSlopeX, -fSlopeY);
    m_edgeScrap := edgeDefineNewEdge(vtxOne, vtxTwo, clrScrap, False);

    { Set up default coloring and visiblity information. }
    m_edgeScrap.m_bVisible := True;
    m_edgeScrap.m_iShapeID := 0;

    { Compute the size of the edge. }
    fLength := fDistance(fScreenX1,
                         fScreenY1,
                         fScreenX2,
                         fScreenY2);

    mtrxTransform.SetTranslation(fMidX, fMidY);
    mtrxTransform.fOrientation := 0.0;
    mtrxTransform.fScale := fScale;
end;

procedure TScrap.Move;
begin
    { Do the default movement stuff. }
    inherited Move;

    { At a certain life completion point, turn to a less visible color. }
    if m_iLifeUsed = m_iLifeFadePoint then
        m_edgeScrap.m_clrEdge := clGray;

    { If our lifespan has been exceeded then destroy this object. }
    if m_iLifeUsed = m_iLifeSpan then
        KillObject(False, 0, 0, 0, 0);

    { Increment our lifespan clicker. }
    m_iLifeUsed := m_iLifeUsed + 1;
end;

end.

