unit Text;

{  ******
   *
   * Module:    Text
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module defines the behavior required by Text objects.  Text objects
   *    actual like other visible objects, but form the basis for fonts within
   *    the game.  All characters on the Rocks screen are polygon-based.  This
   *    object class allows the program to work with text more easily.
   *
   *    Just other visible objects, Text objects can be scaled, rotated, and
   *    moved on the fly.  This is used in the credits screen that comes up
   *    when you leave Rocks open for a minute or two without starting a game.
   *
   ****** }

interface
uses SoundDev, Visible, Graphics, Classes;

type TTextObject = class(TVisibleObject)
    private

        m_szText: String;       { Current text displayed by the object. }
        m_clrColor: TColor;     { Color of the text. }

        { Methods to assign polygon shapes to represent the current text. }
        procedure AssignShapes;
        function szGetShapeFile(cChar: Char): string;

        { Methods for getting/setting property values. }
        procedure SetText(szText: String);
        procedure SetColor(clrColor: TColor);

    public
        { Class constructor. }
        constructor Create(listObjects: TList; szText: String; clrColor: TColor);

        { Exposed properties. }
        property szText: String         Read m_szText       Write SetText;
        property clrColor: TColor       Read m_clrColor     Write SetColor;
end;

implementation
uses General;


constructor TTextObject.Create(listObjects: TList;
                               szText: String; clrColor: TColor);
begin
    { Perform default processing. }
    inherited Create(listObjects, -1);

    { Initialize the text polygons. }
    m_clrColor := clrColor;
    SetText(szText);
end;

procedure TTextObject.SetText(szText: String);
begin
    { Store the new string. }
    m_szText := szText;

    { Display the current score. }
    AssignShapes;
end;

procedure TTextObject.SetColor(clrColor: TColor);
begin
    { Store the new string. }
    m_clrColor := clrColor;

    { Display the current score. }
    AssignShapes;
end;

procedure TTextObject.AssignShapes;
const
    CHARACTER_SPACING = 6;
var
    fRelativePosX: Real;
    iCharIndex: Integer;
    szShapeFile: String;
begin
    { Erase us as we currently appear, and clear all existing vertices and edges
      that define the object. }
    ClearShape;

    { Determine where the left corner of the string will be. }
    fRelativePosX := -(Trunc(Length(m_szText) / 2) * CHARACTER_SPACING);

    { Assign a shape for each character in the string. }
    for iCharIndex := 1 to Length(m_szText) do
    begin
        { Find the file associated with the given character. }
        szShapeFile := szGetShapeFile(m_szText[iCharIndex]);

        { If there is an associated file then add its shape to the object. }
        if Length(szShapeFile) > 0 then
            IncludeShape(szShapeFile, Round(fRelativePosX), 0, m_clrColor, False);

        { Position to the next character on the string. }
        fRelativePosX := fRelativePosX + CHARACTER_SPACING;
    end;

    { Redraw the text now to avoid flickering. }
    Draw;
end;

function TTextObject.szGetShapeFile(cChar: Char): string;
begin
    { Return a shape name based upon the character to be displayed. }
    case UpCase(cChar) of
    '0':  Result := 'Digit0';
    '1':  Result := 'Digit1';
    '2':  Result := 'Digit2';
    '3':  Result := 'Digit3';
    '4':  Result := 'Digit4';
    '5':  Result := 'Digit5';
    '6':  Result := 'Digit6';
    '7':  Result := 'Digit7';
    '8':  Result := 'Digit8';
    '9':  Result := 'Digit9';
    'A':  Result := 'LetterA';
    'B':  Result := 'LetterB';
    'C':  Result := 'LetterC';
    'D':  Result := 'LetterD';
    'E':  Result := 'LetterE';
    'F':  Result := 'LetterF';
    'G':  Result := 'LetterG';
    'H':  Result := 'LetterH';
    'I':  Result := 'LetterI';
    'J':  Result := 'LetterJ';
    'K':  Result := 'LetterK';
    'L':  Result := 'LetterL';
    'M':  Result := 'LetterM';
    'N':  Result := 'LetterN';
    'O':  Result := 'LetterO';
    'P':  Result := 'LetterP';
    'Q':  Result := 'LetterQ';
    'R':  Result := 'LetterR';
    'S':  Result := 'LetterS';
    'T':  Result := 'LetterT';
    'U':  Result := 'LetterU';
    'V':  Result := 'LetterV';
    'W':  Result := 'LetterW';
    'X':  Result := 'LetterX';
    'Y':  Result := 'LetterY';
    'Z':  Result := 'LetterZ';
    '.':  Result := 'Period';
    ':':  Result := 'Colon';
    '_':  Result := 'UnderScore';
    '!':  Result := 'Exclamation';
    else
        Result := '';
    end;
end;

end.

