unit GameCtl;

{  ******
   *
   * Module:    GameCtl
   * Author:    Joe Kessler
   *            IntegrationWare - A New Generation of Extraordinary PC Solutions
   *            www.integrationware.com
   *
   * Purpose:
   *
   *    This module implements the TGameControl class, which is the most
   *    complex class in the program.  It controls the sequencing of the entire
   *    game, from moving individual objects to determining when a level has
   *    been completed.
   *
   ****** }

interface
uses SoundDev, RocksKbd, MultiCh, State, Bart, Score, Visible, Constant, Missle,
     Text, High, Alien, VideoDev, WinProcs, WinTypes, Forms, Graphics, Classes,
     SysUtils, ShapeLib, InputDev, Input, InpCodes, Message, Keyboard;

{ Game control constants. }
const STAR_COUNT = 75;

type TGameControl = class(TMultiChannelStateAware)
    private

        { Internal game-control values. }
        m_scrScoreBoard: TScoreBoard;       { Current score and number of lives. }
        m_hslHighScores: THighScoreList;    { Current high score list. }
        m_voHiScoreText: TTextObject;       { Current hi-score value display. }
        m_iStateChannel1: Integer;          { Primary state channel ID. }
        m_iStateChannel2: Integer;          { Secondary state channel ID. }
        m_voBart: TBart;                    { Currently active Bart object. }
        m_bGameDead: Boolean;               { TRUE if the game object has one full-cycle. }
        m_iCurrentLevel: Integer;           { Current level number during game play. }
        m_iAliensOnLevel: Integer;          { Number of aliens that have appeared on level. }
        m_iLevelBonus: Integer;             { Value of last level bonus given. }
        m_fAlienProbability: Real;          { Probablity that an alien will appear. }
        m_bGamePaused: Boolean;             { TRUE while the game is paused. }
        m_bAlienOnScreen: Boolean;          { TRUE while an alien is on the screen. }
        m_fLevelStartTime: Real;            { Time (in seconds) user started on level. }
        m_lstUniverse: TList;               { List of all objects in the universe. }
        m_inpPlayer: TWindowsKeyboard;      { Current input device, copied for convenience. }

        { Game state variables. }
        m_bEnteringName: Boolean;            { TRUE while the user is entering a hi-score tag. }
        m_bPendingAsteroidCreation: Boolean; { TRUE while we're generating an asteroid field. }
        m_bScoreBoardShownLast: Boolean;     { TRUE if hi-scores were shown last (not credits). }

        m_iCharPos: Integer;                { Current "cursor" position while entering tag. }
        m_szCurrentChar: Char;              { Character at current tag position. }

        { High score name entry values. }
        m_voTagChar: Array[0 .. 2] of TTextObject;

        { Collision lists for processing collision with godd and bad guys.  Alien
          objects (including asteroids) go in one list, while Bart-friendly ones
          go in the other. }
        m_alstCollisionLists: Array [0 .. 1] of TList;

        { Array of points where stars exist on the backdrop. }
        m_astarBackground: Array[0 .. STAR_COUNT] of TPoint;

        { Method to define common shapes used in the game. }
        procedure DefineGameShapes;

        { Methods for controlling game pauses. }
        procedure SetGamePause(bPaused: Boolean);
        procedure EnterPauseState;
        procedure ExitPauseState;

        { Methods for creating and painting on the starry backdrop. }
        procedure MakeStars(iClientWidth, iClientHeight: Integer);
        procedure PaintBackdrop;

        { Method for forming the asteroid field. }
        procedure PopulateAsteroidField(iGroupID: Integer);

        { Method to create a new Bart. }
        procedure InitializeBart;
        procedure ReviveBart;

        { Method to release all objects in a given category.  This is most
          useful when transitioning to a new screen, such as credits. }
        procedure FreeObjects(iGroupID: Integer);

        { This method releases memory used by all objects in the universe that
          are currently dead, and removes them from the master list. }
        procedure CleanUpDeadObjects(var bAlienObjectKilled: Boolean);

        { Other methods. }
        procedure CreateScoreBoard;
        procedure CheckLevelProgression;

        { Collision detection methods. }
        procedure PerformCollisionDetection(listObjects: TList);
        procedure CreateCollisionLists;

        { Methods for high score text entry. }
        procedure ResetTagColors;
        procedure PrepareNameEntryText;
        procedure PerformLetterMovement(iAmount: Integer);
        procedure PerformCursorMovement(iAmount: Integer);
        procedure IncrementCursorPos(iAmount: Integer);

        { This method prepares the text objects used in the credits screen. }
        procedure PrepareCredits;

        { Procedure to create text object and add it to the universe.  Calling
          this method is more convenient that creating text explicitly. }
        function NewTextObject(szText: String;
                                  clrText: TColor;
                                  bFlashing: Boolean;
                                  iPosX, iPosY: Integer;
                                  fScale: Real;
                                  iGroupID: Integer): TTextObject;

        { This method will create a floating, flashing piece of text.  This is
          used for displaying bonus values when the player shoots an alien. }
        procedure DisplayFloatingText(szTextString: String; iPosX, iPosY: Integer);

    protected

        { Virtual methods for customizing state behavior. }
        procedure EnterState(iChannel: Integer; pstCurrent: PState); Override;
        procedure ExitState(iChannel: Integer; pstCurrent: PState); Override;

        { Method to handle keyboard messages. }
        procedure ProcessMessage(iCategory, iType: Integer; bValue0: Boolean; fValue1, fValue2: Real); Override;

    public
        { Class constructor and destructor. }
        constructor Create;
        destructor Destroy;

        { Object control methods. }
        procedure Move; Override;

        { Method to draw a single "Frame". }
        procedure DrawFrame;
        procedure EraseFrame;

        { Methods to tell the game to start it's idle cycle, or break it by
          beginning a new game. }
        procedure DisplaySplashScreen;
        procedure StartNewGame;

        { Exposed properties. }
        property bGamePaused: Boolean       Read m_bGamePaused  Write SetGamePause;
        property bGameDead: Boolean         Read m_bGameDead;
end;

implementation
uses Global, General, Asteroid;

{ Object-grouping constant allow the came control object to operate on objects
  within certain categories. }
const GROUP_ANY = -1;           { All objects. }
const GROUP_SPLASH = 0;         { Objects appearing on the main splash screen. }
const GROUP_CREDITS = 1;        { Objects on the credits screen. }
const GROUP_PAUSE = 2;          { Objects used to display a paused game state. }
const GROUP_GET_READY = 3;      { GET READY text objects. }
const GROUP_HI_SCORE = 4;       { Objects created while the users a high score tag. }
const GROUP_GAME_OVER = 5;      { GAME OVER text object. }
const GROUP_SCOREBOARD = 6;     { Objects used to show current score and lives left. }
const GROUP_PERMANENT = 7;      { Permanent objects thats should never be removed. }
const GROUP_GAME = 8;           { Objects allocated during a level of the game. }

{ Bonus amount for shooting the alien ship. }
const ALIEN_BONUS = 1000;

constructor TGameControl.Create;
var
    iIndex: Integer;
begin
    { Perform default initializations. }
    inherited Create;

    { Initialize the random number generator. }
    Randomize;

    { Initialize the shape library with shapes required by the game. }
    g_slShapeLib := TShapeLibrary.Create;
    DefineGameShapes;

    { Open two state channels: a primary and a secondary.  The secondary one
      will be used to track sub-states within the primary. }
    m_iStateChannel1 := iOpenNewChannel;
    m_iStateChannel2 := iOpenNewChannel;

    { Create the global object list. }
    m_lstUniverse := TList.Create;

    { Create object collision lists.  Each collision-enabled object will belong
      to one of these lists. }
    m_alstCollisionLists[0] := TList.Create;
    m_alstCollisionLists[1] := TList.Create;

    { Determine the position of the starry background. }
    MakeStars(INTERNAL_SCREEN_WIDTH, INTERNAL_SCREEN_HEIGHT);

    { For now, just create a scoreboard and wait. }
    CreateScoreBoard;

    { Create the "current high score" text display in the upper-right corner. }
    m_hslHighScores := THighScoreList.Create(m_lstUniverse, GROUP_SCOREBOARD);
    m_voHiScoreText := NewTextObject( Format('Hi: %6.6d', [m_hslHighScores.lGetNthScore(1)]),
                                      clAqua, False, 435, 10, 1.5, GROUP_PERMANENT);

    { Initialize other general state flags. }
    m_inpPlayer := TWindowsKeyboard(g_envEnviron.devGetInputDevice);
    m_bScoreBoardShownLast := False;
    m_voBart := nil;
    m_fAlienProbability := 0.0;
    m_bAlienOnScreen := False;
    m_bGamePaused := False;
    m_bEnteringName := False;
end;

destructor TGameControl.Destroy;
var
    iIndex: Integer;
begin
    { Erase the entire display frame. }
    g_envEnviron.devGetVideoDevice.ClearFrame;

    { Free all objects still in the universe.  }
    for iIndex := (m_lstUniverse.Count - 1) downto 0 do
        TVisibleObject(m_lstUniverse.Items[iIndex]).Free;
    m_lstUniverse.Free;

    { Free the shape library object. }
    m_hslHighScores.Free;
    g_slShapeLib.Free;

    { Perform default cleanup. }
    inherited Destroy;
end;

procedure TGameControl.PopulateAsteroidField(iGroupID: Integer);
var
    shpNew: TVisibleObject;
    iAsteroidCount: Integer;
    iAsteroidIndex: Integer;

    fPosX, fPosY: Real;
    iPlacementAttempt: Integer;

    rectAsteroid, rectBart: TRect;
    bAsteroidOverlapping, bBartOverlapping: Boolean;
begin
    { Create some asteroids to start off with. }
    if m_iCurrentLevel >= 0 then
        iAsteroidCount := 6 + m_iCurrentLevel
    else
        iAsteroidCount := 3;

    for iAsteroidIndex := 1 to iAsteroidCount do
    begin
        { Create a new asteroid and add him to the global object list. }
        shpNew := TAsteroid.Create(m_lstUniverse, 2.5, m_scrScoreBoard, 1);
        shpNew.iGroupID := iGroupID;
        m_lstUniverse.Add(shpNew);

        { Reset the placement attempt counter. }
        iPlacementAttempt := 0;

        { Position the new asteroid around Bart. }
        repeat
            { Increment the placement attempt counter. }
            iPlacementAttempt := iPlacementAttempt + 1;

            fPosX := Random(iScreenWidth) / 3 + iScreenWidth / 15;
            fPosY := Random(iScreenHeight) / 3 + iScreenHeight / 15;

            if Random >= 0.5 then
                fPosX := iScreenWidth - fPosX;

            if Random >= 0.5 then
                fPosY := iScreenHeight - fPosY;

            rectAsteroid.Left := Trunc(fPosX) - 40;
            rectAsteroid.Right := Trunc(fPosX) + 40;
            rectAsteroid.Top := Trunc(fPosY) - 40;
            rectAsteroid.Bottom := Trunc(fPosY) + 40;
            bAsteroidOverlapping := shpNew.bIsRectSafe(rectAsteroid);

            { Make sure that the new asteroid will not overlap Bart. }
            if m_voBart <> nil then
            begin
                rectBart := m_voBart.rectGetBoundingRect;
                bBartOverlapping := bRectsIntersect(rectBart, rectAsteroid);
            end
            else
                bBartOverlapping := False;

        until ((bAsteroidOverlapping = False) and (bBartOverlapping = False)) or
                ((iPlacementAttempt > 3) and (bBartOverlapping = False));

        shpNew.mtrxTransform.SetTranslation(fPosX, fPosY);

        { Establish a default direction and speed. }
        shpNew.mtrxTransform.fDirection := (Random * TWOPI);
        shpNew.mtrxTransform.fSpeed := (Random * (2.5 + m_iCurrentLevel * 0.20));

        { Make the asteroid rotate a little bit. }
        shpNew.mtrxTransform.fRotation := (0.05 - (Random * 0.1));
    end;

    { We're no longer waiting for the asteroid belt to be created. }
    m_bPendingAsteroidCreation := False;
end;

procedure TGameControl.MakeStars(iClientWidth, iClientHeight: Integer);
var
    iIndex: Integer;
begin
    { Determine the position of the starry background. }
    for iIndex := 0 to (STAR_COUNT - 1) do
    begin
        m_astarBackground[iIndex].x := Random(iClientWidth);
        m_astarBackground[iIndex].y := Random(iClientHeight);
    end;
end;

procedure TGameControl.PaintBackdrop;
var
    devVideo: TVideoDevice;
    iIndex: Integer;
begin
    { Get apointer to the video object. }
    devVideo := g_envEnviron.devGetVideoDevice;

    { Draw a pixel for each star. }
    for iIndex := 0 to (STAR_COUNT - 1) do
        devVideo.SetPixel(m_astarBackground[iIndex].x,
                          m_astarBackground[iIndex].y,
                          clLtGray);
end;

procedure TGameControl.InitializeBart;
var
    shpNew: TVisibleObject;
begin
    { Create Bart. }
    m_voBart := TBart.Create(m_lstUniverse, 0);
    m_lstUniverse.Add(m_voBart);

    { Put Bart right in the middle of screen. }
    m_voBart.mtrxTransform.SetTranslation(iMidScreenX, iMidScreenY);
    m_voBart.mtrxTransform.fScale := 0.80;
    m_voBart.mtrxTransform.fSpeed := 0.25;
end;

procedure TGameControl.CreateScoreBoard;
begin
    { Create the scoreboard object. }
    m_scrScoreBoard := TScoreBoard.Create(m_lstUniverse);
    m_scrScoreBoard.mtrxTransform.SetTranslation(55, 10);
    m_scrScoreBoard.mtrxTransform.fScale := 2;
    m_scrScoreBoard.iGroupID := GROUP_PERMANENT;
    m_lstUniverse.Add(m_scrScoreBoard);
end;

procedure TGameControl.CleanUpDeadObjects(var bAlienObjectKilled: Boolean);
var
    iIndex: Integer;
    voObject: TVisibleObject;
begin
    { Initially, we havent found any dead aliens or asteroids. }                                
    bAlienObjectKilled := False;

    { Clean up any dead objects. }
    for iIndex := (m_lstUniverse.Count - 1) downto 0 do
    begin
        { Get an object from the master list. }
        voObject := m_lstUniverse.Items[iIndex];

        { Check if the object is dead. }
        if voObject.bObjectDead = True then
        begin
            { Make sure the object has been erased before we release it. }
            voObject.Erase;

            { If an alien or asteroid was killed, set a flag to signal it. }
            if ((voObject is TAlien) or (voObject is TAsteroid)) and
               (voObject.iGroupID = GROUP_GAME) then
                bAlienObjectKilled := True;

            { If the object is an Alien, signal that the alien is now gone. }
            if voObject is TAlien then
                m_bAlienOnScreen := False;

            { Release the object, and remove it from the universal list. }
            m_lstUniverse.Delete(iIndex);
            voObject.Free;
        end;
    end;
end;

procedure TGameControl.Move;
var
    bAlienObjectKilled: Boolean;
    iIndex: Integer;
    voAlien: TAlien;
begin
    { If the game has been paused then don't perform any movements. }
    if m_bGamePaused = True then
        Exit;

    { Perform the default actions. }
    Inherited Move;

    { Clean up any objects that are now dead. }
    CleanUpDeadObjects(bAlienObjectKilled);

    { Move all of the objects in the universe. }
    for iIndex := 0 to (m_lstUniverse.Count - 1) do
        TVisibleObject(m_lstUniverse.Items[iIndex]).Move;

    { Conditionally introduce an alien entity. }
    if (m_fAlienProbability > Random) and
       (m_bAlienOnScreen = False) and
       (m_iAliensOnLevel < m_iCurrentLevel) then
    begin
        { Signal that an alien has entered the arena. }
        m_bAlienOnScreen := True;

        { Create the alien object. }
        voAlien := TAlien.Create(m_lstUniverse, 2, 0.05 + m_iCurrentLevel * 0.01);
        m_lstUniverse.Add(voAlien);
        Inc(m_iAliensOnLevel);
    end;

    { Check if it's time to move to another level. }
    if (bAlienObjectKilled = True) then
        CheckLevelProgression;
end;

procedure TGameControl.DrawFrame;
var
    iIndex: Integer;
    voObject: TVisibleObject;
begin
    { Signal the start of a new frame, and paint the starry backdrop. }
    g_envEnviron.devGetVideoDevice.BeginNewFrame;
    PaintBackdrop;

    { Iterate through each object currently in the universe. }
    for iIndex := 0 to (m_lstUniverse.Count - 1) do
    begin
        { Get reference to the next object. }
        voObject := m_lstUniverse.Items[iIndex];

        { Draw the object normally, as long as we are not paused or the object
          is a part of the pause display. }
        if (m_bGamePaused = False) or (voObject.iGroupID = GROUP_PAUSE) then
            voObject.Draw;
    end;

    { Perform collision detection just after a frame draw is completed. }
    PerformCollisionDetection(m_lstUniverse);
end;

procedure TGameControl.EraseFrame;
var
    iIndex: Integer;
begin
    { Erase each object currently in the universe. }
    for iIndex := 0 to (m_lstUniverse.Count - 1) do
        TVisibleObject(m_lstUniverse.Items[iIndex]).Erase;
end;

procedure TGameControl.PerformCollisionDetection(listObjects: TList);
var
    rectObject: TRect;

    iBartIndex, iEnemyIndex: Integer;
    objBart, objEnemy: TVisibleObject;

    txtNew: TTextObject;
begin
    { Separate all collision-enabled objects into two lists: alien objects, and
      hero objects.  Two objects can collide ONLY if they are in separate lists. }
    CreateCollisionLists;

    { Now, detect collisions between objects in the different lists. }
    for iBartIndex := 0 to (m_alstCollisionLists[0].Count - 1) do
    begin
        { Get a reference the Bart object. }
        objBart := m_alstCollisionLists[0].Items[iBartIndex];

        { Process collisions only if the Bart object is still active. }
        if (objBart.bObjectDead = False) and (objBart.bCanCollide = True) then
            for iEnemyIndex := 0 to (m_alstCollisionLists[1].Count - 1) do
            begin
                { Get references the Enemy object. }
                objEnemy := m_alstCollisionLists[1].Items[iEnemyIndex];

                { Process collisions only if the Enemy object is still active. }
                if (objEnemy.bObjectDead = False) and
                   (objEnemy.bCanCollide = True) and
                   (objBart <> objEnemy) and
                   (objBart.iCollisionID <> objEnemy.iCollisionID) then
                begin

                    { If the rectangles do NOT intersect, then no further }
                    { processing is required to determine that the objects }
                    { are NOT colliding. }
                    if objBart.bIsObjectColliding(objEnemy) = True then
                    begin
                        objBart.HandleCollision(objEnemy);
                        objEnemy.HandleCollision(objBart);

                        { If Bart himself was justed killed then try to generate a }
                        { Bart object to take his place. }
                        if objBart is TBart then
                            ReviveBart;

                        { Reward the player for killing the alien. }
                        if (objEnemy is TAlien) and ((objBart is TBart) or (objBart is TMissle)) then
                        begin
                            { Increase the player's score. }
                            m_scrScoreBoard.iCurrentScore := m_scrScoreBoard.iCurrentScore + ALIEN_BONUS;

                            { Display the bonus amount. }
                            DisplayFloatingText(IntToStr(ALIEN_BONUS),
                                                      Round(objEnemy.mtrxTransform.fTranslationX),
                                                      Round(objEnemy.mtrxTransform.fTranslationY));

                        end;

                        { If there are no more enemy objects then progress }
                        { to the next level. }
                        CheckLevelProgression;
                    end;
                end;
            end;
    end;

end;

procedure TGameControl.EnterState(iChannel: Integer; pstCurrent: PState);
var
    iPosX, iPosY: Integer;
    rectBart: TRect;
    iIndex: Integer;
    txtNew: TTextObject;
begin
    { Process states implemented by the base class. }
    inherited EnterState(iChannel, pstCurrent);

    { Clear the secondary channel automatically if the state change is on the
      main command channel. }
    if (iChannel = m_iStateChannel1) then
        saGetChannel(m_iStateChannel2).ClearStateSequence;

    { Process the command. }
    case pstCurrent^.enumCommandState of
    Splash:
        begin
            { Clear any crap off the screen that might still be there from aprevious run. }
            g_envEnviron.devGetVideoDevice.ClearFrame;

            { Free or hide any remaining objects that should not be visible. }
            m_hslHighScores.HideScores;
            FreeObjects(GROUP_CREDITS);

            { Throw a couple of asteroids out there. }
            m_iCurrentLevel := -1;
            PopulateAsteroidField(GROUP_SPLASH);

            { Position the title of the game! }
            NewTextObject('ROCKS', clYellow, False, Round(iScreenWidth / 2), 70, 10.0, GROUP_SPLASH);
            NewTextObject('A FUTURISIC ACTION ARCADE GAME', clWhite, False, iMidScreenX, 130, 2.0, GROUP_SPLASH);
            NewTextObject('VERSION 2.1', clWhite, False, iMidScreenX, 150, 2.0, GROUP_SPLASH);
            NewTextObject('INTEGRATIONWARE INC.',    clWhite, False, iMidScreenX, 200, 2.0, GROUP_SPLASH);
            NewTextObject('WWW.INTEGRATIONWARE.COM', clWhite, False, iMidScreenX, 220, 2.0, GROUP_SPLASH);
            NewTextObject('Press F2 to start...',     clWhite, True,  iMidScreenX, 285, 2.0, GROUP_SPLASH);

            { Show splash screen for a little while, then switch to high scores or credits. }
            AddToStateSequence(m_iStateChannel2, Pause, 15.0, 15.0);
            AddToStateSequence(m_iStateChannel2, EndChannelState, m_iStateChannel1 ,0);

            if m_bScoreBoardShownLast = False then
                begin
                AddToStateSequence(m_iStateChannel1, HighScore, 0, 0);
                m_bScoreBoardShownLast := True;
                end
            else
                begin
                AddToStateSequence(m_iStateChannel1, Credits, 0, 0);
                m_bScoreBoardShownLast := False;
                end;
        end;
    Credits:
        begin
            { Erase anything that shouldn't be showing on this screen. }
            FreeObjects(GROUP_SPLASH);

            { Prepare the text of the credits screen . }
            PrepareCredits;

            { Wait for a few seconds, then put the splash screen back up. }
            AddToStateSequence(m_iStateChannel1, Pause, 30, 0);
            AddToStateSequence(m_iStateChannel1, Splash, 0, 0);
            ExitCurrentState(iChannel);
        end;
    GetReady:
        begin
            { Erase anything that shouldn't be showing on this screen. }
            FreeObjects(GROUP_SPLASH);
            FreeObjects(GROUP_CREDITS);

            { Position the GET READY text. }
            NewTextObject('GET READY!', clWhite, False, iMidScreenX + 30, 150, 5.0, GROUP_GET_READY);

            { Wait for a few seconds, then take down the get ready sign.  Notice
              that since channel #1 is in the GetReady state, channel #2 is used
              to terminate the channel #1 state at the proper time. }
            AddToStateSequence(m_iStateChannel2, Pause, 1.5 ,0);
            AddToStateSequence(m_iStateChannel2, FreeGroupedObjects, GROUP_GET_READY ,0);
            AddToStateSequence(m_iStateChannel2, EndChannelState, m_iStateChannel1, 0);
        end;
    DisplayLevel:
        begin
            { Position the level-end text, and command it to hover for a
              few seconds before disappearing. }
            txtNew := NewTextObject(Format('LEVEL %d COMPLETED', [m_iCurrentLevel - 1]),
                                            clWhite, False, iMidScreenX, 150, 3.0, -1);
            txtNew.AddToStateSequence(Pause, 2.0, 0);
            txtNew.AddToStateSequence(Die, 0, 0);

            ExitCurrentState(iChannel);
        end;
    DisplayBonus:
        begin
            { Add the bonus to the scoreboard! }
            m_scrScoreBoard.iCurrentScore := m_scrScoreBoard.iCurrentScore + m_iLevelBonus;

            { Position the time-bonus text, and command it to hover for a few
              seconds before disappearing. }
            txtNew := NewTextObject(Format('TIME BONUS %d', [m_iLevelBonus]), clWhite,
                                            True, iMidScreenX, 180, 2.0, -1);
            txtNew.AddToStateSequence(Pause, 1.5, 0);
            txtNew.AddToStateSequence(Die, 0, 0);

            ExitCurrentState(iChannel);
        end;
    FreeGroupedObjects:
        begin
            { Free all objects in the given group. }
            FreeObjects(Trunc(pstCurrent^.fValue1));
            ExitCurrentState(iChannel);
        end;
    HighScore:
        begin
            { Erase anything that shouldn't be showing on this screen. }
            FreeObjects(GROUP_SPLASH);
            FreeObjects(GROUP_HI_SCORE);
            FreeObjects(GROUP_GAME);

            { Display the current high scores. }
            m_hslHighScores.DisplayScores;

            { Wait for a few, then put the splash screen back up. }
            AddToStateSequence(m_iStateChannel1, Pause, 10.0, 0);
            AddToStateSequence(m_iStateChannel1, TakeDownHighScores, 0, 0);
            AddToStateSequence(m_iStateChannel1, Pause, 1.0, 0);
            AddToStateSequence(m_iStateChannel1, Splash, 0, 0);

            { That's all we'll do for this state. }
            ExitCurrentState(iChannel);
        end;
    TakeDownHighScores:
        begin
            { Hide the high score screen. }
            m_hslHighScores.HideScores;

            { That's all we'll do for this state. }
            ExitCurrentState(iChannel);
        end;
    GenerateAsteroids:
        begin
            { Populate an asteroid field, and record this as the time that
              the current level began. }
            PopulateAsteroidField(GROUP_GAME);
            m_fLevelStartTime := fGetNow;

            ExitCurrentState(iChannel);
        end;
    GenerateBart:
        begin
            { Initialize a new TBart object, and reduce the remaining count. }
            InitializeBart;
            m_scrScoreBoard.iBartCount := m_scrScoreBoard.iBartCount - 1;

            ExitCurrentState(iChannel);
        end;
    BeginGame:
        begin
            { Erase anything that shouldn't be showing on this screen. }
            m_hslHighScores.HideScores;
            FreeObjects(GROUP_SPLASH);
            FreeObjects(GROUP_CREDITS);

            { Start off at level one. }
            m_fAlienProbability := 0.0;
            m_iAliensOnLevel := 0;
            m_bAlienOnScreen := False;
            m_iCurrentLevel := 1;
            m_scrScoreBoard.iBartCount := 3;
            m_scrScoreBoard.iCurrentScore := 0;
            m_fLevelStartTime := fGetNow;

            { Generate the first asteroid field, then Bart. }
            AddToStateSequence(m_iStateChannel1, Pause, 0.25 ,0);
            AddToStateSequence(m_iStateChannel1, GetReady, 0 ,0);
            AddToStateSequence(m_iStateChannel1, Pause, 0.25 ,0);
            AddToStateSequence(m_iStateChannel1, GenerateAsteroids, 0 ,0);
            AddToStateSequence(m_iStateChannel1, Pause, 0.5, 0);
            AddToStateSequence(m_iStateChannel1, GenerateBart, 0, 0);

            ExitCurrentState(iChannel);
        end;
    GameOver:
        begin
            { Display the GAME OVER message. }
            NewTextObject('GAME OVER', clYellow, False, iMidScreenX, iMidScreenY, 5.0, GROUP_GAME_OVER);

            { Make the high score board show before the credits screen during idle. }
            m_bScoreBoardShownLast := False;

            { Pause for 5 seconds, then go to the next stage. }
            AddToStateSequence(m_iStateChannel2, Pause, 5.0, 0);
            AddToStateSequence(m_iStateChannel2, FreeGroupedObjects, GROUP_GAME_OVER, 0);
            AddToStateSequence(m_iStateChannel2, EndChannelState, m_iStateChannel1, 0);
        end;
    GameDead:
        begin
            { Signal that an entire game has been played and completed. }
            m_bGameDead := True;
        end;
    EnterName:
        begin
            { Display the congratulations screen. }
            PrepareNameEntryText;
            m_bEnteringName := True;

            { Establish default mappings for the game control object. }
            m_inpPlayer.inpGetInput(BART_TURN_LEFT).NotifyOnChange(Self);
            m_inpPlayer.inpGetInput(BART_TURN_RIGHT).NotifyOnChange(Self);
            m_inpPlayer.inpGetInput(BART_THRUST).NotifyOnChange(Self);
            m_inpPlayer.inpGetInput(BART_FIRE).NotifyOnChange(Self);
            m_inpPlayer.inpGetInput(BART_HYPERSPACE).NotifyOnChange(Self);
        end;
    else
    end;
end;

procedure TGameControl.ExitState(iChannel: Integer; pstCurrent: PState);
var
    szTagText: String;
    iRanking: Integer;
begin
    { Perform default processing. }
    inherited ExitState(iChannel, pstCurrent);

    { Perform any required cleanup processing. }
    case pstCurrent^.enumCommandState of
    EnterName:
        Begin
            { Signal that we are no longer accepting tag value input. }
            m_bEnteringName := False;

            { Remove ourselves from input notification. }
            m_inpPlayer.inpGetInput(BART_TURN_LEFT).RemoveNotification(Self);
            m_inpPlayer.inpGetInput(BART_TURN_RIGHT).RemoveNotification(Self);
            m_inpPlayer.inpGetInput(BART_THRUST).RemoveNotification(Self);
            m_inpPlayer.inpGetInput(BART_FIRE).RemoveNotification(Self);
            m_inpPlayer.inpGetInput(BART_HYPERSPACE).RemoveNotification(Self);

            { Get the player's tag. }
            szTagText := m_voTagChar[0].szText + m_voTagChar[1].szText + m_voTagChar[2].szText;

            { Create a high score list object and store the player's new ranking. }
            iRanking := m_hslHighScores.iSetNewHighScore(m_scrScoreBoard.iCurrentScore, szTagText);
            m_hslHighScores.HighlightScore(iRanking);
            m_voHiScoreText.szText := Format('Hi %6.6d', [m_hslHighScores.lGetNthScore(1)]);

            { Show the new score for a few seconds before ending the game cycle. }
            AddToStateSequence(m_iStateChannel1, Pause, 1.0, 0);
            AddToStateSequence(m_iStateChannel1, HighScore, 0, 0);
            AddToStateSequence(m_iStateChannel1, Pause, 15, 0);
            AddToStateSequence(m_iStateChannel1, GameDead, 0, 0);
            ExitCurrentState(m_iStateChannel1);
        end;
    GameOver:
        begin
            { Create a high score list object and get the ranking for the score }
            { that this player acheived. }
            iRanking := m_hslHighScores.iRankScore(m_scrScoreBoard.iCurrentScore);

            { If the player ranked in the top ten then get his name for the }
            { high score table.  Otherwise, just go back to the splash screen. }
            if (iRanking <= 10) then
                AddToStateSequence(m_iStateChannel1, EnterName, 0, 0)
            else
                AddToStateSequence(m_iStateChannel1, Splash, 0, 0);

            { End the current state. }
            ExitCurrentState(iChannel);
        end;
    else
    end;
end;

procedure TGameControl.ReviveBart;
begin
    { The current bart is now NULL until the next one is generated. }
    m_voBart := nil;

    { If there are Barts left then use one.  Otherwise, the game is over! }
    if m_scrScoreBoard.iBartCount > 0 then
    begin
        AddToStateSequence(m_iStateChannel1, Pause, 2.0, 0);
        AddToStateSequence(m_iStateChannel1, GenerateBart, 0, 0);
    end
    else
    begin
        AddToStateSequence(m_iStateChannel1, Pause, 1.0, 0);
        AddToStateSequence(m_iStateChannel1, GameOver, 0.5, 0);
    end;
end;

procedure TGameControl.CheckLevelProgression;
var
    iEnemyCount: Integer;
    iObjectIndex: Integer;
    objThis: TVisibleObject;
    iBartsLeft: Integer;
    fLevelEndTime: Real;
    fTotalLevelTime: Real;
begin
    { If we've already moved to the next level but are waiting for the next }
    { asteroid field to be generated then skip out of here right away. }
    if m_bPendingAsteroidCreation = True then
        Exit;

    { Count the number of asteroids currently on the screen. }
    iEnemyCount := 0;
    for iObjectIndex := 0 to (m_lstUniverse.Count - 1) do
    begin
        { Get a reference to the next object in the list. }
        objThis := m_lstUniverse.Items[iObjectIndex];

        { Increment the asteroid count. }
        if (objThis.bObjectDead = False) and ((objThis is TAsteroid) or (objThis is TAlien)) then
            Inc(iEnemyCount);
    end;

    { Increase the probability of a visiting alien as rocks are destroyed. }
    if (iEnemyCount > 0) then
        m_fAlienProbability := 0.0010 + (m_iCurrentLevel * 0.0004) / iEnemyCount;

    { Don't progress to the next level until all of the Asteroids and Aliens
      have been killed. }
    if (iEnemyCount > 0) then
        Exit;

    { Compute a bonus based upon how long the user took to clear the asteroids. }
    fLevelEndTime := fGetNow;
    fTotalLevelTime := fLevelEndTime - m_fLevelStartTime;
    m_iLevelBonus := 1000 - (Trunc(fTotalLevelTime) * 10);
    if (m_iLevelBonus < 0) then
        m_iLevelBonus := 0;

    { Move to the next level. }
    Inc(m_iCurrentLevel);
    m_fAlienProbability := 0.0;
    m_iAliensOnLevel := 0;

    { Pause for a seconds, then display the completion message. }
    AddToStateSequence(m_iStateChannel1, Pause, 1.0, 0);
    AddToStateSequence(m_iStateChannel1, DisplayLevel, 2, 0);

    { If a bonus was earned then show how much was given. }
    if m_iLevelBonus > 0 then
    begin
        AddToStateSequence(m_iStateChannel1, Pause, 1.0, 0);
        AddToStateSequence(m_iStateChannel1, DisplayBonus, 2, 0);
        AddToStateSequence(m_iStateChannel1, Pause, 0.5, 0);
    end;

    { Finally, wait for asecond while asteroids are generated. }
    AddToStateSequence(m_iStateChannel1, Pause, 2.5, 0);
    AddToStateSequence(m_iStateChannel1, GenerateAsteroids, 0 ,0);
    m_bPendingAsteroidCreation := True;
end;

procedure TGameControl.PrepareNameEntryText;
begin
    { Set up the current cursor position. }
    m_iCharPos := 0;
    m_szCurrentChar := 'A';

    { Set up the initial tag string. }
    m_voTagChar[0] := NewTextObject('A', clYellow, False, iMidScreenX - 30, 230, 5.0, GROUP_HI_SCORE);
    m_voTagChar[1] := NewTextObject('_', clWhite, False, iMidScreenX, 230, 5.0, GROUP_HI_SCORE);
    m_voTagChar[2] := NewTextObject('_', clWhite, False, iMidScreenX + 30, 230, 5.0, GROUP_HI_SCORE);

    { Position the first line of text. }
    NewTextObject('Congratulations!', clYellow, False, iMidScreenX, 70, 3.0, GROUP_HI_SCORE);
    NewTextObject('Your score has distinguished you', clWhite, False, iMidScreenX, 120, 2.0, GROUP_HI_SCORE);
    NewTextObject('as one of our bravest pilots.', clWhite, False, iMidScreenX, 140, 2.0, GROUP_HI_SCORE);
    NewTextObject('Please enter your tag below:', clWhite, False, iMidScreenX, 180, 2.0, GROUP_HI_SCORE);
end;

procedure TGameControl.PerformLetterMovement(iAmount: Integer);
var
    szTagText: String;
    cLastChar: Char;
    iIndex: Integer;
begin
    { Perform the appropriate letter movement. }
    cLastChar := m_szCurrentChar;
    m_szCurrentChar := char(ord(m_szCurrentChar) + iAmount);

    { Jump out if the letter has not changed. }
    if cLastChar = m_szCurrentChar then
        Exit;

    { Make sure that the current character is within range. }
    if m_szCurrentChar > 'Z' then
        m_szCurrentChar := 'A';

    if m_szCurrentChar < 'A' then
        m_szCurrentChar := 'Z';

    { Construct the new string, padding it on the right with underscores. }
    ResetTagColors;
    m_voTagChar[m_iCharPos].szText := m_szCurrentChar;
    m_voTagChar[m_iCharPos].clrColor := clYellow;
end;

procedure TGameControl.PerformCursorMovement(iAmount: Integer);
begin
    { Perform cursor movement. }
    if (m_iCharPos > 0) or (iAmount > 0) then
        IncrementCursorPos(iAmount);
end;

procedure TGameControl.IncrementCursorPos(iAmount: Integer);
var
    szTemp: String[10];
begin
    m_iCharPos := m_iCharPos + iAmount;

    { If the user scrolled past the last letter then use the tag as-is. }
    if m_iCharPos = 3 then
    begin
        { Convert all underscores to spaces. }
        if m_voTagChar[0].szText = '_' then
            m_voTagChar[0].szText := ' ';
        if m_voTagChar[1].szText = '_' then
            m_voTagChar[1].szText := ' ';
        if m_voTagChar[2].szText = '_' then
            m_voTagChar[2].szText := ' ';

        ExitCurrentState(0);
        Exit;
    end;

    { Reset all tag letter colors. }
    ResetTagColors;

    { Stash the current letter in the position we're moving off of.. }
    m_voTagChar[m_iCharPos].szText := m_szCurrentChar;
    m_voTagChar[m_iCharPos].clrColor := clYellow;
end;

procedure TGameControl.ResetTagColors;
begin
    m_voTagChar[0].clrColor := clWhite;
    m_voTagChar[1].clrColor := clWhite;
    m_voTagChar[2].clrColor := clWhite;
end;

procedure TGameControl.PrepareCredits;
var
    txtNew: TTextObject;
    voObject: TVisibleObject;
begin
     { The credits will be "escorted" by a few barts! }
     { Bart #1! }
     voObject := TVisibleObject.Create(m_lstUniverse, GROUP_CREDITS);
     voObject.IncludeShape('Bart', 0, 0, clRed, True);
     voObject.mtrxTransform.fSpeed := 2.0;
     voObject.mtrxTransform.SetTranslation(100, 350);
     m_lstUniverse.Add(voObject);

     { Bart #2! }
     voObject := TVisibleObject.Create(m_lstUniverse, GROUP_CREDITS);
     voObject.IncludeShape('Bart', 0, 0, clRed, True);
     voObject.mtrxTransform.fSpeed := 2.0;
     voObject.mtrxTransform.SetTranslation(400, 350);
     m_lstUniverse.Add(voObject);

    { Position the credits off the screen and let them scroll into view. }
    txtNew := NewTextObject('CREDITS', clWhite, True, iMidScreenX, 350, 3.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('Programming ...  Joe Kessler    ', clWhite, False, iMidScreenX, 450, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('                 Jeff Grammer   ', clWhite, False, iMidScreenX, 475, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('Design ...       Joe Kessler    ', clWhite, False, iMidScreenX, 650, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('Play Testing ... Joe Bielawski  ', clWhite, False, iMidScreenX, 800, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('                 Sean Black     ', clWhite, False, iMidScreenX, 825, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('                 Andy Bonefas   ', clWhite, False, iMidScreenX, 850, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('                 Jim Flury      ', clWhite, False, iMidScreenX, 875, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('                 Flip Nehrt     ', clWhite, False, iMidScreenX, 900, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('                 Rich Otruba    ', clWhite, False, iMidScreenX, 925, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('                 Alycia Paulus  ', clWhite, False, iMidScreenX, 950, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('                 Chuong Pham    ', clWhite, False, iMidScreenX, 975, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('                 Dan Verdeyen   ', clWhite, False, iMidScreenX, 1000, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;

    txtNew := NewTextObject('                 Lee Yelverton  ', clWhite, False, iMidScreenX, 1025, 2.0, GROUP_CREDITS);
    txtNew.mtrxTransform.fDirection := 0;
    txtNew.mtrxTransform.fSpeed := 2.0;
end;

procedure TGameControl.CreateCollisionLists;
var
    iIndex: Integer;
    voObject: TVisibleObject;
    iCollisionID: Integer;
begin
    { Clear out the current contents of the lists. }
    m_alstCollisionLists[0].Clear;
    m_alstCollisionLists[1].Clear;

    { Separate objects into two categories: Bart and Enemy. }
    for iIndex := 0 to (m_lstUniverse.Count - 1) do
    begin
        { Get a reference to the next object in the list. }
        voObject := m_lstUniverse.Items[iIndex];
        iCollisionID := voObject.iCollisionID;

        { Put the object into the appropriate list.  }
        if (iCollisionID >= 0) and (iCollisionID < 2) then
            m_alstCollisionLists[iCollisionID].Add(voObject)
        else
            if (iCollisionID = 2) then
                begin
                m_alstCollisionLists[0].Add(voObject);
                m_alstCollisionLists[1].Add(voObject);
                end;
    end;
end;

function TGameControl.NewTextObject(szText: String;
                                    clrText: TColor;
                                    bFlashing: Boolean;
                                    iPosX, iPosY: Integer;
                                    fScale: Real;
                                    iGroupID: Integer): TTextObject;
begin
    { Create a new text object and add it to the object list. }
    Result := TTextObject.Create(m_lstUniverse, szText, clrText);
    Result.mtrxTransform.SetTranslation(iPosX, iPosY);
    Result.mtrxTransform.fScale := fScale;
    Result.iGroupID := iGroupID;

    if bFlashing = True then
        Result.BeginFlashing;

    { Add the new text object to the object list. }
    m_lstUniverse.Add(Result);
end;

procedure TGameControl.FreeObjects(iGroupID: Integer);
var
    iIndex: Integer;
    voObject: TVisibleObject;
begin
    { Iterate through the entire list of objects. }
    for iIndex := 0 to (m_lstUniverse.Count - 1) do
    begin
        voObject := m_lstUniverse.Items[iIndex];

        { Kill objects that (1) aren't dead, and (2) match the given control }
        { group ID. }
        if (voObject.bObjectDead = False) and
            ((iGroupID = GROUP_ANY) or (voObject.iGroupID = iGroupID)) then
            voObject.KillObject(False, 0, 0, 0, 0);
    end;
end;

procedure TGameControl.DisplayFloatingText(szTextString: String; iPosX, iPosY: Integer);
var
    txtNew: TTextObject;
begin
    { Create a new text object with the given string. }
    txtNew := NewTextObject(szTextString, clWhite, True, iPosX, iPosY, 1.0, -1);

    { Make the string float upward slowly on the screen. }
    txtNew.mtrxTransform.fDirection := 0.0;
    txtNew.mtrxTransform.fSpeed := 1;

    { Take down text amount after a while. }
    txtNew.AddToStateSequence(Pause, 1.5, 0);
    txtNew.AddToStateSequence(Die, 0, 0);
end;

procedure TGameControl.SetGamePause(bPaused: Boolean);
begin
    { If the state is already set to the given value, do nothing now. }
    if (m_bGamePaused = bPaused) then
        Exit
    else
        m_bGamePaused := bPaused;

    { Toggle the current game pause state. }
    if m_bGamePaused = True then
        EnterPauseState
    else
        ExitPauseState;
end;

procedure TGameControl.EnterPauseState;
begin
    { Display the GAME PAUSED text. }
    NewTextObject('Game Paused', clWhite, False, iMidScreenX, 100, 5.0, GROUP_PAUSE);
    NewTextObject('Press F3 to continue...', clRed, False, iMidScreenX, 200, 3.0, GROUP_PAUSE);

    { Set the game pause state and erase the current contents of the screen. }
    m_bGamePaused := True;
    EraseFrame;
end;

procedure TGameControl.ExitPauseState;
begin
    { Free pause-state items, and then resume normal object processing. }
    FreeObjects(GROUP_PAUSE);
    m_bGamePaused := False;
end;

procedure TGameControl.DefineGameShapes;
begin
     { Start by defining the shapes of digits 0-9. }
     g_slShapeLib.DefineShape('Digit0',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  0),
                                    FPoint( 2,  0), FPoint(-2, -2), FPoint( 2, -2) ],
                                 [ Edge(0, 1), Edge(1, 5), Edge(5, 4), Edge(4, 0) ]);
     g_slShapeLib.DefineShape('Digit1',
                                 [ FPoint(0,  2), FPoint(0,  -2) ],
                                 [ Edge(0, 1) ]);
     g_slShapeLib.DefineShape('Digit2',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  0),
                                    FPoint( 2,  0), FPoint(-2, -2), FPoint( 2, -2) ],
                                 [ Edge(0, 1), Edge(1, 3), Edge(3, 2), Edge(2, 4), Edge(4, 5) ]);
     g_slShapeLib.DefineShape('Digit3',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  0),
                                    FPoint( 2,  0), FPoint(-2, -2), FPoint( 2, -2) ],
                                 [ Edge(0, 1), Edge(1, 5), Edge(5, 4), Edge(3, 2) ]);
     g_slShapeLib.DefineShape('Digit4',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  0),
                                    FPoint( 2,  0), FPoint(-2, -2), FPoint( 2, -2) ],
                                 [ Edge(0, 2), Edge(2, 3), Edge(1, 5) ]);
     g_slShapeLib.DefineShape('Digit5',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  0),
                                    FPoint( 2,  0), FPoint(-2, -2), FPoint( 2, -2) ],
                                 [ Edge(1, 0), Edge(0, 2), Edge(2, 3), Edge(3, 5), Edge(5, 4) ]);
     g_slShapeLib.DefineShape('Digit6',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  0),
                                    FPoint( 2,  0), FPoint(-2, -2), FPoint( 2, -2) ],
                                 [ Edge(0, 4), Edge(4, 5), Edge(5, 3), Edge(3, 2) ]);
     g_slShapeLib.DefineShape('Digit7',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  0),
                                    FPoint( 2,  0), FPoint(-2, -2), FPoint( 2, -2) ],
                                 [ Edge(0, 1), Edge(1, 5) ]);
     g_slShapeLib.DefineShape('Digit8',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  0),
                                    FPoint( 2,  0), FPoint(-2, -2), FPoint( 2, -2) ],
                                 [ Edge(0, 1), Edge(1, 5), Edge(5, 4), Edge(4, 0), Edge(2, 3) ]);
     g_slShapeLib.DefineShape('Digit9',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  0),
                                    FPoint( 2,  0), FPoint(-2, -2), FPoint( 2, -2) ],
                                 [ Edge(0, 1), Edge(1, 5), Edge(3, 2), Edge(2, 0) ]);

     { Then, define the each letter of the alphabet. }
     g_slShapeLib.DefineShape('LetterA',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(8, 2), Edge(2, 10), Edge(10, 3), Edge(3, 9), Edge(4, 5) ]);
     g_slShapeLib.DefineShape('LetterB',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(8, 0), Edge(8, 9), Edge(9, 1), Edge(1, 0), Edge(4, 5) ]);
     g_slShapeLib.DefineShape('LetterC',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(1, 0), Edge(0, 8), Edge(8, 9) ]);
     g_slShapeLib.DefineShape('LetterD',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(1, 0), Edge(0, 8), Edge(8, 9), Edge(9, 1) ]);
     g_slShapeLib.DefineShape('LetterE',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(1, 0), Edge(0, 8), Edge(8, 9), Edge(4, 5) ]);
     g_slShapeLib.DefineShape('LetterF',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(1, 0), Edge(0, 8), Edge(4, 5) ]);
     g_slShapeLib.DefineShape('LetterG',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0) ],
                                 [ Edge(1, 0), Edge(0, 8), Edge(8, 9), Edge(9, 5), Edge(5,  12) ]);
     g_slShapeLib.DefineShape('LetterH',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(0, 8), Edge(1, 9), Edge(4, 5) ]);
     g_slShapeLib.DefineShape('LetterI',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(1, 0), Edge(9, 8), Edge(10, 11) ]);
     g_slShapeLib.DefineShape('LetterJ',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(0, 1), Edge(10, 11), Edge(11, 8) ]);
     g_slShapeLib.DefineShape('LetterK',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(0, 8), Edge(4, 12), Edge(12, 1), Edge(12, 9) ]);
     g_slShapeLib.DefineShape('LetterL',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(0, 8), Edge(8, 9) ]);
     g_slShapeLib.DefineShape('LetterM',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(8, 0), Edge(0, 12), Edge(12, 1), Edge(1, 9) ]);
     g_slShapeLib.DefineShape('LetterN',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(8, 0), Edge(0, 9), Edge(9, 1) ]);
     g_slShapeLib.DefineShape('LetterO',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(0, 1), Edge(1, 9), Edge(9, 8), Edge(8, 0) ]);
     g_slShapeLib.DefineShape('LetterP',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(8, 0), Edge(0, 1), Edge(1, 5), Edge(5, 4) ]);
     g_slShapeLib.DefineShape('LetterQ',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(0, 1), Edge(1, 9), Edge(9, 8), Edge(8, 0), Edge(9, 12) ]);
    g_slShapeLib.DefineShape('LetterR',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(8, 0), Edge(0, 1), Edge(1, 5), Edge(5, 4), Edge(12, 9) ]);
     g_slShapeLib.DefineShape('LetterS',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(1, 0), Edge(0, 4), Edge(4, 5), Edge(5, 9), Edge(9, 8) ]);
     g_slShapeLib.DefineShape('LetterT',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(1, 0), Edge(10, 11) ]);
     g_slShapeLib.DefineShape('LetterU',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(0, 8), Edge(8, 9), Edge(9, 1) ]);
     g_slShapeLib.DefineShape('LetterV',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(0, 6), Edge(6, 11), Edge(11, 7), Edge(7, 1) ]);
     g_slShapeLib.DefineShape('LetterW',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(0, 8), Edge(8, 12), Edge(12, 9), Edge(9, 1) ]);
     g_slShapeLib.DefineShape('LetterX',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(0, 9), Edge(8, 1) ]);
     g_slShapeLib.DefineShape('LetterY',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(11, 12), Edge(0, 12), Edge(1, 12) ]);
     g_slShapeLib.DefineShape('LetterZ',
                                 [ FPoint(-2,  2), FPoint( 2,  2), FPoint(-2,  1), FPoint(2, 1),
                                    FPoint(-2,  0), FPoint( 2,  0), FPoint(-2, -1), FPoint(2, -1),
                                    FPoint(-2, -2), FPoint( 2, -2), FPoint( 0,  2), FPoint(0, -2), FPoint(0, 0)  ],
                                 [ Edge(1, 0), Edge(1, 8), Edge(8, 9) ]);

     { ... And punctuation characters ... }
     g_slShapeLib.DefineShape('Period',
                                 [ FPoint(-2,  -1.5), FPoint(-1.5, -1.5), FPoint(-1.5, -2), FPoint(-2, -2) ],
                                 [ Edge(0, 1), Edge(1, 2), Edge(2, 3), Edge(3, 0) ]);
     g_slShapeLib.DefineShape('UnderScore',
                                 [ FPoint(-2,  -2), FPoint(2, -2) ],
                                 [ Edge(0, 1) ]);
     g_slShapeLib.DefineShape('Colon',
                                 [ FPoint(-2, 1),  FPoint(-1.5, 1),  FPoint(-1.5, 0.5),  FPoint(-2, 0.5),
                                    FPoint(-2, -1), FPoint(-1.5, -1), FPoint(-1.5, -0.5), FPoint(-2, -0.5) ] ,
                                 [ Edge(0, 1), Edge(1, 2), Edge(2, 3), Edge(3, 0),
                                    Edge(4, 5), Edge(5, 6), Edge(6, 7), Edge(7, 4) ]);
     g_slShapeLib.DefineShape('Exclamation',
                                 [ FPoint(-2, 2),  FPoint(-2, -1),    FPoint(-2, -1.5),  FPoint(-2, -2) ],
                                 [ Edge(0, 1), Edge(2, 3) ]);

     { Define the shape of Bart and his engine flame. }
     g_slShapeLib.DefineShape('Bart',
                                 [ FPoint(-10, -10), FPoint(0,  -8), FPoint(10, -10), FPoint( 0, 10) ],
                                 [ Edge(0, 1), Edge(1, 2), Edge(2, 3), Edge(3, 0) ]);
     g_slShapeLib.DefineShape('Flame',
                                 [ FPoint(-5, -9), FPoint(0,  -15), FPoint(5, -9) ],
                                 [ Edge(0, 1), Edge(1, 2) ]);

     { Define the shape of an attacking alien spaceship. }
     g_slShapeLib.DefineShape('Alien',
                                 [ FPoint(-2, 2), FPoint(2, 2), FPoint(3, 0), FPoint( 2, -2),
                                    FPoint(-2, -2), FPoint(-3, 0) ],
                                 [ Edge(0, 1), Edge(1, 2), Edge(2, 3), Edge(3, 4), Edge(4, 5),  Edge(5, 0) ]);
end;

procedure TGameControl.DisplaySplashScreen;
begin
     { Start showing the game's splash screen. }
     AddToStateSequence(m_iStateChannel1, Splash, 0 ,0);
end;

procedure TGameControl.StartNewGame;
begin
     // Clear out the current event queue, and add a BeginGame command.
     ExitCurrentState(0);
     ClearStateSequence(0);
     AddToStateSequence(m_iStateChannel1, BeginGame, 0 ,0);
end;

procedure TGameControl.ProcessMessage(iCategory, iType: Integer; bValue0: Boolean; fValue1, fValue2: Real);
begin
    { Process input messages ONLY if we're in tag entry mode.  Notice that
      bValue0 is flag indicating whether the input was just fired, or just
      lifted. }
    if (m_bEnteringName = False) or (bValue0 = False) then
       Exit;

    { Process the input message according to its type. }
    if (iCategory = INPUT_DEVICE) then
        case iType of
            BART_TURN_RIGHT:
                PerformLetterMovement(1);

            BART_TURN_LEFT:
                PerformLetterMovement(-1);

            BART_FIRE:
                PerformCursorMovement(1);

            BART_HYPERSPACE:
                PerformCursorMovement(1);
        else
        end;
end;

end.
