/*
  TelecDLL.c - Telecurs hook DLL.

  Jason Hood, 17 to 22 February, 2003.

  v1.10 - 6 to 9 March, 2003:
    some applications hide dialogs (only closing them when they exit), so use
     SWP_HIDEWINDOW to restore the cursor position;
    instead of processing WM_INITDIALOG & WM_DESTROY messages, test the class
     when showing the window;
    move to the centre of the tabs of property sheets;
    added the OpenLog function and test the file is actually opened.
*/


#include <windows.h>
#include <stdio.h>
#include "Telecurs.h"


Settings* __declspec(dllexport) setting;

static HWND wnd;
static int  ctrl;


FILE* OpenLog( int, HWND, int );


// DLL initialisation: allocate or release the shared memory.
BOOL WINAPI __declspec(dllexport) LibMain( HINSTANCE dll, DWORD reason,
					   LPVOID reserved )
{
  static HANDLE map;

  switch (reason)
  {
    case DLL_PROCESS_ATTACH:
      map = CreateFileMapping( (HANDLE)0xFFFFFFFF, NULL, PAGE_READWRITE, 0,
			       sizeof(Settings), PNAME"Memory" );
      if (map == NULL) return FALSE;
      setting = MapViewOfFile( map, FILE_MAP_WRITE, 0, 0, 0 );
      if (setting == NULL)
      {
	CloseHandle( map );
	return FALSE;
      }
    break;

    case DLL_PROCESS_DETACH:
      UnmapViewOfFile( setting );
      CloseHandle( map );
    break;
  }

  return TRUE;
}


/*
  Determine the control to select:

    the tabs of a property sheet (if ctrl >= 3);
    the (first) default push button;
    the (last) "OK" push button;
    the first push button

  that is visible and enabled. If none of the above are present and there is a
  control with the WS_TABSTOP style, ctrl is set to 1.
*/
BOOL CALLBACK ChildWin( HWND hwnd, LPARAM lParam )
{
  char buf[32];
  LONG style;
  RECT r;

  GetClassName( hwnd, buf, sizeof(buf) );
  style = GetWindowLong( hwnd, GWL_STYLE );

  if (setting->log && lParam)
  {
    GetWindowRect( hwnd, &r );
    fprintf( (FILE*)lParam, "           %04X (%4d,%4d-%4d,%4d) %08lX %s\n",
	     hwnd, r.left, r.top, r.right, r.bottom, style, buf );
  }

  if ((style & WS_VISIBLE) && !(style & WS_DISABLED))
  {
    if (lstrcmpi( buf, "Button" ) == 0)
    {
      if (ctrl < 2)
      {
	style &= 0x0F;
	if (style == BS_DEFPUSHBUTTON)
	{
	  wnd  = hwnd;
	  ctrl = 2;
	}
	else if (style == BS_PUSHBUTTON)
	{
	  GetWindowText( hwnd, buf, sizeof(buf) );
	  if (lstrcmpi( buf, "OK"  ) == 0 ||
	      lstrcmpi( buf, "&OK" ) == 0 ||
	      lstrcmpi( buf, "O&K" ) == 0 ||
	      !wnd)
	  {
	    wnd = hwnd;
	  }
	}
      }
    }
    else if (lstrcmpi( buf, "SysTabControl32" ) == 0)
    {
      if (TabCtrl_GetItemCount( hwnd ) > 1)
      {
	wnd  = hwnd;
	ctrl = 3;
      }
    }
    else if (lstrcmpi( buf, "SysTabControl" ) == 0)
    {
      if (TabCtrl_GetItemCount( hwnd ) > 1)
      {
	wnd  = hwnd;
	ctrl = 4;
      }
    }
    else if (!ctrl && (style & WS_TABSTOP))
      ctrl = 1;
  }

  return TRUE;
}


// The hook: the function that does all the work.
LRESULT CALLBACK __declspec(dllexport) MsgHook( int nCode, WPARAM wParam,
							   LPARAM lParam )
{
  LRESULT    rc;
  CWPSTRUCT* msg;
  BOOL	enabled;
  RECT	r;
  POINT pt;
  UINT	show;
  LONG	style;
  FILE* file;
  char	buf[32];
  int	j;

  rc = CallNextHookEx( setting->hook, nCode, wParam, lParam );
  if (nCode != HC_ACTION)
    return rc;

  // Determine if ScrollLock should override the disabled setting.
  enabled = (setting->override) ? (GetKeyState( VK_SCROLL ) & 1) : 0;
  // Disabled and no ScrollLock or enabled and ScrollLock.
  enabled = !(setting->disabled ^ enabled);

  msg = (CWPSTRUCT*)lParam;
  if (msg->message == WM_WINDOWPOSCHANGED)
  {
    show = ((WINDOWPOS*)msg->lParam)->flags;
    if (show & SWP_SHOWWINDOW)
    {
      if (enabled)
      {
	GetClassName( msg->hwnd, buf, sizeof(buf) );
	if (lstrcmp( buf, "#32770" ) == 0)
	{
	  style = GetWindowLong( msg->hwnd, GWL_STYLE );
	  // Ignore the (probable) pages of a property sheet.
	  if (!(style & WS_CHILD))
	  {
	    if (setting->log)
	      file = OpenLog( setting->count, msg->hwnd, TRUE );

	    wnd  = 0;
	    ctrl = 0;
	    EnumChildWindows( msg->hwnd, (WNDPROC)ChildWin, (LPARAM)file );
	    if (!wnd)
	    {
	      /*
		There are no pushbuttons or tabs, so let's make a guess as to
		where to move:
		  if there are usable controls (ie. something with WS_TABSTOP)
		    and it's not modeless (assuming a modeless dialog is
		    informational), move to the centre;
		  if there is a system menu (and assuming there's a caption),
		    move to the close button;
		  if it's modal, move to the bottom-centre (probably a splash-
		    screen, click-to-close);
		  ignore it (probably an information window).
	      */
	      GetWindowRect( msg->hwnd, &r );
	      if (ctrl && (style & DS_MODALFRAME))
	      {
		pt.x = (r.left + r.right) / 2;
		pt.y = (r.top + r.bottom) / 2;
	      }
	      else if (style & WS_SYSMENU)
	      {
		pt.x = r.right - GetSystemMetrics( SM_CXDLGFRAME )
			       - GetSystemMetrics( SM_CXSIZE ) / 2;
		pt.y = r.top + GetSystemMetrics( SM_CYDLGFRAME )
			     + GetSystemMetrics( SM_CYSIZE ) / 2;
	      }
	      else if (style & DS_MODALFRAME)
	      {
		//pt.x = (r.left + 3 * r.right) / 4;
		//pt.y = (r.top + 3 * r.bottom) / 4;
		pt.x = (r.left + r.right) / 2;
		pt.y = r.bottom - GetSystemMetrics( SM_CYDLGFRAME );
	      }
	      else
	      {
		if (setting->log && file)
		{
		  fputs( "    Ignored (apparently modeless & control-less)\n",
			 file );
		  fclose( file );
		}
		goto hook_exit;
	      }
	    }
	    else
	    {
	      if (ctrl >= 3)	// Property sheet: centre of the tabs
	      {
		j = TabCtrl_GetRowCount( wnd );
		if (ctrl == 3)
		{
		  TabCtrl_GetItemRect( wnd, TabCtrl_GetItemCount( wnd )-1, &r );
		  if (j > 1)
		  {
		    pt.y = (j * (r.bottom - r.top)) / 2;
		    GetWindowRect( wnd, &r );
		    pt.x = (r.right - r.left) / 2;
		  }
		  else
		  {
		    pt.x = r.right / 2;
		    pt.y = (r.top + 3 * r.bottom) / 4;
		  }
		  ClientToScreen( wnd, &pt );
		}
		else
		{
		  // Can't get the rectangle of a SysTabControl,
		  // so just make a guess.
		  GetWindowRect( wnd, &r );
		  if (j > 1)
		  {
		    pt.x = (r.left + r.right) / 2;
		    pt.y = r.top + 2 + j * 18 / 2;
		  }
		  else
		  {
		    pt.x = (3 * r.left + r.right) / 4;
		    pt.y = r.top + 2 + 18 * 3 / 4;
		  }
		}
	      }
	      else		// Pushbutton: a quarter up from the bottom
	      {
		GetWindowRect( wnd, &r );
		pt.x = (r.left + r.right) / 2;
		pt.y = (r.top + 3 * r.bottom) / 4;
	      }
	    }
	    if (setting->ret && setting->count < MAX_DLG)
	    {
	      setting->dlg[setting->count].wnd = msg->hwnd;
	      GetCursorPos( &setting->dlg[setting->count++].pos );
	    }
	    SetCursorPos( pt.x, pt.y );

	    if (setting->log && file)
	    {
	      fprintf( file, "         * %04X (%4d,%4d)\n",
		       (wnd) ? wnd : msg->hwnd, pt.x, pt.y );
	      fclose( file );
	    }
	  }
	}
      }
    }
    else if ((show & SWP_HIDEWINDOW) && setting->ret && setting->count)
    {
      j = setting->count;
      while (setting->dlg[--j].wnd != msg->hwnd)
	if (j == 0)
	  goto hook_exit;

      if (enabled)
	SetCursorPos( setting->dlg[j].pos.x, setting->dlg[j].pos.y );

      if (j != --setting->count)
      {
	memcpy( setting->dlg + j, setting->dlg + j + 1,
		(setting->count - j) * sizeof(Dlg) );
      }

      if (setting->log)
	OpenLog( j, msg->hwnd, FALSE );
    }
  }
  hook_exit:

  return 0;
}


// Open the log file and write a message.
FILE* OpenLog( int num, HWND win, int show )
{
  FILE* file;
  char	buf[40];
  RECT	r;

  file = fopen( setting->logfile, "a" );
  if (file)
  {
    if (setting->ret)
      fprintf( file, "%2d:", num );
    else
      fputs( "   ", file );
    fprintf( file, " %s %04X", (show) ? "Opened" : "Closed", win );
    if (show)
    {
      GetWindowRect( win, &r );
      fprintf( file, " (%4d,%4d-%4d,%4d)", r.left, r.top, r.right, r.bottom );
    }
    GetWindowText( win, buf, sizeof(buf) );
    fprintf( file, " %s\n", buf );

    if (!show)
      fclose( file );
  }

  return file;
}
