/*****************************************************************************
 *                          S T I C K I E S                                  *
 *                         Version 1.6 , updated May-1999                    *
 *                        Written by Steven De Toni                          *
 *****************************************************************************/

#include <windows.h>
#include <mmsystem.h>
#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <commdlg.h>
#include <fcntl.h>
#include <time.h>
#include <mapi.h>      

#include "sticky.h"
#include "printfuncs.h"
#include "filefuncs.h"
#include "mapifuncs.h"
#include "ddemlfuncs.h"
#include "findfuncs.h"
#include "alarmfuncs.h"
#include "mathparser.h"
#include "ball.h"

// Global Variables 
WNDCLASS  StickClassRec;
HWND      ParentStarter;
HANDLE    HInst;
HACCEL    HStickyAccel; // Used in sub-classed edit control to fire sticky menus.
int       StickyNum;

// Global control variables         
BOOL      AutoTextCopy              = FALSE;// Automatically copy text after selection in edit control
BOOL      LatchActive               = TRUE; // Latch to the next window 
int       LatchPixelRange           = 10;    
BOOL      MAPIActive                = TRUE; // Mail-to option available ?
BOOL      SoundActive               = TRUE; // Sound on or off
int       AboutHighScore            = 0;    // Memo hit game in about screen
char      RollupSnd[MAXSNDFILELEN]  = STRROLLUPSND;
char      UnrollSnd[MAXSNDFILELEN]  = STRUNROLLSND;
char      AlarmSnd[MAXSNDFILELEN]   = STRALARMSND;

// Used in LoadSticky to set the inital sticky index upon creation
int       Sticky_IDX_WM_CREATE;

// Find / Replace window globals
HWND      HFndRplDialog   = NULL;
UINT      MFndRplReGMsg   = RegisterWindowMessage (FINDMSGSTRING);

// *** Sticky Global Structures ***
StickyRec StickyRecords[STICKRECNUM+1];

// Colour Scheme data
ColScheme ColourSchemes[MAXCOLSCHEMENUM];
int       ColSchemeNum = 0;
// ----------------------------------------------------------------------------

void SetNULLSForDefaultSticky (void)
{
    // Make sure certain fields are always NULL.
    StickyRecords[STICKRECNUM].editControl = NULL;
    StickyRecords[STICKRECNUM].stickyWnd   = NULL;
    StickyRecords[STICKRECNUM].hFont       = NULL;
    StickyRecords[STICKRECNUM].backGH      = NULL;

    // set txt info to null
    StickyRecords[STICKRECNUM].userMemo.pString = NULL;
    StickyRecords[STICKRECNUM].stickyWnd        = NULL;
    StickyRecords[STICKRECNUM].winCaption[0]    = '\0';
}

// ----------------------------------------------------------------------------

void LoadMainStickyPrefs (void)
{    
    char          profileBuff[MAXSNDFILELEN];
    HKEY          hStickyInfo = NULL;
    unsigned long type        = REG_SZ;
    unsigned long len         = MAXSNDFILELEN;
    LONG          result      = RegOpenKey (HKEY_CURRENT_USER, (LPCTSTR)REGOPENLOC, &hStickyInfo);

    if (result != 0)
        return;

    // Load Misc Stick Glabal Switches
    result = RegQueryValueEx(hStickyInfo, 
                             (LPTSTR)REGMISCINFO, 
                             NULL, 
                             &type, 
                             (LPBYTE)profileBuff, &len);
 
    if (result != 0)
        return;
   
    sscanf (profileBuff, "%d %d %d %d %d", &LatchActive, &LatchPixelRange, &SoundActive, &AutoTextCopy, &AboutHighScore);

    // Load Rollup Wav Sound Effect
    len    = MAXSNDFILELEN;
    result = RegQueryValueEx(hStickyInfo, 
                             (LPTSTR)REGROLLUPWAVE, 
                             NULL, 
                             &type, 
                             (LPBYTE)profileBuff, &len);
 
    if (result != 0)
        return;

    
    sscanf (profileBuff, "%s", RollupSnd);

    // Load Unroll Wav Sound Effect    
    len    = MAXSNDFILELEN;
    result = RegQueryValueEx(hStickyInfo, 
                             (LPTSTR)REGUNROLLWAVE, 
                             NULL, 
                             &type, 
                             (LPBYTE)profileBuff, &len);
 
    if (result != 0)
        return;

    sscanf (profileBuff, "%s", UnrollSnd);

    // Load Alarm Wav Sound Effect
    len    = MAXSNDFILELEN;
    result = RegQueryValueEx(hStickyInfo, 
                             (LPTSTR)REGALARMWAVE, 
                             NULL, 
                             &type, 
                             (LPBYTE)profileBuff, &len);
 
    if (result != 0)
        return;

    sscanf (profileBuff, "%s", AlarmSnd);
    
    RegCloseKey(hStickyInfo); 
}

// ----------------------------------------------------------------------------

void SaveMainStickyPrefs (void)
{    
    char            profileBuff[MAXSNDFILELEN];
    HKEY            hStickyInfo = NULL;
    LONG            result      = RegCreateKey (HKEY_CURRENT_USER, (LPCTSTR)REGOPENLOC, &hStickyInfo);

    if (result != 0)
    {
        MessageBox (NULL, "Unable to open/create registry key!" , "Registry Error", MB_ICONSTOP | MB_OK);
        return;
    }

    // Save Misc Stick Glabal Switches
    sprintf (profileBuff, "%d %d %d %d %d", LatchActive, LatchPixelRange, SoundActive, AutoTextCopy, AboutHighScore);

    RegSetValueEx (hStickyInfo, 
                   (LPCTSTR)REGMISCINFO, 
                   NULL, 
                   REG_SZ, 
                   (CONST BYTE*)profileBuff, 
                   (DWORD) strlen (profileBuff));

    // Save Rollup Wav Sound Effect
    sprintf (profileBuff, "%s", RollupSnd);

    RegSetValueEx (hStickyInfo, 
                   (LPCTSTR)REGROLLUPWAVE, 
                   NULL, 
                   REG_SZ, 
                   (CONST BYTE*)profileBuff, 
                   (DWORD) strlen (profileBuff));

    // Save Unroll Wav Sound Effect
    sprintf (profileBuff, "%s", UnrollSnd);

    RegSetValueEx (hStickyInfo, 
                   (LPCTSTR)REGUNROLLWAVE, 
                   NULL, 
                   REG_SZ, 
                   (CONST BYTE*)profileBuff, 
                   (DWORD) strlen (profileBuff));

    // Save Alarm Wav Sound Effect
    sprintf (profileBuff, "%s", AlarmSnd);

    RegSetValueEx (hStickyInfo, 
                   (LPCTSTR)REGALARMWAVE, 
                   NULL, 
                   REG_SZ, 
                   (CONST BYTE*)profileBuff, 
                   (DWORD) strlen (profileBuff));

    RegCloseKey(hStickyInfo); 
}

// ----------------------------------------------------------------------------

void InitStickyRecords (void)
{
    // Nuke all the stickies
    memset(StickyRecords, 0, sizeof (StickyRec)*STICKRECNUM);
}

// ----------------------------------------------------------------------------

// returns the idx of next free item in sticky record structure, or -1 if
// none available
int GetNextStickyFree (void)
{
    for (int test = 0; test < STICKRECNUM-1; test++)
    {
        if (StickyRecords[test].stickyWnd == NULL)
            return test;
    }
    return -1;
}

// ----------------------------------------------------------------------------

// returns a edit control of sticky that has associated window handle.
HWND GetStickyEditControl (HWND Owner)
{
    for (int test = 0; test < STICKRECNUM; test++)
    {
        if (StickyRecords[test].stickyWnd == Owner)
            return StickyRecords[test].editControl;
    }

    return NULL;
}

// ----------------------------------------------------------------------------

int GetWndIdx (HWND Owner)
{
    for (int test = 0; test < STICKRECNUM; test++)
    {
        if (StickyRecords[test].stickyWnd == Owner)
            return test;
    }

    return -1;
}

// ----------------------------------------------------------------------------

void SaveStickyWinPos (int idx)
{  
    GetWindowRect(StickyRecords[idx].stickyWnd, &StickyRecords[idx].winDim);

    if (StickyRecords[idx].rolledUp != FALSE)
        StickyRecords[idx].winDim.bottom = StickyRecords[idx].winDim.top + 
                                           (StickyRecords[idx].unrolledDim.bottom - StickyRecords[idx].unrolledDim.top);        
}

// ----------------------------------------------------------------------------

void StoreStickyData (int idx, BOOL fullStore)
{
    SaveStickyWinPos (idx);

    if (fullStore != FALSE)
    {
        // Get Window text
        StickyRecords[idx].userMemo.strLen = (int) SendMessage(StickyRecords[idx].editControl, WM_GETTEXTLENGTH, 0, 0L);

        if (StickyRecords[idx].userMemo.strLen >= 0)
        {
            StickyRecords[idx].userMemo.pString = new char[StickyRecords[idx].userMemo.strLen+1];

            if (StickyRecords[idx].userMemo.pString != NULL)
               SendMessage(StickyRecords[idx].editControl, WM_GETTEXT, StickyRecords[idx].userMemo.strLen+1,
                           (LPARAM)(LPSTR) StickyRecords[idx].userMemo.pString);
        }
        else
            StickyRecords[idx].userMemo.pString = NULL;
    }
}

// ----------------------------------------------------------------------------

// *** DESTRUCTOR ***
void NukeRec (HWND Owner)
{
    for (int test = 0; test < STICKRECNUM; test++)
    {
        if (StickyRecords[test].stickyWnd == Owner)
        {
            StickyRecords[test].stickyWnd = StickyRecords[test].editControl = NULL;

            if (StickyRecords[test].hFont != NULL)
                DeleteObject (StickyRecords[test].hFont);

            if (StickyRecords[test].backGH != NULL)
                DeleteObject (StickyRecords[test].backGH);
            return;
        }
    }
}

// ----------------------------------------------------------------------------

void CleanUp (void)
{
    for (int test = 0; test < STICKRECNUM; test++)
    {
        if (StickyRecords[test].stickyWnd != NULL)
        {
            StickyRecords[test].stickyWnd = StickyRecords[test].editControl = NULL;

            if (StickyRecords[test].hFont != NULL)
                DeleteObject (StickyRecords[test].hFont);

            if (StickyRecords[test].backGH != NULL)
                DeleteObject (StickyRecords[test].backGH);
        }
    }

}

// ----------------------------------------------------------------------------

inline void CheckWindowBoundrys (int idx, int inset)
{
    // Make sure sticky window is in the bounds of the screen
    if ((StickyRecords[idx].winDim.left + (StickyRecords[idx].winDim.right - StickyRecords[idx].winDim.left)) < inset) 
    {
        StickyRecords[idx].winDim.left  = -((StickyRecords[idx].winDim.right - StickyRecords[idx].winDim.left)-inset);
        StickyRecords[idx].winDim.right = inset;
    }

    if ((StickyRecords[idx].winDim.top + (StickyRecords[idx].winDim.bottom - StickyRecords[idx].winDim.top)) < inset) 
    {
        StickyRecords[idx].winDim.top    = -((StickyRecords[idx].winDim.bottom - StickyRecords[idx].winDim.top)-inset);
        StickyRecords[idx].winDim.bottom = inset;
    }

    if ((StickyRecords[idx].winDim.left) > GetSystemMetrics(SM_CXSCREEN)-inset)
        StickyRecords[idx].winDim.left = GetSystemMetrics(SM_CXSCREEN)-inset;

    if ((StickyRecords[idx].winDim.top) > GetSystemMetrics(SM_CYSCREEN)-inset)
        StickyRecords[idx].winDim.top  = GetSystemMetrics(SM_CYSCREEN)-inset;
}

// ----------------------------------------------------------------------------

int LoadSticky (HWND hWnd, int idx, BOOL initAlarmInfo)
{
    HWND hChildWin;
    LONG winExtStyle = WS_EX_TOOLWINDOW;
    int  width       = (StickyRecords[idx].winDim.right  - StickyRecords[idx].winDim.left);
    int  height      = (StickyRecords[idx].winDim.bottom - StickyRecords[idx].winDim.top);    

    if (initAlarmInfo != FALSE)
        StickyRecords[idx].stickyAlrm.alarmDlgThread = FALSE;
    
    StickyRecords[idx].stickyWnd                  = NULL;    
    CheckWindowBoundrys (idx, 5);

    if (StickyRecords[idx].thinEdge != FALSE)
        winExtStyle |= WS_EX_STATICEDGE;

    if (StickyRecords[idx].topMostWin != FALSE)
        winExtStyle |= WS_EX_TOPMOST;

    // Pass sticky index parameter, and create the window.
    Sticky_IDX_WM_CREATE = idx;
    hChildWin = CreateWindowEx (winExtStyle,
                                STRWINSTICKYCLASS,                  // window class name
                                "",                                 // window caption                                
                                WS_POPUP | WS_SIZEBOX,
                                StickyRecords[idx].winDim.left,     // initial x position
                                StickyRecords[idx].winDim.top,      // initial y position
                                width,                              // initial x size
                                height,                             // initial y size                                
                                ParentStarter,                      // parent window handle
                                NULL,                               // window menu handle
                                HInst,                              // program instance handle
                                (LPVOID)NULL);                      // creation parameters
    
    if (hChildWin == NULL)
    {
        MessageBox (hWnd, "Unable to create sticky!", 
                    "Window Create Error", MB_OK | MB_ICONSTOP);
        return -1;
    }

    ShowWindow   (hChildWin, StickyRecords[idx].winShowState);
    UpdateWindow (hChildWin);
    return 0;
}

// ----------------------------------------------------------------------------

// returns the number of stickies returned from file
int LoadStickyFromFile ()
{
    FILE* pStickyRecs = fopen (STRSTICKYRECORDS,"rb");
    FILE* pStickyStrs = fopen (STRSTICKYSTRINGS,"rb");
    int   idx         = 0;

    if ((pStickyRecs == NULL) || (pStickyStrs == NULL))
    {
        if (pStickyRecs != NULL)
            fclose (pStickyRecs);

        if (pStickyStrs)
            fclose (pStickyStrs);
        return 0;
    }

    // Load colour information
    if (fread (&ColSchemeNum, sizeof (ColSchemeNum), 1, pStickyRecs) <= 0)
        return 0;

    if (ColSchemeNum > 0)
    {
        if (fread (ColourSchemes, sizeof (ColourSchemes[0]), ColSchemeNum, pStickyRecs) <= 0)
            return 0;
    }

    // Load the sticky defaults first (first Record);
    if (fread (&StickyRecords[STICKRECNUM], sizeof (StickyRecords[STICKRECNUM]), 1, pStickyRecs) <= 0)
        return 0;

    SetNULLSForDefaultSticky();
    
    while ((fread (&StickyRecords[idx], sizeof (StickyRecords[idx]), 1, pStickyRecs) > 0) && (idx < STICKRECNUM))
    {
        // load rest the sticky data
        fread (&StickyRecords[idx].userMemo.strLen, sizeof (StickyRecords[idx].userMemo.strLen), 1, pStickyStrs);

        if (StickyRecords[idx].userMemo.strLen >= 0)
        {
            StickyRecords[idx].userMemo.pString = new char[StickyRecords[idx].userMemo.strLen+1];

            if (StickyRecords[idx].userMemo.pString != NULL)
                fread (StickyRecords[idx].userMemo.pString, StickyRecords[idx].userMemo.strLen+1, 1, pStickyStrs);
        }
        else
            StickyRecords[idx].userMemo.pString = NULL;

        // Create the sticky !
        LoadSticky (ParentStarter, idx, TRUE);
        idx++;
    }

    // Clean up
    fclose (pStickyRecs);
    fclose (pStickyStrs);

    // Test sticky alarms during down time
    DoAlarmDownTimeTest ();

    return idx;
}

// ----------------------------------------------------------------------------

// Returns 0 up a successful save, -1 upon an error.
int SaveStickyToFile ()
{
    FILE* pStickyRecs = fopen (STRSTICKYRECORDS,"wb");
    FILE* pStickyStrs = fopen (STRSTICKYSTRINGS,"wb");

    if ((pStickyRecs == NULL) || (pStickyStrs == NULL))
    {
        fclose (pStickyRecs);
        fclose (pStickyStrs);
        return -1;
    }   
     
    // Save colour information
    fwrite (&ColSchemeNum, sizeof (ColSchemeNum), 1, pStickyRecs);

    if (ColSchemeNum > 0)
        fwrite (ColourSchemes, sizeof (ColourSchemes[0]), ColSchemeNum, pStickyRecs);

    // Save the sticky defaults first (first Record);
    fwrite (&StickyRecords[STICKRECNUM], sizeof (StickyRecords[STICKRECNUM]), 1, pStickyRecs);

    for (int save = 0; save < STICKRECNUM; save++)
    {
        if (StickyRecords[save].stickyWnd != NULL)
        {
            StoreStickyData (save, TRUE);

            // Save record info
            fwrite (&StickyRecords[save], sizeof (StickyRecords[save]), 1, pStickyRecs);

            // Save Memo
            if (StickyRecords[save].userMemo.pString != NULL)
            {
                fwrite (&StickyRecords[save].userMemo.strLen, sizeof (StickyRecords[save].userMemo.strLen), 1, pStickyStrs);
                fwrite (StickyRecords[save].userMemo.pString, StickyRecords[save].userMemo.strLen+1, 1, pStickyStrs);

                delete[] StickyRecords[save].userMemo.pString;
                StickyRecords[save].userMemo.pString = NULL;
            }
        }
    }

    // Clean up
    fclose (pStickyRecs);
    fclose (pStickyStrs);

    return 0;
}

// ----------------------------------------------------------------------------

void DuplicateSticky (int srcIdx, int destIdx)
{
    SaveStickyWinPos (srcIdx);
    StickyRecords[destIdx] = StickyRecords[srcIdx];

    // set default window styles.
    if (StickyRecords[srcIdx].rolledUp != FALSE)
        StickyRecords[destIdx].winDim.bottom =  StickyRecords[destIdx].winDim.top + StickyRecords[destIdx].winDim.bottom;

    StickyRecords[destIdx].rolledUp                = FALSE;
    StickyRecords[destIdx].hFont                   = NULL;
    StickyRecords[destIdx].winShowState            = SW_SHOWNORMAL;

    // set default window pos and title info.
    StickyRecords[destIdx].winCaption[0]           = '\0';
    StickyRecords[destIdx].winDim.top              += 10;
    StickyRecords[destIdx].winDim.left             += 10;
    StickyRecords[destIdx].winDim.right            += 10;    
    StickyRecords[destIdx].winDim.bottom           += 10;

    // Dont duplicate roll and unroll win structures, this can be 
    // painful for the user
    StickyRecords[destIdx].unrolledDim = StickyRecords[destIdx].winDim;
    StickyRecords[destIdx].rolledDim   = StickyRecords[destIdx].winDim;

    // set txt info to null
    StickyRecords[destIdx].userMemo.pString        = NULL;
    StickyRecords[destIdx].stickyWnd               = NULL;
    StickyRecords[destIdx].editControl             = NULL;
    StickyRecords[destIdx].hFont                   = NULL;

    // set alarm clock information.
    // set default for year as year is the only var check
    StickyRecords[destIdx].stickyAlrm.year         = ALRMINIT;
    StickyRecords[destIdx].stickyAlrm.alarmEnabled = FALSE;
    StickyRecords[destIdx].stickyAlrm.alarmBlinker = FALSE;       
}

// ----------------------------------------------------------------------------

void MakeDefaults (void)
{
    int  height = 50;
    int  width  = 200;
    int  x      = (GetSystemMetrics(SM_CXSCREEN) / 2) - (width/2);
    int  y      = (GetSystemMetrics(SM_CYSCREEN) / 2) - (height/2);

    memset (&StickyRecords[STICKRECNUM], 0, sizeof (StickyRecords[STICKRECNUM]));
    StickyRecords[STICKRECNUM].rolledUp      = FALSE;
    StickyRecords[STICKRECNUM].hFont         = NULL;
    StickyRecords[STICKRECNUM].winShowState  = SW_SHOWNORMAL;
    StickyRecords[STICKRECNUM].wordWrapped   = FALSE;
    StickyRecords[STICKRECNUM].hScrollBar    = FALSE;
    StickyRecords[STICKRECNUM].vScrollBar    = FALSE;
    StickyRecords[STICKRECNUM].thinEdge      = FALSE;

    // set default window pos and title info.
    StickyRecords[STICKRECNUM].winCaption[0] = '\0';
    StickyRecords[STICKRECNUM].winDim.top    = y;
    StickyRecords[STICKRECNUM].winDim.left   = x;
    StickyRecords[STICKRECNUM].winDim.right  = x+width;
    StickyRecords[STICKRECNUM].winDim.bottom = y+height;

    // set txt info to null
    StickyRecords[STICKRECNUM].userMemo.pString = NULL;
    StickyRecords[STICKRECNUM].stickyWnd        = NULL;

    // Set Color defaults
    StickyRecords[STICKRECNUM].textCol          = RGB (0, 0, 0);
    StickyRecords[STICKRECNUM].backCol          = RGB (255, 255, 0);

    // set alarm clock information.
    // set default for year as year is the only var check
    StickyRecords[STICKRECNUM].stickyAlrm.year            = ALRMINIT;
    StickyRecords[STICKRECNUM].stickyAlrm.alarmEnabled    = FALSE;
    StickyRecords[STICKRECNUM].stickyAlrm.alarmBlinker    = FALSE;       
    StickyRecords[STICKRECNUM].stickyAlrm.alarmSndRpt     = 1;
    StickyRecords[STICKRECNUM].stickyAlrm.LoopPlayback    = FALSE;
    StickyRecords[STICKRECNUM].stickyAlrm.alarmBlinkCol   = ALARMBACKG;
    StickyRecords[STICKRECNUM].stickyAlrm.alarmBlinking   = TRUE;
    StickyRecords[STICKRECNUM].stickyAlrm.alarmBlinker    = FALSE;

    // Make default for font styles
    StickyRecords[STICKRECNUM].titleFont.lfHeight         = -10;
    StickyRecords[STICKRECNUM].titleFont.lfWidth          = 0;
    StickyRecords[STICKRECNUM].titleFont.lfEscapement     = 0;
    StickyRecords[STICKRECNUM].titleFont.lfOrientation    = 0;
    StickyRecords[STICKRECNUM].titleFont.lfWeight         = 400;
    StickyRecords[STICKRECNUM].titleFont.lfItalic         = 0;
    StickyRecords[STICKRECNUM].titleFont.lfUnderline      = 0;
    StickyRecords[STICKRECNUM].titleFont.lfStrikeOut      = 0;
    StickyRecords[STICKRECNUM].titleFont.lfCharSet        = 0;
    StickyRecords[STICKRECNUM].titleFont.lfOutPrecision   = 1;
    StickyRecords[STICKRECNUM].titleFont.lfClipPrecision  = 2;
    StickyRecords[STICKRECNUM].titleFont.lfQuality        = 1;
    StickyRecords[STICKRECNUM].titleFont.lfPitchAndFamily = 34;    
    strcpy (StickyRecords[STICKRECNUM].titleFont.lfFaceName,"MS Sans Serif");
    StickyRecords[STICKRECNUM].editFont = StickyRecords[STICKRECNUM].titleFont;

    // Global vars 
    LatchActive     = TRUE;
    LatchPixelRange = 10;
    SoundActive     = TRUE;
    strcpy (RollupSnd, STRROLLUPSND);
    strcpy (UnrollSnd, STRUNROLLSND);
    strcpy (AlarmSnd,  STRALARMSND);
}

// ----------------------------------------------------------------------------

BOOL FontChooseFont (HWND hwnd, LOGFONT& curFont)
{
    CHOOSEFONT cf;
    LOGFONT    selFont = curFont;

    cf.lStructSize      = sizeof (CHOOSEFONT);
    cf.hwndOwner        = hwnd;
    cf.hDC              = NULL;
    cf.lpLogFont        = &selFont;
    cf.iPointSize       = 0;
    cf.Flags            = CF_INITTOLOGFONTSTRUCT | CF_SCREENFONTS | CF_EFFECTS;
    cf.rgbColors        = 0L;
    cf.lCustData        = 0L;
    cf.lpfnHook         = NULL;
    cf.lpTemplateName   = NULL;
    cf.hInstance        = NULL;
    cf.lpszStyle        = NULL;
    cf.nFontType        = 0;                      // Returned from ChooseFont
    cf.nSizeMin         = 0;
    cf.nSizeMax         = 0;

    if (ChooseFont (&cf) != FALSE)
    {
        curFont = selFont;
        return TRUE;
    }

    return FALSE;
}

// ----------------------------------------------------------------------------

void PopFontInitialize (HWND hwndEdit, LOGFONT& logfont, HFONT& hFont)
{
    GetObject (GetStockObject (OEM_FIXED_FONT), sizeof (LOGFONT), (LPSTR) &logfont) ;

    if ((hFont = CreateFontIndirect (&logfont)) == NULL)
    {
        MessageBox (hwndEdit, "Problems Creating Font!", "Font Problem", MB_OK | MB_ICONSTOP);
        return;
    }

    SendMessage (hwndEdit, WM_SETFONT, (WPARAM) hFont, 0L) ;
}

// ----------------------------------------------------------------------------

void FontSetFont (HWND hwndEdit, LOGFONT& logfont, HFONT& hFont)
{
    // Remove old font.
    if (hFont != NULL)
        DeleteObject (hFont);

    if ((hFont = CreateFontIndirect (&logfont)) == NULL)
    {
        MessageBox (hwndEdit, "Problems Creating Font!", "Font Problem", MB_OK | MB_ICONSTOP);
        return;
    }
    SendMessage (hwndEdit, WM_SETFONT, (WPARAM) hFont, MAKELONG((WORD)TRUE, 0));
}

// ----------------------------------------------------------------------------

BOOL pickColour (HWND hwnd, COLORREF* pRGB)
{
    /* Color variables */
    CHOOSECOLOR     cc;
    static COLORREF userCols[16] =
    {
        0, 0, 0, 0, 0, 0, 0, 0, 0, 0 ,0,
        0, 0, 0, 0, 0
    };
    /* Set all structure fields to zero. */
    memset(&cc, 0, sizeof(CHOOSECOLOR));

    /* Initialize the necessary CHOOSECOLOR members. */
    cc.lStructSize = sizeof(CHOOSECOLOR);

    cc.hwndOwner    = hwnd;
    cc.rgbResult    = *pRGB;
    cc.lpCustColors = userCols;
    cc.Flags        = CC_RGBINIT | CC_FULLOPEN;

    if (ChooseColor(&cc))
        *pRGB = cc.rgbResult;
    else
        return FALSE;

    return TRUE;
}

// ----------------------------------------------------------------------------

BOOL CALLBACK  DragDropQuery  (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    switch (message)
    {
        case WM_INITDIALOG:
            MessageBeep (MB_ICONQUESTION);
            break;

        case WM_COMMAND:
            switch (LOWORD (wParam))
            {
                case IDC_DDCANCEL:
                case IDC_DDAPPEND:
                case IDC_DDOVERWRITE:
                    EndDialog(hwnd, LOWORD (wParam));
                    return TRUE;
            }
            break;
    }

    return FALSE;
}

// ----------------------------------------------------------------------------

BOOL CALLBACK  StickyTitle  (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    static int    stkIdx;

    switch (message)
    {
        case WM_INITDIALOG:
            // Get sticky index ...
            stkIdx = (int) lParam;
            SetWindowText (GetDlgItem (hwnd, IDC_TITLETEXT), StickyRecords[stkIdx].winCaption);            
            return TRUE;

        case WM_COMMAND :
            switch (LOWORD (wParam))
            {
                // Process the colour select message
                case IDC_CANCEL:
                    EndDialog(hwnd, FALSE);
                    break;

                case IDC_OK:
                    // Load the data from the dialog box text input into sticky.
                    GetWindowText(GetDlgItem (hwnd, IDC_TITLETEXT),
                    StickyRecords[stkIdx].winCaption, CAPTIONSIZE-1);
                    // Set the Windows Title text.
                    SetWindowText (StickyRecords[stkIdx].stickyWnd, StickyRecords[stkIdx].winCaption);
                    EndDialog(hwnd, TRUE);
                    break;
            }
            break;

        case WM_CLOSE:
            EndDialog(hwnd, FALSE);
            break;
    }

    return FALSE;
}

// ----------------------------------------------------------------------------

void PaintStickyPrefsColours (HWND hwnd, COLORREF col, LOGFONT* pTextFont, BOOL paintText)
{
    char   text[] = "TEXT";
    RECT   clientSize;
    HDC    staticDC = GetDC (hwnd);
    HBRUSH brush;

    if (staticDC == NULL)
        return;

    GetClientRect (hwnd, &clientSize);                                
    SetMapMode    (staticDC, MM_TEXT);
    brush  = CreateSolidBrush(col);

    if (paintText == FALSE)   
        FillRect      (staticDC, &clientSize, brush);
    else
    {
        HFONT printFont, oldFont;    
        SIZE  size;

        // Create a font to display
        printFont = CreateFontIndirect(pTextFont);        
        oldFont   = SelectObject (staticDC, printFont);

        SetTextColor (staticDC, col);
        SetBkMode(staticDC, TRANSPARENT);
        
        GetTextExtentPoint32(staticDC, text, sizeof(text), &size);
        
        // adjust rect size
        if (size.cy < clientSize.bottom)
        {
            clientSize.top = ((clientSize.bottom / 2) - (size.cy / 2));
            clientSize.bottom = clientSize.top + size.cy;
        }

        DrawText (staticDC, "TEXT", 4, &clientSize, DT_CENTER | DT_VCENTER);

        // Clean up fonts.
        SelectObject (staticDC, oldFont);
        DeleteObject (printFont);
    }

    ValidateRect  (hwnd, &clientSize);
    DeleteObject  (brush);
    ReleaseDC     (hwnd, staticDC);
}

// ----------------------------------------------------------------------------

void SelectSoundFile (HWND hwnd, char* pSndFile, int buffLen)
{    
    const char*     pFileTypes    = "Wave Sound Files\0*.wav\0"                                    
                                    "All Files\0*.*\0\0"; 
    char            Errstr[256]   = "GetOpenFileName returned Error #";
    OPENFILENAME    fileSelect;
    char            dirLoc[MAXSNDFILELEN];
    char*           pChk;
    DWORD           Errval;         // Error value
    char            buf[5];         // Error buffer    

    
    // Make a directory path.
    strcpy (dirLoc, pSndFile);
    for (pChk = dirLoc + strlen(dirLoc); ((pChk > dirLoc) && (*pChk != '\\')); pChk--)
        *pChk = 0;                    
    
    // Reset return pointers variables to null.    
    memset (&fileSelect, 0, sizeof (fileSelect));

    fileSelect.hwndOwner         = hwnd;
    fileSelect.lStructSize       = sizeof(OPENFILENAME);
    fileSelect.lpstrFilter       = (LPSTR)pFileTypes;  
    fileSelect.lpstrCustomFilter = NULL;
    fileSelect.nFilterIndex      = 1;
    fileSelect.lpstrFile         = (LPSTR)pSndFile;  
    fileSelect.nMaxFile          = buffLen;
    fileSelect.lpstrFileTitle    = "";           
    fileSelect.nMaxFileTitle     = 0;
    fileSelect.lpstrInitialDir   = dirLoc;            
    fileSelect.lpstrTitle        = "Select Wave File";
    fileSelect.Flags             = OFN_HIDEREADONLY  | OFN_FILEMUSTEXIST | 
                                   OFN_PATHMUSTEXIST | OFN_NOCHANGEDIR;
    fileSelect.lpstrDefExt       = "*.wav";

    if(GetOpenFileName(&fileSelect) != TRUE)
    {
        Errval = CommDlgExtendedError();

        if(Errval != 0) // 0 value means user selected Cancel
        {
            sprintf(buf,"%ld",Errval);
            strcat(Errstr,buf);
            MessageBox(NULL,Errstr,"- WARNING - Something Weird Ocurred!", MB_OK | MB_ICONSTOP);
        }        
    }
}

// ----------------------------------------------------------------------------

BOOL CALLBACK  StickyPrefs  (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    const  LOGFONT  textFont  = {-10, 0, 0, 0, FW_BOLD, 0, 0, 0, 0, 1, 2, 1, 32, "MS LineDraw"}; 
    static int      stickyIdx;
    static COLORREF textCol;
    static COLORREF backCol;    
    static COLORREF alarmCol;    

    static LOGFONT  editFont;
    static LOGFONT  titleFont;
    static char     rollupSnd[MAXSNDFILELEN];    
    static char     unrollSnd[MAXSNDFILELEN];    
    static char     alarmSnd[MAXSNDFILELEN];    

    switch (message)
    {
        case WM_INITDIALOG:
            {
                // Set Latch pixel value..
            
                // Check Ranges 
                if (LatchPixelRange < 0) 
                    LatchPixelRange = 0;

                if (LatchPixelRange > 300)
                    LatchPixelRange = 300;

                SetDlgItemInt (hwnd, IDC_LATCHRNG, LatchPixelRange, FALSE);

                // Set Latching method is active or not?
                if (LatchActive == FALSE)
                    CheckDlgButton(hwnd, IDC_LATCHCHK, BST_UNCHECKED);
                else
                    CheckDlgButton(hwnd, IDC_LATCHCHK, BST_CHECKED);

                if (SoundActive == FALSE)
                    CheckDlgButton(hwnd, IDC_SOUNDCHK, BST_UNCHECKED);
                else
                    CheckDlgButton(hwnd, IDC_SOUNDCHK, BST_CHECKED);

                if (AutoTextCopy == FALSE)
                    CheckDlgButton(hwnd, IDC_AUTOCOPYTEXT, BST_UNCHECKED);
                else
                    CheckDlgButton(hwnd, IDC_AUTOCOPYTEXT, BST_CHECKED);

                // Get/set parameters.
                stickyIdx = (int) lParam;

                // set check boxes...
                if (StickyRecords[stickyIdx].wordWrapped != FALSE)
                    CheckDlgButton(hwnd, IDC_WORDWRAPPED, BST_CHECKED);
                else
                    CheckDlgButton(hwnd, IDC_WORDWRAPPED, BST_UNCHECKED);            

                if (StickyRecords[stickyIdx].stickyAlrm.alarmBlinking != FALSE)
                    CheckDlgButton(hwnd, IDC_ALARMBLINKING, BST_CHECKED);
                else
                    CheckDlgButton(hwnd, IDC_ALARMBLINKING, BST_UNCHECKED);            

                if (StickyRecords[stickyIdx].vScrollBar != FALSE)
                    CheckDlgButton(hwnd, IDC_VSCROLLBAR, BST_CHECKED);
                else
                    CheckDlgButton(hwnd, IDC_VSCROLLBAR, BST_UNCHECKED);

                if (StickyRecords[stickyIdx].hScrollBar != FALSE)
                    CheckDlgButton(hwnd, IDC_HSCROLLBAR, BST_CHECKED);
                else
                    CheckDlgButton(hwnd, IDC_HSCROLLBAR, BST_UNCHECKED);            

                if (StickyRecords[stickyIdx].thinEdge != FALSE)
                    CheckDlgButton(hwnd, IDC_THINTHICKBORD, BST_CHECKED);
                else
                    CheckDlgButton(hwnd, IDC_THINTHICKBORD, BST_UNCHECKED);            

                // Load the contents of the combo boxes
                int idx;
                for (idx = 0; idx < ColSchemeNum; idx++)
                    SendMessage (GetDlgItem (hwnd, IDC_COLSCHEME), LB_ADDSTRING, idx, (LPARAM)"");

                // Load colours and fonts
                textCol   = StickyRecords[stickyIdx].textCol;
                backCol   = StickyRecords[stickyIdx].backCol;
                alarmCol  = StickyRecords[stickyIdx].stickyAlrm.alarmBlinkCol;
                editFont  = StickyRecords[stickyIdx].editFont;
                titleFont = StickyRecords[stickyIdx].titleFont;                

                // Load sound settings
                strcpy (rollupSnd, RollupSnd);
                strcpy (unrollSnd, UnrollSnd);
                strcpy (alarmSnd,  AlarmSnd);
            }
            return TRUE;

        case WM_DRAWITEM:
            {                
                LPDRAWITEMSTRUCT pDrwStrct = (LPDRAWITEMSTRUCT) lParam;
                HBRUSH           paintBrush;
                HFONT            printFont, oldFont;
                RECT             textRect;

                if (pDrwStrct->CtlID != IDC_COLSCHEME)
                    break;

                switch (pDrwStrct->itemAction) 
                { 
                    case ODA_SELECT: 
                    case ODA_DRAWENTIRE: 

                        if (pDrwStrct->itemID >= 0)
                        {
                            // Draw background 
                            paintBrush = CreateSolidBrush (ColourSchemes[pDrwStrct->itemID].backCol);
                            FillRect     (pDrwStrct->hDC, &pDrwStrct->rcItem, paintBrush);
                            DeleteObject (paintBrush);

                            // Draw the Text 
                            textRect  = pDrwStrct->rcItem;
                            textRect.left  += 2; textRect.top    += 2;
                            textRect.right -= 2; textRect.bottom -= 2;

                            // Create a font to display
                            printFont = CreateFontIndirect(&textFont);
                            oldFont   = SelectObject (pDrwStrct->hDC, printFont);

                            SetTextColor (pDrwStrct->hDC, ColourSchemes[pDrwStrct->itemID].textCol);
                            SetBkMode(pDrwStrct->hDC, TRANSPARENT);
                            DrawText (pDrwStrct->hDC, "TEXT", 4, &textRect, DT_CENTER | DT_VCENTER);

                            // Clean up fonts.
                            SelectObject (pDrwStrct->hDC, oldFont);
                            DeleteObject (printFont);
                        }
                        else
                            FillRect (pDrwStrct->hDC, &pDrwStrct->rcItem, GetStockObject (WHITE_BRUSH));

                        if (pDrwStrct->itemState & ODS_SELECTED)
                        {
                            pDrwStrct->rcItem.left   += 2;
                            pDrwStrct->rcItem.top    += 2;
                            pDrwStrct->rcItem.right  -= 2;
                            pDrwStrct->rcItem.bottom -= 2;

                            DrawFocusRect (pDrwStrct->hDC, &pDrwStrct->rcItem) ;
                        }
                }
            }
            break;

            // Display the current colour of sticky.
        case WM_PAINT:
            PaintStickyPrefsColours (GetDlgItem (hwnd, IDC_STATICTEXTBACKCOL),  backCol, NULL, FALSE);
            PaintStickyPrefsColours (GetDlgItem (hwnd, IDC_STATICTEXTBACKCOL),  textCol, &editFont, TRUE);
            PaintStickyPrefsColours (GetDlgItem (hwnd, IDC_STATICALARMCOL),     alarmCol, NULL, FALSE);
            break;
            
        case WM_COMMAND :
            switch (LOWORD (wParam))
            {
                // Process the colour select message
                case IDC_TEXTCOL:
                    if (pickColour (hwnd, &textCol) != FALSE)
                    {
                        InvalidateRect (hwnd, NULL, FALSE);
                        UpdateWindow   (hwnd);
                    }
                    break;

                case IDC_BACKCOL:
                    if (pickColour (hwnd, &backCol) != FALSE)
                    {
                        InvalidateRect (hwnd, NULL, FALSE);
                        UpdateWindow   (hwnd);
                    }
                    break;

                case IDC_ALARMCOL:
                    if (pickColour (hwnd, &alarmCol) != FALSE)
                    {
                        InvalidateRect (hwnd, NULL, FALSE);
                        UpdateWindow   (hwnd);
                    }
                    break;

                // Process quick colour schemes
                case IDC_QCSADD:
                    if (ColSchemeNum < MAXCOLSCHEMENUM)
                    {
                        // Load colour info 
                        ColSchemeNum++;
                        ColourSchemes[ColSchemeNum-1].textCol  = textCol;
                        ColourSchemes[ColSchemeNum-1].backCol  = backCol;
                        ColourSchemes[ColSchemeNum-1].alarmCol = alarmCol;                        
                        SendMessage (GetDlgItem (hwnd, IDC_COLSCHEME), LB_ADDSTRING,ColSchemeNum, (LPARAM)"!");
                    }
                    else
                        MessageBox (hwnd, "Out of array space to add colour scheme.", 
                                          "Array Space Error", MB_OK | MB_ICONSTOP);
                    break;

                case IDC_QCSCOPY:
                    {
                        int idx = SendMessage (GetDlgItem (hwnd, IDC_COLSCHEME), LB_GETCURSEL, 0, 0);

                        if (idx != LB_ERR)
                        {
                            textCol  = ColourSchemes[idx].textCol;
                            backCol  = ColourSchemes[idx].backCol;
                            alarmCol = ColourSchemes[idx].alarmCol;
                            InvalidateRect (hwnd, NULL, FALSE);
                            UpdateWindow   (hwnd);
                        }
                        else
                            MessageBox (hwnd, "Select a colour scheme to copy over.", 
                                          "Colour Scheme Information", MB_OK | MB_ICONINFORMATION);

                    }
                    break;

                case IDC_QCSDEL:
                    {
                        int idx = SendMessage (GetDlgItem (hwnd, IDC_COLSCHEME), LB_GETCURSEL, 0, 0);

                        if (idx != LB_ERR)
                        {
                            // Remove it from the list box
                            SendMessage (GetDlgItem (hwnd, IDC_COLSCHEME), LB_DELETESTRING, idx, 0);
                                
                            // Update the colour scheme array, and remove the item and shuffle all others down
                            for (int rm = idx; rm < (ColSchemeNum-1); rm++)
                                ColourSchemes[rm] = ColourSchemes[rm+1];

                            ColSchemeNum--;
                        }
                        else
                            MessageBox (hwnd, "Select a colour scheme to delete.", 
                                          "Colour Scheme Information", MB_OK | MB_ICONINFORMATION);

                    }
                    break;

                // Process Sound playing
                case IDC_ROLLUPPLAY:
                     PlaySound (rollupSnd, NULL, SND_ASYNC | SND_FILENAME);
                     break;

                case IDC_UNROLLPLAY:
                     PlaySound (unrollSnd, NULL, SND_ASYNC | SND_FILENAME);
                     break;

                case IDC_ALARMPLAY:
                     PlaySound (alarmSnd, NULL, SND_ASYNC | SND_FILENAME);
                     break;

                // Select Sound File
                case IDC_ROLLUPSEL:
                    SelectSoundFile (hwnd, rollupSnd, MAXSNDFILELEN);
                    break;

                case IDC_UNROLLSEL:
                    SelectSoundFile (hwnd, unrollSnd, MAXSNDFILELEN);
                    break;

                case IDC_ALARMSEL:
                    SelectSoundFile (hwnd, alarmSnd, MAXSNDFILELEN);
                    break;

                // Select fonts
                case IDC_SETEDITFONT:
                    FontChooseFont (hwnd, editFont);
                    InvalidateRect (hwnd, NULL, FALSE);
                    UpdateWindow   (hwnd);
                    break;

                case IDC_SETTITLEFONT:
                    FontChooseFont (hwnd, titleFont);                                        
                    break;

                // Process user control actions
                case IDC_CANCEL:
                    PostMessage (hwnd, WM_CLOSE, 0, 0);
                    break;

                case IDC_RESET:
                    // load colour values...
                    textCol   = StickyRecords[stickyIdx].textCol;
                    backCol   = StickyRecords[stickyIdx].backCol;
                    alarmCol  = StickyRecords[stickyIdx].stickyAlrm.alarmBlinkCol;

                    editFont  = StickyRecords[stickyIdx].editFont;
                    titleFont = StickyRecords[stickyIdx].titleFont;
                    
                    // set check boxes...
                    if (StickyRecords[stickyIdx].wordWrapped != FALSE)
                        CheckDlgButton(hwnd, IDC_WORDWRAPPED, BST_CHECKED);
                    else
                        CheckDlgButton(hwnd, IDC_WORDWRAPPED, BST_UNCHECKED);            

                    if (StickyRecords[stickyIdx].stickyAlrm.alarmBlinking != FALSE)
                        CheckDlgButton(hwnd, IDC_ALARMBLINKING, BST_CHECKED);
                    else
                        CheckDlgButton(hwnd, IDC_ALARMBLINKING, BST_UNCHECKED);            

                    if (StickyRecords[stickyIdx].vScrollBar != FALSE)
                        CheckDlgButton(hwnd, IDC_VSCROLLBAR, BST_CHECKED);
                    else
                        CheckDlgButton(hwnd, IDC_VSCROLLBAR, BST_UNCHECKED);

                    if (StickyRecords[stickyIdx].hScrollBar != FALSE)
                        CheckDlgButton(hwnd, IDC_HSCROLLBAR, BST_CHECKED);
                    else
                        CheckDlgButton(hwnd, IDC_HSCROLLBAR, BST_UNCHECKED);  

                    if (StickyRecords[stickyIdx].thinEdge != FALSE)
                        CheckDlgButton(hwnd, IDC_THINTHICKBORD, BST_CHECKED);
                    else
                        CheckDlgButton(hwnd, IDC_THINTHICKBORD, BST_UNCHECKED);            

                    InvalidateRect (hwnd, NULL, FALSE);                    
                    UpdateWindow   (hwnd);
                    break;

                case IDC_OK:
                    {
                        BOOL valOk;
                        UINT testPixRng = GetDlgItemInt(hwnd, IDC_LATCHRNG, &valOk, FALSE);

                        // Save window latching Parameters
                        if (valOk == FALSE)
                        {
                            MessageBeep (MB_ICONEXCLAMATION);
                            break;
                        }

                        if (testPixRng < 0) 
                            testPixRng = 0;

                        if (testPixRng > 300)
                            testPixRng = 300;

                        LatchPixelRange = testPixRng;

                        // Determine if latching method is active?
                        if (IsDlgButtonChecked(hwnd, IDC_LATCHCHK) == BST_CHECKED)
                            LatchActive = TRUE;
                        else
                            LatchActive = FALSE;

                        // Determine if sound effects are active?
                        if (IsDlgButtonChecked(hwnd, IDC_SOUNDCHK) == BST_CHECKED)
                            SoundActive = TRUE;
                        else
                            SoundActive = FALSE;

                        // Auto-copy text after select in edit control or ... not
                        if (IsDlgButtonChecked(hwnd, IDC_AUTOCOPYTEXT) == BST_CHECKED)
                            AutoTextCopy = TRUE;
                        else
                            AutoTextCopy = FALSE;

                        // save parameters.
                        StickyRecords[stickyIdx].textCol                  = textCol;
                        StickyRecords[stickyIdx].backCol                  = backCol;
                        StickyRecords[stickyIdx].stickyAlrm.alarmBlinkCol = alarmCol;
                        StickyRecords[stickyIdx].editFont                 = editFont;
                        StickyRecords[stickyIdx].titleFont                = titleFont;                                        
                        strcpy (RollupSnd, rollupSnd);
                        strcpy (UnrollSnd, unrollSnd);
                        strcpy (AlarmSnd,  alarmSnd);

                        // check for word wrap option..
                        if (IsDlgButtonChecked(hwnd, IDC_WORDWRAPPED) == BST_CHECKED)
                            StickyRecords[stickyIdx].wordWrapped = TRUE;
                        else
                            StickyRecords[stickyIdx].wordWrapped = FALSE;
    
                        // check for alarm blinking option.
                        if (IsDlgButtonChecked(hwnd, IDC_ALARMBLINKING) == BST_CHECKED)
                            StickyRecords[stickyIdx].stickyAlrm.alarmBlinking = TRUE;
                        else
                            StickyRecords[stickyIdx].stickyAlrm.alarmBlinking = FALSE;
    
                        // check for horiz scroll bar option.
                        if (IsDlgButtonChecked(hwnd, IDC_HSCROLLBAR) == BST_CHECKED)
                            StickyRecords[stickyIdx].hScrollBar = TRUE;
                        else
                            StickyRecords[stickyIdx].hScrollBar = FALSE;

                        // check for vert scroll bar option.
                        if (IsDlgButtonChecked(hwnd, IDC_VSCROLLBAR) == BST_CHECKED)
                            StickyRecords[stickyIdx].vScrollBar = TRUE;
                        else
                            StickyRecords[stickyIdx].vScrollBar = FALSE;

                        // check for border size option.
                        if (IsDlgButtonChecked(hwnd, IDC_THINTHICKBORD) == BST_CHECKED)
                            StickyRecords[stickyIdx].thinEdge = TRUE;
                        else
                            StickyRecords[stickyIdx].thinEdge = FALSE;

                        EndDialog(hwnd, TRUE);
                    }   
                    break;
            }
            break;

        case WM_CLOSE:
            EndDialog(hwnd, FALSE);
            break;
    }

    return FALSE;
}

// ----------------------------------------------------------------------------

int GetStickyTitleFontHeight (HWND hwnd, int idx)
{
    HDC        dc     = GetDC (hwnd);
    TEXTMETRIC tm;
    int        fontHeight;

    if (dc != NULL)
    {
        HFONT    printFont, oldFont;

        printFont = CreateFontIndirect(&StickyRecords[idx].titleFont);
        oldFont   = SelectObject (dc, printFont);

        if (GetTextMetrics(dc,&tm) != FALSE)
            fontHeight = tm.tmHeight;
        else
            fontHeight = abs(StickyRecords[idx].titleFont.lfHeight);

        DeleteObject (SelectObject (dc, oldFont));
        ReleaseDC (hwnd, dc);
    }
    else 
        fontHeight = abs(StickyRecords[idx].titleFont.lfHeight);

    return fontHeight;
}

// ----------------------------------------------------------------------------

void ProcessDragDropFiles (WPARAM wParam, int mode, int wndIdx)
{
    char* pFileName = NULL;    
    int   fileLen   = 1024;
    int   dropNum   = DragQueryFile((HDROP) wParam, (UINT)-1, NULL, 0);

    for (int idx = 0; idx < dropNum; idx++)
    {
        fileLen = DragQueryFile((HDROP) wParam, idx, NULL, 0 );
        if ((pFileName = new char[fileLen+1]) == NULL)
            return;

        //copy of the file name 
        DragQueryFile((HDROP) wParam, idx, pFileName, fileLen+1 ); 
        
        ImportDragDropFile (pFileName,  mode, wndIdx);

        // Clean up and make ready for any additional files.
        if (pFileName != NULL)
        {
            delete[] pFileName;
            pFileName = NULL;
        }
    }

    DragFinish((HDROP) wParam);
}

// ----------------------------------------------------------------------------

inline BOOL IsWndCloser (RECT* pAssignedRect, RECT* pTestRect)
{
    if (pAssignedRect->top > pTestRect->top)
        return TRUE;           
    else if (pAssignedRect->bottom < pTestRect->bottom)
            return TRUE;

    if (pAssignedRect->left < pTestRect->left)
        return TRUE;        
    else if (pAssignedRect->right > pTestRect->right)
        return TRUE;          

    return FALSE;
}

void ProcessWindowMoving (HWND hwnd, LPARAM lParam, POINT lastLButtonDn)
{
    int  xMove = (lastLButtonDn.x - (short)LOWORD(lParam));
    int  yMove = (lastLButtonDn.y - (short)HIWORD(lParam));                                        
    int  curWndWidth, curWndHeight;                    
    int  wndIdx;

    BOOL assigned  = FALSE;
    BOOL xLocked   = FALSE;
    BOOL yLocked   = FALSE;
    int  assignIdx;
    int  xAssigned, yAssigned;
    RECT wndRect,  testRect, assignedRect;

    GetWindowRect(hwnd, &wndRect);
    xAssigned    = xMove = wndRect.left-xMove;
    yAssigned    = yMove = wndRect.top-yMove;
    curWndWidth  = wndRect.right  - wndRect.left;
    curWndHeight = wndRect.bottom - wndRect.top;

    // Test for window alignment feature...
    if (LatchActive == FALSE) 
        goto MoveTheWindow;

    // Do window alignment processing, may take chuck of the CPU if there are alot windows!
    for (wndIdx = 0; wndIdx < STICKRECNUM; wndIdx++)
    {                                             
         // Match all other valid windows and check their size and location.
         if ((StickyRecords[wndIdx].stickyWnd != NULL) &&
             (StickyRecords[wndIdx].stickyWnd != hwnd))
         {
             // Get other sticky windows location and size
             GetWindowRect(StickyRecords[wndIdx].stickyWnd, &testRect);  

             if  ( (((xMove             < (testRect.left  - LatchPixelRange)) && 
                     (xMove+curWndWidth < (testRect.left  - LatchPixelRange)))
                    ||
                    ((xMove             > (testRect.right + LatchPixelRange)) && 
                     (xMove+curWndWidth > (testRect.right + LatchPixelRange))))
                    == 0)
             {
                 // test top range
                 if ((((yMove+curWndHeight) >= (testRect.top - LatchPixelRange)) && 
                      ((yMove+curWndHeight) <= testRect.top)) &&
                       (yLocked == FALSE))
                 {
                     if (assigned == FALSE)
                     {
                        yAssigned = testRect.top - curWndHeight;
                        assigned  = TRUE;
                        assignIdx = wndIdx;
                     }
                     else
                     {
                         GetWindowRect(StickyRecords[assignIdx].stickyWnd, &assignedRect);
                         if (IsWndCloser (&assignedRect, &testRect) != FALSE)
                         {
                            yAssigned = testRect.top - curWndHeight;
                            assigned  = TRUE;
                            assignIdx = wndIdx;                                
                         }
                     }                             
                 }
                                     
                 // test bottom range
                 if (((yMove   <= (testRect.bottom + LatchPixelRange)) &&
                      (yMove   >= testRect.bottom)) &&
                      (yLocked == FALSE))
                 {
                     if (assigned == FALSE)
                     {
                        yAssigned = testRect.bottom;
                        assigned  = TRUE;
                        assignIdx = wndIdx;
                     }
                     else
                     {
                         GetWindowRect(StickyRecords[assignIdx].stickyWnd, &assignedRect);
                         if (IsWndCloser (&assignedRect, &testRect) != FALSE)
                         {
                            yAssigned = testRect.bottom;
                            assigned  = TRUE;
                            assignIdx = wndIdx;                                
                         }
                     }                             
                 }

                 // --------------------------------------------------------
/*
                 // test bottom corners
                 if ((yMove+curWndHeight >= testRect.bottom) &&
                     (yMove+curWndHeight <= testRect.bottom + LatchPixelRange))
                 {
                     yAssigned = testRect.bottom - curWndHeight;                             
                     yLocked   = TRUE;
                 }

                 // test top corners
                 if ((yMove <= testRect.top) &&
                     (yMove >= testRect.top - LatchPixelRange))
                 {                                         
                    yAssigned = testRect.top;
                    yLocked   = TRUE;
                 }
*/
             } 

             // ###############################################

             if  ( (((yMove              < (testRect.top    - LatchPixelRange)) && 
                     (yMove+curWndHeight < (testRect.top    - LatchPixelRange)))
                    ||
                    ((yMove              > (testRect.bottom + LatchPixelRange)) && 
                     (yMove+curWndHeight > (testRect.bottom + LatchPixelRange))))
                    == 0)
             { 
                 // test left range
                 if ((((xMove+curWndWidth) >= (testRect.left - LatchPixelRange)) && 
                       (xMove+curWndWidth  <=  testRect.left)) &&
                       (xLocked == FALSE))
                 {
                     if (assigned == FALSE)
                     {
                        xAssigned = testRect.left - curWndWidth;
                        assigned  = TRUE;
                        assignIdx = wndIdx;
                     }
                     else
                     {
                         GetWindowRect(StickyRecords[assignIdx].stickyWnd, &assignedRect);
                         if (IsWndCloser (&assignedRect, &testRect) != FALSE)
                         {
                            xAssigned = testRect.left - curWndWidth;
                            assigned  = TRUE;
                            assignIdx = wndIdx;                                
                         }
                     }                             
                 }

                 // test right range
                 if (((xMove <= (testRect.right + LatchPixelRange)) &&
                      (xMove >= testRect.right)) &&
                      (xLocked == FALSE))
                 {
                     if (assigned == FALSE)
                     {
                        xAssigned = testRect.right;
                        assigned  = TRUE;
                        assignIdx = wndIdx;
                     }
                     else
                     {
                         GetWindowRect(StickyRecords[assignIdx].stickyWnd, &assignedRect);
                         if (IsWndCloser (&assignedRect, &testRect) != FALSE)
                         {
                            xAssigned = testRect.right;
                            assigned  = TRUE;
                            assignIdx = wndIdx;                                
                         }
                     }                             
                 }

                 // --------------------------------------------------------
/*
                 // test left corners
                 if ((xMove  <=  testRect.left) &&
                     (xMove  >=  testRect.left - LatchPixelRange))
                 {
                     xAssigned = testRect.left;                             
                     xLocked   = TRUE;
                 }

                 // test right corners
                 if ((xMove+curWndWidth  >=  testRect.right) &&
                     (xMove+curWndWidth  <=  testRect.right + LatchPixelRange))
                 {                             
                    xAssigned = testRect.right - curWndWidth;
                    xLocked   = TRUE;
                 }
*/
             }
         }
    }

MoveTheWindow:    
    MoveWindow(hwnd, xAssigned, yAssigned, curWndWidth, curWndHeight, TRUE);
}

void CalculateSelection (HWND hwnd)
{
    const int buffLen = 1024;
    char   buff[buffLen];
    int    startPos;
    int    endPos;    
    int    lineLen;
    double result;
    parser parse;
    int    idx  =  GetWndIdx (hwnd);

    // Get the current selection
    SendMessage (StickyRecords[idx].editControl, EM_GETSEL,   (WPARAM)&startPos, (LPARAM)&endPos);

    // if no selection then get the current line
    if (endPos == startPos)
    {             
        // Get the current line of text as the math expression
        int lineIdx = SendMessage (StickyRecords[idx].editControl, EM_LINEFROMCHAR, endPos, 0);                           
        
        *((WORD*) buff) = buffLen;
        lineLen = SendMessage (StickyRecords[idx].editControl, EM_GETLINE, (WPARAM)lineIdx, (LPARAM) (LPSTR)buff);
        
        // add terminate to parser string, parser is a little funny...                           
        buff[lineLen]   = ' ';
        buff[lineLen+1] = '\0';                            

        // Make sure we are at the end of the current line.
        int tmp = SendMessage (StickyRecords[idx].editControl, EM_LINEINDEX, (WPARAM)lineIdx, 0);
        if (tmp >= 0)
            endPos = tmp + lineLen;
    }
    else if ((endPos-startPos) > (buffLen-1))
    {
        MessageBox (hwnd, "Unable to complete operation. Temporary memory exceed.", 
                    "Memory Error", MB_ICONSTOP | MB_OK);
        return;
    }
    else
    {
        // process the selected text only
        int startLine;
        int endLine;
        int lineStartIdx;

        startLine = SendMessage (StickyRecords[idx].editControl, EM_LINEFROMCHAR, startPos, 0);                           
        endLine   = SendMessage (StickyRecords[idx].editControl, EM_LINEFROMCHAR, endPos, 0);                           

        if (startLine != endLine) // multi-line selection
        {            
            char tmpBuff[buffLen];

            for (int count = startLine; count <= endLine; count++)
            {                
                // get the current line 
                *((WORD*) tmpBuff) = buffLen;
                lineLen            = SendMessage (StickyRecords[idx].editControl, EM_GETLINE, (WPARAM)count, (LPARAM) (LPSTR)tmpBuff);
                tmpBuff[lineLen]   = '\0'; 
                
                if (count == startLine)             // process the first line
                {
                    // get the start of the line
                    lineStartIdx  = SendMessage (StickyRecords[idx].editControl, EM_LINEINDEX, (WPARAM)startLine, 0);                       
                    strcpy (buff, tmpBuff+(startPos - lineStartIdx));
                }
                else if (count == endLine)          // process the last line 
                {
                    lineStartIdx  = SendMessage (StickyRecords[idx].editControl, EM_LINEINDEX, (WPARAM)startLine, 0);                       
                    tmpBuff[(endPos-lineStartIdx)] = ' ';
                    tmpBuff[(endPos-lineStartIdx)] = '\0'+1;
                    strcat (buff, tmpBuff);                    
                }
                else
                    strcat (buff, tmpBuff);
            }          
        }
        else  // single-line selection
        {
            // get the current line 
            *((WORD*) buff) = buffLen;
            lineLen         = SendMessage (StickyRecords[idx].editControl, EM_GETLINE, (WPARAM)startLine, (LPARAM) (LPSTR)buff);
            buff[lineLen]   = '\0'; 

            // get the start of the line
            lineStartIdx  = SendMessage (StickyRecords[idx].editControl, EM_LINEINDEX, (WPARAM)startLine, 0);                       
        
            char* pCpy   = buff+(startPos - lineStartIdx);
            for (int cpyNum = 0; cpyNum < (endPos - startPos); cpyNum++, pCpy++)
                buff[cpyNum] = *pCpy;
              
            buff[cpyNum]   = ' ';
            buff[cpyNum+1] = '\0';            
        }        
    }

    // calculate equation
    if (parse.analyser(buff, &result) != 0)
        sprintf (buff, "%s at char [%c, %d]", ErrorStr[parse.parseErrorCode], buff[parse.expStrIdx-1], parse.expStrIdx);
    else
    {
        sprintf (buff, "%-20.20g", result);        
        // remove trailing spaces if there any
        for (char* pSpc = buff+(strlen(buff)-1); ((pSpc > buff) && (*pSpc == ' ')); pSpc--)
            *pSpc = '\0';
    }

    // output results
    SendMessage (StickyRecords[idx].editControl, EM_SETSEL, endPos, endPos);
    SendMessage (StickyRecords[idx].editControl, EM_REPLACESEL, FALSE, (LPARAM)(LPCTSTR)"\r\n");
    SendMessage (StickyRecords[idx].editControl, EM_REPLACESEL, TRUE,  (LPARAM)(LPCTSTR)buff);
}

// ************************************************************************************
// ************************************************************************************
// ************************************************************************************

// Sub class the edit control to provide accelerator functions to the main sticky window.
int CALLBACK SubClassEditCntl (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    /* get the pointer to the extra bytes structure out of the user data. */
    SubClassInfo*   pSBInfo = (SubClassInfo*) GetWindowLong (hwnd, GWL_USERDATA);    

    switch (message) 
    {
        case WM_HELP:
            WinHelp(hwnd, STRSTICKYHELP, HELP_FINDER, 0);
            break;


        // Free up the sub-class windows structure which was allocated at subclass time.         
        case WM_DESTROY:
            LocalUnlock (LocalHandle (pSBInfo));
            break;

        // process the return key in calculator mode
        case WM_CHAR:            
            if ((StickyRecords[pSBInfo->stickyIdx].calculatorMode != FALSE) && ((TCHAR)wParam == '\r'))
            {
                // process key and each it
                CalculateSelection (pSBInfo->hOwnerWnd);                
                return 0;
            }
            break;

        case WM_SYSKEYDOWN:
            MSG            accelMsg;

            // Translate the virtual keys to accelerator keys to menu keys and 
            // redirect them to the edit control's owner sticky window.
            accelMsg.hwnd    = hwnd;
            accelMsg.message = message;
            accelMsg.wParam  = wParam;
            accelMsg.lParam  = lParam;    
            accelMsg.time    = 0; 
            accelMsg.pt.x    = 0;
            accelMsg.pt.y    = 0;

            // if message was processed, then eat the key.
            if (TranslateAccelerator(pSBInfo->hOwnerWnd, HStickyAccel, &accelMsg) != FALSE)
                return 0;
            break;
        
        case WM_LBUTTONUP:
            // check to see if control keys, shift or control are depressed,
            // if they are then process the message here.
            switch (wParam)
            {
                // auto cut
                case MK_SHIFT: 
                    {
                        DWORD startPos = 0;
                        DWORD endPos   = 0;

                        SendMessage (hwnd, EM_GETSEL, (WPARAM) &startPos, (LPARAM) &endPos);

                        if ((startPos != 0) || (endPos != 0))
                            PostMessage (hwnd, WM_CUT, 0, 0);
                    }
                    break;


                default:
                    // auto copy
                    if ((AutoTextCopy != FALSE) || (wParam = MK_CONTROL))
                    {
                        DWORD startPos = 0;
                        DWORD endPos   = 0;

                        SendMessage (hwnd, EM_GETSEL, (WPARAM) &startPos, (LPARAM) &endPos);

                        if ((startPos != 0) || (endPos != 0))
                            PostMessage (hwnd, WM_COPY, 0, 0);
                    }
            }
            break;

        case WM_RBUTTONUP:
            // check to see if control keys, shift or control are depressed,
            // if they are then process the message here and dont pass it onto the edit control
            switch (wParam)
            {
                // auto paste
                case MK_CONTROL:
                    PostMessage (hwnd, WM_PASTE, 0, 0);
                    return 0;

                case MK_SHIFT:
                    SendMessage (pSBInfo->hOwnerWnd, WM_CONTEXTMENU, wParam, lParam);
                    return 0;
            }
            break;

    }

    /**********************************************************************
     * for messages that are not handled explicitly here, pass them
     *  back to the old window procedure.
     **********************************************************************/
    return CallWindowProc ((FARPROC)pSBInfo->pOrigSubclassProc, hwnd, message, wParam, lParam);
}

// ----------------------------------------------------------------------------

inline void InitEditSubclass (HWND hOwnerWnd, HWND hEditWnd, int idx)
{
    SubClassInfo* pSBInfo;

    pSBInfo = (SubClassInfo*) LocalAlloc (LPTR, sizeof (SubClassInfo));

    if (pSBInfo == NULL)
        return;

    pSBInfo->stickyIdx         = idx;
    pSBInfo->hOwnerWnd         = hOwnerWnd;
    pSBInfo->pOrigSubclassProc = (WNDPROC) GetWindowLong (hEditWnd, GWL_WNDPROC);

    SetWindowLong (hEditWnd, GWL_USERDATA, (LONG) pSBInfo);
    SetWindowLong (hEditWnd, GWL_WNDPROC,  (LONG) SubClassEditCntl);
}

// ************************************************************************************
// ************************************************************************************
// ************************************************************************************

int WINAPI Sticky (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
#define EDITGLOBALINITSIZE 512
#define EDITWINID 1
    static FARPROC fpfnPrefsDialog    = NULL;
    static FARPROC fpfnSetAlarmDialog = NULL;
    static FARPROC fpfnSetStickyTitle = NULL;
    static HMENU   hPopupMenu         = NULL;
    static HMENU   hPopupFndMenu      = NULL;
    static HMENU   hPopupMailMenu     = NULL;
    static HMENU   hPopupCalcMenu     = NULL;
    static HMENU   hPopupIOMenu       = NULL;
    static int     handleCounter      = 0;
    static POINT   lastLButtonDn;               // used in moving the window and icon button clicking
    static RECT    closeButRect;
    static BOOL    moveIsOk           = FALSE;  // test to see if use click the left mouse button in correct place
    static HICON   IconBut1           = NULL;
    static HICON   IconBut2           = NULL;
    static HICON   IconAlarm          = NULL;
    static HICON   IconCalcMode       = NULL;

    // Find Text Buffer memory     
    static char        findWhat[FINDBUFFLEN] = {'\0'};
    static char        replWhat[FINDBUFFLEN] = {'\0'};
    static FINDREPLACE fndRplInfo = {sizeof (fndRplInfo), 0, HInst, 0, findWhat, 
                                     replWhat, FINDBUFFLEN, FINDBUFFLEN, 0, 0, 0};

    // Indexs to sticky windows array references.
    static int      prefsIdx;
    int             idx;
    
    switch (message)
    {
        case WM_CREATE:
            {
                DWORD  editWinStyle = 0;
                HANDLE hEdGlobalMem = HInst;      
                                

                // initialise sticky program stuff
                if (hPopupMenu == NULL)
                {
                    // Add sub-menu item to popupmenu and add menu items to popup menu.
                    hPopupFndMenu = CreateMenu();
                    InsertMenu(hPopupFndMenu, 0, MF_STRING, FINDTEXTID,    (LPCSTR) STRFINDTEXT);
                    InsertMenu(hPopupFndMenu, 0, MF_STRING, FINDREPLACEID, (LPCSTR) STRFINDREPLACE);

                    // Add sub-menu item to popupmenu and add menu items to popup menu.
                    hPopupMailMenu = CreateMenu();
                    InsertMenu(hPopupMailMenu, 0, MF_STRING, MAILTONOINSTALLID, (LPCSTR) STRMAILTONOINSTALL);
                    InsertMenu(hPopupMailMenu, 0, MF_STRING, MAILTOINSTALLID,(LPCSTR) STRMAILTOINSTALL);

                    // Add sub-menu item to popupmenu and add menu items to popup menu.
                    hPopupIOMenu = CreateMenu();
                    InsertMenu(hPopupIOMenu, 0, MF_STRING, IMPORTSTICKYID,(LPCSTR) STRIMPORTSTICKY);
                    InsertMenu(hPopupIOMenu, 0, MF_STRING, EXPORTSTICKYID,(LPCSTR) STREXPORTSTICKY);

                    // Add sub-menu item to popupmenu and add menu items to popup menu.
                    hPopupCalcMenu = CreateMenu();
                    InsertMenu(hPopupCalcMenu, 0, MF_STRING, CALCULATORMODEID, (LPCSTR) STRCALMODEONOFF);
                    InsertMenu(hPopupCalcMenu, 0, MF_STRING, CALCULATESELECTID,(LPCSTR) STRCALSELECTION);

                    // Add menu item to popupmenu and add menu items to popup menu.
                    hPopupMenu    = CreatePopupMenu();
                    InsertMenu(hPopupMenu, 0, MF_STRING, NEWSTICKYID,  (LPCSTR) STRNEWSTICKY);
                    InsertMenu(hPopupMenu, 0, MF_STRING, ROLLUPID,     (LPCSTR) STRROLLUP);                    
                    InsertMenu(hPopupMenu, 0, MF_STRING, SETTITLEID,   (LPCSTR) STRSETTITLE);
                    InsertMenu(hPopupMenu, 0, MF_STRING, PREFSID,      (LPCSTR) STRPREFS);
                    InsertMenu(hPopupMenu, 0, MF_STRING, SETTOPMOSTID, (LPCSTR) STRSETTOPMOST);
                    InsertMenu(hPopupMenu, 0, MF_STRING, SETALARMID,   (LPCSTR) STRSETALARM);   
                    InsertMenu(hPopupMenu, 0, MF_STRING, PRINTSTICKYID,  (LPCSTR) STRPRINTSTICKY);      
                    
                    InsertMenu(hPopupMenu, 0, MF_POPUP, (UINT)hPopupCalcMenu, (LPCSTR) STRCALSUBMENU);  
                    InsertMenu(hPopupMenu, 0, MF_POPUP, (UINT)hPopupFndMenu,(LPCSTR) STRFINDSUBMENU);  
                   
                    if (MAPIActive == FALSE)
                        InsertMenu(hPopupMenu, 0, MF_POPUP | MF_DISABLED | MF_GRAYED, (UINT)hPopupMailMenu, (LPCSTR) STRMAILTOSUBMENU);  
                    else
                        InsertMenu(hPopupMenu, 0, MF_POPUP, (UINT)hPopupMailMenu, (LPCSTR) STRMAILTOSUBMENU);  
                    
                    InsertMenu(hPopupMenu, 0, MF_POPUP, (UINT)hPopupIOMenu,   (LPCSTR) STREXPORTIMPORT); 
                    
                    InsertMenu(hPopupMenu, 0, MF_STRING, MAKEASDEFAULTID,(LPCSTR) STRMAKEASDEFAULT);                      
                    InsertMenu(hPopupMenu, 0, MF_STRING, CLOSESTICKYID,  (LPCSTR) STRCLOSESTICKY);                                      

                    // Load close sticky icons
                    IconBut1      = LoadIcon (HInst, MAKEINTRESOURCE (IDI_CLOSEBUT1));
                    IconBut2      = LoadIcon (HInst, MAKEINTRESOURCE (IDI_CLOSEBUT2));
                    IconAlarm     = LoadIcon (HInst, MAKEINTRESOURCE (IDI_ALARMSET));
                    IconCalcMode  = LoadIcon (HInst, MAKEINTRESOURCE (IDI_CALCMODE));
                    handleCounter = 1;
                }
                else
                    handleCounter++;

                // *************************************************
                // ********* Init Individual Sticky Window *********
                // *************************************************                

                idx = Sticky_IDX_WM_CREATE; // Set the current Index.                                
                editWinStyle = WS_CHILD       | WS_VISIBLE    | ES_LEFT       | ES_MULTILINE | 
                               ES_AUTOVSCROLL | ES_WANTRETURN | ES_OEMCONVERT | DS_LOCALEDIT | 
                               ES_NOHIDESEL;
                
                // If sticky was loaded from file, then initalise it.
                // Determine if edit control needs to be in word wrapped mode
                if (StickyRecords[idx].wordWrapped == FALSE)
                    editWinStyle |= ES_AUTOHSCROLL;

                if (StickyRecords[idx].vScrollBar != FALSE)
                    editWinStyle |= WS_VSCROLL;

                if (StickyRecords[idx].hScrollBar != FALSE)
                    editWinStyle |= WS_HSCROLL;


                // Allocate global memory for sticky edit control to allow text > 30k, this method
                // is only supported in Windows NT! Start for 512bytes and grow...
                if (GetVersion() <  0x80000000) // Windows NT
                {                    
                    hEdGlobalMem = GlobalAlloc(GMEM_DDESHARE | GMEM_MOVEABLE | GMEM_ZEROINIT,  EDITGLOBALINITSIZE);
                    if (hEdGlobalMem == NULL)
                        hEdGlobalMem = HInst;   // use local memory within application
                }

                StickyRecords[idx].editControl = CreateWindowEx (WS_EX_RTLREADING | WS_EX_LEFTSCROLLBAR, 
                                                                 "edit", NULL,
                                                                 editWinStyle,
                                                                 0, 0, 0, 0,
                                                                 hwnd, (HMENU)EDITWINID,
                                                                 hEdGlobalMem,
                                                                 NULL);
                                
                if (StickyRecords[idx].editControl == NULL)
                {
                    MessageBox    (hwnd, "Can't Create Edit Control", "Out Of Memory", MB_ICONSTOP | MB_OK);
                    DestroyWindow (hwnd);
                    return 0;
                }

                // Sub-class the edit control to process key press, mainly alt keys to 
                // act as accelerators to the associated sticky window (i.e. key short cuts).
                InitEditSubclass (hwnd, StickyRecords[idx].editControl, idx);

                // The EM_LIMITTEXT function will allow the edit field to
                // hold more than 32K.  The default for an edit field is
                // approximately 32K, but by sending a 0, 0L through as
                // the parameters, the default then becomes all available
                // memory or 64K which comes first.                    
                SendMessage(StickyRecords[idx].editControl, EM_LIMITTEXT, 0, 0L);
                                
                // Add sticky record ...
                StickyRecords[idx].stickyWnd   = hwnd;
                StickyRecords[idx].backGH      = CreateSolidBrush(StickyRecords[idx].backCol);

                // Set the font for edit control
                FontSetFont (StickyRecords[idx].editControl, StickyRecords[idx].editFont, StickyRecords[idx].hFont);

                // Determine what state the window is .. i.e rolled or unrolled.
                if (StickyRecords[idx].rolledUp != FALSE)
                {                    
                    MoveWindow   (StickyRecords[idx].stickyWnd, 
                                  StickyRecords[idx].winDim.left, 
                                  StickyRecords[idx].winDim.top,
                                  StickyRecords[idx].winDim.right - StickyRecords[idx].winDim.left, 
                                  0, 
                                  TRUE);

                    ModifyMenu(GetSystemMenu (StickyRecords[idx].stickyWnd, FALSE),
                                              ROLLUPID, MF_STRING, ROLLUPID, STRUNROLL);
                }
                
                // Load string info into edit control.
                if (StickyRecords[idx].userMemo.pString != NULL)
                {
                    SetWindowText(StickyRecords[idx].editControl, (LPCSTR) StickyRecords[idx].userMemo.pString);

                    delete[]  StickyRecords[idx].userMemo.pString;
                    StickyRecords[idx].userMemo.pString = NULL;
                }

                // Set the caption if there is one.
                if (strlen (StickyRecords[idx].winCaption) > 0)
                    SetWindowText (StickyRecords[idx].stickyWnd, StickyRecords[idx].winCaption);

                if (StickyRecords[idx].winShowState == SW_SHOWMINIMIZED)
                    StickyRecords[idx].minimised = TRUE;                    

                DragAcceptFiles(hwnd , TRUE);  // initalise drag and drop                              
            }
            return 0;

        case WM_DROPFILES:
            {
                FARPROC fpfnDDQuery = MakeProcInstance ((FARPROC)DragDropQuery, HInst);
                int mode = 0;                
            
                idx = GetWndIdx (hwnd);

                // Check to see if there is anything to over write text?
                if (GetWindowTextLength(StickyRecords[idx].editControl) <= 0)
                    mode = IDC_DDOVERWRITE;
                else
                    mode = DialogBox(HInst, MAKEINTRESOURCE(IDD_DRAGDROPQUERY), hwnd, (DLGPROC) fpfnDDQuery);

                switch (mode)
                {
                    case IDC_DDAPPEND:
                    case IDC_DDOVERWRITE:
                        // determine where this message came from, user drag/drop
                        // or from import menu item
                        if ((lParam != FALSE) && (wParam == NULL))
                            ImportStickyFileIntoEditControl (mode, GetWndIdx (hwnd));
                        else
                            ProcessDragDropFiles (wParam, mode, GetWndIdx (hwnd));
                        break;

                    case IDC_DDCANCEL:
                    default:
                        break;
                }
                
                FreeProcInstance (fpfnDDQuery);
            }
            return TRUE;

        case WM_SETFOCUS:            
            SetFocus (GetStickyEditControl (hwnd));

            // this is used to notify possible alarm blinking threads to 
            //stop ... another simple hack!
            StickyRecords[GetWndIdx (hwnd)].stickyAlrm.alarmBlinker = FALSE;
            return 0;

        case WM_SIZE:            
            {                
                idx = GetWndIdx (hwnd);
                int titleHeight = GetStickyTitleFontHeight (hwnd, idx);

                MoveWindow (GetStickyEditControl (hwnd), 
                            0, 
                            titleHeight,
                            LOWORD (lParam), 
                            HIWORD (lParam) - titleHeight, TRUE);            
            }
            return TRUE;
           
        case WM_GETMINMAXINFO:
            {
                idx = GetWndIdx (hwnd);

                LPMINMAXINFO lpmmi       = (LPMINMAXINFO) lParam;
                int          fontHeight  = GetStickyTitleFontHeight (hwnd, idx);                

                // If in window is in roll state then ignore y size.
                // don't change the below code or things don't work right!
                if ((StickyRecords[idx].rolledUp != FALSE) &&
                   ((StickyRecords[idx].minimised == FALSE) &&
                    (StickyRecords[idx].restored == FALSE)))
                {
                    if (StickyRecords[idx].thinEdge != FALSE)
                        lpmmi->ptMaxTrackSize.y = (fontHeight + (GetSystemMetrics (SM_CYEDGE) * 2));
                    else
                        lpmmi->ptMaxTrackSize.y = (fontHeight + (GetSystemMetrics (SM_CYSIZEFRAME) * 2)) - GetSystemMetrics (SM_CYEDGE);
                }

                if (StickyRecords[idx].thinEdge != FALSE)
                    lpmmi->ptMinTrackSize.y = (fontHeight + (GetSystemMetrics (SM_CYEDGE) * 2));
                else
                    lpmmi->ptMinTrackSize.y = (fontHeight + (GetSystemMetrics (SM_CYSIZEFRAME) * 2)) - GetSystemMetrics (SM_CYEDGE);
             }
            break;

        // Set the colour of the windows ...
        case WM_CTLCOLOREDIT:
            idx = GetWndIdx (hwnd);

            // Set text to what every is within the sticky record. 
            SetTextColor((HDC) wParam, StickyRecords[idx].textCol);
            SetBkColor  ((HDC) wParam, StickyRecords[idx].backCol);
            return (long) StickyRecords[idx].backGH;

        case WM_PAINT:
            {
                idx = GetWndIdx (hwnd);
                RECT     paintArea;
                HDC      hdc     = (HDC) wParam;
                HBRUSH   paintBrush;
                COLORREF testCapCol;        // used to determine flashing nore normal caption colouring.
                COLORREF captionBKCol;              
                HFONT    printFont, oldFont;
                int      r, g, b;
                int      fontHeight = GetStickyTitleFontHeight (hwnd, idx);

                // If wParam HDC is null, then allocate one, is that is null then break out
                if (hdc == NULL)
                {
                    if ((hdc = GetDC (hwnd)) == NULL)
                        break;
                }

                // Outputmode
                SetMapMode(hdc, MM_TEXT);

                // Set the background color for the title bar
                if (StickyRecords[idx].stickyAlrm.alarmBlinker != FALSE)
                    testCapCol = StickyRecords[idx].stickyAlrm.alarmBlinkCol;
                else
                    testCapCol = StickyRecords[idx].backCol;

                if ((r = (GetRValue (testCapCol) - 30)) < 0)
                    r = 0;
                if ((g = (GetGValue (testCapCol) - 30)) < 0)
                    g = 0;
                if ((b = (GetBValue (testCapCol) - 30)) < 0)
                    b = 0;
                
                captionBKCol = GetNearestColor(hdc, RGB(r,g,b));
                
                // Define the paint area                
                GetClientRect(hwnd, &paintArea);
                paintArea.left   = 0;
                paintArea.top    = 0;
                paintArea.bottom = fontHeight;
                paintBrush       = CreateSolidBrush (captionBKCol);
                FillRect (hdc, &paintArea, paintBrush);

                // output sticky caption
                printFont = CreateFontIndirect(&StickyRecords[idx].titleFont);
                oldFont   = SelectObject (hdc, printFont);
                SetTextColor (hdc, StickyRecords[idx].textCol); 
                SetBkColor   (hdc, captionBKCol);
                DrawText(hdc, 
                         (LPCTSTR) StickyRecords[idx].winCaption, 
                         strlen (StickyRecords[idx].winCaption), 
                         &paintArea, 
                         DT_LEFT | DT_VCENTER );

                // Output icon button thingie
                DrawIconEx(hdc, (paintArea.right-fontHeight)-1, 
                                0, 
                                IconBut1, 
                                fontHeight-1, 
                                fontHeight-1, 
                                0, 
                                NULL, 
                                DI_NORMAL );

                // Output icon if alarm is set for sticky
                if (StickyRecords[idx].stickyAlrm.alarmEnabled != FALSE)                
                    DrawIconEx(hdc, (paintArea.right-(fontHeight*2))-1, 
                                    0, 
                                    IconAlarm, 
                                    fontHeight-1, 
                                    fontHeight-1, 
                                    0, 
                                    NULL, 
                                    DI_NORMAL);

                // Output icon if sticky is in calculator mode
                if (StickyRecords[idx].calculatorMode != FALSE)                
                    DrawIconEx(hdc, (paintArea.right-(fontHeight*2))-1, 
                                    0, 
                                    IconCalcMode, 
                                    fontHeight-1, 
                                    fontHeight-1, 
                                    0, 
                                    NULL, 
                                    DI_NORMAL);


                // if in 256 colour mode, then define the regions between title and edit control
                if ((testCapCol == captionBKCol) && (StickyRecords[idx].rolledUp == FALSE))
                    DrawEdge (hdc, &paintArea, EDGE_SUNKEN, BF_BOTTOM); 

                // Release and free objects
                DeleteObject (SelectObject (hdc, oldFont));
                DeleteObject (paintBrush);
                ReleaseDC (hwnd, hdc);                
            }   
            break;

        case WM_MOUSEMOVE:
            if ((wParam == MK_LBUTTON) && (moveIsOk != FALSE)) 
                ProcessWindowMoving (hwnd, lParam, lastLButtonDn);
            else
                ReleaseCapture();
            break;

        case WM_LBUTTONDBLCLK:
            idx = GetWndIdx (hwnd);

            // notify this window to roll up
            if ((int)HIWORD(lParam) <= GetStickyTitleFontHeight (hwnd, idx))
            {
                PostMessage (hwnd, WM_COMMAND, ROLLUPID, 0);
                // Determine what key is current down, if its the shift key then 
                // reset roll and unroll window width sizes.
                if ((wParam & MK_SHIFT) != 0)
                {                
                    RECT curRect;

                    GetWindowRect (hwnd, &curRect);
                    StickyRecords[idx].unrolledDim.left  = curRect.left;
                    StickyRecords[idx].unrolledDim.right = curRect.right;
                    StickyRecords[idx].rolledDim.left    = curRect.left;
                    StickyRecords[idx].rolledDim.right   = curRect.right;
                }
            }
            break;
    
        case WM_LBUTTONDOWN:
            {
                idx = GetWndIdx (hwnd);

                RECT clientRect;
                int  fontHeight = GetStickyTitleFontHeight (hwnd, idx);

                lastLButtonDn.x = LOWORD(lParam);
                lastLButtonDn.y = HIWORD(lParam);                
                GetClientRect(hwnd, &clientRect);                

                // Test to see if the user is able to move the window
                if ((lastLButtonDn.y <= fontHeight) && 
                    (lastLButtonDn.x < (clientRect.right - fontHeight)))
                {
                    SetCapture(hwnd);
                    moveIsOk = TRUE;
                }
                else if (lastLButtonDn.x >= (clientRect.right - fontHeight))
                {
                    HDC hdc = GetDC (hwnd);

                    // output close sticky button thingie
                    DrawIconEx(hdc, 
                               (clientRect.right - fontHeight)-1, 
                               0, 
                               IconBut2, 
                               fontHeight-1,
                               fontHeight-1, 
                               0, 
                               NULL, 
                               DI_NORMAL);

                    ReleaseDC (hwnd, hdc);
                     
                    // fill in rect of where the current icon is                                        
                    closeButRect.left   = clientRect.right - fontHeight,
                    closeButRect.top    = 0;
                    closeButRect.right  = clientRect.right;
                    closeButRect.bottom = fontHeight;                    
                }     
                else
                    moveIsOk = FALSE;            
            }
            break;

        case WM_LBUTTONUP:            
            {            
                idx = GetWndIdx (hwnd);
                
                RECT   clientRect;
                int    fontHeight = GetStickyTitleFontHeight (hwnd, idx);
                HDC    hdc = GetDC (hwnd);

                lastLButtonDn.x = LOWORD(lParam);
                lastLButtonDn.y = HIWORD(lParam);                
                GetClientRect(hwnd, &clientRect);                

                if (moveIsOk != FALSE)
                {
                    moveIsOk = FALSE;
                    ReleaseCapture ();
                }

                // Test if mouse is still within area to close sticky
                if (PtInRect (&closeButRect, lastLButtonDn) != FALSE)
                    PostMessage (hwnd, WM_COMMAND, CLOSESTICKYID, 0);

                DrawIconEx(hdc, 
                           (clientRect.right - fontHeight)-1, 
                           0, 
                           IconBut1, 
                           fontHeight-1,
                           fontHeight-1, 
                           0, 
                           NULL, 
                           DI_NORMAL);

                ReleaseDC (hwnd, hdc);
            }
            break;

        case WM_CONTEXTMENU:
            {
                idx = GetWndIdx (hwnd);
                POINT pos;
               
                // Set the checkable items in the menu
                if (StickyRecords[idx].topMostWin != FALSE)
                    CheckMenuItem(hPopupMenu, SETTOPMOSTID, MF_CHECKED);
                else
                    CheckMenuItem(hPopupMenu, SETTOPMOSTID, MF_UNCHECKED);

                if (StickyRecords[idx].stickyAlrm.alarmEnabled != FALSE)
                    CheckMenuItem(hPopupMenu, SETALARMID, MF_CHECKED);
                else
                    CheckMenuItem(hPopupMenu, SETALARMID, MF_UNCHECKED);

                if (StickyRecords[idx].calculatorMode != FALSE)
                    CheckMenuItem(hPopupMenu,  (UINT)hPopupCalcMenu, MF_CHECKED);
                else
                    CheckMenuItem(hPopupMenu,  (UINT)hPopupCalcMenu, MF_UNCHECKED);

                if (StickyRecords[idx].rolledUp == FALSE)                    
                    ModifyMenu(hPopupMenu, ROLLUPID, MF_STRING, ROLLUPID, STRROLLUP);
                else
                    ModifyMenu(hPopupMenu, ROLLUPID, MF_STRING, ROLLUPID, STRUNROLL);
                   
                GetCursorPos(&pos); 

                TrackPopupMenuEx(hPopupMenu, TPM_TOPALIGN | TPM_LEFTALIGN , pos.x, pos.y, hwnd, NULL);                
            }
            break;

        case WM_HELP:
            WinHelp(hwnd, STRSTICKYHELP, HELP_FINDER, 0);
            break;

        // ---------------------------------------------------------------------------------------------------------------------------
        // ---------------------------------------------------------------------------------------------------------------------------
        // Process user commands ...
        case WM_COMMAND:
            switch (LOWORD(wParam))
            {
                // -------------------------------------------------------------------------------------------------------------------
                case MAKEASDEFAULTID:
                    if (MessageBox (hwnd, 
                                    "This is an unrecoverable action.\n"
                                    "Are you sure you want to make this\n"
                                    "sticky your default sticky style?", 
                                    "*** Look Here ***", 
                                    MB_ICONINFORMATION | MB_YESNO) == IDYES)
                    {
                        DuplicateSticky (GetWndIdx (hwnd), STICKRECNUM);
                    }
                    break;
                // -------------------------------------------------------------------------------------------------------------------
                case EXPORTSTICKYID:
                    ExportStickyFile (GetWndIdx (hwnd));
                    break;

                case IMPORTSTICKYID:
                    SendMessage (hwnd, WM_DROPFILES, NULL, TRUE);                    
                    break;

                // -------------------------------------------------------------------------------------------------------------------
                case PREFSID:
                    static int     result;

                    if (fpfnPrefsDialog == NULL)
                    {
                        
                        fpfnPrefsDialog = MakeProcInstance ((FARPROC)StickyPrefs, HInst);
                        result          = DialogBoxParam (HInst, 
                                                          MAKEINTRESOURCE(IDD_PREFS), 
                                                          hwnd,
                                                          (DLGPROC) fpfnPrefsDialog, 
                                                          (LPARAM) GetWndIdx (hwnd));

                        FreeProcInstance (fpfnPrefsDialog);
                        fpfnPrefsDialog = NULL;
                        
                        if (result != 0)
                        {
                            int copyIdx  = GetNextStickyFree ();
                            int prefsIdx = GetWndIdx (hwnd);

                            if (copyIdx < 0)
                                return 0;

                            // Create a new instance of the window with user's changes.
                            StoreStickyData (prefsIdx, TRUE);                            
                            DestroyWindow   (StickyRecords[prefsIdx].stickyWnd);
                            StickyRecords[prefsIdx].stickyWnd = NULL;                                                      
                            LoadSticky      (ParentStarter, prefsIdx, FALSE);
                        }
                    }
                    else
                        MessageBox (hwnd, "Only one instance of the preference dialog box\ncan be open at a time.",
                        "Preference Open Error",
                            MB_ICONINFORMATION | MB_OK);
                    break;
                // -------------------------------------------------------------------------------------------------------------------
                case SETTITLEID:
                    if (fpfnSetStickyTitle == NULL)
                    {
                        fpfnSetStickyTitle = MakeProcInstance ((FARPROC)StickyTitle, HInst);

                        DialogBoxParam(HInst, MAKEINTRESOURCE(IDD_SETTITLE), hwnd,
                        (DLGPROC) fpfnSetStickyTitle, (LPARAM) GetWndIdx (hwnd));
                        FreeProcInstance (fpfnSetStickyTitle);
                        fpfnSetStickyTitle = NULL;
                        InvalidateRect(hwnd, NULL, TRUE);
                        UpdateWindow (hwnd);
                    }
                    else
                        MessageBox (hwnd, "Only one instance of the sticky title change dialog box\ncan be open at a time.",
                            "Sticky Title Open Error", MB_ICONINFORMATION | MB_OK);
                    break;
                // -------------------------------------------------------------------------------------------------------------------
                case SETALARMID:
                    if (fpfnSetAlarmDialog == NULL)
                    {
                        fpfnSetAlarmDialog = MakeProcInstance ((FARPROC)SetAlarmDialog, HInst);

                        DialogBoxParam(HInst, 
                                       MAKEINTRESOURCE(IDD_ALARM), 
                                       hwnd, 
                                       (DLGPROC)fpfnSetAlarmDialog, 
                                       (LPARAM) GetWndIdx (hwnd));
                        FreeProcInstance (fpfnSetAlarmDialog);
                        fpfnSetAlarmDialog = NULL;

                        // Make sure alarm icon is displayed if it needs to be
                        InvalidateRect (hwnd, NULL, FALSE);
                    }
                    else
                        MessageBox (hwnd, "Only one instance of the alarm dialog box\ncan be open at a time.",
                            "Alarm Open Error", MB_ICONINFORMATION | MB_OK);
                    break;
                // -------------------------------------------------------------------------------------------------------------------
                case SETTOPMOSTID:
                    {
                        // Change the window state to top/bottom most window.
                        // Get window parameters .. and change them.
                        RECT area;

                        idx = GetWndIdx (hwnd);

                        GetWindowRect (hwnd, &area);
                        if (StickyRecords[idx].topMostWin == FALSE)
                        {
                            StickyRecords[idx].topMostWin = TRUE;
                            SetWindowPos(hwnd, HWND_TOPMOST,  
                                         area.left,
                                         area.top,
                                         0, 0, SWP_NOSIZE | SWP_SHOWWINDOW);
                        }
                        else
                        {
                             StickyRecords[idx].topMostWin = FALSE;
                             SetWindowPos(hwnd, HWND_NOTOPMOST,  
                                          area.left,
                                          area.top,
                                          0, 0, SWP_NOSIZE | SWP_SHOWWINDOW);
                        }                        
                    }
                    break;
                // -------------------------------------------------------------------------------------------------------------------
                case NEWSTICKYID:
                    {
                        int newIdx = GetNextStickyFree ();
                        
                        if (newIdx < 0)
                        {
                            MessageBox    (hwnd, "Can't Create Any More Stickies", "Out Of Array Space", MB_ICONSTOP | MB_OK);
                            break;
                        }
                        
                        DuplicateSticky    (GetWndIdx (hwnd), newIdx);
                        LoadSticky         (ParentStarter, newIdx, TRUE);                        
                        SetForegroundWindow(StickyRecords[newIdx].stickyWnd);
                        SetFocus           (StickyRecords[newIdx].editControl);
                    }
                    break;
                // -------------------------------------------------------------------------------------------------------------------
                case CLOSESTICKYID:
                    if (MessageBox (hwnd, 
                                    "Are you sure you want to close and kill sticky?", 
                                    "*** Look Here ***", 
                                    MB_ICONINFORMATION | MB_YESNO) == IDYES)
                    {
                        // Make sure parent has focus or all other child windows disappear because
                        // the current focus is lost.
                        SetForegroundWindow(ParentStarter);
                        DestroyWindow(hwnd);
                    }
                    break;                   
                // -------------------------------------------------------------------------------------------------------------------                
                case ROLLUPID:
                    {
                        idx = GetWndIdx (hwnd);

                        int fontHeight = GetStickyTitleFontHeight (hwnd, idx);

                        if (idx < 0) 
                            break;               // some funny error!

                        // Save and restore the window size
                        if (StickyRecords[idx].rolledUp == FALSE)
                        {                                                    
                            int   titleHeight;

                            if (SoundActive != FALSE)
                                PlaySound (RollupSnd, NULL, SND_ASYNC | SND_FILENAME);

                            // save window size
                            GetWindowRect(hwnd, &StickyRecords[idx].winDim);
                            StickyRecords[idx].unrolledDim     = StickyRecords[idx].winDim;
                            StickyRecords[idx].unrolledDimInit = TRUE; 

                            // change size of the window
                            if (StickyRecords[idx].thinEdge != FALSE)
                                titleHeight = fontHeight + (GetSystemMetrics (SM_CYEDGE) * 2);
                            else
                                titleHeight = abs(fontHeight + (GetSystemMetrics (SM_CYSIZEFRAME) * 2)) - GetSystemMetrics (SM_CYEDGE);

                            if (StickyRecords[idx].rolledDimInit == FALSE)
                            {
                                StickyRecords[idx].rolledDimInit = TRUE;
                                StickyRecords[idx].rolledDim     = StickyRecords[idx].winDim;
                            }

                            MoveWindow(hwnd, 
                                       StickyRecords[idx].unrolledDim.left, 
                                       StickyRecords[idx].unrolledDim.top,
                                       StickyRecords[idx].rolledDim.right - StickyRecords[idx].rolledDim.left, 
                                       titleHeight,
                                       TRUE);

                            StickyRecords[idx].rolledUp = TRUE;                            
                        }
                        else
                        {                            
                            int   titleHeight = GetStickyTitleFontHeight (hwnd, idx);

                            if (SoundActive != FALSE)
                                PlaySound (UnrollSnd, NULL, SND_ASYNC | SND_FILENAME);

                            StickyRecords[idx].rolledUp = FALSE;

                            // restore window size, and save the current window size
                            GetWindowRect(hwnd, &StickyRecords[idx].rolledDim);
                            StickyRecords[idx].rolledDimInit = TRUE;                            
                            
                            if (StickyRecords[idx].unrolledDimInit == FALSE)
                            {
                                StickyRecords[idx].unrolledDimInit = TRUE;
                                StickyRecords[idx].unrolledDim     = StickyRecords[idx].winDim;
                            }

                            MoveWindow(hwnd, 
                                       StickyRecords[idx].rolledDim.left, 
                                       StickyRecords[idx].rolledDim.top,
                                       StickyRecords[idx].unrolledDim.right  - StickyRecords[idx].unrolledDim.left,
                                       StickyRecords[idx].unrolledDim.bottom - StickyRecords[idx].unrolledDim.top, TRUE);

                            // Resize the edit control
                            RECT  newRect;
                            GetClientRect (hwnd, &newRect);
                            MoveWindow (GetStickyEditControl (hwnd), 
                                        0, 
                                        titleHeight,
                                        newRect.right   - newRect.left, 
                                        (newRect.bottom - newRect.top) - titleHeight, TRUE);            
                        }
                    }
                    return TRUE;
                // -------------------------------------------------------------------------------------------------------------------  
                case PRINTSTICKYID:
                    {
                        idx = GetWndIdx (hwnd);

                        PrntPrintFile (HInst, 
                                       hwnd,  
                                       StickyRecords[idx].winCaption, 
                                       GetStickyEditControl (hwnd),
                                       &StickyRecords[idx].editFont);

                    }
                    return TRUE;
                // -------------------------------------------------------------------------------------------------------------------  
                case FINDTEXTID:
                    if (HFndRplDialog != NULL)
                    {
                        MessageBox (hwnd, "Only one instance of the find/replace dialog box\ncan be open at a time.",
                        "Find/Replace Open Error", MB_ICONINFORMATION | MB_OK);
                        break;
                    }

                    fndRplInfo.hwndOwner = hwnd;
                    fndRplInfo.Flags     = FR_HIDEWHOLEWORD | FR_DOWN;
                    HFndRplDialog        = FindText (&fndRplInfo);
                    return TRUE;
                // -------------------------------------------------------------------------------------------------------------------  
                case FINDREPLACEID:
                    if (HFndRplDialog != NULL)
                    {
                        MessageBox (hwnd, "Only one instance of the find/replace dialog box\ncan be open at a time.",
                        "Find/Replace Open Error", MB_ICONINFORMATION | MB_OK);
                        break;
                    }

                    fndRplInfo.hwndOwner = hwnd;
                    fndRplInfo.Flags     = FR_HIDEWHOLEWORD | FR_DOWN;
                    HFndRplDialog        = ReplaceText (&fndRplInfo);
                    return TRUE;
                // -------------------------------------------------------------------------------------------------------------------  
                case CALCULATORMODEID:
                    idx = GetWndIdx (hwnd);
                    if (StickyRecords[idx].calculatorMode != FALSE)
                        StickyRecords[idx].calculatorMode = FALSE;
                    else
                        StickyRecords[idx].calculatorMode = TRUE;

                    // Make sure alarm icon is displayed if it needs to be
                    InvalidateRect (hwnd, NULL, FALSE);
                    break;

                case CALCULATESELECTID:
                    CalculateSelection (hwnd);
                    break;

                // -------------------------------------------------------------------------------------------------------------------  
                case MAILTONOINSTALLID:
                case MAILTOINSTALLID:
                    {
                        int  rtnCode;

                        if (LOWORD(wParam) == MAILTONOINSTALLID)
                            rtnCode = SendStickyDocument (GetWndIdx (hwnd), FALSE);
                        else
                            rtnCode = SendStickyDocument (GetWndIdx (hwnd), TRUE);

                        if (rtnCode != 0)                        
                        {
                            char errorTxt[128] = {'\0'};

                            switch (rtnCode)
                            {
                                case -1:
                                    sprintf (errorTxt, "Memory allocation error in MAPI mail\r\n"
                                                    "construction. Error code = %d", -1);
                                    break;

                                case -2:        // user abort
                                    return TRUE;                                    

                                case -3:
                                    sprintf (errorTxt, "WARNING: Failed to create temparary trojan file.\r\n"
                                                       "Sticky export partially completed. Error code = %d", -3);
                                    break;

                                default:                                
                                    sprintf (errorTxt, "Unkown error occurred in MAPI mail\r\n"
                                                    "construction. Error code = %d", rtnCode);                                    
                            }

                            MessageBox (hwnd, errorTxt, 
                                        "Mail To Operation Failed!", MB_ICONINFORMATION | MB_OK);
                        }                        
                    }
                    return TRUE;

            }
            break;

        case WM_DESTROY:
            NukeRec (hwnd);
            DragAcceptFiles(hwnd , FALSE);

            handleCounter--;
            if (handleCounter <= 0)
            {
                if (hPopupMenu != NULL)
                    DestroyMenu (hPopupMenu);

                if (hPopupFndMenu != NULL)
                    DestroyMenu (hPopupFndMenu);

                if (hPopupMailMenu != NULL)
                    DestroyMenu (hPopupMailMenu);                

                if (hPopupCalcMenu != NULL)
                    DestroyMenu (hPopupCalcMenu);                

                if (hPopupIOMenu != NULL)
                    DestroyMenu (hPopupIOMenu);

                hPopupIOMenu   = hPopupMenu     = hPopupFndMenu = 
                hPopupMailMenu = hPopupCalcMenu = NULL;
            }
            break;

        default:
            if (message == MFndRplReGMsg)
            {
                FINDREPLACE* pFR    = (FINDREPLACE*) lParam;
                static BOOL  msgBox = FALSE; // used to stop display of multiple message boxes.
               
                idx  = GetWndIdx (hwnd);

                if (pFR->Flags & FR_DIALOGTERM)
                { 
                    HFndRplDialog = NULL;                     
                    return TRUE;
                }

                if (pFR->Flags & FR_FINDNEXT)
                {
                    if (msgBox != FALSE)
                        MessageBeep (MB_ICONEXCLAMATION);
                    else if (FindText (StickyRecords[idx].editControl, pFR) < 0)
                    {
                        msgBox = TRUE;                        
                        MessageBox (hwnd, "Text Not Found In Sticky.", 
                                    "Text not found!", MB_ICONINFORMATION | MB_OK | MB_APPLMODAL);
                        msgBox = FALSE;
                    }
                }

                if (pFR->Flags & FR_REPLACE)
                {
                    if (msgBox != FALSE)
                        MessageBeep (MB_ICONEXCLAMATION);
                    else if (ReplaceText (StickyRecords[idx].editControl, pFR) < 0)
                    {
                        msgBox = TRUE;                        
                        MessageBox (hwnd, "Text Not Found In Sticky.", 
                                    "Text not found!", MB_ICONINFORMATION | MB_OK | MB_APPLMODAL);
                        msgBox = FALSE;
                    }
                }

                if (pFR->Flags & FR_REPLACEALL)
                {
                    SendMessage (StickyRecords[idx].editControl, EM_SETSEL, 0, 0);
                    
                    // Raplace all occurances
                    while (ReplaceText (StickyRecords[idx].editControl, pFR) >= 0);
                }

            }
    }

    return DefWindowProc (hwnd, message, wParam, lParam);
}

// ************************************************************************************
// ************************************************************************************
// ************************************************************************************

DWORD WINAPI AboutScrollerThread (LPVOID pData) 
{
    const int            shift     = 10;
    const int            degNum    = 360;
    unsigned long        sine[degNum];
    HWND                 scroller  = GetDlgItem ((HWND) pData, IDC_SCROLLER);
    HDC                  hDispDC, hVirtDC;
    int                  xPlotPos  = 0;
    int                  yPlotPos;
    RECT                 wndRect;
    HBITMAP              hDrawBitmap, hOldBitmap;    
    DWORD                startTime, endTime;
    char*                pStr =         
        "This program was written by Steven De Toni from 1997 to May 1999 and"
        " classed as FREEWARE. As long as the program code is not part of any"
        " commercial product. You may use any part of this program and/or"
        " source code for other FREEWARE products. Use this program at your own"
        " risk! The author will not be held responsible for any damage this program"
        " may cause you or your system.............................Special thanks"
        " go to the many people who have given constructive comments to improve stickies:"
        " Keith Davies, Fernand Maquet, Kenny Hardie,"
        " Lars Albertsen, Lucich, Marcel vanBreeden, Mark Zec, Peter Wiwe, Shawn,"
        " Woody Lesmana, my wife Amy for putting up with me and any others I"
        " have replied e-mail to.........................."
        " This will be the FINAL release of this program by me, my initial"
        " design of the program was to be small, fast, compact and usable."
        " I feel I have achieved this; any more additional features to the program"
        " will probably hinder my original design. That's it.............................Enjoy "
        " Stickies V1.60 Bye.............................Oh, I forgot the catch phrase of the"
        " century.............................This product is Y2K friendly........"
        ".....................(what a joke!)............................."
        "How to make quick scones: INGREDIENTS... 4 cups self-raising flour"
        ",  300ml cream (1 carton or small bottle of cream)"
        ",  1 can Lemon & Paeroa or lemonade (355ml)"
        ",  1/2 teaspoon salt............................."
        "[STEP 1] Mix all ingredients in a bowl to a smooth dough."
        " [STEP 2] Tip out onto a well-floured surface and cut into squares or"
        " press out with a round cookie cutter. "
        "[STEP 3 ] Bake at 220C (fan bake if possible) for about 15-20 minutes"
        " until starting to colour golden. Check they are cooked through and"
        " cool on a wire rack. Serve warm with jam or whole fruit preserves and"
        " whipped cream, garnish with fresh fruit as desired...... YUMMY!";

       
    if ((scroller == NULL) || (pData == NULL))
        return FALSE;
   
    hDispDC = GetDC (scroller);    

    // Build sine code table ...
    for (int build = 0; build < degNum; build++)    
        sine[build] = (long) (sin((float)build / 57.29577951) * (float)(1 << shift));
    
    // Get output size...
    SetMapMode    (hDispDC, MM_TEXT);
    GetClientRect (scroller, &wndRect);
    
    // Build off screen bitmap
    hVirtDC      = CreateCompatibleDC  (hDispDC);
    SetMapMode (hVirtDC, MM_TEXT);
    hDrawBitmap = CreateCompatibleBitmap  (hDispDC, wndRect.right, wndRect.bottom);
    hOldBitmap  = SelectObject (hVirtDC, hDrawBitmap);    

    // sinus scrolling variables
    int  sPos;
    int  freq = 3;
    int  mid  = 4;
    int  yPut;   
    int  xPixelLine;
    int  yDisplacement;
    SIZE txtInfo;

    GetTextExtentPoint(hVirtDC, pStr, strlen(pStr), &txtInfo);
    yPlotPos      = (wndRect.bottom / 2) - (txtInfo. cy / 2);
    xPlotPos      = wndRect.right + 1;
    yDisplacement = yPlotPos -  ((sine[45] * mid) >> shift);  // (char pos) - (max sine height), approx calculation
    
    while (IsWindow (scroller) != FALSE)
    {
        
        // profile performance of the machine.
        startTime = GetTickCount ();

        FillRect(hVirtDC, &wndRect, GetStockObject(WHITE_BRUSH));
        TextOut (hVirtDC, xPlotPos, yPlotPos, pStr, strlen(pStr));
        sPos = 1;
        for (xPixelLine = 0; xPixelLine < wndRect.right; xPixelLine++)
        {
            yPut = ((int) (sine [sPos % degNum] * mid) >> shift) + yPlotPos;

            // Plaster bitmap onto sine wave!
            BitBlt (hVirtDC, xPixelLine, yPut, 1, wndRect.bottom,
                    hVirtDC, xPixelLine, yDisplacement, SRCCOPY);

            sPos += freq;
        }

        // Blast bitmap onto screen
        BitBlt (hDispDC, 0, 0, wndRect.right, wndRect.bottom,
                hVirtDC, 0, 0, SRCCOPY);
        
        // Adjust time to 60 FPS
        endTime = GetTickCount ();

        if ((endTime-startTime) < 15)
            Sleep(15 - (endTime-startTime));

        if (xPlotPos < -txtInfo.cx)
            xPlotPos = wndRect.right + 1;
        else
            xPlotPos--;
    }

    DeleteObject (SelectObject (hVirtDC, hOldBitmap));
    ReleaseDC (scroller, hVirtDC);
    ReleaseDC (scroller, hDispDC);

    return TRUE;
}

BOOL CALLBACK AboutDialog (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
#define TIMERICONUPDATE 1
#define TIMERROLLWINDOW 2

    static Ball iconBall[4];    
    static int  idx;    
    static int  curScore;
    static int  timeLimit;
    static char buff[32];
    static int  rollInc;

    switch (message)
    {
        case WM_INITDIALOG:
            {
                RECT area, areaScroll;                
                int  top;
                
                GetWindowRect (hwnd, &area);
                SetWindowPos(hwnd, HWND_TOPMOST,  
                             (GetSystemMetrics(SM_CXSCREEN) / 2) - ((area.right - area.left)/2),
                             (GetSystemMetrics(SM_CYSCREEN) / 2) - ((area.bottom - area.top)/2),
                             0, 0, SWP_NOSIZE | SWP_SHOWWINDOW);

                
                // Get scroll area window top position 
                GetWindowRect (hwnd, &areaScroll);                
                top = areaScroll.top;
                GetWindowRect (GetDlgItem (hwnd, IDC_SCROLLER), &areaScroll);                

                GetClientRect (hwnd, &area);
                for (idx = IDC_MEMOICON1; idx <= IDC_MEMOICON4; idx++)
                    iconBall[idx-IDC_MEMOICON1].Init (3,         3,
                                                      area.left, area.right,
                                                      area.top,  (areaScroll.top - top) - 10,
                                                      hwnd,      idx);

                // Load the scores.
                sprintf (buff, "High Score: %d", AboutHighScore);
                SetWindowText (GetDlgItem (hwnd, IDC_ABOUTHIGHSCORE), (LPCTSTR)buff);
                curScore = 0;
                sprintf (buff, "Score: %d", curScore);
                SetWindowText (GetDlgItem (hwnd, IDC_ABOUTSCORE), (LPCTSTR)buff);

                rollInc = 1;

                // Start scroller thread                
                DWORD threadID;

                CreateThread (NULL, 512, AboutScrollerThread, (LPDWORD) hwnd, 0, &threadID);                                                
            }
            break;

        case WM_TIMER:   
            switch (wParam)
            {
                case TIMERICONUPDATE:
                    if (timeLimit > 0)
                    {
                        for (idx = IDC_MEMOICON1; idx <= IDC_MEMOICON4; idx++)
                        {
                            iconBall[idx-IDC_MEMOICON1].Move();
                            iconBall[idx-IDC_MEMOICON1].Show();
                        }

                        timeLimit--;                
                        sprintf (buff, "Timer: %d", timeLimit);
                        SetWindowText (GetDlgItem (hwnd, IDC_ABOUTTIMELIMIT), (LPCTSTR)buff);
                    }
                    else
                    {
                        KillTimer(hwnd, TIMERICONUPDATE);
                        ShowWindow (GetDlgItem (hwnd, IDC_RESTART), SW_SHOW); 
                    }
                    break;

                case TIMERROLLWINDOW:
                    // Rollup window
                    {
                        RECT winRect;

                        rollInc += 3;
                        GetWindowRect (hwnd, &winRect);
                        MoveWindow (hwnd, winRect.left, winRect.top, 
                                    winRect.right - winRect.left, 
                                    (winRect.bottom - winRect.top) -rollInc, TRUE);

                        if ((winRect.bottom - winRect.top) -rollInc <= 0)
                        {
                            KillTimer(hwnd, TIMERROLLWINDOW);
                            KillTimer(hwnd, TIMERICONUPDATE);
                            EndDialog(hwnd, FALSE);
                        }
                    }
                    break;
            }
            break;

        case WM_COMMAND:
            switch (LOWORD(wParam))
            {
                case IDC_RESTART:
                    ShowWindow (GetDlgItem (hwnd, IDC_RESTART), SW_HIDE); 
                    // Setup game system;
                    timeLimit = 2000; 
                    SetTimer (hwnd, TIMERICONUPDATE, 10, (TIMERPROC)AboutDialog);                
                    curScore = 0;
                    sprintf (buff, "Score: %d", curScore);
                    SetWindowText (GetDlgItem (hwnd, IDC_ABOUTSCORE), (LPCTSTR)buff);
                    break;
            }
            break;
            
        case WM_LBUTTONDOWN:
            if (timeLimit > 0)
            {
                POINT p;
                HWND hChild;
                
                p.x = LOWORD(lParam);
                p.y = HIWORD(lParam);

                if ((hChild = ChildWindowFromPointEx(hwnd, p, CWP_SKIPTRANSPARENT)) == NULL)
                    break;

                for (idx = IDC_MEMOICON1; idx <= IDC_MEMOICON4; idx++)
                {                    
                    if (GetDlgItem (hwnd, idx) == hChild)
                    {
                        curScore++;
                        goto displayScore;        
                    }
                }

                curScore--;
            displayScore:
                sprintf (buff, "Score: %d", curScore);
                SetWindowText (GetDlgItem (hwnd, IDC_ABOUTSCORE), (LPCTSTR)buff);

                // New high score.
                if (curScore > AboutHighScore)
                {
                    AboutHighScore  = curScore;
                    // Load the high score.
                    sprintf (buff, "High Score: %d", AboutHighScore);
                    SetWindowText (GetDlgItem (hwnd, IDC_ABOUTHIGHSCORE), (LPCTSTR)buff);
                }
            }
            break;

        case WM_CLOSE:
            SetTimer (hwnd, TIMERROLLWINDOW, 10, (TIMERPROC)AboutDialog);                            
            break;
    }

    return FALSE;
}

// ********************************************************************************************
// ********************************************************************************************
// ********************************************************************************************

BOOL TrayMessage(HWND hDlg, DWORD dwMessage, UINT uID, HICON hIcon, PSTR pszTip)
{
    BOOL res;

    NOTIFYICONDATA tnd;

    tnd.cbSize           = sizeof(NOTIFYICONDATA);
    tnd.hWnd             = hDlg;
    tnd.uID              = uID;
    tnd.uFlags           = NIF_MESSAGE|NIF_ICON|NIF_TIP;
    tnd.uCallbackMessage = MYWM_NOTIFYICON;
    tnd.hIcon            = hIcon;
    if (pszTip)
        lstrcpyn(tnd.szTip, pszTip, sizeof(tnd.szTip));
    else
        tnd.szTip[0] = '\0';

    res = Shell_NotifyIcon(dwMessage, &tnd);

    if (hIcon)
        DestroyIcon(hIcon);

    return res;
}

// ********************************************************************************************
// ********************************************************************************************
// ********************************************************************************************

inline void SetExecutePath (void)
{
    char path[MAXSNDFILELEN];    

    strcpy (path, __argv[0]);
    
    for(char* pC = (path + strlen (path)); (pC > path); pC--)
    {
        if (*pC == '\\')
            break;
        else
            *pC = '\0';
    }
    
    SetCurrentDirectory (path);   
}

int WINAPI WinMain (HANDLE hInstance, HANDLE hPrevInstance,
                    LPSTR lpszCmdParam, int nCmdShow)
{
    HWND        hwnd;
    MSG         msg;
    WNDCLASS    wndclass;    

    HInst = hInstance;
    SetExecutePath ();

    // Check if command line contains parameter to allow multiple instances of stickies
    if (strstr (lpszCmdParam, "-multinstances") == NULL) 
    {
        // Test for previous instances... if so then exit, only 1 instances necessary  
        // for DDE operations from explorer.
        if (FindWindow(STRWINAPPNAMECLASS, STRWINAPPNAMECLASS) != NULL)
            return -1;
    }

    memset(&wndclass, 0, sizeof (wndclass));
    memset(&StickClassRec, 0, sizeof (StickClassRec));        

    // **** Main Cordinator Window ****
    wndclass.style         = CS_HREDRAW | CS_VREDRAW;
    wndclass.lpfnWndProc   = (WNDPROC) MainWindow;
    wndclass.cbClsExtra    = 0;
    wndclass.cbWndExtra    = 0;
    wndclass.hInstance     = hInstance;
    wndclass.hIcon         = NULL;
    wndclass.hCursor       = LoadCursor     (NULL, IDC_ARROW);
    wndclass.lpszMenuName  = NULL;
    wndclass.lpszClassName = STRWINAPPNAMECLASS;

    // **** Create Sticky Work Class ****
    StickClassRec.style         = CS_HREDRAW | CS_VREDRAW | CS_DBLCLKS;
    StickClassRec.lpfnWndProc   = (WNDPROC) Sticky;
    StickClassRec.cbClsExtra    = 0;
    StickClassRec.cbWndExtra    = 0;
    StickClassRec.hInstance     = hInstance;
    StickClassRec.hIcon         = NULL;        
    StickClassRec.hCursor       = LoadCursor     (NULL, IDC_ARROW);
    StickClassRec.lpszMenuName  = NULL;
    StickClassRec.lpszClassName = STRWINSTICKYCLASS;

    RegisterClass (&wndclass) ;
    RegisterClass (&StickClassRec) ;

    // Load MAPI services...
    if (InitMAPI() != 0)
        MAPIActive = FALSE;

    // Startup DDE Server
    DDE_InitServer ();

    // Load Sticky Accelerators
    HStickyAccel = LoadAccelerators (HInst, MAKEINTRESOURCE(IDR_STICKYACCEL));

    hwnd = CreateWindow(STRWINAPPNAMECLASS,           // window class name
                        STRWINAPPNAMECLASS,           // window caption
                        WS_OVERLAPPEDWINDOW,          // WS_CAPTION |    WS_SYSMENU | WS_MINIMIZEBOX ,     // window style
                        0,                            // initial x position
                        0,                            // initial y position
                        50,                           // initial x size
                        50,                           // initial y size
                        GetDesktopWindow(),           // parent window handle
                        NULL,                         // window menu handle
                        hInstance,                    // program instance handle
                        NULL) ;                       // creation parameters

    if (hwnd == NULL)
        return -1;
  
    while (GetMessage (&msg, NULL, 0, 0))
    {
        // process find / replace dialog message if dialog box is open.
        if ((msg.hwnd == HFndRplDialog) && (HFndRplDialog != NULL))
            IsDialogMessage (HFndRplDialog, &msg);
        else
        {
            TranslateMessage (&msg);
            DispatchMessage  (&msg);
        }
    }

    // Cleanup registered information
    UnregisterClass(STRWINAPPNAMECLASS, HInst);
    UnregisterClass(STRWINSTICKYCLASS, HInst);

    // Unload MAPI services...
    if (MAPIActive != FALSE)
         DeInitMAPI ();

    // Shutdown DDE Server
    DDE_UninitServer ();

    return msg.wParam ;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

int WINAPI MainWindow (HWND hwnd, UINT message, WPARAM wParam, LPARAM lParam)
{
#define TIMERALARM 0
#define TIMERSAVE  1

    static FARPROC fpfnAboutDialog   = NULL;
    static FARPROC   fpfnPrefsDialog = NULL;
    static TIMERPROC lpfnTimerProc   = NULL;  
    static HMENU     hPopupMenu      = NULL;
    static HMENU     hPopupUtilsMenu = NULL;

    switch (message)
    {
        case WM_CREATE:
            {
                // Initialise Globals ...                
                ParentStarter = hwnd;                
                InitStickyRecords ();
                MakeDefaults ();
                LoadMainStickyPrefs ();                                

                // *** Start the timer to test alarms on current open windows...
                lpfnTimerProc = (TIMERPROC) MakeProcInstance((FARPROC) MainWindow, HInst);
                
                // update every 60 seconds, and hour
                SetTimer (hwnd, TIMERALARM, 60000,  lpfnTimerProc);
                SetTimer (hwnd, TIMERSAVE,  600000, lpfnTimerProc);

                // ---- Setup the icon in the tray ----
                TrayMessage(hwnd, NIM_ADD, TRAY_ICON, LoadIcon (HInst, MAKEINTRESOURCE (IDI_MEMO)), "Sticky Stuff");
                
                // Create utils sub menu first
                hPopupUtilsMenu = CreateMenu();
                InsertMenu(hPopupUtilsMenu, 0, MF_STRING, SAVESTICKIESID,    (LPCSTR) STRSAVESTICKIES);
                InsertMenu(hPopupUtilsMenu, 0, MF_STRING, IMPORTSTICKYID,    (LPCSTR) STRIMPORTSTICKYMAIN);
                InsertMenu(hPopupUtilsMenu, 0, MF_STRING, REALIGNSTICKIESID, (LPCSTR) STRREALIGNSTICKIES);
                InsertMenu(hPopupUtilsMenu, 0, MF_STRING, NUKEALLSTICKIESID, (LPCSTR) STRNUKEALLSTICKIES);
                InsertMenu(hPopupUtilsMenu, 0, MF_STRING, HELPMEID,          (LPCSTR) STRHELPME);

                // Add menu items to popup menu.
                hPopupMenu      = CreatePopupMenu();
                InsertMenu(hPopupMenu, 0, MF_STRING, NEWSTICKYID,         (LPCSTR) STRNEWSTICKYMAINWIN);                
                InsertMenu(hPopupMenu, 0, MF_STRING, DEFSTICKIESPREFSID,  (LPCSTR) STRDEFSTICKIESPREFS);
                InsertMenu(hPopupMenu, 0, MF_POPUP, (UINT)hPopupUtilsMenu,(LPCSTR) STRUTILSSUBMENU);  
                InsertMenu(hPopupMenu, 0, MF_STRING, ABOUTID,             (LPCSTR) STRABOUT);                       
                InsertMenu(hPopupMenu, 0, MF_STRING | MF_SEPARATOR, 1,    (LPCSTR) NULL);                
                InsertMenu(hPopupMenu, 0, MF_STRING, QUITSTICKIESID,      (LPCSTR) STRQUITSTICKIES);                
                InsertMenu(hPopupMenu, 0, MF_STRING | MF_SEPARATOR, 1,    (LPCSTR) NULL);                
                InsertMenu(hPopupMenu, 0, MF_STRING ,               1,    (LPCSTR) STRLEAVEMENU);                                                

                LoadStickyFromFile ();
            }
            return TRUE;

        case WM_TIMER:
            switch (wParam)
            {
                case TIMERALARM:
                    DoAlarmTest();
                    break;

                case TIMERSAVE:
                    // Save current state of stickies to file
                    SaveMainStickyPrefs ();
                    SaveStickyToFile ();
                    break;
            }            
            return TRUE;

        // User has selected the icon within the system tray
        case MYWM_NOTIFYICON:
            switch (lParam)
            {
                // Bring forward all the sticky windows ...
                case WM_LBUTTONDOWN:
                    {
                        SetForegroundWindow(hwnd);

                        for (int test = 0; test < STICKRECNUM-1; test++)
                        {
                            if (StickyRecords[test].stickyWnd != NULL)
                                BringWindowToTop(StickyRecords[test].stickyWnd);
                        }
                    }
                    return TRUE;

                // popup the menu ...
                case WM_RBUTTONDOWN:                
                    {
                        POINT pos;
                        
                        GetCursorPos(&pos); 
                        TrackPopupMenuEx(hPopupMenu, TPM_TOPALIGN | TPM_LEFTALIGN, pos.x, pos.y, hwnd, NULL);
                    }
                    return TRUE;

                case WM_LBUTTONDBLCLK:// left mouse double click
                    PostMessage (hwnd, WM_COMMAND, LOWORD(NEWSTICKYID), 0);
                    return TRUE;
            }
            break;

        case WM_COMMAND:
            switch (LOWORD(wParam))
            {
                case SAVESTICKIESID:
                    // Save stickies here
                    if (SaveStickyToFile () != 0)
                        MessageBox (hwnd, "Unable to save sticky info", "I/O Error", MB_ICONSTOP | MB_OK);
                    return TRUE;

                case ABOUTID:                                           
                    if (fpfnAboutDialog == NULL)
                    {
                        fpfnAboutDialog = MakeProcInstance ((FARPROC)AboutDialog, HInst);
                        DialogBox(HInst, MAKEINTRESOURCE(IDD_ABOUT), hwnd, (DLGPROC) fpfnAboutDialog);
                        FreeProcInstance (fpfnAboutDialog);
                        fpfnAboutDialog = NULL;
                    }
                    return TRUE;

                case NEWSTICKYID:
                    {
                        int newIdx = GetNextStickyFree ();

                        if (newIdx < 0)
                        {
                            MessageBox    (hwnd, "Can't Create Any More Stickies", "Out Of Array Space", MB_ICONSTOP | MB_OK);
                            break;
                        }
                    
                        DuplicateSticky    (STICKRECNUM,   newIdx);
                        LoadSticky         (ParentStarter, newIdx, TRUE);
                        SetForegroundWindow(StickyRecords[newIdx].stickyWnd);
                        SetFocus           (StickyRecords[newIdx].editControl);
                    }
                    return TRUE;

                case QUITSTICKIESID:
                    PostMessage (hwnd, WM_CLOSE, 0, 0);
                    break;

                case REALIGNSTICKIESID:
                    {
                        int responce = MessageBox (hwnd, "This action will re-align sticky windows "
                                                         "from the top left hand corner of the screen, downwards.\r\n"
                                                         "This action is unrecoverable, it is used to re-position lost stickies "
                                                         "from outside the windows desktop display area.\r\n"
                                                         "Are you sure you want to do this?", 
                                                         "Re-align Sticky Windows", 
                                                         MB_ICONSTOP | MB_YESNO);

                        if (responce != IDYES)
                            break;

                        // re-position all sticky windows 
                        int  yPos     = 0;
                        int  maxWidth = 0;
                        int  moveLeft = 0;
                        RECT repo;
                        
                        for (int idx = 0; idx < STICKRECNUM; idx++)
                        {
                            if (StickyRecords[idx].stickyWnd != NULL)
                            {
                                ShowWindow(StickyRecords[idx].stickyWnd, SW_HIDE);

                                if ((yPos + GetStickyTitleFontHeight (StickyRecords[idx].stickyWnd, idx) + GetSystemMetrics(SM_CYSIZEFRAME) + 2) >= 
                                    GetSystemMetrics (SM_CYSCREEN))
                                {
                                    moveLeft += maxWidth + 1;
                                    maxWidth = 0;
                                    yPos     = 0;
                                }

                                if (StickyRecords[idx].rolledUp == FALSE)
                                {
                                    GetWindowRect(StickyRecords[idx].stickyWnd, &StickyRecords[idx].winDim); 
                                    StickyRecords[idx].rolledUp      = TRUE;                                        
                                    StickyRecords[idx].winDim.bottom = (StickyRecords[idx].winDim.bottom - StickyRecords[idx].winDim.top);
                                }
                                
                                repo                        = StickyRecords[idx].winDim;
                                repo.left                   = moveLeft;
                                repo.right                  = moveLeft + (StickyRecords[idx].winDim.right - StickyRecords[idx].winDim.left);
                                repo.top                    = yPos;                                    
                                StickyRecords[idx].winDim   = repo;

                                // change the window position.                                    
                                MoveWindow   (StickyRecords[idx].stickyWnd, 
                                              StickyRecords[idx].winDim.left,
                                              StickyRecords[idx].winDim.top,
                                              (StickyRecords[idx].winDim.right - StickyRecords[idx].winDim.left), 
                                              0, 
                                              FALSE);

                                yPos = yPos + GetStickyTitleFontHeight (StickyRecords[idx].stickyWnd, idx) + 2;
                                
                                if (StickyRecords[idx].thinEdge != FALSE)
                                    yPos += GetSystemMetrics(SM_CYEDGE);                                    
                                 else 
                                    yPos += GetSystemMetrics(SM_CYSIZEFRAME);                                    

                                // Check screen boundries
                                if ((StickyRecords[idx].winDim.right - StickyRecords[idx].winDim.left) > maxWidth)
                                    maxWidth = (StickyRecords[idx].winDim.right - StickyRecords[idx].winDim.left);

                                ShowWindow(StickyRecords[idx].stickyWnd, SW_SHOW);
                            }
                        }                            
                    }
                    break;

                case DEFSTICKIESPREFSID:
                    static int     result;

                    if (fpfnPrefsDialog == NULL)
                    {                            
                        fpfnPrefsDialog = MakeProcInstance ((FARPROC)StickyPrefs, HInst);
                        result          = DialogBoxParam (HInst, MAKEINTRESOURCE(IDD_PREFS), hwnd,
                                                          (DLGPROC) fpfnPrefsDialog, (LPARAM) STICKRECNUM);
                        FreeProcInstance (fpfnPrefsDialog);
                        fpfnPrefsDialog = NULL;
                    }
                    else
                        MessageBox (hwnd, "Only one instance of the preference dialog\ncan be open at a time",
                        "Preference Open Error",
                            MB_ICONINFORMATION | MB_OK);
                    break;

                case IMPORTSTICKYID:
                    ImportStickyFile ();
                    break;

                case NUKEALLSTICKIESID:
                    {
                        int responce = MessageBox (hwnd, "This action will delete all sticky windows!\r\n"
                                                         "Are you sure you want to do this?",
                                                         "Nuke em'all", 
                                                         MB_ICONSTOP | MB_YESNO);

                        if (responce != IDYES)
                            break;

                        for (int idx = 0; idx < STICKRECNUM; idx++)
                        {
                            if (StickyRecords[idx].stickyWnd != NULL)
                            {
                                SetForegroundWindow(ParentStarter);
                                DestroyWindow(StickyRecords[idx].stickyWnd);
                            }
                        }
                    }
                    break;

                case HELPMEID:
                    WinHelp(hwnd, STRSTICKYHELP, HELP_FINDER, 0);
                    break;
            }
            break;

        case WM_ENDSESSION:
            if ((BOOL) wParam == FALSE)
                break;

        // alarm thread creation calls. Threads need to be created from 
        // a stable parent process to prevent Zombie process occuring.
        // That is, the parent thread terminating before its child thread.
        case WM_USER:
            // wParam thread call
            // lParam sticky window index
            switch (wParam)
            {
                case ALARM_THRD_SOUNDPLAY:
                    {
                        DWORD threadID;
                        
                        // Create a separate thread to handle number sound sample playbacks.
                        CreateThread (NULL, 512, AlarmSndPlaybackThread, (LPDWORD) (DWORD)lParam, 0, &threadID);
                    }
                    break;
                case ALARM_THRD_BLINKER:
                    {
                        DWORD threadID;

                        // Create a separate thread to handle number blinking windows.
                        CreateThread (NULL, 512, AlarmBlinkStickyThread, (LPDWORD) (DWORD)lParam, 0, &threadID);
                    }
                    break;
            }
            break;

        case WM_CLOSE:
        case WM_DESTROY:
            // *** Stop alarm timer ***
            KillTimer(hwnd, TIMERSAVE);
            KillTimer(hwnd, TIMERALARM);
            FreeProcInstance(lpfnTimerProc);            

            // Save stickies here
            SaveMainStickyPrefs  ();
            if (SaveStickyToFile () != 0)
                MessageBox (hwnd, "Unable to save sticky info", "I/O Error", MB_ICONSTOP | MB_OK);

            CleanUp ();

            // Remove icon from system tray
            TrayMessage(hwnd, NIM_DELETE, TRAY_ICON, NULL, NULL);
            DestroyMenu(hPopupMenu);

            PostQuitMessage (0);
            return 0 ;
    }


    return DefWindowProc (hwnd, message, wParam, lParam) ;
}