/*    CDMATH.H

      Complex library for the languages C and C++.

      This header file contains all definitions for
      double-precision complex numbers (complex double).

      Copyright (c) 1996-2001 by Martin Sander
      Address of the author:
           Dr. Martin Sander Software Dev.
           Steinachstr. 9A
           D-69198 Schriesheim
           Germany
           MartinSander@Bigfoot.com
           http://www.optivec.com
*/


#ifndef __CDMATH_H
#define __CDMATH_H

#if !defined( _CMATH_DEFS )
   #ifdef __BORLANDC__
       #pragma option -a-
   #else /* Visual C++, Optima++ */
       #pragma pack( push,1 )
   #endif /* avoid insertion of dummy bytes  */
   typedef struct {float    Re,  Im;}  fComplex;
   typedef struct {double   Re,  Im;}  dComplex;
   typedef struct {float    Mag, Arg;} fPolar;
   typedef struct {double   Mag, Arg;} dPolar;
   #ifdef __BORLANDC__
       typedef long double    extended;
       typedef struct {extended Re,  Im;}  eComplex;
       typedef struct {extended Mag, Arg;} ePolar;
       #pragma option -a.
   #else /* Visual C++ */
       typedef  double extended; /* Visual C++ does not support
                                 80-bit IEEE numbers. So make
                                 extended equal to double    */
       typedef dComplex  eComplex;
       typedef dPolar    ePolar;
       #pragma pack( pop )
   #endif    /* restore default data packing  */
   typedef fComplex fcomplex;
   typedef dComplex dcomplex;
   typedef eComplex ecomplex;
   typedef fPolar fpolar;
   typedef dPolar dpolar;
   typedef ePolar epolar;  // tolerate all-lower case
   #define _CMATH_DEFS
#endif
#ifdef __BORLANDC__
    #include <_defs.h>
    #if defined __TINY || defined __SMALL__ || defined __MEDIUM__
        #if defined(_RTLDLL) || defined(_CLASSDLL)
            #error Must use static BC Runtime Library with OptiVec and CMATH in models TINY, SMALL, MEDIUM
        #endif
        #define   _VFAR  near   /* even in case of DS!=SS  */
    #elif defined __FLAT__
        #define  _VFAR
    #else
        #define   _VFAR  far
    #endif
    #if (__BORLANDC__ >= 0x450)
         #define __cmf _RTLENTRY _EXPFUNC
    #else
         #define __cmf  _Cdecl _FARFUNC
    #endif
    #if __BORLANDC__ < 0x500
        #define VBOOL int
    #else
        #define VBOOL bool
    #endif
#else  /* Visual C++, Optima++ */
    #define _VFAR
    #define __cmf  __cdecl
    #define VBOOL int
#endif
#define _VFARC const _VFAR

/*  first the constructors:  */
#ifdef __cplusplus
   /* since dComplex and dPolar are declared as a structs instead of classes,
      the constructors cannot get the names "dComplex" and "dPolar" here.   */
  #ifndef _DCPLX_DEFINED
  inline dComplex __cmf dcplx( double __ReVal )
  {   dComplex Result;
      Result.Re = __ReVal;
      Result.Im = 0.0;
      return Result;
  }

  inline dPolar __cmf dpolr( double __MagVal )
  {   dPolar Result;
      Result.Mag = __MagVal;
      Result.Arg = 0.0;
      return Result;
  }

      // up-conversions from single precision
  inline dComplex __cmf dcplx( fComplex _VFARC & __zf )
  {   dComplex Result;
      Result.Re = __zf.Re;
      Result.Im = __zf.Im;
      return Result;
  }

  inline dPolar __cmf dpolr( fPolar _VFARC & __pf )
  {   dPolar Result;
      Result.Mag = __pf.Mag;
      Result.Arg = __pf.Arg;
      return Result;
  }
      // down-conversion from extended precision
      // (with OVERFLOW error handling):
  #ifdef __BORLANDC__
      dComplex __cmf dcplx( eComplex __ze );
      dPolar   __cmf dpolr( ePolar   __pe );
  #else
      #ifdef _CMATH_CLASSDEFS
          inline dComplex _VFAR & __cmf dcplx( complex<long double> __ze )
		  { return( complex<double>(__ze.Re, __ze.Im )); }
          inline dPolar _VFAR & __cmf dpolr( polar<long double> __pe )
		  { return( polar<double>(__pe.Mag, __pe.Arg )); }
      #endif
  #endif
	  //  identity:
  inline dComplex _VFAR & __cmf dcplx( dComplex _VFAR & __zd )
  {   return __zd;  } 
  inline dPolar   _VFAR & __cmf dpolr( dPolar _VFAR & __pd )
  {   return __pd;  }

      // interconversions of Cartesian and polar
  dComplex __cmf dcplx( fPolar   __pf );
  dComplex __cmf dcplx( dPolar   __pd );
  dPolar   __cmf dpolr( fComplex __zf );
  dPolar   __cmf dpolr( dComplex __zd );
  #ifdef __BORLANDC__
      dComplex __cmf dcplx( ePolar   __pe );
      dPolar   __cmf dpolr( eComplex __ze );
  #else
      #ifdef _CMATH_CLASSDEFS
          dComplex __cmf dcplx( polar<long double> __pe );
          dPolar __cmf dpolr( complex<long double> __ze );
      #endif
  #endif

  #define _DCPLX_DEFINED
  #endif  // _DCPLX_DEFINED
#endif
       /* basic form of constructor for C and C++ : */
dComplex __cmf dcplx( double __ReVal,  double __ImVal);
dPolar   __cmf dpolr( double __MagVal, double __ArgVal );

      /* plain-C version of conversions: 
         (with OVERFLOW handling, where appropriate)      */
#if !defined _CMATH_CLASSDEFS /* declared as friend functions in <nexcplx.h> */
    dComplex __cmf cftocd( fComplex __zf );
    dPolar   __cmf pftopd( fPolar   __pf );
    dComplex __cmf pftocd( fPolar   __pf );
    dComplex __cmf pdtocd( dPolar   __pd );
    dPolar   __cmf cftopd( fComplex __zf );
    dPolar   __cmf cdtopd( dComplex __zd );
    #ifdef __BORLANDC__
        dComplex __cmf cetocd( eComplex __ze );
        dPolar   __cmf petopd( ePolar   __pe );
        dComplex __cmf petocd( ePolar   __pe );
        dPolar   __cmf cetopd( eComplex __ze );
    #else   /* extended = double */
        #define cetocd(z) (z)
        #define petopd(z) (z)
        #define petocd pdtocd
        #define cetopd cdtopd
    #endif
#endif

   /* Basic complex operations. They are defined both
   for C and C++. However, for C++ you may as well use the
   overloaded operators and functions defined further below. */
#define         cd_real( z )  (z).Re
#define         cd_imag( z )  (z).Im
#define         pd_abs(  p )  (p).Mag
#define         pd_arg(  p )  (p).Arg
#if defined __cplusplus && !defined _CMATH_CLASSDEFS
extern "C" {  // the following functions cannot be "extern C",
#endif        // if dComplex identical to complex<double>
dComplex __cmf  cd_neg(  dComplex __z );
dComplex __cmf  cd_conj( dComplex __z );
dPolar   __cmf  pd_neg(  dPolar   __p );
dPolar   __cmf  pd_conj( dPolar   __p );
dPolar   __cmf  pd_principal( dPolar __p );
#ifdef __cplusplus  // even if _CMATH_CLASSDEFS
    extern "C" {
#endif
double __cmf  cd_norm( dComplex __z );
double __cmf  cd_arg(  dComplex __z );
double __cmf  pd_norm( dPolar __p );
double __cmf  pd_real( dPolar __p );
double __cmf  pd_imag( dPolar __p );
#ifdef __cplusplus
    }
#endif
dComplex __cmf  cd_polar( double __mag, double __angle );
dComplex __cmf  cd_magargtoc( double __mag, double __angle ); /* same as cd_polar */
dPolar   __cmf  pd_complex( double __re,  double __im );
dPolar   __cmf  pd_reimtop( double __re,  double __im ); /* same as pd_complex */

dComplex __cmf  cd_add(   dComplex __x, dComplex __y );
dComplex __cmf  cd_addRe( dComplex __x, double __yRe );
dComplex __cmf  cd_sub(   dComplex __x, dComplex __y );
dComplex __cmf  cd_subRe( dComplex __x, double __yRe );  /*  x - yRe  */
dComplex __cmf  cd_subrRe( dComplex __x, double __yRe ); /*  yRe - x  */
dComplex __cmf  cd_mul(   dComplex __x, dComplex __y );
dComplex __cmf  cd_mulRe( dComplex __x, double __yRe );
dComplex __cmf  cd_div(   dComplex __x, dComplex __y );
dComplex __cmf  cd_divRe( dComplex __x, double __yRe );   /*  x / yRe  */
dComplex __cmf  cd_divrRe( dComplex __x, double __yRe );  /*  yRe / x  */

dPolar   __cmf  pd_mul(    dPolar   __x, dPolar   __y );
dPolar   __cmf  pd_mulRe(  dPolar   __x, double __yRe );
dPolar   __cmf  pd_div(    dPolar   __x, dPolar   __y );
dPolar   __cmf  pd_divRe(  dPolar   __x, double __yRe );   /*  x / yRe  */
dPolar   __cmf  pd_divrRe( dPolar   __x, double __yRe );  /*  yRe / x  */

/*  mathematical functions with error handling through _matherr: */
#ifdef __cplusplus 
    extern "C" double   __cmf  cd_abs(  dComplex __z );
#else
	double   __cmf  cd_abs(  dComplex __z );
#endif
dComplex __cmf  cd_acos( dComplex __z );
dComplex __cmf  cd_asin( dComplex __z );
dComplex __cmf  cd_atan( dComplex __z );
dComplex __cmf  cd_cos(  dComplex __z );
dComplex __cmf  cd_cosh( dComplex __z );
dComplex __cmf  cd_cubic( dComplex __z );  /* raise to the third power */
dComplex __cmf  cd_exp(  dComplex __z );
dPolar   __cmf  cd_exptop(  dComplex __z ); /* exp with result as dPolar */
dComplex __cmf  cd_inv(  dComplex __z );    /*   1.0 / z   */
dComplex __cmf  cd_ipow( dComplex __z, int __exponent );
                                             /* raise z to integer power */
dComplex __cmf  cd_ln(   dComplex __z );
dComplex __cmf  cd_log(   dComplex __z );  /* sane as cd_ln */
dComplex __cmf  cd_log2(  dComplex __z );
dComplex __cmf  cd_log10( dComplex __z );
dComplex __cmf  cd_pow( dComplex __base, dComplex __exponent );
dComplex __cmf  cd_powReBase( double __base, dComplex __exponent ); /* power of real base */
dComplex __cmf  cd_powReExpo( dComplex __base, double __exponent ); /* raise z to real power */
                         /* for integer exponents, use cd_ipow ! */
dComplex __cmf  cd_quartic( dComplex __z );  /* raise to the fourth power */
dComplex __cmf  cd_sin(  dComplex __z );
dComplex __cmf  cd_sinh( dComplex __z );
dComplex __cmf  cd_square( dComplex __z );
dComplex __cmf  cd_sqrt( dComplex __z );
dComplex __cmf  cd_tan(  dComplex __z );
dComplex __cmf  cd_tanh( dComplex __z );

       /* some of the mathematical functions also for polar: */
dPolar   __cmf  pd_cubic( dPolar   __p );  /* raise to the third power */
dPolar   __cmf  pd_inv(  dPolar   __p );    /*   1.0 / z   */
dPolar   __cmf  pd_ipow( dPolar   __p, int __exponent );
                                             /* raise z to integer power */
dComplex __cmf  pd_lntoc(    dPolar   __p );
dComplex __cmf  pd_logtoc(   dPolar   __p ); /* same as pd_lntocd */
dComplex __cmf  pd_log2toc(  dPolar   __p );
dComplex __cmf  pd_log10toc( dPolar   __p );
dPolar   __cmf  pd_powReExpo( dPolar   __base, double __exponent ); /* raise z to real power */
                         /* for integer exponents, use pd_ipow ! */
dPolar   __cmf  pd_quartic( dPolar   __p );  /* raise to the fourth power */
dPolar   __cmf  pd_square( dPolar   __p );
dPolar   __cmf  pd_sqrt( dPolar   __p );


#if defined __cplusplus && !defined _CMATH_CLASSDEFS
}    //  end of the extern "C" statement
dPolar   __cmf  pd_principal( double __mag, double __angle );
       // this overload cannot be extern C
#endif

#if defined __cplusplus && !defined __STD_COMPLEX && !defined __COMPLEX_H && !defined __NEWCPLX_H
   /* in addition to the basic operations defined above for C,
      here is the same complete set of overloaded operators and
      functions as offered by <newcplx.h> for the complex classes.  */

    inline double real( dComplex _VFARC & __z )
    {
        return __z.Re;
    }

    inline double imag( dComplex _VFARC & __z )
    {
        return __z.Im;
    }

    inline dComplex neg( dComplex _VFARC & __z1 )
    {   dComplex Result;
        Result.Re = -__z1.Re;
        Result.Im = -__z1.Im;
        return Result;
    }

    inline dComplex conj( dComplex _VFARC & __z)
    {   dComplex Result;
        Result.Re =  __z.Re;
        Result.Im = -__z.Im;
        return Result;
    }

    double   __cmf  norm( dComplex __z );
    double   __cmf  arg(  dComplex __z );
    dComplex __cmf  magargtoc( double Mag, double Angle );


  //  unary operators:

  inline dComplex _VFAR & operator +( dComplex _VFAR & __z1 )
  {
      return __z1;
  }

  inline dComplex operator -( dComplex _VFARC & __z1 )
  {   dComplex Result;
      Result.Re = -__z1.Re;
      Result.Im = -__z1.Im;
      return Result;
  }


  //  binary operators:

  inline dComplex operator +( dComplex _VFARC & __z1, dComplex _VFARC & __z2 )
  {   dComplex Result;
      Result.Re = __z1.Re + __z2.Re;
      Result.Im = __z1.Im + __z2.Im;
      return Result;
  }

  inline dComplex operator +( dComplex _VFARC & __z1, double __z2Re )
  {   dComplex Result;
      Result.Re = __z1.Re + __z2Re;
      Result.Im = __z1.Im;
      return Result;
  }

  inline dComplex operator +( double __z1Re, dComplex _VFARC & __z2 )
  {   dComplex Result;
      Result.Re = __z1Re + __z2.Re;
      Result.Im = __z2.Im;
      return Result;
  }

  inline dComplex operator -( dComplex _VFARC & __z1, dComplex _VFARC & __z2 )
  {   dComplex Result;
      Result.Re = __z1.Re - __z2.Re;
      Result.Im = __z1.Im - __z2.Im;
      return Result;
  }

  inline dComplex operator -( dComplex _VFARC & __z1, double __z2Re )
  {   dComplex Result;
      Result.Re = __z1.Re - __z2Re;
      Result.Im = __z1.Im;
      return Result;
  }

  inline dComplex operator -( double __z1Re, dComplex _VFARC & __z2 )
  {   dComplex Result;
      Result.Re = __z1Re - __z2.Re;
      Result.Im = -__z2.Im;
      return Result;
  }

  inline dComplex operator *( dComplex _VFARC & __z1, dComplex _VFARC & __z2 )
  {   dComplex Result;
      Result.Re  = __z1.Re * __z2.Re - __z1.Im * __z2.Im;
      Result.Im  = __z1.Re * __z2.Im + __z1.Im * __z2.Re;
      return Result;
  }

  inline dComplex operator *( dComplex _VFARC & __z1, double __z2Re )
  {   dComplex Result;
      Result.Re = __z1.Re * __z2Re;
      Result.Im = __z1.Im * __z2Re;
      return Result;
  }

  inline dComplex operator *( double __z1Re, dComplex _VFARC & __z2 )
  {   dComplex Result;
      Result.Re  = __z1Re * __z2.Re;
      Result.Im  = __z1Re * __z2.Im;
      return Result;
  }

  inline dComplex operator /( dComplex _VFARC & __z1, double __z2Re )
  {   dComplex Result;
      Result.Re = __z1.Re / __z2Re;
      Result.Im = __z1.Im / __z2Re;
      return Result;
  }


  dComplex __cmf operator /( dComplex _VFARC & __z1, dComplex _VFARC & __z2 );
  dComplex __cmf operator /( double __z1Re, dComplex _VFARC & __z2 );

    /* compound-assignment operators:  */
  inline dComplex _VFAR & operator +=( dComplex _VFAR & __z1, dComplex _VFARC & __z2 )
  {
      __z1.Re += __z2.Re;
      __z1.Im += __z2.Im;
      return __z1;
  }

  inline dComplex _VFAR & operator +=( dComplex _VFAR & __z1, double __z2Re )
  {
      __z1.Re += __z2Re;
      return __z1;
  }

  inline dComplex _VFAR & operator -=( dComplex _VFAR & __z1, dComplex _VFARC & __z2 )
  {
      __z1.Re -= __z2.Re;
      __z1.Im -= __z2.Im;
      return __z1;
  }

  inline dComplex _VFAR & operator -=( dComplex _VFAR & __z1, double __z2Re )
  {
      __z1.Re -= __z2Re;
      return __z1;
  }

  inline dComplex _VFAR & operator *=( dComplex _VFAR & __z1, dComplex _VFARC & __z2 )
  {
      double tmpRe;
      tmpRe   = __z1.Re * __z2.Re - __z1.Im * __z2.Im;
      __z1.Im = __z1.Re * __z2.Im + __z1.Im * __z2.Re;
      __z1.Re = tmpRe;
      return __z1;
  }

  inline dComplex _VFAR & operator *=( dComplex _VFAR & __z1, double __z2Re )
  {
      __z1.Re *= __z2Re;
      __z1.Im *= __z2Re;
      return __z1;
  }

  dComplex _VFAR & __cmf operator /=( dComplex _VFAR & __z1, dComplex _VFARC & __z2 );

  inline dComplex _VFAR & operator /=( dComplex _VFAR & __z1, double __z2Re )
  {
      __z1.Re /= __z2Re;
      __z1.Im /= __z2Re;
      return __z1;
  }


  inline VBOOL operator ==( dComplex _VFARC & __z1, dComplex _VFARC & __z2 )
  {
      return (__z1.Re == __z2.Re) && (__z1.Im == __z2.Im );
  }

  inline VBOOL operator ==( dComplex _VFARC & __z1, double __z2Re )
  {
      return (__z1.Re == __z2Re) && (__z1.Im == 0.0 );
  }

  inline VBOOL operator !=( dComplex _VFARC & __z1, dComplex _VFARC & __z2 )
  {
      return (__z1.Re != __z2.Re) || (__z1.Im != __z2.Im );
  }

  inline VBOOL operator !=( dComplex _VFARC & __z1, double __z2Re )
  {
      return (__z1.Im != 0.0 ) || (__z1.Re != __z2Re);
  }


  /* now the operators for polar: */

    double  __cmf real(  dPolar __p );
    double  __cmf imag(  dPolar __p );
    dPolar __cmf reimtop( double __Re, double __Im );
    inline double norm( dPolar _VFARC & __p )
    {
        return __p.Mag * __p.Mag;
    }
    inline double arg( dPolar _VFARC & __p )
    {
        return __p.Arg;
    }
    dPolar  __cmf neg(  dPolar __p );

    inline dPolar conj( dPolar _VFARC & __p)
    {   dPolar Result;
        Result.Mag =  __p.Mag;
        Result.Arg = -__p.Arg;
        return Result;
    }


  //  unary operators:

  inline dPolar _VFAR & operator +( dPolar _VFAR & __p1 )
  {
      return __p1;
  }

  inline dPolar operator -( dPolar _VFARC & __p1 )
  {
      return neg( __p1 );
  }

  //  binary operators:

  inline dPolar operator *( dPolar _VFARC & __p1, dPolar _VFARC & __p2 )
  {   dPolar Result;
      Result.Mag  = __p1.Mag * __p2.Mag;
      Result.Arg  = __p1.Arg + __p2.Arg;
      return Result;
  }

  dPolar __cmf operator *( dPolar _VFARC & __p1, double __p2Re );
  dPolar __cmf operator *( double __p1Re, dPolar _VFARC & __p2 );

  inline dPolar operator /( dPolar _VFARC & __p1, dPolar _VFARC & __p2 )
  {   dPolar Result;
      Result.Mag  = __p1.Mag / __p2.Mag;
      Result.Arg  = __p1.Arg - __p2.Arg;
      return Result;
  }

  dPolar __cmf operator /( dPolar _VFARC & __p1, double __p2Re );
  dPolar __cmf operator /( double __p1Re, dPolar _VFARC & __p2 );

      /*  compound assinment operators:  */
  inline dPolar _VFAR & operator *=( dPolar _VFAR & __p1, dPolar _VFARC & __p2 )
  {
      __p1.Mag *= __p2.Mag;
      __p1.Arg += __p2.Arg;
      return __p1;
  }

  dPolar _VFAR & __cmf operator *=( dPolar _VFAR & __p1, double __p2Re );

  inline dPolar _VFAR & operator /=( dPolar _VFAR & __p1, dPolar _VFARC & __p2 )
  {
      __p1.Mag /= __p2.Mag;
      __p1.Arg -= __p2.Arg;
      return __p1;
  }

  dPolar _VFAR & __cmf operator /=( dPolar _VFAR & __p1, double __p2Re );

  inline VBOOL operator ==( dPolar _VFARC & __p1, dPolar _VFARC & __p2 )
  {
      return (__p1.Mag == __p2.Mag) &&
             (pd_arg(__p1) == pd_arg(__p2) );
  }

  inline VBOOL operator ==( dPolar _VFARC & __p1, double __p2Re )
  {
      return (__p1.Mag == __p2Re) && (pd_arg(__p1) == 0.0 );
  }

  inline VBOOL operator !=( dPolar _VFARC & __p1, dPolar _VFARC & __p2 )
  {
      return (__p1.Mag != __p2.Mag) || (pd_arg(__p1) != pd_arg(__p2) );
  }

  inline VBOOL operator !=( dPolar _VFARC & __p1, double __p2Re )
  {
      return (__p1.Mag != __p2Re) || (pd_arg(__p1) != 0.0 );
  }

  /*  C++ version of the mathematical functions defined above.
      They use the same code as the C versions. In case of an error,
      you get a message in which the name of the C version is
      stated.
      Note that these functions require complex arguments to be
      passed by value, not by reference, as it is done in the member
      functions of the class complex. In terms of efficiency, this
      is about the same. (The math functions of the class complex
      store complex results at intermediate addresses and copy them
      to the desired address afterwards. This final copy is not
      necessary here.)                                             */

  double   __cmf  abs(  dComplex __z );
  dComplex __cmf  acos( dComplex __z );
  dComplex __cmf  asin( dComplex __z );
  dComplex __cmf  atan( dComplex __z );
  dComplex __cmf  cos(  dComplex __z );
  dComplex __cmf  cosh( dComplex __z );
  dComplex __cmf  cubic( dComplex __z );  /* raise to the third power */
  dComplex __cmf  exp(  dComplex __z );
  dPolar   __cmf  exptop(  dComplex __z );
  dComplex __cmf  inv(  dComplex __z );    /*   1.0 / z   */
  dComplex __cmf  ipow( dComplex __z, int __exponent );
                                            /* raise z to integer power */
  dComplex __cmf  ln(  dComplex __z );
  dComplex __cmf  log(  dComplex __z );  /* sane as ln */
  dComplex __cmf  log2( dComplex __z );
  dComplex __cmf  log10( dComplex __z );
  dComplex __cmf  pow( dComplex __z, dComplex __exponent );
  dComplex __cmf  pow( dComplex __z,  double __exponent ); // identical to powReExpo
  dComplex __cmf  pow( double __base,  dComplex __exponent ); // identical to powReBase
  dComplex __cmf  powReBase( double __base, dComplex __exponent ); // power of real base
  dComplex __cmf  powReExpo( dComplex __z, double __exponent );    // raise z to real power
                            // for integer exponents, use ipow !
  dComplex __cmf  quartic( dComplex __z );  // raise to the fourth power
  dComplex __cmf  sin(  dComplex __z );
  dComplex __cmf  sinh( dComplex __z );
  dComplex __cmf  square( dComplex __z );
  dComplex __cmf  sqrt( dComplex __z );
  dComplex __cmf  tan(  dComplex __z );
  dComplex __cmf  tanh( dComplex __z );

        /*  polar math functions:  */
  inline double abs( dPolar _VFARC & __p )
  {
      return __p.Mag;
  }
  dPolar   __cmf  cubic( dPolar   __p );  /* raise to the third power */
  dPolar   __cmf  inv(  dPolar   __p );    /*   1.0 / z   */
  dPolar   __cmf  ipow( dPolar   __p, int __exponent );
                                             /* raise z to integer power */
  dComplex __cmf  lntoc(    dPolar   __p );
  dComplex __cmf  logtoc(   dPolar   __p ); /* same as lntoc */
  dComplex __cmf  log2toc(  dPolar   __p );
  dComplex __cmf  log10toc( dPolar   __p );
  dPolar   __cmf  pow( dPolar   __base, double __exponent ); /* raise z to real power */
  dPolar   __cmf  powReExpo( dPolar   __base, double __exponent ); /* raise z to real power */
                         /* for integer exponents, use ipow ! */
  dPolar   __cmf  quartic( dPolar   __p );  /* raise to the fourth power */
  dPolar   __cmf  square( dPolar   __p );
  dPolar   __cmf  sqrt( dPolar   __p );
#endif //  __cplusplus, not __STD_COMPLEX , __COMPLEX_H, __NEWCPLX_H


/***  user-accessible error handling functions, borrowed from VectorLib  ****/

#ifdef __cplusplus
extern "C" {
#endif
void  __cmf  V_noteError( char _VFAR *fname, unsigned why );
void  __cmf  V_printErrorMsg( char _VFAR *ErrMsg );
void  __cmf  V_setErrorEventFile( char _VFAR *filename,  unsigned ScreenAndFile );
void  __cmf  V_closeErrorEventFile( void );

/*** translation of calls to matherr() into _matherr() for BorlandC 4.0+ ***/

#if (__BORLANDC__ >= 0x450) && !defined (__FLAT__)
     #if !defined( __MATH_H )
            #include <math.h>
     #endif
     int  _Cdecl _FARFUNC matherr (struct exception _VFAR *__e);
     #define NEWMATHERR  \
         int matherr( struct exception _VFAR *__e ) \
         {  return( _matherr( __e )); }
#else
     #define NEWMATHERR
#endif

#ifdef __cplusplus
}   // end of extern "C"
#endif
#endif   /*  __CDMATH_H  */
