/** \file CON_Resource.h 
    Data Resources Stream.
    Enables packing data into one file, optionally compressed.

Copyright (c) 1998-1999 by Amir Geva.
This file is part of the Photon Game Development library,
beta release version 0.25.  
Permission is granted to use and copy this file for non-commercial use only.  
Please contact the author concerning commercial usage. 
Amir Geva makes no representations about the suitability of this software for any purpose.
It is provided "as is" without express or implied warranty.

*/

class ResourceStream;

#ifndef H_CON_RESOURCE
#define H_CON_RESOURCE

#include <fstream.h>
#include <CON_Basic.h>

/** This class gives a simple way of storing application's data files into
    one archive, with easy access to each file at run time.
    Each resource is identified by a name (file name !?) */
class ResourceStream : public Interface
{
public:
  /** Get a stream for a specific resource */
  virtual istream* getStream(const char* Name) = 0;

  /** Free the stream, when done with it. */
  virtual void     freeStream(istream* is) = 0;

  /** Get a resource data copied into a buffer */
  virtual char*    getData(const char* Name, char* cData, int MaxLen=-1) = 0;

  /** Get the length of a specific resource */
  virtual int      getLength(const char* Name) = 0;

  /** Write a resource to the file. */
  virtual long     putData(const char* Name, char* Buffer, int Length) = 0;

  /** Write a resource to the file using an input stream for source. */
  virtual long     putData(const char* Name, istream* is, int Length) = 0;

  /** Set this to non-zero to compress all data put into the resource file. */
  virtual long     setCompression(int Compression) = 0;

  /** Returns a non-zero value if an error occured. */
  virtual long     status() = 0;

};

////////////////////////////////////////////////////////////
// ResourceStream Functions
////////////////////////////////////////////////////////////
#ifndef FUNC_RESOURCE
#define FUNC_RESOURCE


/** \relates ResourceStream
    Create a new ResourceStream that uses disk files directly. */
DLLExport ResourceStream* newResourceStream();

/** \relates ResourceStream
    Create a new ResourceStream for writing. 
    Requires stream of the output resource file 
*/
DLLExport ResourceStream* newResourceStream(ofstream* os);

/** \relates ResourceStream
    Create a new ResourceStream for reading (Create=0) or writing (Create=1).  
    Will open a disk file by the given name.
*/
DLLExport ResourceStream* newResourceStream(const char* Name, int Create=0);

/** \relates ResourceStream
    Set a resource stream as the system default resource stream.
    It will be released when the program terminates, or when setting 
    a different stream as the system default resource stream.
*/
DLLExport ResourceStream* setDefaultResourceStream(ResourceStream* RS);


#endif // FUNC_RESOURCE


#endif // H_CON_RESOURCE