//
//
//
//
//
//
//
//
//
//
// Microsoft Windows 95/98/NT Version 
//
//Copyright (c) 1994-1999 by Dan Higdon, Tim Little, and Chuck Walbourn
//
//
//
// This file and all associated files are subject to the terms of the
// GNU Lesser General Public License version 2 as published by the
// Free Software Foundation (http://www.gnu.org).   They remain the
// property of the authors: Dan Higdon, Tim Little, and Chuck Walbourn.
// See LICENSE.TXT in the distribution for a copy of this license.
//
// THE AUTHORS MAKE NO WARRANTIES, EXPRESS OR IMPLIED, AS TO THE CORRECTNESS
// OF THIS CODE OR ANY DERIVATIVE WORKS WHICH INCORPORATE IT.  THE AUTHORS
// PROVIDE THE CODE ON AN "AS-IS" BASIS AND EXPLICITLY DISCLAIMS ANY
// LIABILITY, INCLUDING CONSEQUENTIAL AND INCIDENTAL DAMAGES FOR ERRORS,
// OMISSIONS, AND OTHER PROBLEMS IN THE CODE.
//
//
//
//                        http://www.mythos-engine.org/
//
//
//
// Created by Tim Little & Chuck Walbourn
//
// eslpoint.cpp
//
// The EschLight and associated classes are used for implementing the
// lighting model.  EschLights affect objects by controlling the
// shade level and potentially color of a vertex.  EschLight objects
// are maintained in a linked list.
//
// An EschPointLight shines in all directions equally, but has a specific
// location in 3-space.
//
//

//
//
//                                Includes
//
//

#include "escher.hpp"
#include "esfile.hpp"

//
//
//                                Equates
//
//

#define max(a,b)  (((a) > (b)) ? (a) : (b))

//
//
//                                 Code
//
//

//
//  Constructors/Destructors  
//

//Ŀ
// EschPointLight - Constructor                                             
//                                                                          
// Initializes the light object with full intesity and sets type.           
//
EschPointLight::EschPointLight(const char *fname, const char *lname) :
    EschLight(),
    pos(0,0,0)
{
    ltyp = ESCH_LGTT_POINT;
    intensity = 255;

    if (fname)
    {
        load(fname,lname);
    }
};

EschPointLight::EschPointLight(const EschPoint *pnt) :
    EschLight(),
    pos(*pnt)
{
    ltyp = ESCH_LGTT_POINT;
    intensity = 255;
};

EschPointLight::EschPointLight(float ix, float iy, float iz) :
    EschLight(),
    pos(ix,iy,iz)
{
    ltyp = ESCH_LGTT_POINT;
    intensity = 255;
};



//
//  Operations  
//

//Ŀ
// EschPointLight - setup                                                   
//                                                                          
// Performs the setup of the light on an object by object basis.            
//
void EschPointLight::setup(const EschFrameRef *fr)
{
    if (fr
        && EschCurrent
        && EschCurrent->camera
        && EschCurrent->camera->flags & ESCH_CAM_MODELSPACE)
    {
        pos.transform(&fr->iorient,&tpos);
    }
    else
    {
        tpos = pos;
    }

    EschLight::setup(fr);
}


//Ŀ
// EschPointLight - shine                                                   
//                                                                          
// Sets the shade level in the Van Gogh point based on the intesity of the  
// light and the given vertex information.                                  
//                                                                          
// The point light source computes a direction vector relative to the       
// vertex and then computes a dot product.                                  
//
void EschPointLight::shine(const EschVertex *v, VngoPoint *vpt, int revnrml) const
{
    if (!(flags & ESCH_LGT_OFF))
    {
        // Compute point source lighting vector
        EschVector vtx(tpos.x - v->x,
                       tpos.y - v->y,
                       tpos.z - v->z);
        vtx.normalize();

        // Compute dot product
        float dot = v->normal DOT vtx;

        if (dot < 0)
        {
            dot = (revnrml) ? -dot : 0;
        }

        // Compute shade value
        vpt->shade += long(dot * _intensity);
    }

// Chain for true multiple lights
    if (inext)
    {
        inext->shine(v,vpt,revnrml);
    }
// Clamp shade value
    else
    {
        if (vpt->shade > mp)
            vpt->shade = mp;
        else if (vpt->shade < 0)
            vpt->shade = 0;
    }
}


//Ŀ
// EschPointLight - hishine                                                 
//                                                                          
// Sets the shade level, allowing specular highlights to overdrive the      
// color based on the intensity of the light and the given vertex info.     
//                                                                          
// The point light source computes a direction vector relative to the       
// vertex and then computes a dot product.                                  
//
void EschPointLight::hishine(const EschVertex *v, VngoPoint *vpt, int revnrml) const
{
    if (!(flags & ESCH_LGT_OFF))
    {
        // Compute point source lighting vector
        EschVector vtx(tpos.x - v->x,
                       tpos.y - v->y,
                       tpos.z - v->z);
        vtx.normalize();

        // Compute dot product
        float dot = v->normal DOT vtx;

        if (dot < 0)
        {
            dot = (revnrml) ? -dot : 0;
        }

        // Compute shade value
        vpt->shade += long(dot * _hintensity);
    }

// Chain for true multiple lights
    if (inext)
    {
        inext->hishine(v,vpt,revnrml);
    }
// Clamp shade value
    else
    {
        if (vpt->shade > VNGO_SHADE_FULL)
            vpt->shade = VNGO_SHADE_FULL;
        else if (vpt->shade < 0)
            vpt->shade = 0;
    }
}



//
//  I/O Routines  
//

//Ŀ
// EschPointLight - load                                                    
//                                                                          
// Loads the position and other data of the point light from an IFF file.   
//                                                                          
// The IFF version assumes that the IFF is open and pointing at the proper  
// chunk.  If a 'lname' is given and the name doesn't match that in the     
// chunk, an error is returned.                                             
//
esch_error_codes EschPointLight::load(const char *fname, const char *lname)
{
    esch_error_codes    err;
    XFParseIFF          iff;

// Open file
    if (iff.open(fname,XF_OPEN_READ | XF_OPEN_DENYWRITE))
        return ESCH_ERR_FILEERROR;

// Loop until a header with the correct name is found
    err=ESCH_ERR_NONE;

    for(;;)
    {
        if (iff.next() != XF_ERR_NONE)
        {
            if (!iff.depth)
                return ESCH_ERR_NOTFOUND;
            iff.leaveform();
            continue;
        }

        if (iff.chunkid == iff.makeid('E','P','N','T')
            || iff.chunkid == iff.makeid('E','P','N','1'))
        {
            err=load(&iff,lname);
            if (!err
                || err != ESCH_ERR_NOTFOUND)
                break;
        }

        // Must scan inside any ESEN Scene forms.
        else if (iff.chunkid == iff.makeid('F','O','R','M')
                 && iff.formid == iff.makeid('E','S','E','N'))
        {
            iff.enterform();
        }
    }

// Return ok
    iff.close();
    return err;
}

esch_error_codes EschPointLight::load(XFParseIFF *iff, const char *lname)
{
    assertMyth("EschPointLight::load requries iff pointer",
               iff);

// Floating-point format
    if (iff->chunkid == iff->makeid('E','P','N','1'))
    {
        EschFileLightPoint  pnt;

        if (iff->chunkSize != sizeof(pnt))
            return ESCH_ERR_INVALIDDATA;

        // Read chunk and verify name, if given.
        if (iff->read(&pnt))
            return ESCH_ERR_FILEERROR;

        if (lname && strcmp(lname,pnt.name))
            return ESCH_ERR_NOTFOUND;

        // Found, so setup Point Light
        strcpy(name,pnt.name);
        flags = pnt.flags;

        VngoColor24bit  c(pnt.color);
        set_intensity( max( max(c.r, c.g), c.b ) );

        set_position( pnt.x, pnt.y, pnt.z );
    }
// Fixed-point format
    else if (iff->chunkid == iff->makeid('E','P','N','T'))
    {
        EschFileLightPointV1  pnt;

        if (iff->chunkSize != sizeof(pnt))
            return ESCH_ERR_INVALIDDATA;

        // Read chunk and verify name, if given.
        if (iff->read(&pnt))
            return ESCH_ERR_FILEERROR;

        if (lname && strcmp(lname,pnt.name))
            return ESCH_ERR_NOTFOUND;

        // Found, so setup Point Light
        strcpy(name,pnt.name);
        flags = pnt.flags;

        VngoColor24bit  c(pnt.color);
        set_intensity( max( max(c.r, c.g), c.b ) );

        set_position(pnt.x / 65536.0f, pnt.y / 65536.0f, pnt.z / 65536.0f);
    }
    else
    {
        return ESCH_ERR_NOTFOUND;
    }

// Return ok
    return ESCH_ERR_NONE;
}

// End of module - eslpoint.cpp 

