/*
 * This file is part of the MGUI (Morello Graphic User Interface)
 * library. Copyright 1995-2000 Vincenzo Morello.
 *
 */


#ifndef _MGUI_H
#define _MGUI_H

#include <stdarg.h>

#include "mguidefs.h"

#ifdef __cplusplus
extern "C" {
#endif

void	MGUIMain(int argc, char **argv);

void MClearDiskError(void);
int MGetDiskError(int *disk, int *code);
int MShouldRetryDueToDiskError(void);

int	MPaletteMode(void);
int	MGetScreenWidth(void);
int	MGetScreenHeight(void);
int	MGetNPlanes(void);
long	MGetNColors(void);
MTColor	MBlackColor(void);
MTColor	MWhiteColor(void);

char	*MGetSysDir(void);

void MEnableCustomizing(void);
void MEnableCharBrowser(void);

char	*MGetStringOption(const char *obj_name, const char *opt_name);
char	*MGetTextOption(const char *obj_name);
int	MGetIntOption(const char *obj_name, const char *opt_name, int *ret);
int	MGetFontOption(const char *obj_name, MTFont *ret);
int	MGetBGColorOption(const char *obj_name, int *pr, int *pg, int *pb);
int	MGetFGColorOption(const char *obj_name, int *pr, int *pg, int *pb);
int	MGetShadowOption(const char *obj_name, int *shadow, int *i_t, int *o_t);

void	MSetStringOption(const char *obj_name, const char *opt_name, const char *value);
void	MSetTextOption(const char *obj_name, const char *text);
void	MSetIntOption(const char *obj_name, const char *rname, int val);
void	MSetFGColorOption(const char *obj_name, int r, int g, int b);
void	MSetBGColorOption(const char *obj_name, int r, int g, int b);
void	MSetShadowOption(const char *obj_name, int shadow, int in_t, int out_t);
void	MSetFontOption(const char *obj_name, MTFont font);
void	MSaveOptions(void);

void	MMainLoop(void);
void	MLoopWhileFlag(int *pflag, int value);
void	MLoopWhileEvents(int discard);
void	MNextEvent(MEvent *pe, int wait);
void	MProcessEvent(MEvent *pe);
void	MGUITerm(void);

MTCursor MCreateCursor(MIMAGE bits, int xs, int ys);
MTFont	MLoadFont(const char *family, int h, int bold, int italic);
void	MUnloadFont(MTFont font);
const char *MGetFirstFontFamily(void);
const char *MGetNextFontFamily(void);

int     MTextCharX(const char *text, MTFont font, int x);
int     MTextHeight(const char *text, MTFont font);
int     MCharHeight(unsigned int car, MTFont font);
int     MTextWidth(const char *text, MTFont font);
int     MTextNWidth(const char *text, MTFont font, int len);
int     MCharWidth(unsigned int car, MTFont font);

void    MBeep(void);

MTColor	MAllocColor(int r, int g, int b);
MTColor	MAllocShiftColor(MTColor col, int shift);
void	MFreeColor(MTColor col);
MTColor	MAllocCell(int r, int g, int b);
void	MSetCellColor(MTColor c, int r, int g, int b);
void    MFreeCell(MTColor c);
void    MQueryColor(MTColor col, int *pr, int *pg, int *pb);

int	MGetKeyboardState(void);

void MGetDefaultBackgroundRGB(int *pr, int *pg, int *pb);
void MSetDefaultBackgroundRGB(int r, int g, int b);
void MGetDefaultForegroundRGB(int *pr, int *pg, int *pb);
void MSetDefaultForegroundRGB(int r, int g, int b);
MTColor  MAllocDefaultBackgroundColor(void);

/* MIMAGE */

MIMAGE MAllocImage(int img_w, int img_h, int nplanes);
MIMAGE MReadBMPFile(const char *fname);
MIMAGE MReadTIFFFile(const char *fname);
MIMAGE MReadJPEGFile(const char *fname);
MIMAGE MReadGIFFile(const char *fname);
MIMAGE MImageClone(MIMAGE img);
MIMAGE MCreateImageMaskByCMapIndex(MIMAGE img, int index);
MIMAGE MCreateImageMaskByPixel(MIMAGE img, int x, int y);
MIMAGE MCreateImageMaskByRGB(MIMAGE img, int r, int g, int b);

int MWriteTIFFFile(MIMAGE img, const char *name, int compression);
int MWriteBMPFile(MIMAGE img, const char *fname);
int MWriteJPEGFile(MIMAGE img, const char *name, int quality);

void	MFreeImage(MIMAGE data);
void	MInstallImage(MIMAGE data);
void	MUninstallImage(MIMAGE data);
void	MImageGetInfo(MIMAGE pmi, int *pw, int *ph, int *pnplanes);
void	MImageGetPixelRGB(MIMAGE pmi, int x, int y, int *pr, int *pg, int *pb);
void	MImageSetPixelRGB(MIMAGE pmi, int x, int y, int r, int g, int b);
int	MImageGetPixelIndex(MIMAGE pmi, int x, int y);
void	MImageSetPixelIndex(MIMAGE pmi, int x, int y, int i);
void	MImageGetCMapEntry(MIMAGE pmi, int i, int *pr, int *pg, int *pb);
void	MImageSetCMapEntry(MIMAGE pmi, int i, int r, int g, int b);

void	MImageSetMask(MIMAGE dst, MIMAGE mask);
void	MImageMerge(MIMAGE dst, MIMAGE src, int xo, int yo, int transparent);
void	MImageComputeBackgroundRGB(MIMAGE img, int *pr, int *pg, int *pb);

void MImageSetMaskByCMapIndex(MIMAGE img, int id);
void MImageSetMaskByPixel(MIMAGE img, int x, int y);
void MImageSetMaskByRGB(MIMAGE img, int r, int g, int b);


/* MLOG */

void	MLogInit(int level, int nr, int nvr, int nc, int nvc, const char *fname);
void	MLog(int level, const char *fmt, ...);
void	MMaskedLog(int level, CARD32 mask, const char *fmt, ...);
void	vMLog(int level, const char *fmt, va_list vl);
void	vMMaskedLog(int level, CARD32 mask, const char *fmt, va_list vl);
void	MLogSetLevel(int level);
void	MLogSetMask(CARD32 a);
CARD32	MLogGetMask(void);
void	MLogSetMaskBits(CARD32 a);
void	MLogResetMaskBits(CARD32 a);
void	MLogSetTimeFormat(const char *strftime_format);
void	MSetTraceFileName(const char *filename);
void	MTraceSetTimeFormat(const char *strftime_format);
void	Trace(const char *fmt,...);
void	vTrace(const char *fmt, va_list vl);

/* SPOOL */

int	MSpoolFile(const char *fname);
int	MSpoolTempFile(const char *fname);


void MDrawText(MOBJECT p, int x, int y, int w, int h, const char *text,
		MTFont font, MTColor fgc, int align);
void MDrawNText(MOBJECT p, int x, int y, int w, int h, const char *text,
		MTFont font, MTColor fgc, int align, int n);
void MDrawBGText(MOBJECT p, int x, int y, int w, int h, const char *text,
		MTFont font, MTColor bgc, MTColor fgc, int align);
void MDrawBGNText(MOBJECT p, int x, int y, int w, int h, const char *text,
		MTFont font, MTColor bgc, MTColor fgc, int align, int n);
void MDrawShadowedText(MOBJECT p, int x, int y, int w, int h, const char *text,
		MTFont font, int sh, MTColor fgc1, MTColor fgc2, int align);
void MDrawShadowedNText(MOBJECT p, int x, int y, int w, int h, const char *text,
		MTFont font, int sh, MTColor fgc1, MTColor fgc2, int align, int n);

void MDrawPoint(MOBJECT p, int x, int y, MTColor col);
void MDrawLine(MOBJECT p, int x1, int y1, int x2, int y2, MTColor col);
void MDrawPolyline(MOBJECT p, MPOINT *points, int n, MTColor col);
void MDrawFilledPolyline(MOBJECT p, MPOINT *points, int n, MTColor col);
void MDrawBox(MOBJECT p, int x, int y, int w, int h, MTColor bgc);
void MDrawFilledBox(MOBJECT p, int x, int y, int w, int h, MTColor bgc);
void MDrawCircle(MOBJECT p, int x, int y, int r, MTColor col);
void MDrawFilledCircle(MOBJECT p, int x, int y, int r, MTColor col);
void MDrawArc(MOBJECT p, int x, int y, int r, int a1, int a2, MTColor col);
void MDrawFilledArc(MOBJECT p, int x, int y, int r, int a1, int a2, MTColor c);
void MDrawImage(MOBJECT p, MIMAGE gi, int pm_x, int pm_y, int w, int h,
		int x, int y);
void MDrawXorLine(MOBJECT p, int x1, int y1, int x2, int y2);
void MDrawXorBox(MOBJECT p, int x, int y, int w, int h);

void MDrawWindow(MOBJECT o);
void MDrawShadow(MOBJECT o);
void MDrawSubShadow(MOBJECT o, int x, int y, int w, int h, MTColor tsc, MTColor bsc, int thick);
void MDrawBorder(MOBJECT o, int x, int y, int w, int h, MTColor col, int thick);


/* Object properties */

char *MObjectGetText(MOBJECT obj, char *text);
void MObjectSetText(MOBJECT obj, const char *text);
char *MObjectGetName(MOBJECT obj);
void MObjectSetName(MOBJECT obj, const char *name);
void *MObjectGetUserData(MOBJECT obj);
void MObjectSetUserData(MOBJECT obj, void *ud);
int  MObjectType(MOBJECT obj);
MOBJECT MObjectParent(MOBJECT obj);
void MObjectEnableOptions(MOBJECT obj, const char *name, CARD32 mask);
void MObjectSaveOptions(MOBJECT obj);
CARD32 MObjectGetEnabledOptions(MOBJECT obj);
void MObjectSetPopupHelp(MOBJECT obj, const char *str, int delay, int duration);
void MObjectSetLabelHelp(MOBJECT obj, const char *str, MOBJECT label);
void MObjectChangePopupHelpText(MOBJECT obj, const char *str);
void MObjectChangeLabelHelpText(MOBJECT obj, const char *str);


/* Object appereance */

void MObjectSetColor(MOBJECT obj, MTColor bgc, MTColor fgc);
MTColor MObjectBackgroundColor(MOBJECT obj);
MTColor MObjectForegroundColor(MOBJECT obj);
void MObjectSetBackgroundRGB(MOBJECT obj, int r, int g, int b);
void MObjectSetForegroundRGB(MOBJECT obj, int r, int g, int b);
MTFont	MObjectGetFont(MOBJECT obj);
void MObjectSetFont(MOBJECT obj, MTFont font, int resize);
void MObjectGetShadow(MOBJECT obj, int *type, int *in, int *out);
void MObjectSetShadow(MOBJECT obj, int type, int in, int out);
void MObjectSetTransparency(MOBJECT obj, CARD8 level);
CARD8 MObjectGetTransparency(MOBJECT obj);
void MObjectSetBackgroundImage(MOBJECT obj, MIMAGE img, int mode);
void MObjectSetBackgroundImageFile(MOBJECT obj, const char *fname, int mode);
MIMAGE MObjectGetBackgroundImage(MOBJECT obj);
void MObjectSetColorFromBackgroundImage(MOBJECT obj);
void MObjectSetBGImageMaskByCMapIndex(MOBJECT obj, int id);
void MObjectSetBGImageMaskByPixel(MOBJECT obj, int x, int y);
void MObjectSetBGImageMaskByRGB(MOBJECT obj, int r, int g, int b);

/* Object geometry */

int  MObjectGetWidth(MOBJECT obj);
int  MObjectGetHeight(MOBJECT obj);
int  MObjectGetCompositeWidth(MOBJECT obj);
int  MObjectGetCompositeHeight(MOBJECT obj);
int  MObjectGetX(MOBJECT obj);
int  MObjectGetY(MOBJECT obj);
void MObjectSetSize(MOBJECT obj, int w, int h);
void MObjectSetWidth(MOBJECT obj, int w);
void MObjectSetHeight(MOBJECT obj, int h);
void MObjectSetCompositeSize(MOBJECT obj, int w, int h);
void MObjectSetCompositeWidth(MOBJECT obj, int w);
void MObjectSetCompositeHeight(MOBJECT obj, int h);
void MObjectSetPosition(MOBJECT obj, int x, int y);
void MObjectSetX(MOBJECT obj, int x);
void MObjectSetY(MOBJECT obj, int y);
void MObjectSetAttachment(MOBJECT obj, int top, int bot, int left, int rig);
void MObjectSetOffset(MOBJECT obj, int t_o, int b_o, int l_o, int r_o);
void MObjectAlign(MOBJECT obj, int h_align, int v_align);


/* Object behaviour */

void MObjectSetUnsensitive(MOBJECT obj);
void MObjectSetSensitive(MOBJECT obj);
void MObjectSetSensitivity(MOBJECT obj, int status);
int MObjectGetSensitivity(MOBJECT obj);
void MObjectSetUnselectable(MOBJECT obj);
void MObjectSetSelectable(MOBJECT obj);
void MObjectSetResize(MOBJECT obj, int resize_w, int resize_h);
void MObjectSetCursor(MOBJECT obj, MTCursor cursor);


/* Object control */

void MObjectRedraw(MOBJECT obj);
void MObjectMap(MOBJECT obj);
void MObjectUnmap(MOBJECT obj);
void MObjectResize(MOBJECT obj, int w, int h);
void MObjectDestroy(MOBJECT obj);
void MDispatchObjEvent(MOBJECT p, MEvent *pe);
void MObjectAddEventNotifyCallback(MOBJECT obj, OBJ_EVENT_CB cb, CARD32 event_mask, void *a_data);
void MObjectAddDestroyCallback(MOBJECT obj, MGUI_CB cb, void *a_data);
void MObjectRemoveEventNotifyCallback(MOBJECT obj, OBJ_EVENT_CB cb, void *a_data);
void MObjectRemoveDestroyCallback(MOBJECT obj, MGUI_CB cb, void *a_data);
void MInsertObject(MOBJECT parent, MOBJECT obj, MOBJECT where, int resize);
void MDeleteObject(MOBJECT obj, int resize);
void MReparentObject(MOBJECT parent, MOBJECT obj, MOBJECT where);
void MSelectObject(MOBJECT f);
void MSelectNextBrother(MOBJECT f);
void MSelectPreviousBrother(MOBJECT f);
void MSelectNextCousin(MOBJECT p);
void MSelectPreviousCousin(MOBJECT p);
void MTopAttach(MOBJECT child, int att_type, int off, MOBJECT obj);
void MBottomAttach(MOBJECT child, int att_type, int off, MOBJECT obj);
void MRightAttach(MOBJECT child, int att_type, int off, MOBJECT obj);
void MLeftAttach(MOBJECT child, int att_type, int off, MOBJECT obj);


MOBJECT MObjectGetVScrollBar(MOBJECT obj);
MOBJECT MObjectGetHScrollBar(MOBJECT obj);


/* PIXMAP */

MOBJECT MCreatePixmapFromData(MOBJECT parent, MIMAGE data);
MOBJECT MCreatePixmapFromFile(MOBJECT parent, const char *fname);
void	MPixmapSetImage(MOBJECT pixmap, MIMAGE gi);
void	MPixmapSetImageFile(MOBJECT pixmap, const char *fname);
MIMAGE  MPixmapGetImage(MOBJECT pixmap);


/* SHELL */

MOBJECT MCreateShell(const char *title, int flags);
MOBJECT MObjectShell(MOBJECT obj);
Window  MShellGetWindow(MOBJECT s);
void    MShellDestroy(MOBJECT obj);
void MShellSetTitle(MOBJECT obj, const char *title);
void MShellGetTitle(MOBJECT obj, char *title);
void MShellRealize(MOBJECT obj);
void MShellRealizeXY(MOBJECT obj, int x, int y);
void MShellRealizeXYWH(MOBJECT obj, int x, int y, int w, int h);
void MShellUnrealize(MOBJECT obj);
void MShellSetWMCloseCallback(MOBJECT p, WM_CLOSE_CB cb, void *a_data);
void MShellRaise(MOBJECT obj);
void MShellConfigure(MOBJECT obj, int x, int y, int w, int h);


/* FORM */

MOBJECT MCreatePosForm(MOBJECT parent);
MOBJECT MCreateRowForm(MOBJECT parent);
MOBJECT MCreateColForm(MOBJECT parent);
MOBJECT MCreatePagedPosForm(MOBJECT parent, int npage);
MOBJECT MCreatePagedRowForm(MOBJECT parent, int npage);
MOBJECT MCreatePagedColForm(MOBJECT parent, int npage);
MOBJECT MCreatePagedPosFormExt(MOBJECT parent, int npage);
MOBJECT MCreatePagedRowFormExt(MOBJECT parent, int npage);
MOBJECT MCreatePagedColFormExt(MOBJECT parent, int npage);
MOBJECT MCreateTabbedPosForm(MOBJECT parent, int npage, MTFont f);
MOBJECT MCreateTabbedRowForm(MOBJECT parent, int npage, MTFont f);
MOBJECT MCreateTabbedColForm(MOBJECT parent, int npage, MTFont f);
MOBJECT MCreateScrolledPosForm(MOBJECT parent, int clip_w, int clip_h);
MOBJECT MCreateScrolledRowForm(MOBJECT parent, int clip_w, int clip_h);
MOBJECT MCreateScrolledColForm(MOBJECT parent, int clip_w, int clip_h);
MOBJECT MCreateClippedPosForm(MOBJECT parent, int clip_w, int clip_h);
MOBJECT MCreateClippedRowForm(MOBJECT parent, int clip_w, int clip_h);
MOBJECT MCreateClippedColForm(MOBJECT parent, int clip_w, int clip_h);

void    MFormSetSpacing(MOBJECT p, int offset);
void    MFormSetChildSelectCallback(MOBJECT p, OBJ_VALUE_CB cb, void *data);
void	MScrolledFormSetIncr(MOBJECT obj, int h_inc, int v_inc);
int	MSelectFirstChild(MOBJECT f);
int	MSelectLastChild(MOBJECT f);
MOBJECT MGetSelectedObject(MOBJECT f);
void    MFormSetPrefAtt(MOBJECT p, int pref_att);
void    MFormSetFocusWrap(MOBJECT f, int true_false);
void 	MFormSetFocusHighlight(MOBJECT f, int true_false);
void    MFormFocusLink(MOBJECT obj1, MOBJECT obj2);
void    MPagedFormSetActivePage(MOBJECT form, int page);
int     MPagedFormGetActivePage(MOBJECT form);
void	MPagedFormDisablePageKeys(MOBJECT form);
void	MPagedFormEnablePageKeys(MOBJECT form);
void	MPagedFormSetNPages(MOBJECT form, int n_pages);
void    MPagedFormSetActivePageCallback(MOBJECT p, OBJ_VALUE_CB cb, void *data);
void    MFormSetClipX(MOBJECT obj, int clip_x);
void    MFormSetClipY(MOBJECT obj, int clip_y);
int	MFormGetClipX(MOBJECT obj);
int	MFormGetClipY(MOBJECT obj);
void    MFormAddKeyCallback(MOBJECT shell, int key, OBJ_VALUE_CB cb, void *ad);
MOBJECT MFormGetFirstChild(MOBJECT form);
MOBJECT MFormGetNextChild(MOBJECT form);
int     MFormGetNChildren(MOBJECT form);

MOBJECT MCreateTableForm(MOBJECT parent, int n_cols);
MOBJECT MCreateScrolledTableForm(MOBJECT parent, int n_cols,
	int clip_w, int clip_h,	int head_rows, int head_cols);
void MTableFormShowGrid(MOBJECT form);
void MTableFormHideGrid(MOBJECT form);
void MTableFormSetRowHeight(MOBJECT form, int row, int h);
void MTableFormSetColWidth(MOBJECT form, int col, int w);
void MTableFormSetColFont(MOBJECT form, int col, MTFont f);
void MTableFormSetColAlign(MOBJECT form, int col, int al);
void MTableFormSetRowAlign(MOBJECT form, int row, int al);
void MTableFormSetColBackgroundRGB(MOBJECT form, int col, int r, int g, int b);
void MTableFormSetColForegroundRGB(MOBJECT form, int col, int r, int g, int b);
void MTableFormSetCellText(MOBJECT form, int x, int y, const char *text);
void MTableFormGetCellText(MOBJECT form, int x, int y, char *text);
void MTableFormSetCellFont(MOBJECT form, int x, int y, MTFont f);
void MTableFormSetCellAlign(MOBJECT form, int x, int y, int h_al, int v_al);
void MTableFormSetCellBackgroundRGB(MOBJECT form, int x, int y, int r, int g, int b);
void MTableFormSetCellForegroundRGB(MOBJECT form, int x, int y, int r, int g, int b);
MOBJECT MTableFormGetCellObject(MOBJECT form, int x, int y);
void MTableFormSetNCol(MOBJECT form, int nc);
int  MTableFormGetNCol(MOBJECT form);
void MTableFormDeleteRow(MOBJECT form, int row);
void MTableFormDeleteDataRows(MOBJECT form);
void MTabbedFormSetTabText(MOBJECT form, int page, const char *buff);
void MTabbedFormGetTabText(MOBJECT form, int page, char *buff);

/* DRAWAREA */

MOBJECT MCreateDrawArea(MOBJECT parent, int width, int height);
void    MDrawAreaSetExposeCallback(MOBJECT obj, OBJ_EVENT_CB cb, void *ad);
void    MDrawAreaSetInputCallback(MOBJECT obj, OBJ_EVENT_CB cb, void *ad);
void    MDrawAreaSetResizeCallback(MOBJECT obj, DRAWAREA_PREF_CB cb, void *ad);


/* Formatted Text objects */

MOBJECT MCreateFormattedText(MOBJECT par, const char *text, int clip_w, MTFont f);
MOBJECT MCreateScrolledFormattedText(MOBJECT par, const char *text,
									 int clip_w, int clip_h, MTFont f);

void MFormattedTextSetMargin(MOBJECT obj, int w_m, int h_m);
void MFormattedTextSetHSpacing(MOBJECT obj, int h_s);
void MFormattedTextSetTabSize(MOBJECT obj, int ts);

/* LABEL */

MOBJECT MCreateLabel(MOBJECT parent, const char *text, MTFont font);
void	MLabelSetMargin(MOBJECT p, int wm, int hm);
void	MLabelSetHSpacing(MOBJECT p, int hs);
void	MLabelSetAlignment(MOBJECT p, int align);
const char *MLabelValue(MOBJECT label);


/* PBUTTON */

MOBJECT MCreateDrawnPButton(MOBJECT parent, MIMAGE pn, MIMAGE pa, MIMAGE pu);
MOBJECT MCreateDrawnPButtonExt(MOBJECT parent,
				   MIMAGE pn, MIMAGE pa, MIMAGE pu,
				   MIMAGE pn_on, MIMAGE pa_on, MIMAGE pu_on);
MOBJECT MCreateImagePButton(MOBJECT parent, const char *npf, const char *apf,
							const char *upf);
MOBJECT MCreateImagePButtonExt(MOBJECT parent,
				const char *npf, const char *apf, const char *upf,
				const char *non, const char *aon, const char *uon);
MOBJECT MCreateDrawnTextPButton(MOBJECT parent, const char *text, MTFont font,
				MIMAGE pn, MIMAGE pa, MIMAGE pu);
MOBJECT MCreateDrawnTextPButtonExt(MOBJECT parent, const char *text, MTFont font,
				   MIMAGE pn, MIMAGE pa, MIMAGE pu,
				   MIMAGE pn_on, MIMAGE pa_on, MIMAGE pu_on);
MOBJECT MCreateImageTextPButton(MOBJECT parent, const char *text, MTFont font,
				const char *npf, const char *apf, const char *upf);
MOBJECT MCreateImageTextPButtonExt(MOBJECT parent, const char *text, MTFont font,
				const char *npf, const char *apf, const char *upf,
				const char *non, const char *aon, const char *uon);
MOBJECT MCreatePButton(MOBJECT parent, const char *text, MTFont font);
void    MPButtonSetCallback(MOBJECT button, MGUI_CB cb, void *data);
void    MPButtonSetRepeatDelay(MOBJECT button, int first, int repeat);
void    MPButtonSetMargin(MOBJECT p, int wm, int hm);
void	MPButtonSetHSpacing(MOBJECT p, int hs);
void	MPButtonSetAlignment(MOBJECT p, int align);
void	MPButtonSetImage(MOBJECT obj, const char *npf, const char *apf,
						 const char *upf);

/* RBUTTON */

MOBJECT MCreateDrawnRButton(MOBJECT parent, MIMAGE np, MIMAGE ap, MIMAGE up);
MOBJECT MCreateDrawnRButtonExt(MOBJECT parent,
				   MIMAGE pn, MIMAGE pa, MIMAGE pu,
				   MIMAGE pn_on, MIMAGE pa_on, MIMAGE pu_on);
MOBJECT MCreateImageRButton(MOBJECT parent,
				const char *npf, const char *apf, const char *upf);
MOBJECT MCreateImageRButtonExt(MOBJECT parent,
				const char *npf, const char *apf, const char *upf,
				const char *non, const char *aon, const char *uon);
MOBJECT MCreateDrawnTextRButton(MOBJECT parent, const char *text, MTFont font,
				MIMAGE np, MIMAGE ap, MIMAGE up);
MOBJECT MCreateDrawnTextRButtonExt(MOBJECT parent, const char *text, MTFont font,
				   MIMAGE pn, MIMAGE pa, MIMAGE pu,
				   MIMAGE pn_on, MIMAGE pa_on, MIMAGE pu_on);
MOBJECT MCreateImageTextRButton(MOBJECT parent, const char *text, MTFont font,
				const char *npf, const char *apf, const char *upf);
MOBJECT MCreateImageTextRButtonExt(MOBJECT parent, const char *text, MTFont font,
				const char *npf, const char *apf, const char *upf,
				const char *non, const char *aon, const char *uon);
MOBJECT MCreateRButton(MOBJECT parent, const char *text, MTFont font);
void	MRButtonActivate(MOBJECT o);
void	MRButtonDeactivate(MOBJECT o);
void	MRButtonSetCallback(MOBJECT button, OBJ_VALUE_CB cb, void *data);
int	MRButtonGetActivationStatus(MOBJECT o);
void	MRButtonSetImage(MOBJECT obj, const char *npf, const char *apf, const char *upf);


/* TBUTTON */

MOBJECT MCreateDrawnTButton(MOBJECT parent, MIMAGE np, MIMAGE ap, MIMAGE up);
MOBJECT MCreateDrawnTButtonExt(MOBJECT parent,
				   MIMAGE pn, MIMAGE pa, MIMAGE pu,
				   MIMAGE pn_on, MIMAGE pa_on, MIMAGE pu_on);
MOBJECT MCreateImageTButton(MOBJECT parent,
				const char *npf, const char *apf, const char *upf);
MOBJECT MCreateImageTButtonExt(MOBJECT parent,
				const char *npf, const char *apf, const char *upf,
				const char *non, const char *aon, const char *uon);
MOBJECT MCreateDrawnTextTButton(MOBJECT parent, const char *text, MTFont font,
					MIMAGE np, MIMAGE ap, MIMAGE up);
MOBJECT MCreateDrawnTextTButtonExt(MOBJECT parent, const char *text, MTFont font,
				   MIMAGE pn, MIMAGE pa, MIMAGE pu,
				   MIMAGE pn_on, MIMAGE pa_on, MIMAGE pu_on);
MOBJECT MCreateImageTextTButton(MOBJECT parent, const char *text, MTFont font,
				const char *npf, const char *apf, const char *upf);
MOBJECT MCreateImageTextTButtonExt(MOBJECT parent, const char *text, MTFont font,
				const char *npf, const char *apf, const char *upf,
				const char *non, const char *aon, const char *uon);
MOBJECT MCreateTButton(MOBJECT parent, const char *text, MTFont font);
void	MTButtonActivate(MOBJECT o);
void	MTButtonDeactivate(MOBJECT o);
void	MTButtonSetCallback(MOBJECT button, OBJ_VALUE_CB cb, void *data);
void	MTButtonSetActivationStatus(MOBJECT o, int status);
int	MTButtonGetActivationStatus(MOBJECT o);
void	MTButtonSetImage(MOBJECT obj, const char *npf, const char *apf, const char *upf);

/* Generic Drawn BUTTON functions (work on Push/Toggle/Radio) */

void MButtonSetTextVsImageAlignment(MOBJECT button, CARD8 align);
void MButtonSetOnMouseOverForegroundColor(MOBJECT button, MTColor cc);
void MButtonSetOnMouseOverForegroundRGB(MOBJECT button, int r, int g, int b);
MIMAGE  MDrawnButtonChangeNormalImage(MOBJECT button, MIMAGE new_img);
MIMAGE  MDrawnButtonChangeArmedImage(MOBJECT button, MIMAGE new_img);
MIMAGE  MDrawnButtonChangeUnsensitiveImage(MOBJECT button, MIMAGE new_img);
MIMAGE  MDrawnButtonChangeNormalOnImage(MOBJECT button, MIMAGE new_img);
MIMAGE  MDrawnButtonChangeArmedOnImage(MOBJECT button, MIMAGE new_img);
MIMAGE  MDrawnButtonChangeUnsensitiveOnImage(MOBJECT button, MIMAGE new_img);
void MButtonChangeImageFiles(MOBJECT obj, 
				const char *npf, const char *apf, const char *upf,
				const char *npfo, const char *apfo, const char *upfo);
void MButtonSetShadowOnMouseOver(MOBJECT obj, int thick);
void MButtonSetAccelerator(MOBJECT button, int key);

void MButtonSetImageMaskByCMapIndex(MOBJECT obj, int id);
void MButtonSetImageMaskByPixel(MOBJECT obj, int x, int y);
void MButtonSetImageMaskByRGB(MOBJECT obj, int r, int g, int b);
void MButtonUnsetOnMouseOverForeground(MOBJECT obj);
void MButtonUnsetShadowOnMouseOver(MOBJECT obj);

/*STATUSBAR */

MOBJECT MCreateStatusBar(MOBJECT parent, const char *text, int max, MTFont font);
void    MStatusBarSetMax(MOBJECT p, int max);
void    MStatusBarSetPos(MOBJECT p, int pos);
void	MStatusBarSetSliderColor(MOBJECT p, MTColor col);
void	MStatusBarSetSliderRGB(MOBJECT p, int r, int g, int b);

/*  MStatusBarSetWidth() is obsoleted by MObjectSetWidth().
 *  The following macro is only present for backward compatibility.
 */

#define MStatusBarSetWidth(obj, w)	MObjectSetWidth(obj, w)

/* SCROLLBAR */

MOBJECT MCreateScrollBar(MOBJECT parent, int type, long n, long max);
void    MScrollBarSetMax(MOBJECT p, long max);
void    MScrollBarSetPos(MOBJECT p, long pos);
void    MScrollBarSetN(MOBJECT p, long n);
void    MScrollBarSetIncr(MOBJECT p, int incr);
void    MScrollBarSetPosCallback(MOBJECT p, OBJ_VALUE_CB cb, void *data);
void	MScrollBarSetValues(MOBJECT p, long max, long n, long pos);
long    MScrollBarGetMax(MOBJECT p);
long    MScrollBarGetPos(MOBJECT p);
long    MScrollBarGetN(MOBJECT p);


/* SBLIST */

MOBJECT MCreateSText(MOBJECT parent, const char *text, MTFont font,
		int nrow, int ncol);
MOBJECT MCreateSTextClip(MOBJECT parent, NEW_BLOCK_CB nb, void *nbd,
		MTFont font, int nrow, int ncol, int xc, long xn);
MOBJECT MCreateSFile(MOBJECT parent, const char *fname, MTFont font,
		int nrow, int ncol);
MOBJECT MCreateSList(MOBJECT parent, MTFont font, int row_len,
		int nrow, int ncol);
MOBJECT MCreateSListMultiSel(MOBJECT parent, MTFont font,
		int row_len, int nrow, int ncol);
MOBJECT MCreateSListClip(MOBJECT parent, NEW_BLOCK_CB nb, void *nbd,
		ITEM_SEEK_CB is, void *isd, MTFont font,
		int nrow, int ncol, int xc, long xn);
MOBJECT MCreateSListSkip(MOBJECT parent, NEW_BLOCK_CB nb, void *nbd,
		ITEM_SEEK_CB is, void *isd, MTFont font,
		int nrow, int ncol, int xc);

MOBJECT MCreateEditList(MOBJECT parent, MTFont font, EL_TEMPLATE *elt,
		int ne, int nrow, int ncol);

LIST_ACT *MSListGetFirstItem(MOBJECT p);
LIST_ACT *MSListGetFirstSelectedItem(MOBJECT p);
LIST_ACT *MSListGetNextItem(MOBJECT p);
LIST_ACT *MSListGetNextSelectedItem(MOBJECT p);
LIST_ACT *MSListGetFirstUnselectedItem(MOBJECT p);
LIST_ACT *MSListGetNextUnselectedItem(MOBJECT p);
long MSListGetNItem(MOBJECT p);

void MSListSelectItem(MOBJECT p, const char *item);
void MSListDeselectItem(MOBJECT p, const char *item);
void MSListSelectElement(MOBJECT p, long n);
void MSListDeselectElement(MOBJECT p, long n);
void    MSListClear(MOBJECT p);
int     MSListAddItem(MOBJECT p, const char *item, void *data);
int     MSListDelItem(MOBJECT p, const char *item);
int     MSListAddElement(MOBJECT p, long n, const char *item, void *data);
int     MSListDelElement(MOBJECT p, long n);
void    MSListChangeItem(MOBJECT p, const char *old, const char *new_item, void *u_data);
void    MSListChangeElement(MOBJECT p, long n, const char *new_item, void *u_data);
LIST_ACT *MSListGetItem(MOBJECT p, const char *item);
LIST_ACT *MSListGetElement(MOBJECT p, long n);
LIST_ACT *MSListGetSelected(MOBJECT p);
void    MSListSetCallback(MOBJECT obj, LIST_ACT_CB cb, void *data);
void	MSListSetPickCallback(MOBJECT obj, LIST_ACT_CB cb, void *value);
void    MSListSetSelectionCallback(MOBJECT obj, LIST_SEL_CB cb, void *data);
void    MSListSetSortCallback(MOBJECT obj, STRCMP_CB cb);
void    MSListFreeze(MOBJECT obj);
void    MSListThaw(MOBJECT obj);
void    MSBLSetHead(MOBJECT obj, const char *text);
void    MSBLGetHead(MOBJECT obj, char *buff);
void    MSBLHeadSetColor(MOBJECT obj, MTColor bgc, MTColor fgc);
void    MSBLHeadSetBackgroundImage(MOBJECT obj, MIMAGE img, int mode);
void	MSListSetHighLightColors(MOBJECT obj, MTColor bgc, MTColor fgc);

void    MSTextSetString(MOBJECT p, const char *text);

void    MSFileSetFile(MOBJECT obj, const char *fname);

void    MSBLSetHMargin(MOBJECT p, int hm);

int     MEditListAddItem(MOBJECT p, char **argv, void *u_data);
void    MEditListChangeElement(MOBJECT p, long n, char **new_ele, void *data);
void    MEditListDelElement(MOBJECT p, long n);
void    MEditListSeekItem(MOBJECT p, char **argv, long *pos, void **u_data);
void    MEditListGetItemField(MOBJECT el, const char *item, int ii, char *buff);

int	MEditListAddItemFromEdits(MOBJECT p, void *u_data);
void MEditListGetEditText(MOBJECT p, int n, char *buff);
void MEditListSetEditText(MOBJECT p, int n, const char *buff);

int  MSListGetItemCheckStatus(MOBJECT p, const char *item);
void MSListSetItemCheckStatus(MOBJECT p, const char *item, int status);
void MSListReset(MOBJECT p);
void MSListEnableMultiSel(MOBJECT obj);
void MSListEnableItemMotion(MOBJECT p);
void MSListDisableMultiSel(MOBJECT obj);
void MSListDisableItemMotion(MOBJECT obj);

MOBJECT MSListGetVScrollBar(MOBJECT p);
MOBJECT MSListGetHScrollBar(MOBJECT p);

/* EDIT */

MOBJECT MCreateClippedEdit(MOBJECT parent, const char *text, int len, int clip_len,
		MTFont font);
MOBJECT MCreateEdit(MOBJECT parent, const char *text, int len, MTFont font);
MOBJECT MCreateClippedHiddenEdit(MOBJECT parent, const char *text, int len, int clip_len,
		MTFont font, char shown_char);
MOBJECT MCreateHiddenEdit(MOBJECT parent, const char *text, int len, MTFont font,
		char shown_char);
MOBJECT MCreateSpinEdit(MOBJECT parent, const char *text, int len, int clip_len, MTFont font, OBJ_VALUE_CB cb, void *cd);
void MEditSetMargin(MOBJECT edit, int w_margin, int h_margin);
void MEditSetFilter(MOBJECT edit, int filter, const char *extra_chars);

MOBJECT MCreateClippedMLEdit(MOBJECT parent, const char *text,
		int width, int height, MTFont font);
MOBJECT MCreateScrolledMLEdit(MOBJECT parent, const char *text,
		int width, int height, MTFont font);

void	MEditChangeText(MOBJECT edit, int off, int len, const char *new_t, int chg_len);
void	MEditGetSubString(MOBJECT edit, int start, int end, char *str);
int	MEditFindSubStringForward(MOBJECT edit, int off, const char *str, int icase);
int	MEditFindSubStringBackward(MOBJECT edit, int off, const char *str, int icase);
void	MEditSetSelection(MOBJECT edit, int start, int end);
void	MEditGetSelection(MOBJECT edit, int *pstart, int *pend);
void	MEditSetCursorPos(MOBJECT edit, int pos);
int	MEditGetCursorPos(MOBJECT edit);
void	MEditSetCursorXY(MOBJECT edit, int x, int y);
void	MEditGetCursorXY(MOBJECT edit, int *px, int *py);
void	MEditSetAutoIndent(MOBJECT edit, int on_off);
void	MEditSetTabSize(MOBJECT edit, int size);
void	MEditSetClipWindow(MOBJECT edit, int clip_w, int clip_h);

void	MEditSetCursorPosCallback(MOBJECT edit, EDIT_CURS_CB cb, void *data);

void MEditSetTextExt(MOBJECT edit, const char *text, int validated, int notify);
void MEditSetActCallback(MOBJECT edit, OBJ_STRING_CB cb, void *data);
void MEditSetValCallback(MOBJECT edit, EDIT_VAL_CB cb, void *data);
void MEditSetValueChangeCallback(MOBJECT edit, EDIT_CHANGE_CB cb, void *data);
void MEditSetNewValueCallback(MOBJECT edit, OBJ_STRING_CB cb, void *data);
void MEditSetHotKeyCallback(MOBJECT edit, int key, MGUI_CB cb, void *data);
void MEditSetValidatedColors(MOBJECT edit, MTColor bg_color, MTColor fg_color);
void MEditSetNotValidatedColors(MOBJECT edit, MTColor bgc, MTColor fgc);
void MEditSetValidationStatus(MOBJECT edit, int validated);
void MEditSetReplace(MOBJECT edit);
int MIsEditValidated(MOBJECT p);
int MEditGetFilter(MOBJECT p);
const char *MEditValue(MOBJECT edit);
void MEditHide(MOBJECT edit, char shown_char);
void MEditUnhide(MOBJECT edit);
void MSpinEditSetCallback(MOBJECT spinedit, OBJ_VALUE_CB cb, void *data);
void MEditEnableEditing(MOBJECT edit);
void MEditDisableEditing(MOBJECT edit);


/* OPTIONEDIT */

MOBJECT MCreateOptionEdit(MOBJECT parent, const char *text, int len, int clip_len,
		int nrow, MTFont font);
MOBJECT MOptionEditGetList(MOBJECT ce);
void    MOptionEditSetListCallback(MOBJECT obj, LIST_ACT_CB cb, void *data);


/* MENU */

MOBJECT MCreateMenuBar(MOBJECT parent, MTFont font);
MOBJECT MCreateMenu(MOBJECT menu, const char *text);
MOBJECT MCreateSubMenu(MENU_ITEM item);
MOBJECT MCreatePopupMenu(MTFont font);
void MDestroyMenu(MOBJECT menu);
MENU_ITEM MMenuAddItemObsolete(MOBJECT pd, const char *item, MGUI_CB cb, void *ad);
MENU_ITEM MMenuAddItem(MOBJECT pd, const char *item, MENU_ITEM_CB cb, void *ad);
MENU_ITEM MMenuAddRadioItem(MOBJECT pd, const char *item, MENU_ITEM_CB cb, void *ad);
MENU_ITEM MMenuAddToggleItem(MOBJECT pd, const char *item, MENU_ITEM_CB cb, void *ad);
MENU_ITEM MMenuAddSeparator(MOBJECT pd);
void MPopupMenu(MOBJECT pm, int x, int y);
void MMenuSetTitleText(MOBJECT pd, const char *text);
void MMenuItemSetText(MENU_ITEM item, const char *text);
void MMenuItemSetAccel(MENU_ITEM item, int acc, const char *at);
void MMenuItemSetSensitive(MENU_ITEM item);
void MMenuItemSetUnsensitive(MENU_ITEM item);
void MMenuItemSetSelectCB(MOBJECT p, MGUI_CB cb, void *ad);
void MMenuBarSetAcc(MOBJECT menu, int acc);
const char *MMenuItemGetText(MENU_ITEM item);
const char *MMenuGetTitleText(MOBJECT obj);
int     MMenuItemSetCheckStatus(MENU_ITEM item, int TrueFalse);
int     MMenuItemGetCheckStatus(MENU_ITEM item);
int     MMenuItemToggleCheckStatus(MENU_ITEM item);
MOBJECT MMenuItemGetMenu(MENU_ITEM item);
void	MMenuBarSetHighLightColors(MOBJECT obj, MTColor bgc, MTColor fgc);
void	MMenuSetHighLightColors(MOBJECT obj, MTColor bgc, MTColor fgc);


/* MGUIUTIL */

int MMessageDialog(const char *t, const char *msg, const char *btn1, const char *btn2, ...);
int vMMessageDialog(const char *t, const char *msg, const char *btn1, const char *btn2, va_list vl);
int MInputPrompt(const char *t, const char *msg, char *text, int len, const char *ok_btn,
	const char *canc_btn);
char *MTextPrompt(const char *t, const char *msg, const char *text, int w, int h, const char *ok,
	const char *cancel);
int	MFileSelection(const char *title, const char *filter, char *fname, char *dname,
		int change_cur_dir);
char	*MGetCurrentDirectory(void);

void	MSetInputPromptRGB(int r, int g, int b);
void	MSetMessageDialogRGB(int r, int g, int b);
void	MSetFileSelectionRGB(int r, int g, int b);
void	MGetInputPromptRGB(int *pr, int *pg, int *pb);
void	MGetMessageDialogRGB(int *pr, int *pg, int *pb);
void	MGetFileSelectionRGB(int *pr, int *pg, int *pb);
void    MSetFileSelectionTextStrings(
			const char *file_list, const char *dir_list,
			const char *drive_list, const char *filter,
			const char *filename, const char *curr_dir,
			const char *ok_button, const char *close_button);
void    MSetDiskErrorTextStrings(
			const char *write_protect,
			const char *not_ready,
			const char *unformatted,
			const char *not_responding);
void MSetMessageDialogFgRGB(int r, int g, int b);
void MGetMessageDialogFgRGB(int *pr, int *pg, int *pb);
void MSetMessageDialogFonts(MTFont text_font, MTFont buttons_font);
void MGetMessageDialogFonts(MTFont *text_font, MTFont *buttons_font);
void MSetInputPromptFgRGB(int r, int g, int b);
void MGetInputPromptFgRGB(int *pr, int *pg, int *pb);
void MSetInputPromptFonts(MTFont text, MTFont edit, MTFont btn);
void MGetInputPromptFonts(MTFont *text_font, MTFont *edit_font, MTFont *buttons_font);


/* Timeout */

TIMEOUT_ID MAddTimeout(CARD32 tmout, TIMEOUT_CB cb, void *ud);
void MRemoveTimeout(TIMEOUT_ID id);
void MCallAllTimeoutCallbacks(void);
void MRefreshTimeout(TIMEOUT_ID *pid, CARD32 msec, TIMEOUT_CB cb, void *ud);
void MClearTimeout(TIMEOUT_ID *pid);
void MCallExpiredTimeoutCallbacks(void);

/* File values read/write */

void MWriteCARD8(FILE *fp, CARD8 val);
void MWriteCARD16(FILE *fp, CARD16 val);
void MWriteCARD32(FILE *fp, CARD32 val);
CARD8 MReadCARD8(FILE *fp);
CARD16 MReadCARD16(FILE *fp);
CARD32 MReadCARD32(FILE *fp);

/* Socket facilities */

void MAddSocketHandler(int s, unsigned events, SOCKET_CB cb, void *data);
void MRemoveSocketHandler(int s);
void MSocketSetEventsMask(int s, unsigned events);
unsigned int MSocketGetEventsMask(int s);

/*
 *  Graphic printing currently works only under Windows
 */

#ifdef WIN_MGUI

DOC_ID MOpenPrintDoc(char *doc_name, int open_dialog, int orientation, MPAGE_OPTIONS *pages);
void MClosePrintDoc(DOC_ID id);
void MPrintDocNextPage(DOC_ID id);
void MPrintObject(MOBJECT obj, DOC_ID id, MPRINT_OPTIONS *po);
void MPrintImage(MIMAGE mpm, int full);
int MPrintSetup(int orientation, int *xres, int *yres, int *pwidth, int *pheight, int *nplanes);

#endif /* WIN_MGUI */

/*
 * for backwards compatibility... (release > 1.8.0)
 */
#ifdef OLD_MGUI

#define PULLDOWN_ITEM	MENU_ITEM
#define OT_PULLDOWN	OT_MENU
#undef OT_MENU
#define OT_MENU		OT_MENUBAR

#define MCreatePullDown(menubar, text) \
	MCreateMenu(menubar, text)
#define MCreateSubPullDown(pd, item) \
	MCreateSubMenu(item)
#define MCreatePopupPullDown(font) \
	MCreatePopupMenu(font)
#define MPullDownAddItem(pm, t, c, d) \
	MMenuAddItemObsolete(pm, t, c, d)
#define MPopupPullDown(pm, x, y) \
	MPopupMenu(pm, x, y)
#define MPullDownSetTitleText(pd, text) \
	MMenuSetTitleText(pd, text)
#define MPullDownSetItemText(pd, item, text) \
	MMenuItemSetText(item, text)
#define MPullDownSetItemAccel(pd, item, acc, at) \
	MMenuItemSetAccel(item, acc, at)
#define MPullDownSetItemSensitive(pd, item) \
	MMenuItemSetSensitive(item)
#define MPullDownSetItemUnsensitive(pd, item) \
	MMenuItemSetUnsensitive(item)
#define MPullDownSetItemSelectCB(p, cb, ad) \
	MMenuItemSetSelectCB(p, cb, ad)
#define MMenuSetAcc(menu, acc) \
	MMenuBarSetAcc(menu, acc)
#define MPullDownSetItemCallback(pdm, id, cb, ad) \
	MMenuItemSetCallback(id, cb, ad)
#define MPullDownGetItemText(pd, item) \
	MMenuItemGetText(item)
#define MPullDownGetTitleText(obj) \
	MMenuGetTitleText(obj)
#define MSetItemCheckStatus(pd, item, TrueFalse) \
	MMenuItemSetCheckStatus(item, TrueFalse)
#define MGetItemCheckStatus(pd, item) \
	MMenuItemGetCheckStatus(item)
#define MToggleItemCheckStatus(pd, item) \
	MMenuItemToggleCheckStatus(item)
#define MDestroyPullDown(pd) \
	MDestroyMenu(pd)

#define MCreatePagedForm(parent, npage) \
	MCreatePagedRowForm(parent, npage)

#define MCreatePagedFormExt(parent, npage) \
	MCreatePagedRowFormExt(parent, npage)

#define MCreateTabbedForm(parent, npage, f) \
	MCreateTabbedRowForm(parent, npage, f)

#define MCreateScrolledForm(parent, clip_w, clip_h) \
	MCreateScrolledRowForm(parent, clip_w, clip_h)

#define MCreateClippedForm(parent, clip_w, clip_h) \
	MCreateClippedRowForm(parent, clip_w, clip_h)

#define MObjectWidth(parent) \
	MObjectGetWidth(parent)

#define MObjectHeight(parent) \
	MObjectGetHeight(parent)

#define MObjectX(parent) \
	MObjectGetX(parent)

#define MObjectY(parent) \
	MObjectGetY(parent)

#define MPButtonChangeNormalImage(button, new_img) \
	MDrawnButtonChangeNormalImage(button, new_img)

#define MPButtonChangeArmedImage(button, new_img) \
	MDrawnButtonChangeArmedImage(button, new_img)

#define MPButtonChangeUnsensitiveImage(button, new_img) \
	MDrawnButtonChangeUnsensitiveImage(button, new_img)

#define MPButtonSetAccelerator(button, key) \
	MButtonSetAccelerator(button, key)

#endif /* OLD_MGUI */

#ifdef __cplusplus
}
#endif

#endif
