/*Data Access Library. v. 1.1.
 *JDBC types support. Translates between human readable Types description and
 *codes from java.sql.Types, java data types etc.
 *AUTHOR:  Alexander Jaremenko <jarem@altavista.net>
 *RELEASE DATE: 
 */

package JProjects.eab.data.wizzard;

import java.sql.Types;
import java.util.*;

/**JDBC types support. Translates between human readable Types description and
 *codes from java.sql.Types, java data types etc. This class is for internal
 *use and gets public due to implementation constarints.
 *@see DCGenerator
 *@author Alexander Jaremenko  <address><a href="mailto:jarem@altavista.net">&lt jarem@altavista.net &gt</a></address>
 */
public class JDBCTypes {
    private static Hashtable _sqlTypMap = null;
    private static Hashtable _strToSQLTypeMap = null;

  /**Convert string, naming JDBC type to type code.
   *@param n human readable JDBC type's name
   *@return JDBC type's code as defined in java.sql.Types or -1 if
   *arg is not valid type name.
   */
    public static int mapToSQLType(String n) {
	if (_strToSQLTypeMap==null)
	    fillStrToSQLTypeMap();
	Integer res = (Integer)_strToSQLTypeMap.get(n);
	if (res==null)
	    return -1;
	return res.intValue();
    }

  /**Perform reverse mapping: from JDBC type code to name.
   *@param sqlT JDBC type code as defined in java.sql.Types
   *@return human readable JDBC type's name
   */
    public static String getSQLType(int sqlT) {
	if (_sqlTypMap==null)
	    fillTypMap();
	String[] ts = (String[])_sqlTypMap.get(new Integer(sqlT));
	if (ts!=null)
	    return ts[2];
	return null;
    }

    static String[] getSQLStrings(int sqlT) {
	if (_sqlTypMap==null)
	    fillTypMap();
	String[] ts = (String[])_sqlTypMap.get(new Integer(sqlT));
	return ts;
    }

  /**Do same thing that mapToSQLType plus parses scale specification.
   *@param sqlT JDBC type name with optional scale specification (e.g.
   *"NUMBER.2")
   *@return 2 - dimentional array. First element of an row holds JDBC
   *type code and the second one scale value.
   */
  public static int[] parseDT(String sqlT) {
    String sT;
    int j = sqlT.indexOf('.');
    int sc=0;
    if (j > 0) {
      sT = sqlT.substring(0,j);
      try {
	sc = Integer.parseInt(sqlT.substring(j+1));
      } catch (NumberFormatException ex) {}
    } else
      sT = sqlT;
    j = mapToSQLType(sT);
    return new int[] {j,sc};
  }

    private static int nextSQLTyp(int n) {
	switch(n) {
	case(0):  return Types.TINYINT;
	case(1):  return Types.SMALLINT;
	case(2):  return Types.INTEGER;
	case(3):  return Types.BIGINT;
	case(4):  return Types.REAL;
	case(5):  return Types.FLOAT;
	case(6):  return Types.DOUBLE;
	case(7):  return Types.DECIMAL;
	case(8):  return Types.NUMERIC;
	case(9):  return Types.BIT;
	case(10):  return Types.CHAR;
	case(11):  return Types.VARCHAR;
	case(12):  return Types.LONGVARCHAR;
	case(13):  return Types.BINARY;
	case(14):  return Types.VARBINARY;
	case(15):  return Types.LONGVARBINARY;
	case(16):  return Types.DATE;
	case(17):  return Types.TIME;
	case(18):  return Types.TIMESTAMP;
	}
	return Integer.MIN_VALUE;
    }

    private static void fillTypMap() {
	_sqlTypMap = new Hashtable(19);
	for (int i = 0;i<19;i++)
	    _sqlTypMap.put( new Integer(nextSQLTyp(i)),nextTypStrings(i));
    }


    private static String[] nextTypStrings(int n) {
	switch(n) {
	case(0):  return new String[] {"byte","Byte","TINYINT"};
	case(1):  return new String[] {"short","Short","SMALLINT"};
	case(2):  return new String[] {"int","Integer","INTEGER"};
	case(3):  return new String[] {"long","Long","BIGINT"};
	case(4):  
	case(5):  return new String[] {"float","Float","FLOAT"};
	case(6):  return new String[] {"double","Double","DOUBLE"};
	case(7):
	case(8):  return new String[] {"BigDecimal","BigDecimal","DECIMAL"};
	case(9):  return new String[] {"boolean","Boolean","BIT"};
	case(10): return new String[] {"String","String","CHAR"};
	case(11):  return new String[] {"String","String","VARCHAR"};
	case(12): return new String[] {"AsciiStream","DAIOStream","LONGVARCHAR"};
	case(13): return new String[] {"byte[]","byte[]","BINARY"};
	case(14):  return new String[] {"byte[]","byte[]","VARBINARY"};
	case(15):  return new String[] {"BinaryStream","DAIOStream","LONGVARBINARY"};
	case(16):  return new String[] {"Date","Date","DATE"};
	case(17):  return new String[] {"Time","Time","TIME"};
	case(18):  return new String[] {"Timestamp","Timestamp","TIMESTAMP"};
	}
	return null;
    }

    private static  void fillStrToSQLTypeMap() {
	_strToSQLTypeMap = new Hashtable(19);
	for (int i=0;i<19;i++) {
	    switch(i) {
	    case(0): _strToSQLTypeMap.put("TINYINT",new Integer(nextSQLTyp(i)));
		break;
	    case(1): _strToSQLTypeMap.put("SMALLINT",new Integer(nextSQLTyp(i)));
		break;
	    case(2): _strToSQLTypeMap.put("INTEGER",new Integer(nextSQLTyp(i)));
		break;
	    case(3): _strToSQLTypeMap.put("BIGINT",new Integer(nextSQLTyp(i)));
		break;
	    case(4): _strToSQLTypeMap.put("REAL",new Integer(nextSQLTyp(i)));
		break;
	    case(5): _strToSQLTypeMap.put("FLOAT",new Integer(nextSQLTyp(i)));
		break;
	    case(6): _strToSQLTypeMap.put("DOUBLE",new Integer(nextSQLTyp(i)));
		break;
	    case(7): _strToSQLTypeMap.put("DECIMAL",new Integer(nextSQLTyp(i)));
		break;
	    case(8): _strToSQLTypeMap.put("NUMERIC",new Integer(nextSQLTyp(i)));
		break;
	    case(9): _strToSQLTypeMap.put("BIT",new Integer(nextSQLTyp(i)));
		break;
	    case(10): _strToSQLTypeMap.put("CHAR",new Integer(nextSQLTyp(i)));
		break;
	    case(11): _strToSQLTypeMap.put("VARCHAR",new Integer(nextSQLTyp(i)));
		break;
	    case(12): _strToSQLTypeMap.put("LONGVARCHAR",new Integer(nextSQLTyp(i)));
		break;
	    case(13): _strToSQLTypeMap.put("BINARY",new Integer(nextSQLTyp(i)));
		break;
	    case(14): _strToSQLTypeMap.put("VARBINARY",new Integer(nextSQLTyp(i)));
		break;
	    case(15): _strToSQLTypeMap.put("LONGVARBINARY",new Integer(nextSQLTyp(i)));
		break;
	    case(16): _strToSQLTypeMap.put("DATE",new Integer(nextSQLTyp(i)));
		break;
	    case(17): _strToSQLTypeMap.put("TIME",new Integer(nextSQLTyp(i)));
		break;
	    case(18): _strToSQLTypeMap.put("TIMESTAMP",new Integer(nextSQLTyp(i)));
	    }
	}
    }


}
