/*  
 * Copyright (c) 2002-2003 MIIK Ltd. All rights reserved.  
 *  
 * Use is subject to license terms.  
 *   
 * The complete licence text can be found at   
 * http://www.jniwrapper.com/license.jsp?prod=winpack  
 */
package com.jniwrapper.win32.gdi;

import com.jniwrapper.*;
import com.jniwrapper.util.EnumItem;
import com.jniwrapper.win32.FunctionName;
import com.jniwrapper.win32.Handle;
import com.jniwrapper.win32.LastErrorException;
import com.jniwrapper.win32.ui.User32;

/**
 * Class DC
 *
 * @author Serge Piletsky
 */
public class DC extends Handle
{
    static final String FUNCTION_CREATE_COMPATIBLE_DC = "CreateCompatibleDC";
    static final String FUNCTION_SELECT_OBJECT = "SelectObject";
    static final String FUNCTION_BIT_BLT = "BitBlt";
    static final String FUNCTION_STRETCH_BLT = "StretchBlt";
    static final String FUNCTION_PAT_BLT = "PatBlt";
    static final String FUNCTION_DELETE_DC = "DeleteDC";
    static final String FUNCTION_SET_DC_BRUSH_COLOR = "SetDCBrushColor";
    static final String FUNCTION_SET_DC_PEN_COLOR = "SetDCPenColor";
    static final FunctionName FUNCTION_TEXT_OUT = new FunctionName("TextOut");
    static final String FUNCTION_DRAW_ICON_EX = "DrawIconEx";
    static final String FUNCTION_DRAW_ICON = "DrawIcon";
    static final String FUNCTION_SET_BACKGROUND_MODE = "SetBkMode";
    static final String FUNCTION_SET_TEXT_COLOR = "SetTextColor";

    static final String FUNCTION_SELECT_CLIP_RGN = "SelectClipRgn";
    static final String FUNCTION_FILL_REGION = "FillRgn";
    static final String FUNCTION_FRAME_RGN = "FrameRgn";
    static final String FUNCTION_GET_POLY_FILL_MODE = "GetPolyFillMode";
    static final String FUNCTION_INVERT_RGN = "InvertRgn";
    static final String FUNCTION_PAINT_RGN = "PaintRgn";
    static final String FUNCTION_SET_POLY_FILL_MODE = "SetPolyFillMode";
    static final String FUNCTION_GET_PIXEL = "GetPixel";

    public DC()
    {
        super();
    }

    public DC(long value)
    {
        super(value);
    }

    public static DC createCompatibleDC(DC dc)
    {
        Function function = Gdi32.getInstance().getFunction(FUNCTION_CREATE_COMPATIBLE_DC);
        DC result = new DC();
        function.invoke(result, dc == null ? (Parameter)new Pointer(null, true) : dc);
        return result;
    }

    /**
     * @return Result of execution of <code>SelectObject</code> API function.
     */
    private long selectObject(GdiObject object)
    {
        Function function = Gdi32.get(FUNCTION_SELECT_OBJECT);
        Handle result = new Handle();
        function.invoke(result, this, object);
        return result.getValue();
    }

    public Brush selectObject(Brush brush)
    {
        final long handle = selectObject((GdiObject)brush);
        final Brush result = new Brush(handle);
        return result;
    }

    public Bitmap selectObject(Bitmap bitmap)
    {
        final long handle = selectObject((GdiObject)bitmap);
        final Bitmap result = new Bitmap(handle);
        return result;
    }

    /**
     * Selects the passed region on the device context.
     * @param region the region to select
     * @return region
     */
    public Region selectObject(Region region)
    {
        final long handle = selectObject((GdiObject)region);
        final Region result = new Region(handle);
        return result;
    }

    /**
     * Performs a bit-block transfer of the color data corresponding to a
     * rectangle of pixels from the specified source device context into a
     * destination device context.
     * All coordinate parameters are in logical units.
     * <br>
     * <b>NOTE:</b> <code>bitBlt</code> only does clipping on the destination DC.<br>
     *
     * @param xDest the x-coordinate of the upper-left corner of the destination rectangle.
     * @param yDest the y-coordinate of the upper-left corner of the destination rectangle.
     * @param width the width of the source and destination rectangles.
     * @param height the height of the source and the destination rectangles.
     * @param xSrc the x-coordinate of the upper-left corner of the source rectangle.
     * @param ySrc the y-coordinate of the upper-left corner of the source rectangle.
     * @param rop a raster-operation code.
     */
    public static void bitBlt(DC hdcDest,
                              int xDest,
                              int yDest,
                              int width,
                              int height,
                              DC hdcSrc,
                              int xSrc,
                              int ySrc,
                              RasterOperation rop)
    {
        Function function = Gdi32.get(FUNCTION_BIT_BLT);
        function.invoke(null, new Parameter[]
        {
            hdcDest,
            new Int(xDest),
            new Int(yDest),
            new Int(width),
            new Int(height),
            hdcSrc,
            new Int(xSrc),
            new Int(ySrc),
            new UInt32(rop.getValue())
        });
    }

    /**
     * Copies a bitmap from a source rectangle into a destination rectangle,
     * stretching or shrinking to fit the dimensions of the destination rectangle,
     * if necessary.
     * The image transformation is determined by the stretching mode currently
     * set in the device context.
     * All coordinate parameters are in logical units.
     *
     * @param hDCDest destination device context.
     * @param xOriginDest the x-coordinate of the upper-left corner of the destination
     * rectangle.
     * @param yOriginDest the y-coordinate of the upper-left corner of the destination
     * rectangle.
     * @param widthDest the width of the destination rectangle.
     * @param heightDest the height of the destination rectangle.
     * @param hDCSrc source device context.
     * @param xOriginSrc the x-coordinate of the upper-left corner of the source rectangle.
     * @param yOriginSrc the y-coordinate of the upper-left corner of the source rectangle.
     * @param widthSrc specifies the width of the source rectangle.
     * @param heightSrc specifies the height of the source rectangle.
     * @param rasterOp specifies the raster operation to be performed. Raster operation codes define how the system
     * combines colors in output operations that involve a brush, a source bitmap, and a destination bitmap.
     * @return nonzero if the method succeeds, zero otherwise.
     */
    public static boolean stretchBlt(DC hDCDest,
                                     int xOriginDest,
                                     int yOriginDest,
                                     int widthDest,
                                     int heightDest,
                                     DC hDCSrc,
                                     int xOriginSrc,
                                     int yOriginSrc,
                                     int widthSrc,
                                     int heightSrc,
                                     RasterOperation rasterOp)
    {
        Int result = new Int();
        Function function = Gdi32.get(FUNCTION_STRETCH_BLT);
        function.invoke(result, new Parameter[]
        {
            hDCDest,
            new Int(xOriginDest),
            new Int(yOriginDest),
            new Int(widthDest),
            new Int(heightDest),
            hDCSrc,
            new Int(xOriginSrc),
            new Int(yOriginSrc),
            new Int(widthSrc),
            new Int(heightSrc),
            new UInt32(rasterOp.getValue())
        });
        if (result.getValue() > 0)
        {
            return true;
        }
        return false;
    }

    /**
     * This method paints the specified rectangle using the brush that is currently selected into the specified device
     * context. The brush color and the surface color or colors are combined by using the specified raster operation.
     * <br><br>
     * <b>NOTE:</b> The values of the <code>rastOps</code> parameter for this method are a limited subset of the full
     * 256 ternary raster-operation codes; in particular, an operation code that refers to a source rectangle cannot be
     * used.
     * @param hDC an instance of {@link com.jniwrapper.win32.gdi.DC} class representing current device context.
     * @param xLeft specifies the x-coordinate, in logical units, of the upper-left corner of the rectangle to be filled.
     * @param yLeft specifies the y-coordinate, in logical units, of the upper-left corner of the rectangle to be filled.
     * @param width specifies the width, in logical units, of the rectangle.
     * @param height specifies the height, in logical units, of the rectangle.
     * @param rastOps specifies the raster operation code. This code can be one of the following values:<br>
     * PATCOPY - Copies the specified pattern into the destination bitmap.<br>
     * PATINVERT - Combines the colors of the specified pattern with the colors of the destination rectangle by using
     * the Boolean XOR operator.<br>
     * DSTINVERT - Inverts the destination rectangle.<br>
     * BLACKNESS - Fills the destination rectangle using the color associated with index 0 in the physical palette.
     * (This color is black for the default physical palette.)<br>
     * WHITENESS - Fills the destination rectangle using the color associated with index 1 in the physical palette.
     * (This color is white for the default physical palette.)
     * @return If the method succeeds, the return value is nonzero. If the method fails, the return value is zero.
     */
    public static boolean patBlt(DC hDC,
                                 int xLeft,
                                 int yLeft,
                                 int width,
                                 int height,
                                 long rastOps)
    {
        Int result = new Int();
        Function function = Gdi32.get(FUNCTION_PAT_BLT);
        function.invoke(result,
                new Parameter[]
                {
                    hDC,
                    new Int(xLeft),
                    new Int(yLeft),
                    new Int(width),
                    new Int(height),
                    new UInt32(rastOps)
                });
        if (result.getValue() > 0)
        {
            return true;
        }
        return false;
    }

    public void release()
    {
        deleteDC(this);
    }

    public static void deleteDC(DC dc)
    {
        Function function = Gdi32.get(FUNCTION_DELETE_DC);
        Int result = new Int();
        function.invoke(result, dc);
        if (result.getValue() == 0)
        {
            throw new LastErrorException();
        }
    }

    /**
     * Draws the text at the given coordinates.
     *
     * @param x the x-coordinate of the top-left corner.
     * @param y the y-coordinate of the top-left corner.
     * @param text text to display.
     */
    public void textOut(int x, int y, String text)
    {
        final Function function = Gdi32.get(FUNCTION_TEXT_OUT.toString());
        UShortInt returnValue = new UShortInt();
        function.invoke(returnValue, new Parameter[]
        {
            this,
            new Int(x),
            new Int(y),
            Gdi32.getInstance().stringParam(text),
            new Int(text.length())
        });
        if (returnValue.getValue() == 0)
        {
            throw new LastErrorException();
        }
    }

    public void drawIconEx(int x,
                           int y,
                           Icon icon,
                           int width,
                           int heght,
                           int frameIndex,
                           Handle bgBrush,
                           int flags)
    {
        final Function function = User32.getInstance().getFunction(FUNCTION_DRAW_ICON_EX);
        UShortInt returnValue = new UShortInt();
        function.invoke(returnValue, new Parameter[]
        {
            this,
            new Int(x),
            new Int(y),
            icon,
            new Int(width),
            new Int(heght),
            new UInt(frameIndex),
            bgBrush,
            new UInt(flags)
        });
        if (returnValue.getValue() == 0)
        {
            throw new LastErrorException();
        }
    }

    public boolean drawIcon(int x, int y, Icon icon)
    {
        final Function function = User32.getInstance().getFunction(FUNCTION_DRAW_ICON);
        Bool result = new Bool();
        function.invoke(result, this, new Int(x), new Int(y), icon);
        return result.getValue();
    }

    public void setBkMode(int bgMode)
    {
        final Function function = Gdi32.get(FUNCTION_SET_BACKGROUND_MODE);
        UShortInt returnValue = new UShortInt();
        function.invoke(returnValue, this, new Int(bgMode));
        if (returnValue.getValue() == 0)
        {
            throw new LastErrorException();
        }
    }

    public void setTextColor(ColorRef color)
    {
        ColorRef returnValue = new ColorRef();
        final Function function = Gdi32.get(FUNCTION_SET_TEXT_COLOR);
        function.invoke(returnValue, this, color);
        if (returnValue.isInvalid())
        {
            throw new LastErrorException("Error while changing text color");
        }
    }

    /**
     * Selects the given region as the current clipping region.
     *
     * @param region region to be selected
     */
    public void selectClipRgn(Region region)
    {
        Int resultRgnType = new Int();
        final Function function = Gdi32.getInstance().getFunction(FUNCTION_SELECT_CLIP_RGN);
        function.invoke(resultRgnType, this, region);
        if (Region.RGN_ERROR == resultRgnType.getValue())
        {
            throw new LastErrorException("Failed to select region");
        }
    }

    /**
     * Fills a region by using the specified brush.
     *
     * @param region region to fill.
     * @param brush brush to be used for painting.
     */
    public void fillRegion(Region region, Brush brush)
    {
        Bool result = new Bool();
        final Function function = Gdi32.getInstance().getFunction(FUNCTION_FILL_REGION);
        function.invoke(result, this, region, brush);
        if (!result.getValue())
        {
            throw new LastErrorException("Failed to fill region");
        }
    }

    /**
     * Draws a border around the specified region by using the specified brush.
     *
     * @param region
     * @param brush
     * @param width
     * @param heigh
     */
    public void frameRegion(Region region, Brush brush, int width, int heigh)
    {
        Bool result = new Bool();
        final Function function = Gdi32.getInstance().getFunction(FUNCTION_FRAME_RGN);
        function.invoke(result, new Parameter[]
        {
            this,
            region,
            brush,
            new Int(width),
            new Int(heigh)
        });
        if (!result.getValue())
        {
            throw new LastErrorException("Failed to frame region");
        }
    }

    /**
     * Retrieves the current polygon fill mode.
     *
     * @return polygon fill mode.
     */
    public int getPolyFillMode()
    {
        Int result = new Int();
        final Function function = Gdi32.getInstance().getFunction(FUNCTION_GET_POLY_FILL_MODE);
        function.invoke(result, this);
        return (int)result.getValue();
    }

    /**
     * Inverts the colors in the specified region.
     *
     * @param region
     * @return If the function succeeds, the return value is true; otherwise false
     */
    public boolean invertRegion(Region region)
    {
        Bool result = new Bool();
        final Function function = Gdi32.getInstance().getFunction(FUNCTION_INVERT_RGN);
        function.invoke(result, this, region);
        return result.getValue();
    }

    /**
     * Paints the specified region by using the brush currently selected into the device context.
     * @param region
     * @return If the function succeeds, the return value is true; otherwise false
     */
    public boolean paintRegion(Region region)
    {
        Bool result = new Bool();
        final Function function = Gdi32.getInstance().getFunction(FUNCTION_PAINT_RGN);
        function.invoke(result, this, region);
        return result.getValue();
    }

    /**
     * Sets the polygon fill mode for functions that fill polygons.
     *
     * @param polyFillMode
     */
    public void setPolyFillMode(PolyFillMode polyFillMode)
    {
        final Function function = Gdi32.getInstance().getFunction(FUNCTION_SET_POLY_FILL_MODE);
        function.invoke(null, this, new Int(polyFillMode.getValue()));
    }

    public ColorRef getPixel(int x, int y)
    {
        ColorRef result = new ColorRef();
        final Function function = Gdi32.get(FUNCTION_GET_PIXEL);
        function.invoke(result, this, new Int(x), new Int(y));
        return result;
    }

    /**
     * This method sets the current device context (DC) brush color to the specified color value. If the device cannot
     * represent the specified color value, the color is set to the nearest physical color.
     * @param hDC an instance of {@link com.jniwrapper.win32.gdi.DC} class representing current device context.
     * @param clrref an instance of {@link com.jniwrapper.win32.gdi.ColorRef} class representing color for brush.
     * @return If the method succeeds, the return value specifies the previous DC brush color as a
     * <code>ColorRef</code> value. If the method fails, the return value is CLR_INVALID.
     */
    public static ColorRef setDCBrushColor(DC hDC, ColorRef clrref)
    {
        ColorRef result = new ColorRef();
        Function function = Gdi32.get(FUNCTION_SET_DC_BRUSH_COLOR);
        function.invoke(result, hDC, clrref);
        return result;
    }

    /**
     * Method sets the current device context (DC) pen color to the specified color value or the nearest to specified
     * color value in case when device cannot set specified color value.
     *
     * @param hDC instance of {@link com.jniwrapper.win32.gdi.DC} class.
     * @param colorRef instance of {@link com.jniwrapper.win32.gdi.ColorRef} class.
     * @return If the method succeeds, the return value specifies the previous DC pen color as instance of
     * {@link com.jniwrapper.win32.gdi.ColorRef} class.
     */
    public ColorRef setDCPenColor(DC hDC, ColorRef colorRef)
    {
        ColorRef color = new ColorRef();
        Function function = Gdi32.get(FUNCTION_SET_DC_PEN_COLOR);
        function.invoke(color, hDC, colorRef);
        return color;
    }

    public static class RasterOperation extends EnumItem
    {
        public static final RasterOperation SRCCOPY = new RasterOperation(0x00CC0020);
        public static final RasterOperation SRCPAINT = new RasterOperation(0x00EE0086);
        public static final RasterOperation SRCAND = new RasterOperation(0x008800C6);
        public static final RasterOperation SRCINVERT = new RasterOperation(0x00660046);
        public static final RasterOperation SRCERASE = new RasterOperation(0x00440328);
        public static final RasterOperation NOTSRCCOPY = new RasterOperation(0x00330008);
        public static final RasterOperation NOTSRCERASE = new RasterOperation(0x001100A6);
        public static final RasterOperation MERGECOPY = new RasterOperation(0x00C000CA);
        public static final RasterOperation MERGEPAINT = new RasterOperation(0x00BB0226);
        public static final RasterOperation PATCOPY = new RasterOperation(0x00F00021);
        public static final RasterOperation PATPAINT = new RasterOperation(0x00FB0A09);
        public static final RasterOperation PATINVERT = new RasterOperation(0x005A0049);
        public static final RasterOperation DSTINVERT = new RasterOperation(0x00550009);
        public static final RasterOperation BLACKNESS = new RasterOperation(0x00000042);
        public static final RasterOperation WHITENESS = new RasterOperation(0x00FF0062);

        private RasterOperation(int value)
        {
            super(value);
        }
    }
}
