
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;
import org.webmacro.util.java2.*;
import java.util.*;
import java.io.*;
import org.webmacro.util.*;

/**
  * A parsed string is a vector of strings and macros. When parsing,
  * it begins with a quotation mark and extends until a matching 
  * close quotation mark (single or double quotes accepted, the 
  * close quote must match the open quote). 
  * <p>
  * When parsing, you can use the escape character to protect values
  * that might otherwise be interepreted. The escape character is \
  * <p>
  * It usually contains a series of strings and variables. If you put
  * a Macro into it, the Macro.evaluate()/Macro.write() method will 
  * be used to include its contents. If you include a non-Macro, 
  * its Object.toString() method will be used instead.
  * <p>
  * Unix users should note that there is no difference between double 
  * and single quotes with parsed string. It is more like the quoting 
  * used for HTML attributes. Internal variables are alway evaluated.
  * <p>
  * Examples:<pre>
  *
  *    #include 'this is a parsed string with a $variable in it'
  *    #include "use double quotes and you can put don't in it"
  *    #include "use the escape char to write \$10 in a ParsedString"
  *
  * </pre>Here the text inside the quotes is the ParsedString.
  */
public final class ParsedString extends Vector implements Macro 
{

   /**
     * conditionally include debugging statements in the compiled code
     */
   private static final boolean debug = false;

   /**
     * Create a new parsed string
     */
   ParsedString() { }

   /**
     * Return the value of the parsed string, after substituting all 
     * contained variables and removing the quotation marks.
     * @exception InvalidContextException is required data is missing
     */
   public Object evaluate(Object data)
      throws InvalidContextException
   {
      Object o;
      StringBuffer str = new StringBuffer(96); 
      //System.out.println("number of elements " + elementCount);
      for (int i=0; i<elementCount; i++) {
	 o = elementData[i];
	 if (! (o instanceof Macro)) {
	    //System.out.println("not a  macro");
	    str.append(o.toString());
	 } else {    // should only contain Variables and Strings 
            try {
	       str.append(((Variable) o).evaluate(data));
            } catch (ClassCastException e) {
               throw new InvalidContextException(
                     "ParsedString: Expected variable or string, got: " + o);
            }
	 }
      }
      return str.toString(); // never null, we created it above
   }


   /**
     * Write the parsed string out. Performs the same operation as 
     * evaluate(context) but writes it to the stream. Although this is 
     * required by the Macro superclass, we don't expect it to be used much
     * since a parsed string does not really appear in a Block (it appears
     * as the argument to a function or directive.)
     * @exception InvalidContextException is required data is missing
     * @exception IOException if could not write to output stream
     */
   final public void write(Writer out, Object data) 
	throws InvalidContextException, IOException
   {
      out.write(evaluate(data).toString()); // evaluate never returns null
   }

   /**
     * Beginning with the quotation mark, parse everything up until the 
     * close quotation mark. It is a parse error if EOL or EOF happen 
     * before the end of the close quotation mark. The parsed string can 
     * begin with either a single or double quotation mark, and then it 
     * must end with the same mark.
     * @exception ParseException on unrecoverable parse error
     * @exception IOException on failure to read from parseTool
     */
   final public static Object parse(ParseTool in) 
      throws ParseException, IOException
   {
      boolean quoted = false;
      boolean singleQuoted = false;
      ParsedString qString = new ParsedString();
 
      // eat open single or double quote
      singleQuoted = in.parseChar('\'');
      if (singleQuoted) { 
         //System.out.println("single quotes");
         quoted = true;
      } else if (in.parseChar('\"')) {
         //System.out.println("double quotes");
         quoted = true;
      } else {
         //System.out.println("undefined quotes");
         return null; 
      } 

      StringBuffer str = new StringBuffer(96);
      Object child = new Object();
      int tok;

      boolean quoteEnded = false;
      while (!quoteEnded && (in.ttype != in.TT_EOF) && (in.ttype != in.TT_EOL)) {
	 child = null;
         switch(in.ttype) {
            case (in.TT_WORD):
               //System.out.println("word: " + in.sval);
               str.append(in.sval); 
               break;
            case '$': //variable
               //System.out.println("variable");
               child = Variable.parse(in); 
               if (child == null) {
                  Engine.log.warning(
                        "ParsedString: Expected a variable got nothing");
                  str.append("<!--\n expected variable got nothing \n-->");
               }
	       break;
            /* case '\\':  // escape character 
               // ignore the next escape
	       if (in.eat(\\)) {
	          str.append(ttype);  
               // ignore the next single quote
	       } else if (in.eat(\')) {
	          str.append(ttype);  
               // ignore the next double quote
	       } else if (in.eat(\")) {
                  str.append(ttype);
	       // assume \ is just a ordinary char
	       } else {
	          str.append('\\');
	       }
	    */
            case '\'': // end single quote
               if (singleQuoted) {
                  //System.out.println("hit end single quote");
                  quoteEnded = true;
               } else {
                  //System.out.println("did not hit end single");
	          str.append((char)in.ttype);
	       }
	       break;
            case '"': // end single quote
               if (!singleQuoted && quoted) {
                  quoteEnded = true;
               } else {
	          str.append((char)in.ttype);
	       }
               break;	       
            default:
               if (debug) {
                  Engine.log.debug("ParsedString other: /" + (char)in.ttype + "/");
               }
	       // all other types assume to be text
               str.append((char)in.ttype);
               break;	       
         }

	 // add to self only if have child 
         if (child != null) {
            qString.addElement(str.toString());
            qString.addElement(child);
            str.setLength(0);
         } else {
            tok = in.nextToken();
         }
      }
   
      // no variables found in string, Empty strings are not errors
      if ((child == null) && (str.length() > 0)) {
         qString.addElement(str.toString());
      } 

      // only three ways to exit loop: end quote, TT_EOF, TT_EOL
      if (!quoteEnded) {
         if (in.ttype == in.TT_EOF) {
	     throw new ParseException(in, "expected close quote but got EOF"); 
	 } else {
	     throw new ParseException(in, "expected close quote but got EOL"); 
	 }
      }

      return qString;

   }

   /**
     * test harness
     */ 
   public static void main(String args[]) {
      Log.setLevel(Log.DEBUG);
      Log.traceExceptions(true);
      try {
         Map contextA = new HashMap();
         Integer ten = new Integer(10);
         Integer twenty = new Integer(20);
         contextA.put("a", ten);
         contextA.put("b", twenty);
   
         Reader readerA = new StringReader("\"expect no variable at all\"");
         ParseTool parseToolA = new ParseTool("string",readerA);
	 parseToolA.nextToken();
         ParsedString QCA =  (ParsedString) ParsedString.parse(parseToolA);
         System.out.println("expected string: /expect no variable at all/");
         System.out.println("evaluated string: " + QCA.evaluate(contextA));
         System.out.println("");
     
         Reader readerB = new StringReader("\'single \"quote no variable\'");
         ParseTool parseToolB = new ParseTool("string",readerB);
	 parseToolB.nextToken();
         ParsedString QCB =  (ParsedString) ParsedString.parse(parseToolB);
         System.out.println("expected string: /single \"quote no variable/");
         System.out.println("evaluated string: " + QCB.evaluate(contextA));
         contextA.put(QCB.toString(),QCB);
         System.out.println("");

         Reader readerD = new StringReader("\"$a ParsedString at beginning\"");
         ParseTool parseToolD = new ParseTool("string",readerD);
	 parseToolD.nextToken();
         ParsedString QCD =  (ParsedString) ParsedString.parse(parseToolD);
         System.out.println("expected string: /10 ParsedString at beginning/");
         System.out.println("evaluated string: " + QCD.evaluate(contextA));
         System.out.println("");

         //Reader readerE = new StringReader("\'two variables $b at end $a\'");
         Reader readerE = new StringReader("\'\'");
         ParseTool parseToolE = new ParseTool("string",readerE);
	 parseToolE.nextToken();
         ParsedString QCE =  (ParsedString) ParsedString.parse(parseToolE);
         //System.out.println("expected string: /two variables 20 at end 10/");
         System.out.println("expected string: //");
         System.out.println("evaluated string: " + QCE.evaluate(contextA));
         System.out.println("");

         Reader readerC = new StringReader("\'unfinished ParsedString at end $\'");
         ParseTool parseToolC = new ParseTool("string",readerC);
	 parseToolC.nextToken();
         ParsedString QCC =  (ParsedString) ParsedString.parse(parseToolC);
         System.out.println("expected string: /unfinished ParseString at end <!--\n expected variable got nothing \n-->/");
         System.out.println("evaluated string: " + QCC.evaluate(contextA));
         contextA.put(QCC.toString(),QCC);
         System.out.println("");

      } catch (Exception e){
         System.out.println("MAIN CAUGHT AN EXCEPTION");
         e.printStackTrace();
      }

   }
}
