
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.engine;
import org.webmacro.util.java2.*;
import java.util.*;
import java.io.*;
import org.webmacro.util.*;

/**
  * QuotedText is a parse method that returns a string enclosed in 
  * singe or double quotation marks
  */
final class QuotedText 
{

   /**
     * Beginning with the quotation mark, include everything up until the 
     * close quotation mark or end of file if EOF occurred before the 
     * close quote. All '$' must be escaped otherwise a parse 
     * error is thrown. This is to clearly distinguish the behaviour of
     * a QuotedText and a ParsedString 
     *
     * @exception ParseException on unrecoverable parse error
     * @exception IOException on failure to read from parseTool
     */
   final public static String parseQuotedText(ParseTool in) 
      throws ParseException, IOException
   {
      boolean quoted = false;
      boolean singleQuoted = false;
 
      // eat open single or double quote
      singleQuoted = in.parseChar('\'');
      if (singleQuoted) { 
         //System.out.println("single quotes");
         quoted = true;
      } else if (in.parseChar('\"')) {
         //System.out.println("double quotes");
         quoted = true;
      } else {
         //System.out.println("undefined quotes");
         return null; 
      } 

      StringBuffer str = new StringBuffer(96);

      boolean quoteEnded = false;
      while (!quoteEnded && (in.ttype != in.TT_EOF) && (in.ttype != in.TT_EOL)) {
         switch(in.ttype) {
            case '$': // '$' not allowed in quotedString must be escaped 
	        throw new ParseException(in, "'$' are not allowed to appear " +
                   "in current quoted String. This is exceptional behaviour " + 
                   "for quoted strings. Parse rules expect a literal string " +
                   "in current position."); 
            case '\'': // end single quote
               if (singleQuoted) {
                  //System.out.println("hit end single quote");
                  quoteEnded = true;
               } else {
                  //System.out.println("did not hit end single");
	          str.append((char)in.ttype);
	       }
	       break;
            case '"': // end single quote
               if (!singleQuoted && quoted) {
                  quoteEnded = true;
               } else {
	          str.append((char)in.ttype);
	       }
               break;	       
	    case in.TT_WORD:
               str.append(in.sval);
	       break;
            default:
               if (Engine.debug) {
                  Engine.log.debug("other: /" + (char)in.ttype + "/");
               }
	       // all other types assume to be text
               str.append((char)in.ttype);
               break;	       
         }
         in.nextToken();
      }
   
      // no variables found in string, Empty strings are not errors
      if (str.length() == 0) {
         if (Engine.debug) {
            Engine.log.debug("empty QuotedText");
         }
      } 

      // only three ways to exit loop: end quote, TT_EOF, TT_EOL
      if (!quoteEnded) {
         if (in.ttype == in.TT_EOF) {
	     throw new ParseException(in, "expected close quote but got EOF"); 
	 } else {
	     throw new ParseException(in, "expected close quote but got EOL"); 
	 }
      }

      return str.toString();
   }

   /**
     * Test harness
     */
   public static void main(String args[]) {

      Log.setLevel(Log.DEBUG);
      Log.traceExceptions(true);
      try {
         Map contextA = new HashMap();
         Integer ten = new Integer(10);
         Integer twenty = new Integer(20);
         contextA.put("a", ten);
         contextA.put("b", twenty);
   
         Reader readerC = new StringReader(
            "\'unfinished QuotedText at end $ r\'");
         ParseTool parseToolC = new ParseTool("string",readerC);
	 parseToolC.nextToken();
         String QCC =  QuotedText.parseQuotedText(parseToolC);
         System.out.println(
            "expected string: /unfinished QuotedText at end $ r/"); 
         System.out.println("evaluated string: " + QCC);
         contextA.put(QCC.toString(),QCC);
         System.out.println("");


         Reader readerA = new StringReader("\"expect no variable at all\"");
         ParseTool parseToolA = new ParseTool("string",readerA);
	 parseToolA.nextToken();
         String QCA =  QuotedText.parseQuotedText(parseToolA);
         System.out.println("expected string: /expect no variable at all/");
         System.out.println("evaluated string: " + QCA);
         System.out.println("");
     
         Reader readerB = new StringReader("\'single \"quote no variable\'");
         ParseTool parseToolB = new ParseTool("string",readerB);
	 parseToolB.nextToken();
         String QCB =   QuotedText.parseQuotedText(parseToolB);
         System.out.println("expected string: /single \"quote no variable/");
         System.out.println("evaluated string: " + QCB);
         contextA.put(QCB.toString(),QCB);
         System.out.println("");

         Reader readerD = new StringReader("\"$a QuotedText at beginning\"");
         ParseTool parseToolD = new ParseTool("string",readerD);
	 parseToolD.nextToken();
         String QCD =  QuotedText.parseQuotedText(parseToolD);
         System.out.println("expected string: /$a QuotedText at beginning/");
         System.out.println("evaluated string: " + QCD);
         System.out.println("");

         //Reader readerE = new StringReader("\'two variables $b at end $a\'");
         Reader readerE = new StringReader("\'\'");

         ParseTool parseToolE = new ParseTool("string",readerE);
	 parseToolE.nextToken();
         String QCE =  QuotedText.parseQuotedText(parseToolE);
         System.out.println("expected string: //");
         //System.out.println("expected string: /two variables $b at end $a/");
         System.out.println("evaluated string: " + QCE);
         System.out.println("");

      } catch (Exception e){
         System.out.println("MAIN CAUGHT AN EXCEPTION");
         e.printStackTrace();
      }

   }
}
