
/*
 * Copyright (c) 1998, 1999 Semiotek Inc. All Rights Reserved.
 *
 * This software is the confidential intellectual property of
 * of Semiotek Inc.; it is copyrighted and licensed, not sold.
 * You may use it under the terms of the GNU General Public License,
 * version 2, as published by the Free Software Foundation. If you 
 * do not want to use the GPL, you may still use the software after
 * purchasing a proprietary developers license from Semiotek Inc.
 *
 * This software is provided "as is", with NO WARRANTY, not even the 
 * implied warranties of fitness to purpose, or merchantability. You
 * assume all risks and liabilities associated with its use.
 *
 * See the attached License.html file for details, or contact us
 * by e-mail at info@semiotek.com to get a copy.
 */


package org.webmacro.resource;

import java.util.*;
import java.io.*;
import org.webmacro.broker.*;
import org.webmacro.util.*;
import java.security.MessageDigest;

/**
  * This is the reference implementation of a "user" ResourceProvider. It
  * operates under a ResourceBroker and doles out objects of type "user" 
  * by returning resource objects which implement the User interface. 
  * <p>
  * You should implement your own UserProvider using a database or other
  * source, as this one is not particularly good. It's provided as an 
  * example so that you can write one that is compatible with code that
  * may want to use it.
  * <p>
  * It is recommended that implementations of the "user" resource be
  * compatible (from a ResourceProvider standpoint) with this one. That
  * way WebMacro developers can write applications using the "user" resource
  * type and be independent of the actual mechanism used to save and 
  * retreive User objects.
  * <p>
  * This implementation is fairly limited. It simply  maintains a 
  * hashtable of users, which it serializes out to disk on shutdown, and 
  * reads on startup. This has obvious performance problems if you intend 
  * to have a large number of users. It does, however, store passwords
  * as a cryptographically strong one-way hash so that they are not 
  * accessible to people who manage to get a copy of the underlying data. 
  * It is recommended that all implementation of the User interface 
  * follow this security practice as well (many people naively use the same 
  * password everywhere, so even if your servlet is not security sensitive
  * you may be exposing your users up to undue risk elsewhere).
  * <p>
  * You could implement a much better UserProvider and use it instead of this 
  * one simply by registering your ResourceProvider as the "user" with the 
  * ResourceBroker rather than this one.
  * <p>
  * For example, you could implement a version of the "user" resource 
  * which retreives and saves data to an LDAP server, SQL database, PSE,
  * or other mechanism.
  * <p>
  * @see User
  * @see ResourceBroker
  * @see ResoruceProvider
  */
public class UserProvider implements ResourceProvider
{
   /**
     * Constant containing the ResourceBroker and Log type for this class
     */
   final static public String TYPE = "user";

   final static boolean debug_ = false;
   final static Log log_ = new Log(TYPE,"User and User Databsae Events");
   static {
      log_.setLevel(Log.DEBUG);
   }

   private File _userdb;
   private Hashtable _users = null;

   final static private String[] types_ = { TYPE };

   /**
     * Create a new User database that reads and writes its 
     * information from the supplied file.
     * @exception InitException if initialization of provider failed
     */
   public  void init( ResourceBroker broker )
      throws InitException
   {

      String dbFile;
      try {
         dbFile = (String) broker.getValue(Config.TYPE, Config.USER_DB_FILE);
      } catch (Exception e) {
         log_.exception(e);
         throw new InitException("Could not find my config info");
      }

      try {
         _userdb = new File(dbFile);
         if (! _userdb.exists()) {
            _users = new Hashtable();
            save();
         }
      } catch (Exception e) {
         log_.exception(e);
         throw new InitException("Could not read my user file: " + dbFile);
      }
   }


   // RESOURCE PROVIDER INTERFACE


   /**
     * Desired level of concurrency--we are all in memory, so none.
     */
   final public int resourceThreads()
   {
      return 0;
   }

   /**
     * How long to cache things (1 minute)
     */
   final public int resourceExpireTime()
   {
      return 1000 * 60;
   }

   /**
     * Get user
     */
   synchronized final public void resourceRequest(RequestResourceEvent evt)
   {
      if (debug_) {
         log_.debug("request: " + evt);
      }
      try {
         evt.set(get(evt.getName()));
      } catch (Exception e) {
         return;
      }
   }

   /**
     * Create user
     * @exception ResourceUnavailableException to indicate user cannot be found and the search should be aborted
     */
   synchronized final public void resourceCreate(CreateResourceEvent evt)
      throws ResourceUnavailableException
   {
      if (debug_) {
         log_.debug("create: " + evt);
      }
      try {   

         // get the database

         if (_users == null) {
            try {
               load();
            } catch (Exception e) {
               log_.exception(e);
               log_.error("Could not load user database:" + this);
            }
         }

         // see if such a user already exists

         User u = (User) _users.get(evt.getName());
         if (u != null) {
            throw new UserException("User already exists: " + u);
         }        String name = evt.getName();

         // create a user

         String password;
         if (evt.getArgument() != null) {
            password = evt.getArgument().toString();
         } else {
            password = null;
         }
         u = new UserImpl(name,password);
         _users.put(name,u);
         evt.set(u);

      } catch (UserException e) {
         throw new ResourceUnavailableException("User already exists");
      } catch (InvalidArgumentException e) {
         return;
      } catch (ResourceRevokedException e) {
         try {
            remove(evt.getName());
         } catch (Exception e2) {
            // we failed to add...?
         }
      }
   }

   /**
     * Save user
     */
   synchronized final public boolean resourceSave(ResourceEvent evt) 
   {
      if (debug_) {
         log_.debug("save: " + evt);
      }
      try {
         if (_users.get(evt.getName()) != null) {
             _users.put(evt.getName(),evt.getValue());
            return true;
         } else {
            return false;
         }
      } catch (Exception e) {
         return false;
      }
   }

   /**
     * Delete user
     */
   synchronized final public boolean resourceDelete(ResourceEvent evt)
   {
      if (null != remove(evt.getName())) {
         if (debug_) {
            log_.debug("delete: " + evt);
         }
         return true;
      } else {
         return false;
      }
   }

   /**
     * Handles requests of type REMOVE_USER, ADD_USER, and NEW_USER
     */
   synchronized final public String[] getTypes() {
      return types_; 
   }

   /**
     * Shut down means save state
     */
   synchronized final public void destroy()
   {
      if (debug_) {
         log_.debug("shutdown");
      }
      try {
         save();
      } catch (Exception e) {
         log_.exception(e);
         log_.error("Unable to save user database");
      }
   }


   // IMPLEMENTATION

   /**
     * Returns the filename of this user database
     */
   final public String toString() {
      return _userdb.toString();
   }

   /**
     * Find the record for the user with the supplied name. The 
     * resulting user object is NOT authenticated--you need to 
     * use the User.login method in order to authenticate.
     * @IOException if unable to read from users file
     */
   private User get(String name)
   {
      if (_users == null) {
         try {
            load();
         } catch (Exception e) {
            log_.exception(e);
            log_.error("Could not load user database:" + this);
            return null;
         }
      }
      return (User) _users.get(name);
   }

   /**
     * Return an enumeration listing all the users
     * @IOException if unable to read from users file
     */
   private Enumeration elements() 
   {
      try {
         if (_users == null) {
            load();
         }
      } catch (Exception e) {
         log_.exception(e);
         log_.error("Could not load user database:" + this);
      }
      return _users.elements();
   }

   /**
     * Remove the specified user
     * @IOException if unable to read from users file
     */
   private User remove(String name)
   {
      if (_users == null) {
         try {
            load();
            if (debug_) {
               log_.debug("remove() loaded database");
            }
         } catch (Exception e) {
            log_.exception(e);
            log_.error("remove could not load user database:" + this);
         }
      }
      User ret = (User) _users.remove(name);
      try {
         save();
      } catch (Exception e) {
         log_.exception(e);
         log_.error("Could not save user database:" + this);
      }
      return ret;
   }

   /**
     * Save the user database
     * @IOException if unable to read/write users file
     */
   private void save()
   {
      if (_users == null) {
         return; // don't save it unless we loaded it
      }
      File tmpFile = null;
      try {
         tmpFile = new File(_userdb.getName() + "-tmp" + 
               Config.getUniqueString() );
         FileOutputStream fileOut = new FileOutputStream(tmpFile.getName(), false);
         ObjectOutputStream out = new ObjectOutputStream(fileOut);

         synchronized(_users) {
            out.writeObject(_users);
         }
         out.close();
         tmpFile.renameTo(_userdb);
      } catch (IOException e) {
         log_.exception(e); 
         log_.error("Unable to save UserProvider, removing tempfile");
         if (tmpFile != null) {
            tmpFile.delete();
         }
      }
   }

   /**
     * Load the user database
     * @IOException if unable to read from users file
     */
   private void load()
      throws IOException
   {
      FileInputStream fileIn = new FileInputStream(_userdb);
      ObjectInputStream in = new ObjectInputStream(fileIn);
      try {
         Hashtable tmp = (Hashtable) in.readObject();
         _users = tmp;
         if (_users == null) {
            log_.warning("Could not read user database");
         } 
      } catch (Exception e) {
         throw new IOException(fileIn + " has an invalid format.");
      }
      in.close();
   }


   /**
     * Test harness
     */
   final public static void main(String arg[])
   {

      System.out.println("Type commands of the format:");
      System.out.println("   add user password");
      System.out.println("   check user password");
      System.out.println("   remove user");
      System.out.println("   list");

      Log.setLevel(Log.ALL);
      Log.traceExceptions(true);
      try {
         ResourceBroker b = new ResourceBroker();
         b.join(new Config());
         UserProvider db = new UserProvider();

         b.join(db);

         String line;
         StringTokenizer st;
         Vector v;
         BufferedReader in = 
            new BufferedReader(new InputStreamReader(System.in));
         while ((line = in.readLine()) != null)
         {
            st = new StringTokenizer(line);
            v = new Vector();
            while (st.hasMoreTokens()) {
               v.addElement( st.nextToken() );
            }
            String word[] = new String[ v.size() ];
            v.copyInto(word);

            if (word.length < 1) {
               System.out.println("Commands: add, check, remove, list");
               continue;
            }

            try {
               if (word[0].equals("add")) {
                  String userName = word[1];
                  String password = word[2];
                  ResourceEvent re = b.create(TYPE,userName);
                  User u = (User) re.getValue();
                  u.setPassword(password);
                  System.out.println(userName + " added");
               } else if (word[0].equals("check")) {
                  String userName = word[1];
                  String password = word[2];
                  User u = (User) b.getValue(TYPE,userName);
                  System.out.println(userName + " authenticated: " +
                        u.authenticate(password));
               } else if (word[0].equals("remove")) {
                  ResourceMap map = b.get(TYPE);
                  String userName = word[1];
                  map.remove(userName);
                  System.out.println(userName + " removed");
               } else if (word[0].equals("list")) {
                  Enumeration users = db.elements();
                  while (users.hasMoreElements()) {
                     System.out.println(users.nextElement());
                  }
               } else if (word[0].equals("show")) {
                  String userName = word[1];
                  User u = (User) b.getValue(TYPE,userName);
                  System.out.println(u);
               } else {
                  System.out.println("I don't understand " + word[0]);
               }
            } catch(Exception e) {
               e.printStackTrace();
            }
            Thread.sleep(500);
         }

         System.out.println("Shutting down....");
	 b.shutdown();
         System.out.println("--Done--");
      } catch (Exception e) {
         e.printStackTrace();
      }
   }

}


/**
  * Represents a web user. You do not normally create user objects 
  * directly, instead you get them from a UserProvider. A user is a 
  * place to store objects related to a particular user, so it has a 
  * dictionary like API. Since the UserProvider must persist the user 
  * object somehow, User impelements Serializable.
  */
class UserImpl implements User, java.io.Serializable
{

   private String myName;
   private String myDigestPwd;
   private Hashtable myMap;

   static final boolean debug_ = false;
   static final Log log_ = new Log(UserProvider.TYPE,"user information");
  
   /**
     * Every user has a name and a password. Note that the password
     * will not be stored in plain text, and is not recoverable.
     * Future implementations might use something better than 
     * passwords to authenticate users, but for now this will 
     * have to do. You can pass a null password in, which will
     * create a user that cannot be authenticated--use setPassword
     * to enable login by providing a non-null password.
     */
   protected UserImpl(String username, String password)
   {
      if (debug_) {
         log_.debug("new User(" + username + ", PASSWORD)");
      }
      myName = username;
      setPassword(password);
   }

   /**
     * Set this users password to the submitted string. Setting a users 
     * password to null prevents them from being authenticated.
     */
   public void setPassword(String password)
   {
      if (password != null) {
         myDigestPwd = crypt(password);
      } else {
         myDigestPwd = null;
      }
      if (debug_) {
         log_.debug("setPassword(PASSWORD) " + myName + " -> " + myDigestPwd);
      }
   }

   /**
     * Get the name of this user
     */
   public String getName() {
      return myName;
   }

   /**
     * Check whether the supplied authorization key authenticates
     * this user or not. In the current implementation it checks 
     * whether a digest of the authKey matches a digest of the 
     * authKey originally used to create the user.
     */
   public boolean authenticate(String authKey)
   {
       return ((myDigestPwd != null) && crypt(authKey).equals(myDigestPwd));
   }

   /**
     * Get an enumeration of the elements 
     */
   public Enumeration elements() {
      return myMap.elements();
   }

   /**
     * Get an object stored in this user
     */
   public Object get(String key)
   {
      return myMap.get(key);
   }

   /**
     * Store an object in the user
     */
   public Object put(String key, Object value)
   {
      return myMap.put(key,value);
   }

   /**
     * Remove an object from the user
     */
   public Object remove(String key)
   {
      return myMap.remove(key);
   }

   /**
     * How many objects are stored in the user?
     */
   public int size() 
   {
      return myMap.size();
   }

   /**
     * Are there any object stored in the user?
     */
   public boolean isEmpty()
   {
      return myMap.isEmpty();
   }

   /**
     * Get an encrypted version of the passphrase
     */
   private String crypt(String plainText)
   {
      try {
         MessageDigest md = MessageDigest.getInstance("SHA");
         return Base64.encode( md.digest( plainText.getBytes() ) );
      } catch (Exception e) {
         UserProvider.log_.error("Encryption services unavailable.");
         return null;
      } 
   }

   /**
     * Return the name of this user
     */
   public String toString() {
      return "User(" + myName + "," + myDigestPwd + ")";
   }

}
