/* cfMatObj.H

  object-oriented interface for OptiVec matrix functions of type "float"

  Copyright (c) 1996-2003 by OptiCode
  All Rights Reserved.
*/

#ifndef CFMATOBJ_H

#include <VecObj.h>
#include <cfVecObj.h>
#include <fMatObj.h>
#include <uVecObj.h>
#include <iVecObj.h>

#define cvta const vector<fComplex>&
#define cmta const matrix<fComplex>&
#define cua  const unsigned&
#define cuint const unsigned
#define cta  const fComplex&

#define vta  vector<fComplex>&
#define mta  matrix<fComplex>&
#define ta   fComplex&

#define vt   vector<fComplex>
#define mt   matrix<fComplex>
#define t    fComplex
#define tVector cfVector
#define tMatrix cfMatrix
#define tMatObj cfMatObj
#define tVecObj cfVecObj

#define tbase  float

#ifdef __BORLANDC__
#pragma warn -inl
#endif

#ifdef _MSC_VER
namespace OptiVec{
#endif

	VECOBJ_NEW_TEMPLATE class matrix<t>
	{
#ifdef _MSC_VER
     public:
		tMatrix  mx_matr;
        tVector  vx_m0;
		unsigned u_ht, u_len;
        ui       ui_size;
#else
		tMatrix  mx_matr;
        tVector  vx_m0;
		unsigned u_ht, u_len;
        ui       ui_size;
		friend class matrix<char>;
		friend class matrix<short>;
		friend class matrix<int>;
		friend class matrix<long>;
		friend class matrix<quad>;
		friend class matrix<unsigned char>;
		friend class matrix<unsigned short>;
		friend class matrix<unsigned int>;
		friend class matrix<unsigned long>;
		friend class matrix<ui>;
		friend class matrix<float>;
		friend class matrix<double>;
		friend class matrix<extended>;
		friend class matrix<fComplex>;
		friend class matrix<dComplex>;
		friend class matrix<eComplex>;
		friend class matrix<fPolar>;
		friend class matrix<dPolar>;
		friend class matrix<ePolar>;
		friend class vector<float>;
		friend class vector<double>;
		friend class vector<extended>;
		friend class vector<fComplex>;
		friend class vector<dComplex>;
		friend class vector<eComplex>;
#endif
	public:
////////////////////////////////////////////////////////////////////////////////////////////////////
//constructors and destructors
//
		matrix(){ mx_matr=0; vx_m0=0; u_ht=0; u_len=0; ui_size=0;}

		matrix(cua ht, cua len)
		{	if((ht > 0) && (len > 0))
			{	mx_matr=MCF_matrix(ht,len); vx_m0=mx_matr[0];
				u_ht=ht; u_len=len; ui_size=((ui)ht)*len;
			}
			else{	mx_matr=0; vx_m0=0; u_ht=0; u_len=0; ui_size=0;}
		}

		matrix(cua ht, cua len, cta fill)
		{	if(!ht || !len) //size == 0
			{	mx_matr=0; vx_m0=0; u_ht=0; u_len=0; ui_size=0;}
			else
			{
				u_ht=ht; u_len=len; ui_size=((ui)ht)*len;
				if(!fill.Re && !fill.Im) //fill == 0.0
                {	mx_matr=MCF_matrix0(ht,len); vx_m0=mx_matr[0];}
				else
				{	mx_matr=MCF_matrix(ht,len);  vx_m0=mx_matr[0];
					VCF_equC(vx_m0,ui_size,fill);}
            }
		}

		matrix(cmta init)
		{	if(init.ui_size)
			{	mx_matr= MCF_matrix(init.u_ht, init.u_len); vx_m0=mx_matr[0];
				u_ht=init.u_ht; u_len=init.u_len; ui_size=init.ui_size;
				VCF_equV(vx_m0, init.vx_m0, ui_size);
			} else
			{ mx_matr=0; vx_m0=0; u_ht=0; u_len=0; ui_size=0;}
		}

		~matrix(){ if(mx_matr) M_free(mx_matr);}


////////////////////////////////////////////////////////////////////////////////////////////////////
//management
//
		void enforceSize()const{ if(!ui_size) throw OptiVec_EmptySizeErr();}
		void enforceSize(unsigned ht, unsigned len)const{
			if(u_len != len || u_ht != ht || !ui_size) throw OptiVec_SizeErr();}

		void enforceLen(unsigned len)const{ if(u_len != len) throw OptiVec_SizeErr();}
		void enforceHt(unsigned ht)const{ if(u_ht != ht) throw OptiVec_SizeErr();}

		void enforcePow2()const{ if(!isipow2(u_len) || !isipow2(u_ht)) throw OptiVec_Pow2SizeErr();}

		void setSize(cua ht, cua len)
		{	ui size;
            size = ((ui)ht)*len;
			if(!size) free(); //can set size to zero
			if(u_len != len || u_ht != ht) //only allocate if different sizes
			{	if(mx_matr) M_free(mx_matr);
				mx_matr= MCF_matrix(ht, len); vx_m0=mx_matr[0];
				u_len=len; u_ht=ht; ui_size= size;
			}
		}

		void setSize(cmta X)
		{	if(!X.ui_size) free(); //can set size to zero
			if(u_len != X.u_len || u_ht != X.u_ht) //only allocate if different sizes
			{	if(mx_matr) M_free(mx_matr);
				mx_matr= MCF_matrix(X.u_ht, X.u_len); vx_m0=mx_matr[0];
				u_len=X.u_len; u_ht=X.u_ht; ui_size= X.ui_size;
			}
		}

		ui getHt()   const { return u_ht; }
		ui getLen()  const { return u_len; }
		ui getSize() const { return ui_size; }
		tMatrix getMatrix() const { return mx_matr; }
        tVector getM0()     const { return vx_m0; }

		void free()
		{	if(mx_matr) M_free(mx_matr);
			mx_matr=0; vx_m0=0; u_ht=0; u_len=0; ui_size=0;
		}

		ta at(unsigned m, unsigned n)
		{
			if(u_ht<=m || u_len<=n) throw OptiVec_SmallSizeErr();
			return mx_matr[m][n];
		}


///////////////////////////////////////////////////////////////////////////////////////////////////
// Overloaded operators
//
		mta operator=(cmta matr)
		{	setSize(matr.u_ht, matr.u_len);
			enforceSize();
			VCF_equV(vx_m0, matr.vx_m0, ui_size);
			return *this;}

		mta operator=(cta scalar)
		{	enforceSize();
			VCF_equC(vx_m0, ui_size, scalar);
			return *this;}

		//this is the only really dangerous operator
		tVector operator[](const int& index){return mx_matr[index];}

		mta operator+=(cmta X)
		{
			enforceSize(X.u_ht, X.u_len);
			VCF_accV(vx_m0, X.vx_m0, ui_size);
			return *this;}

		mta operator+=(cta scalar)
		{	enforceSize();
			VCF_addC(vx_m0, vx_m0, ui_size, scalar);
			return *this;}

		mta operator*=(cta scalar)
		{	enforceSize();
			VCF_mulC(vx_m0,vx_m0,ui_size,scalar);
			return *this;}

		mt operator+(cmta rhs)
		{	enforceSize(rhs.u_ht, rhs.u_len);
			tMatObj tmp(this->u_ht, this->u_len);
			VCF_addV(tmp.vx_m0,this->vx_m0,rhs.vx_m0,tmp.ui_size);
			return tmp;}

		mt operator+(cta C)
		{	tMatObj tmp(this->u_ht, this->u_len);
			tmp.enforceSize();
			VCF_addC(tmp.vx_m0,this->vx_m0,tmp.ui_size,C);
			return tmp;}

		mt operator-(cmta rhs)
		{	enforceSize(rhs.u_ht, rhs.u_len);
			tMatObj tmp(this->u_ht, this->u_len);
			VCF_subV(tmp.vx_m0,this->vx_m0,rhs.vx_m0,tmp.ui_size);
			return tmp;}

		mt operator-(cta C)
		{	tMatObj tmp(this->u_ht, this->u_len);
			tmp.enforceSize();
			VCF_subC(tmp.vx_m0,this->vx_m0,tmp.ui_size,C);
			return tmp;}

		mt operator*(cmta rhs)
		{	enforceLen(rhs.u_ht);
			tMatObj tmp(this->u_ht, rhs.u_len);
			tmp.enforceSize();
			MCFmulM(tmp.vx_m0,this->vx_m0,rhs.vx_m0,this->u_ht, this->u_len,rhs.u_len);
			return tmp;}

		vt operator*(cvta rhs)
		{	enforceLen(rhs.ui_size);
			tVecObj tmp(this->u_ht);
			tmp.enforceSize();
			MCFmulV(tmp.vx_vect,this->vx_m0,rhs.vx_vect,this->u_ht, this->u_len);
			return tmp;}

		// operator*(cvta VX, cmta MY) is member of cfVecObj, see below

		mt operator*(cta C)
		{	tMatObj tmp(this->u_ht, this->u_len);
			tmp.enforceSize();
			VCF_mulC(tmp.vx_m0,this->vx_m0,tmp.ui_size,C);
			return tmp;}

		mt operator/(cta C)
		{	tMatObj tmp(this->u_ht, this->u_len);
			tmp.enforceSize();
			VCF_divC(tmp.vx_m0,this->vx_m0,tmp.ui_size,C);
			return tmp;}

		friend mt operator+(cta C, cmta X)
		{	tMatObj tmp(X.u_ht,X.u_len);
			VCF_addC(tmp.vx_m0,X.vx_m0,tmp.ui_size,C);
			return tmp;}

		friend mt operator-(cta C, cmta X)
		{	tMatObj tmp(X.u_ht,X.u_len);
			VCF_subrC(tmp.vx_m0,X.vx_m0,tmp.ui_size,C);
			return tmp;}

		friend mt operator*(cta C, cmta X)
		{	tMatObj tmp(X.u_ht,X.u_len);
			VCF_mulC(tmp.vx_m0,X.vx_m0,tmp.ui_size,C);
			return tmp;}

/////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////////////////
//OptiVec functions in member format
//from MCXstd.h
//

/////////////////////////////////////////////////////////////////////////////////////////
//Generation
//
		//these are not recommended, better to use constructors
		void _matrix(cuint ht, cuint len) //underscore necessary to avoid confusing with constructor
		{
			free();
			ui_size=((ui)ht)*len;
			if(ui_size){ u_ht=ht;u_len=len;mx_matr= MCF_matrix(ht,len);vx_m0=mx_matr[0];}
		}

		void _matrix0(cuint ht, cuint len) //underscore necessary to avoid confusing with constructor
		{
			free();
			ui_size=((ui)ht)*len;
			if(ui_size){ u_ht=ht;u_len=len;mx_matr= MCF_matrix0(ht,len);vx_m0=mx_matr[0];}
		}


/////////////////////////////////////////////////////////////////////////////////////////
//Addressing single matrix elements
//
		t* Pelement(cuint m, cuint n)
		{
			if(u_ht<=m || u_len<=n) throw OptiVec_SmallSizeErr();
			return MCFPelement(vx_m0,u_ht,u_len,m,n);
		}

		t element(cuint m, cuint n)
		{
			if(u_ht<=m || u_len<=n) throw OptiVec_SmallSizeErr();
			return MCFelement(vx_m0,u_ht,u_len,m,n);
		}


/////////////////////////////////////////////////////////////////////////////////////////
//Initialization
//

		void equ0()
		{
			enforceSize();
			VCF_equ0(vx_m0,ui_size);
		}

		void equ1()
		{
			enforceSize();
			MCFequ1(vx_m0,u_len);
		}

		void equm1()
		{
			enforceSize();
			MCFequm1(vx_m0,u_len);
		}

		long random(const long& seed, cta Min, cta Max)
		{
			enforceSize();
			return VCF_random(vx_m0, ui_size, seed, Min, Max);
		}

		void outerprod(cvta X, cvta Y)
		{
			enforceSize( X.ui_size, Y.ui_size);
			MCFouterprod(vx_m0, X.vx_vect, Y.vx_vect, u_ht, u_len);
		}

		void Row_equ0(cuint iRow)
		{
			if( !u_len || (u_ht <= iRow) )throw OptiVec_SmallSizeErr();
			MCFRow_equ0(vx_m0, u_ht, u_len, iRow);
		}

		void Col_equ0(cuint iCol)
		{
			if( !u_ht || (u_len <= iCol) )throw OptiVec_SmallSizeErr();
			MCFCol_equ0(vx_m0, u_ht, u_len, iCol);
		}

		void Dia_equ0()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SmallSizeErr();
			MCFDia_equ0(vx_m0, u_len);
		}

		void Row_equC(cuint iRow, cta C)
		{
			if( !u_len || (u_ht <= iRow) )throw OptiVec_SmallSizeErr();
			MCFRow_equC(vx_m0, u_ht, u_len, iRow, C);
		}

		void Col_equC(cuint iCol, cta C)
		{
			if( !u_ht || (u_len <= iCol) )throw OptiVec_SmallSizeErr();
			MCFCol_equC(vx_m0, u_ht, u_len, iCol, C);
		}

		void Dia_equC(cta C)
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SmallSizeErr();
			MCFDia_equC(vx_m0, u_len, C);
		}

		void Row_equV(cuint iRow, cvta X)
		{
			if( !u_len || (u_ht <= iRow) )throw OptiVec_SmallSizeErr();
			MCFRow_equV(vx_m0, u_ht, u_len, iRow, X.vx_vect);
		}

		void Col_equV(cuint iCol, cvta X)
		{
			if( !u_ht || (u_len <= iCol) )throw OptiVec_SmallSizeErr();
			MCFCol_equV(vx_m0, u_ht, u_len, iCol, X.vx_vect);
		}

		void Dia_equV(cvta X)
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SmallSizeErr();
			MCFDia_equV(vx_m0, u_len, X.vx_vect);
		}

		void Trd_equM(cmta MTrd)
		{
			if( MTrd.u_ht != 3 || u_ht != u_len || u_len != MTrd.u_len )throw OptiVec_SizeErr();
			MCFTrd_equM(vx_m0, MTrd.vx_m0, u_len);
		}

		void Trd_extract(cmta MA)
		{
			if( u_ht != 3 || MA.u_ht != MA.u_len || MA.u_len != u_len )throw OptiVec_SizeErr();
			MCFTrd_extract(vx_m0, MA.vx_m0, u_len);
		}

		void equM(cmta X)
		{
			enforceSize(X.u_ht, X.u_len);
			MCFequM(vx_m0, X.vx_m0, u_ht, u_len);
		}

		void neg(cmta X)
		{
			enforceSize(X.u_ht, X.u_len);
			MCFneg(vx_m0, X.vx_m0, u_ht, u_len);
		}

		void conj(cmta X)
		{
			enforceSize(X.u_ht, X.u_len);
			MCFconj(vx_m0, X.vx_m0, u_ht, u_len);
		}

		void UequL()
		{
			if( !u_len || u_len != u_ht )throw OptiVec_SizeErr();
			MCFUequL(vx_m0, u_len);
		}

		void LequU()
		{
			if( !u_len || u_len != u_ht )throw OptiVec_SizeErr();
			MCFLequU(vx_m0, u_len);
		}

/////////////////////////////////////////////////////////////////////////////////////////
//Data-type interconversions
//
		//defined by including cdMatObj.h
		void CDtoCF(const cdMatObj& X);

		//defined by including ceMatObj.h
		void CEtoCF(const ceMatObj& X);



/////////////////////////////////////////////////////////////////////////////////////////
//Submatrices
//
		void submatrix(cmta MSrce, cuint firstRowInCol, cuint sampInCol, cuint firstColInRow, cuint sampInRow )
		{
			if( (MSrce.u_ht  <= firstRowInCol+(u_ht-1) *sampInCol) ||
                (MSrce.u_len <= firstColInRow+(u_len-1)*sampInRow) ) throw OptiVec_SizeErr();
			MCFsubmatrix(vx_m0, u_ht, u_len, MSrce.vx_m0, MSrce.u_ht, MSrce.u_len,
                 firstRowInCol, sampInCol, firstColInRow, sampInRow );
		}

		void submatrix_equM(cuint firstRowInCol, cuint sampInCol, cuint firstColInRow, cuint sampInRow, cmta MSrce)
		{
			if( (u_ht  <= firstRowInCol+(MSrce.u_ht-1) *sampInCol) ||
                (u_len <= firstColInRow+(MSrce.u_len-1)*sampInRow) ) throw OptiVec_SizeErr();
			MCFsubmatrix_equM(vx_m0, u_ht, u_len,
                 firstRowInCol, sampInCol, firstColInRow, sampInRow,
                 MSrce.vx_m0, MSrce.u_ht, MSrce.u_len );
		}

		void equMblock(cmta MSrce, cuint firstRowInCol, cuint firstColInRow )
		{
			if( (MSrce.u_ht  < firstRowInCol+u_ht) ||
                (MSrce.u_len < firstColInRow+u_len) ) throw OptiVec_SizeErr();
			MCFequMblock(vx_m0, u_ht, u_len, MSrce.vx_m0, MSrce.u_ht, MSrce.u_len,
                 firstRowInCol, firstColInRow );
		}

		void equMblockT(cmta MSrce, cuint firstRowInCol, cuint firstColInRow )
		{
			if( (MSrce.u_ht  < firstRowInCol+u_len) ||
                (MSrce.u_len < firstColInRow+u_ht) ) throw OptiVec_SizeErr();
			MCFequMblockT(vx_m0, u_ht, u_len, MSrce.vx_m0, MSrce.u_ht, MSrce.u_len,
                 firstRowInCol, firstColInRow );
		}

		void block_equM(cuint firstRowInCol, cuint firstColInRow, cmta MSrce )
		{
			if( (u_ht  < firstRowInCol+MSrce.u_ht) ||
                (u_len < firstColInRow+MSrce.u_len) ) throw OptiVec_SizeErr();
			MCFblock_equM(vx_m0, u_ht, u_len, firstRowInCol, firstColInRow,
                 MSrce.vx_m0, MSrce.u_ht, MSrce.u_len );
		}

		void block_equMT(cuint firstRowInCol, cuint firstColInRow, cmta MSrce )
		{
			if( (u_ht  < firstRowInCol+MSrce.u_len) ||
                (u_len < firstColInRow+MSrce.u_ht) ) throw OptiVec_SizeErr();
			MCFblock_equMT(vx_m0, u_ht, u_len, firstRowInCol, firstColInRow,
                 MSrce.vx_m0, MSrce.u_ht, MSrce.u_len );
		}

/////////////////////////////////////////////////////////////////////////////////////////
// Deleting or inserting a row or a column
//

		void Row_delete(cmta MA, cuint iRow)
		{
            if( MA.u_ht <= iRow ) throw OptiVec_SmallSizeErr();
            if( mx_matr == MA.mx_matr ) // overwrite MA?
			{  	ui_size -= u_len;  // don't reallocate, as row pointers remain intact
				MCFRow_delete(vx_m0, vx_m0, u_ht--, u_len, iRow);
            }
            else
            {	enforceSize(MA.u_ht-1, MA.u_len);
				MCFRow_delete(vx_m0, MA.vx_m0, MA.u_ht, MA.u_len, iRow);
            }
		}

		void Col_delete(cmta MA, cuint iCol)
		{
            if( MA.u_len <= iCol ) throw OptiVec_SmallSizeErr();
            if( mx_matr == MA.mx_matr ) // overwrite MA?
            {
            	tMatrix tmp = MCF_matrix( u_ht, u_len-1 );
                vx_m0 = tmp[0];
				MCFCol_delete(vx_m0, MA.vx_m0, u_ht, u_len--, iCol);
                M_free( mx_matr );
                ui_size -= u_len;
                mx_matr = tmp;
            }
            else
            {
				enforceSize(MA.u_ht, MA.u_len-1);
				MCFCol_delete(vx_m0, MA.vx_m0, u_ht, u_len, iCol);
            }
		}

		void Row_insert(cmta MA, cuint iRow, cvta X)
		{
            if( u_ht <= iRow ) throw OptiVec_SmallSizeErr();
            if( mx_matr == MA.mx_matr ) // overwrite MA?
            {
            	tMatrix tmp = MCF_matrix( u_ht, u_len );
                vx_m0 = tmp[0];
				MCFRow_insert(vx_m0, MA.vx_m0, u_ht, u_len, iRow, X.vx_vect);
                M_free( mx_matr );
                ui_size += u_len;
                mx_matr = tmp;
            }
            else
            {
				enforceSize(MA.u_ht, MA.u_len+1);
				MCFRow_insert(vx_m0, MA.vx_m0, u_ht, u_len, iRow, X.vx_vect);
            }
		}

		void Col_insert(cmta MA, cuint iCol, cvta X)
		{
            if( u_len <= iCol ) throw OptiVec_SmallSizeErr();
            if( mx_matr == MA.mx_matr ) // overwrite MA?
            {
            	tMatrix tmp = MCF_matrix( u_ht, u_len );
                vx_m0 = tmp[0];
				MCFCol_insert(vx_m0, MA.vx_m0, u_ht, u_len, iCol, X.vx_vect);
                M_free( mx_matr );
                ui_size += u_len;
                mx_matr = tmp;
            }
            else
            {
				enforceSize(MA.u_ht, MA.u_len+1);
				MCFCol_insert(vx_m0, MA.vx_m0, u_ht, u_len, iCol, X.vx_vect);
            }
		}

/////////////////////////////////////////////////////////////////////////////////////////
//Basic arithmetic operations performed on one single row, or one single column
// of any matrix, or on the diagonal of a square matrix
//

		void Row_addC(cuint iRow, cta C)
		{
			if( !u_len || (u_ht <= iRow) )throw OptiVec_SmallSizeErr();
			MCFRow_addC(vx_m0, u_ht, u_len, iRow, C);
		}

		void Col_addC(cuint iCol, cta C)
		{
			if( !u_ht || (u_len <= iCol) )throw OptiVec_SmallSizeErr();
			MCFCol_addC(vx_m0, u_ht, u_len, iCol, C);
		}

		void Dia_addC(cuint iCol, cta C)
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SmallSizeErr();
			MCFDia_addC(vx_m0, u_len, C);
		}

		void Row_addV(cuint iRow, cvta X)
		{
			if( !u_len || (u_ht <= iRow) || (X.ui_size < u_len))throw OptiVec_SmallSizeErr();
			MCFRow_addV(vx_m0, u_ht, u_len, iRow, X.vx_vect);
		}

		void Col_addV(cuint iCol, cvta X)
		{
			if( !u_ht || (u_len <= iCol) || (X.ui_size < u_ht) )throw OptiVec_SmallSizeErr();
			MCFCol_addV(vx_m0, u_ht, u_len, iCol, X.vx_vect);
		}

		void Dia_addV(cuint iCol, cvta X)
		{
			if( !u_len || (u_len != u_ht) || (X.ui_size < u_len) )throw OptiVec_SmallSizeErr();
			MCFDia_addV(vx_m0, u_len, X.vx_vect);
		}

		void Row_subC(cuint iRow, cta C)
		{
			if( !u_len || (u_ht <= iRow) )throw OptiVec_SmallSizeErr();
			MCFRow_subC(vx_m0, u_ht, u_len, iRow, C);
		}

		void Col_subC(cuint iCol, cta C)
		{
			if( !u_ht || (u_len <= iCol) )throw OptiVec_SmallSizeErr();
			MCFCol_subC(vx_m0, u_ht, u_len, iCol, C);
		}

		void Dia_subC(cuint iCol, cta C)
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SmallSizeErr();
			MCFDia_subC(vx_m0, u_len, C);
		}

		void Row_subV(cuint iRow, cvta X)
		{
			if( !u_len || (u_ht <= iRow) || (X.ui_size < u_len))throw OptiVec_SmallSizeErr();
			MCFRow_subV(vx_m0, u_ht, u_len, iRow, X.vx_vect);
		}

		void Col_subV(cuint iCol, cvta X)
		{
			if( !u_ht || (u_len <= iCol) || (X.ui_size < u_ht) )throw OptiVec_SmallSizeErr();
			MCFCol_subV(vx_m0, u_ht, u_len, iCol, X.vx_vect);
		}

		void Dia_subV(cuint iCol, cvta X)
		{
			if( !u_len || (u_len != u_ht) || (X.ui_size < u_len) )throw OptiVec_SmallSizeErr();
			MCFDia_subV(vx_m0, u_len, X.vx_vect);
		}

		void Row_subrC(cuint iRow, cta C)
		{
			if( !u_len || (u_ht <= iRow) )throw OptiVec_SmallSizeErr();
			MCFRow_subrC(vx_m0, u_ht, u_len, iRow, C);
		}

		void Col_subrC(cuint iCol, cta C)
		{
			if( !u_ht || (u_len <= iCol) )throw OptiVec_SmallSizeErr();
			MCFCol_subrC(vx_m0, u_ht, u_len, iCol, C);
		}

		void Dia_subrC(cuint iCol, cta C)
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SmallSizeErr();
			MCFDia_subrC(vx_m0, u_len, C);
		}

		void Row_subrV(cuint iRow, cvta X)
		{
			if( !u_len || (u_ht <= iRow) || (X.ui_size < u_len))throw OptiVec_SmallSizeErr();
			MCFRow_subrV(vx_m0, u_ht, u_len, iRow, X.vx_vect);
		}

		void Col_subrV(cuint iCol, cvta X)
		{
			if( !u_ht || (u_len <= iCol) || (X.ui_size < u_ht) )throw OptiVec_SmallSizeErr();
			MCFCol_subrV(vx_m0, u_ht, u_len, iCol, X.vx_vect);
		}

		void Dia_subrV(cuint iCol, cvta X)
		{
			if( !u_len || (u_len != u_ht) || (X.ui_size < u_len) )throw OptiVec_SmallSizeErr();
			MCFDia_subrV(vx_m0, u_len, X.vx_vect);
		}

		void Row_mulC(cuint iRow, cta C)
		{
			if( !u_len || (u_ht <= iRow) )throw OptiVec_SmallSizeErr();
			MCFRow_mulC(vx_m0, u_ht, u_len, iRow, C);
		}

		void Col_mulC(cuint iCol, cta C)
		{
			if( !u_ht || (u_len <= iCol) )throw OptiVec_SmallSizeErr();
			MCFCol_mulC(vx_m0, u_ht, u_len, iCol, C);
		}

		void Dia_mulC(cuint iCol, cta C)
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SmallSizeErr();
			MCFDia_mulC(vx_m0, u_len, C);
		}

		void Row_mulV(cuint iRow, cvta X)
		{
			if( !u_len || (u_ht <= iRow) || (X.ui_size < u_len))throw OptiVec_SmallSizeErr();
			MCFRow_mulV(vx_m0, u_ht, u_len, iRow, X.vx_vect);
		}

		void Col_mulV(cuint iCol, cvta X)
		{
			if( !u_ht || (u_len <= iCol) || (X.ui_size < u_ht) )throw OptiVec_SmallSizeErr();
			MCFCol_mulV(vx_m0, u_ht, u_len, iCol, X.vx_vect);
		}

		void Dia_mulV(cuint iCol, cvta X)
		{
			if( !u_len || (u_len != u_ht) || (X.ui_size < u_len) )throw OptiVec_SmallSizeErr();
			MCFDia_mulV(vx_m0, u_len, X.vx_vect);
		}

		void Row_divC(cuint iRow, cta C)
		{
			if( !u_len || (u_ht <= iRow) )throw OptiVec_SmallSizeErr();
			MCFRow_divC(vx_m0, u_ht, u_len, iRow, C);
		}

		void Col_divC(cuint iCol, cta C)
		{
			if( !u_ht || (u_len <= iCol) )throw OptiVec_SmallSizeErr();
			MCFCol_divC(vx_m0, u_ht, u_len, iCol, C);
		}

		void Dia_divC(cuint iCol, cta C)
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SmallSizeErr();
			MCFDia_divC(vx_m0, u_len, C);
		}

		void Row_divV(cuint iRow, cvta X)
		{
			if( !u_len || (u_ht <= iRow) || (X.ui_size < u_len))throw OptiVec_SmallSizeErr();
			MCFRow_divV(vx_m0, u_ht, u_len, iRow, X.vx_vect);
		}

		void Col_divV(cuint iCol, cvta X)
		{
			if( !u_ht || (u_len <= iCol) || (X.ui_size < u_ht) )throw OptiVec_SmallSizeErr();
			MCFCol_divV(vx_m0, u_ht, u_len, iCol, X.vx_vect);
		}

		void Dia_divV(cuint iCol, cvta X)
		{
			if( !u_len || (u_len != u_ht) || (X.ui_size < u_len) )throw OptiVec_SmallSizeErr();
			MCFDia_divV(vx_m0, u_len, X.vx_vect);
		}

		void Row_divrC(cuint iRow, cta C)
		{
			if( !u_len || (u_ht <= iRow) )throw OptiVec_SmallSizeErr();
			MCFRow_divrC(vx_m0, u_ht, u_len, iRow, C);
		}

		void Col_divrC(cuint iCol, cta C)
		{
			if( !u_ht || (u_len <= iCol) )throw OptiVec_SmallSizeErr();
			MCFCol_divrC(vx_m0, u_ht, u_len, iCol, C);
		}

		void Dia_divrC(cuint iCol, cta C)
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SmallSizeErr();
			MCFDia_divrC(vx_m0, u_len, C);
		}

		void Row_divrV(cuint iRow, cvta X)
		{
			if( !u_len || (u_ht <= iRow) || (X.ui_size < u_len))throw OptiVec_SmallSizeErr();
			MCFRow_divrV(vx_m0, u_ht, u_len, iRow, X.vx_vect);
		}

		void Col_divrV(cuint iCol, cvta X)
		{
			if( !u_ht || (u_len <= iCol) || (X.ui_size < u_ht) )throw OptiVec_SmallSizeErr();
			MCFCol_divrV(vx_m0, u_ht, u_len, iCol, X.vx_vect);
		}

		void Dia_divrV(cuint iCol, cvta X)
		{
			if( !u_len || (u_len != u_ht) || (X.ui_size < u_len) )throw OptiVec_SmallSizeErr();
			MCFDia_divrV(vx_m0, u_len, X.vx_vect);
		}


/////////////////////////////////////////////////////////////////////////////////////////
//One-dimensional vector operations
//performed along all rows or columns, or along the diagonal of a square matrix
//(some of these functions belong to fVecObj.h)

		t Dia_maxReIm()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_maxReIm(vx_m0, u_len);
		}

		t Dia_minReIm()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_minReIm(vx_m0, u_len);
		}

		tbase Dia_absmax()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_absmax(vx_m0, u_len);
		}

		tbase Dia_absmin()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_absmin(vx_m0, u_len);
		}

		t Dia_absmaxReIm()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_absmaxReIm(vx_m0, u_len);
		}

		t Dia_absminReIm()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_absminReIm(vx_m0, u_len);
		}

		t Dia_sabsmax()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_sabsmax(vx_m0, u_len);
		}

		t Dia_sabsmin()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_sabsmin(vx_m0, u_len);
		}

		t Dia_cabsmax()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_cabsmax(vx_m0, u_len);
		}

		t Dia_cabsmin()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_absmin(vx_m0, u_len);
		}

		t Dia_sum()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_sum(vx_m0, u_len);
		}

		t Dia_prod()
		{
			if( !u_len || (u_len != u_ht) )throw OptiVec_SizeErr();
			return MCFDia_prod(vx_m0, u_len);
		}

		void Rows_runsum()
		{
			enforceSize();
			MCFRows_runsum(vx_m0, u_ht, u_len);
		}

		void Cols_runsum()
		{
			enforceSize();
			MCFCols_runsum(vx_m0, u_ht, u_len);
		}

		void Rows_runprod()
		{
			enforceSize();
			MCFRows_runprod(vx_m0, u_ht, u_len);
		}

		void Cols_runprod()
		{
			enforceSize();
			MCFCols_runprod(vx_m0, u_ht, u_len);
		}

		void Rows_rotate(const int pos)
		{
			enforceSize();
			MCFRows_rotate(vx_m0, u_ht, u_len, pos);
		}

		void Cols_rotate(const int pos)
		{
			enforceSize();
			MCFCols_rotate(vx_m0, u_ht, u_len, pos);
		}

		void Rows_reflect()
		{
			enforceSize();
			MCFRows_reflect(vx_m0, u_ht, u_len);
		}

		void Cols_reflect()
		{
			enforceSize();
			MCFCols_reflect(vx_m0, u_ht, u_len);
		}


/////////////////////////////////////////////////////////////////////////////////////////
//Operations involving two rows or two colums of one matrix
//
		void Rows_exchange(cuint row1, cuint row2)
		{
			if( !u_len || (u_ht <= row1) || (u_ht <= row2))throw OptiVec_SizeErr();
			MCFRows_exchange(vx_m0, u_ht, u_len, row1, row2);
		}

		void Cols_exchange(cuint col1, cuint col2)
		{
			if( !u_len || (u_ht <= col1) || (u_ht <= col2))throw OptiVec_SizeErr();
			MCFCols_exchange(vx_m0, u_ht, u_len, col1, col2);
		}

		void Rows_add(cuint destRow, cuint srceRow)
		{
			if( !u_len || (u_ht <= destRow) || (u_ht <= srceRow))throw OptiVec_SizeErr();
			MCFRows_add(vx_m0, u_ht, u_len, destRow, srceRow);
		}

		void Cols_add(cuint destCol, cuint srceCol)
		{
			if( !u_len || (u_ht <= destCol) || (u_ht <= srceCol))throw OptiVec_SizeErr();
			MCFCols_add(vx_m0, u_ht, u_len, destCol, srceCol);
		}

		void Rows_sub(cuint destRow, cuint srceRow)
		{
			if( !u_len || (u_ht <= destRow) || (u_ht <= srceRow))throw OptiVec_SizeErr();
			MCFRows_sub(vx_m0, u_ht, u_len, destRow, srceRow);
		}

		void Cols_sub(cuint destCol, cuint srceCol)
		{
			if( !u_len || (u_ht <= destCol) || (u_ht <= srceCol))throw OptiVec_SizeErr();
			MCFCols_sub(vx_m0, u_ht, u_len, destCol, srceCol);
		}

		void Rows_Cadd(cuint destRow, cuint srceRow, cta C)
		{
			if( !u_len || (u_ht <= destRow) || (u_ht <= srceRow))throw OptiVec_SizeErr();
			MCFRows_Cadd(vx_m0, u_ht, u_len, destRow, srceRow, C);
		}

		void Cols_Cadd(cuint destCol, cuint srceCol, cta C)
		{
			if( !u_len || (u_ht <= destCol) || (u_ht <= srceCol))throw OptiVec_SizeErr();
			MCFCols_Cadd(vx_m0, u_ht, u_len, destCol, srceCol, C);
		}

		void Rows_lincomb(cuint destRow, cta destC, cuint srceRow, cta srceC)
		{
			if( !u_len || (u_ht <= destRow) || (u_ht <= srceRow))throw OptiVec_SizeErr();
			MCFRows_lincomb(vx_m0, u_ht, u_len, destRow, destC, srceRow, srceC);
		}

		void Cols_lincomb(cuint destCol, cta destC, cuint srceCol, cta srceC)
		{
			if( !u_len || (u_ht <= destCol) || (u_ht <= srceCol)) throw OptiVec_SizeErr();
			MCFCols_lincomb(vx_m0, u_ht, u_len, destCol, destC, srceCol, srceC);
		}

/////////////////////////////////////////////////////////////////////////////////////////
//Matrix transposition
//
		void transpose(cmta MA)
		{
			enforceSize(MA.u_len, MA.u_ht);
			MCFtranspose(vx_m0, MA.vx_m0, u_ht, u_len);
		}

		void hermconj(cmta MA)
		{
			enforceSize(MA.u_len, MA.u_ht);
			MCFhermconj(vx_m0, MA.vx_m0, u_ht, u_len);
		}


/////////////////////////////////////////////////////////////////////////////////////////
//Matrix arithmetics
//(The following functions are more efficient than the arithmetic operators defined above)
//
		void addM(cmta MA, cmta MB)
		{
			enforceSize(MA.u_ht, MA.u_len);
			enforceSize(MB.u_ht, MB.u_len);
			MCFaddM(vx_m0, MA.vx_m0, MB.vx_m0, u_ht, u_len);
		}

		void addMT(cmta MA, cmta MB)
		{
			enforceSize(MA.u_ht,  MA.u_len);
			enforceSize(MB.u_len, MB.u_ht);
			MCFaddMT(vx_m0, MA.vx_m0, MB.vx_m0, u_ht, u_len);
		}

		void subM(cmta MA, cmta MB)
		{
			enforceSize(MA.u_ht, MA.u_len);
			enforceSize(MB.u_ht, MB.u_len);
			MCFsubM(vx_m0, MA.vx_m0, MB.vx_m0, u_ht, u_len);
		}

		void subMT(cmta MA, cmta MB)
		{
			enforceSize(MA.u_ht,  MA.u_len);
			enforceSize(MB.u_len, MB.u_ht);
			MCFsubMT(vx_m0, MA.vx_m0, MB.vx_m0, u_ht, u_len);
		}

		void subrM(cmta MA, cmta MB)
		{
			enforceSize(MA.u_ht, MA.u_len);
			enforceSize(MB.u_ht, MB.u_len);
			MCFsubM(vx_m0, MB.vx_m0, MA.vx_m0, u_ht, u_len);
		}

		void subrMT(cmta MA, cmta MB)
		{
			enforceSize(MA.u_ht,  MA.u_len);
			enforceSize(MB.u_len, MB.u_ht);
			MCFsubrMT(vx_m0, MA.vx_m0, MB.vx_m0, u_ht, u_len);
		}

		void mulC(cmta MA, cta C)
		{
			enforceSize(MA.u_ht, MA.u_len);
			MCFmulC(vx_m0, MA.vx_m0, u_ht, u_len, C);
		}

		void divC(cmta MA, cta C)
		{
			enforceSize(MA.u_ht, MA.u_len);
			MCFdivC(vx_m0, MA.vx_m0, u_ht, u_len, C);
		}

		void lincomb(cmta MA, cmta MB, cta CA, cta CB)
		{
			enforceSize(MA.u_ht, MA.u_len);
			enforceSize(MB.u_ht, MB.u_len);
			MCFlincomb(vx_m0, MA.vx_m0, MB.vx_m0, u_ht, u_len, CA, CB);
		}

		//mixed multiplications matrix * vector belong to fVecObj
        void mulM( cmta MA, cmta MB )
        {
            if( u_ht != MA.u_ht || u_len != MB.u_len || MB.u_ht != MA.u_len )throw OptiVec_SizeErr();
            MCFmulM( vx_m0, MA.vx_m0, MB.vx_m0, MA.u_ht, MA.u_len, MB.u_len);
        }

        void mulMT( cmta MA, cmta MB )
        {
            if( u_ht != MA.u_ht || u_len != MB.u_ht || MB.u_len != MA.u_len )throw OptiVec_SizeErr();
            MCFmulMT( vx_m0, MA.vx_m0, MB.vx_m0, MA.u_ht, MA.u_len, MB.u_ht);
        }

        void mulMH( cmta MA, cmta MB )
        {
            if( u_ht != MA.u_ht || u_len != MB.u_ht || MB.u_len != MA.u_len )throw OptiVec_SizeErr();
            MCFmulMH( vx_m0, MA.vx_m0, MB.vx_m0, MA.u_ht, MA.u_len, MB.u_ht);
        }

        void TmulM( cmta MA, cmta MB )
        {
            if( u_ht != MA.u_len || u_len != MB.u_len || MB.u_ht != MA.u_ht )throw OptiVec_SizeErr();
            MCFTmulM( vx_m0, MA.vx_m0, MB.vx_m0, MA.u_ht, MA.u_len, MB.u_len);
        }

        void HmulM( cmta MA, cmta MB )
        {
            if( u_ht != MA.u_len || u_len != MB.u_len || MB.u_ht != MA.u_ht )throw OptiVec_SizeErr();
            MCFHmulM( vx_m0, MA.vx_m0, MB.vx_m0, MA.u_ht, MA.u_len, MB.u_len);
        }

        void TmulMT( cmta MA, cmta MB )
        {
            if( u_ht != MA.u_len || u_len != MB.u_ht || MB.u_len != MA.u_ht )throw OptiVec_SizeErr();
            MCFTmulMT( vx_m0, MA.vx_m0, MB.vx_m0, MA.u_ht, MA.u_len, MB.u_ht);
        }

        void dia_mulM( cvta MADia, cmta MB )
        {
            if( u_ht != MADia.ui_size || u_len != MB.u_len || MB.u_ht != MADia.ui_size )throw OptiVec_SizeErr();
            MCFdiamulM( vx_m0, MADia.vx_vect, MB.vx_m0, MB.u_ht, MB.u_len );
        }

        void dia_mulMT( cvta MADia, cmta MB )
        {
            if( u_ht != MADia.ui_size || u_len != MB.u_ht || MB.u_len != MADia.ui_size )throw OptiVec_SizeErr();
            MCFdiamulMT( vx_m0, MADia.vx_vect, MB.vx_m0, MB.u_ht, MB.u_len );
        }

        void mulMdia( cmta MA, cvta MBDia )
        {
            if( u_ht != MA.u_ht || u_len != MBDia.ui_size || MBDia.ui_size != MA.u_len )throw OptiVec_SizeErr();
            MCFmulMdia( vx_m0, MA.vx_m0, MBDia.vx_vect, MA.u_ht, MA.u_len );
        }

        void TmulMdia( cmta MA, cvta MBDia )
        {
            if( u_ht != MA.u_len || u_len != MBDia.ui_size || MBDia.ui_size != MA.u_ht )throw OptiVec_SizeErr();
            MCFTmulMdia( vx_m0, MA.vx_m0, MBDia.vx_vect, MA.u_ht, MA.u_len );
        }

/////////////////////////////////////////////////////////////////////////////////////////
//Linear Algebra
//
        int LUdecompose( uVecObj Ind, cmta MA )
        {
            if( u_ht != MA.u_ht || u_len != MA.u_len || u_ht != u_len || u_ht != Ind.ui_size )
                throw OptiVec_SizeErr();
            return MCFLUdecompose( vx_m0, Ind.vx_vect, MA.vx_m0, u_len );
        }

        int LUdecompose( uVecObj* Ind, cmta MA )
        {
            if( u_ht != MA.u_ht || u_len != MA.u_len || u_ht != u_len || u_ht != Ind->ui_size )
                throw OptiVec_SizeErr();
            return MCFLUdecompose( vx_m0, Ind->vx_vect, MA.vx_m0, u_len );
        }

        int  LUDresult()
        {
            return MCF_LUDresult();
        }

		void LUDsetEdit(const tbase& Thresh)
		{
			MCF_LUDsetEdit(Thresh);
		}

		tbase LUDgetEdit()
		{
			return MCF_LUDgetEdit();
		}

        void LUinv( cmta MLU, const uVecObj& Ind )
        {
            if( u_ht != MLU.u_ht || u_len != MLU.u_len || u_ht != u_len || u_ht != Ind.ui_size )
                throw OptiVec_SizeErr();
            MCFLUinv( vx_m0, MLU.vx_m0, Ind.vx_vect, u_len );
        }

        t LUdet( int permut )
        {
            if( u_ht != u_len ) throw OptiVec_SizeErr();
            return MCFLUdet( vx_m0, u_len, permut );
        }

/*  SVD not yet implemented:
        int SVdecompose( tMatObj *MV, tVecObj *W, cmta MA )
        {
            if( (MA.u_ht < MA.u_len ? (u_ht != u_len) : (u_ht != MA.u_ht))
              || u_len != MA.u_len || u_len != MV->u_len || u_len != MV->u_ht
              || u_len != W->ui_size )   throw OptiVec_SizeErr();
            return MCFSVdecompose( vx_m0, MV->vx_m0, W->vx_vect,
                                   MA.vx_m0, MA.u_ht, MA.u_len );
        }

        int SVdecompose( tMatObj MV, tVecObj W, cmta MA )
        {
            if( (MA.u_ht < MA.u_len ? (u_ht != u_len) : (u_ht != MA.u_ht))
              || u_len != MA.u_len || u_len != MV.u_len || u_len != MV.u_ht
              || u_len != W.ui_size )   throw OptiVec_SizeErr();
            return MCFSVdecompose( vx_m0, MV.vx_m0, W.vx_vect,
                                   MA.vx_m0, MA.u_ht, MA.u_len );
        }

		void SVDsetEdit(cta Thresh)
		{
			MCF_SVDsetEdit(Thresh);
		}

		t SVDgetEdit()
		{
			return MCF_SVDgetEdit();
		}
*/

        int inv( cmta MA )
        {
            if( u_ht != MA.u_ht || u_len != MA.u_len || u_ht != u_len )
                throw OptiVec_SizeErr();
            return MCFinv( vx_m0, MA.vx_m0, u_len );
        }

        t det()
        {
            if( u_ht != u_len ) throw OptiVec_SizeErr();
            return MCFdet( vx_m0, u_len );
        }


/////////////////////////////////////////////////////////////////////////////////////////
//Fourier Transforms, Convolutions, Filtering
//

		void p_FFT(cmta X, int dir=1)
		{
			enforcePow2();
			MCFpFFT(vx_m0,X.vx_m0,u_ht,u_len,dir);
		}

		void p_FFTtoC(const matrix<tbase>& X)
		{
			enforcePow2();
			MFpFFTtoC(vx_m0,X.vx_m0,u_ht,u_len);
		}

		void p_filter(cmta X, cmta Flt)
		{
			enforcePow2();
			MCFpfilter(vx_m0, X.vx_m0, Flt.vx_m0, u_ht, u_len);
		}

		void l_FFT(cmta X, int dir=1)
		{
			enforcePow2();
			MCFlFFT(vx_m0,X.vx_m0,u_ht,u_len,dir);
		}

		void l_FFTtoC(const matrix<tbase>& X)
		{
			enforcePow2();
			MFlFFTtoC(vx_m0,X.vx_m0,u_ht,u_len);
		}

		void l_filter(cmta X, cmta Flt)
		{
			enforcePow2();
			MCFlfilter(vx_m0, X.vx_m0, Flt.vx_m0, u_ht, u_len);
		}

		void s_FFT(cmta X, int dir=1)
		{
			enforcePow2();
			MCFsFFT(vx_m0,X.vx_m0,u_ht,u_len,dir);
		}

		void s_FFTtoC(const matrix<tbase>& X)
		{
			enforcePow2();
			MFsFFTtoC(vx_m0,X.vx_m0,u_ht,u_len);
		}

		void s_filter(cmta X, cmta Flt)
		{
			enforcePow2();
			MCFsfilter(vx_m0, X.vx_m0, Flt.vx_m0, u_ht, u_len);
		}

		void FFT(cmta X, int dir=1)
		{
			enforcePow2();
			MCFFFT(vx_m0,X.vx_m0,u_ht,u_len,dir);
		}

		void FFTtoC(const matrix<tbase>& X)
		{
			enforcePow2();
			MFFFTtoC(vx_m0,X.vx_m0,u_ht,u_len);
		}

		void filter(cmta X, cmta Flt)
		{
			enforcePow2();
			MCFfilter(vx_m0, X.vx_m0, Flt.vx_m0, u_ht, u_len);
		}


/////////////////////////////////////////////////////////////////////////////////////////
//Input and Output
//
		void fprint(FILE* stream, unsigned linewidth) const
		{
			enforceSize();
			MCFfprint(stream,vx_m0,u_ht, u_len,linewidth);
		}

#if !defined _Windows || defined __FLAT__ || defined _WIN32
		void cprint() const
		{
			enforceSize();
			MCFcprint(vx_m0,u_ht,u_len);
		}
#endif
		void print() const
		{
			fprint(stdout,80);
		}

		void write(FILE* stream) const
		{
			enforceSize();
			MCFwrite(stream,vx_m0,u_ht,u_len);
		}

		void read(FILE* stream)
		{
			enforceSize();
			MCFread(vx_m0,u_ht,u_len,stream);
		}

		void setWriteFormat(char* formatString) const
		{
			VCF_setWriteFormat(formatString);
		}

		void setWriteSeparate(char* sepString) const
		{
			VCF_setWriteSeparate(sepString);
		}

		void store(FILE* stream) const
		{
			enforceSize();
			fwrite(&u_ht,sizeof(unsigned),1,stream);
			fwrite(&u_len,sizeof(unsigned),1,stream);
			VCF_store(stream,vx_m0,ui_size);
		}

		void recall(FILE* stream)
		{
			fread(&u_ht,sizeof(unsigned),1,stream);
			fread(&u_len,sizeof(unsigned),1,stream);
			setSize(u_ht,u_len);
			VCF_recall(vx_m0,ui_size,stream);
		}
	};


	inline vt tVecObj::operator*(cmta MY)
    {
		tVecObj tmp(this->ui_size);
           tmp.enforceSize();
		VCFmulM(tmp.vx_vect,vx_vect,MY.vx_m0,MY.u_ht,MY.u_len);
		return tmp;
	}

	inline void tVecObj::Row_extract(cmta MA, const unsigned iRow)
    {
        if( MA.u_ht <= iRow || ui_size < MA.u_len || !ui_size ) throw OptiVec_SizeErr();
        MCFRow_extract( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len, iRow );
    }

	inline void tVecObj::Col_extract(cmta MA, const unsigned iCol)
    {
        if( MA.u_len <= iCol || ui_size < MA.u_ht || !ui_size ) throw OptiVec_SizeErr();
        MCFCol_extract( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len, iCol );
    }

	inline void tVecObj::Dia_extract(cmta MA)
    {
		if( !ui_size || ui_size != MA.u_len || (MA.u_len != MA.u_ht) )throw OptiVec_SizeErr();
        MCFDia_extract( vx_vect, MA.vx_m0, MA.u_len );
    }

	inline void tVecObj::Rows_maxReIm(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_maxReIm( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Cols_maxReIm(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_maxReIm( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Rows_minReIm(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_minReIm( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Cols_minReIm(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_minReIm( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Rows_absmaxReIm(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_absmaxReIm( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Cols_absmaxReIm(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_absmaxReIm( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Rows_absminReIm(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_absminReIm( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Cols_absminReIm(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_absminReIm( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Rows_sabsmax(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_sabsmax( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Cols_sabsmax(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_sabsmax( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Rows_sabsmin(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_sabsmin( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Cols_sabsmin(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_sabsmin( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Rows_cabsmax(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_cabsmax( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Cols_cabsmax(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_cabsmax( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Rows_cabsmin(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_cabsmin( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Cols_cabsmin(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_cabsmin( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void vector<tbase>::Rows_absmax(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_absmax( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void vector<tbase>::Cols_absmax(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_absmax( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void vector<tbase>::Rows_absmin(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_absmin( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void vector<tbase>::Cols_absmin(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_absmin( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Rows_sum(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_sum( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Cols_sum(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_sum( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Rows_prod(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_ht ) throw OptiVec_SizeErr();
        MCFRows_prod( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

	inline void tVecObj::Cols_prod(cmta MA)
    {
        if( !MA.u_ht || !MA.u_len || ui_size < MA.u_len ) throw OptiVec_SizeErr();
        MCFCols_prod( vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

    inline void tVecObj::MmulV( cmta MA, cvta X )
    {
        if( ui_size != MA.u_ht || X.ui_size != MA.u_len ) throw OptiVec_SizeErr();
        MCFmulV( vx_vect, MA.vx_m0, X.vx_vect, MA.u_ht, MA.u_len );
    }

    inline void tVecObj::MTmulV( cmta MA, cvta X )
    {
        if( ui_size != MA.u_len || X.ui_size != MA.u_ht ) throw OptiVec_SizeErr();
        MCFTmulV( vx_vect, MA.vx_m0, X.vx_vect, MA.u_ht, MA.u_len );
    }

    inline void tVecObj::mulM( cvta X, cmta MA )
    {
        if( ui_size != MA.u_len || X.ui_size != MA.u_ht ) throw OptiVec_SizeErr();
        VCFmulM( vx_vect, X.vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

    inline void tVecObj::mulMT( cvta X, cmta MA )
    {
        if( ui_size != MA.u_ht || X.ui_size != MA.u_len ) throw OptiVec_SizeErr();
        VCFmulMT( vx_vect, X.vx_vect, MA.vx_m0, MA.u_ht, MA.u_len );
    }

    inline void tVecObj::LUsolve( cmta MLU, cvta B, const uVecObj& Ind )
    {
        if( ui_size != MLU.u_len || ui_size != MLU.u_ht ||
            ui_size != B.ui_size || ui_size != Ind.ui_size ) throw OptiVec_SizeErr();
        MCFLUsolve( vx_vect, MLU.vx_m0, B.vx_vect, Ind.vx_vect, MLU.u_len );
    }

    inline void tVecObj::LUimprove( cvta B, cmta MA, cmta MLU,
                          const uVecObj& Ind )
    {
        if( ui_size != MLU.u_len || ui_size != MLU.u_ht ||
            ui_size != MA.u_len || ui_size != MA.u_ht ||
            ui_size != B.ui_size || ui_size != Ind.ui_size ) throw OptiVec_SizeErr();
        MCFLUimprove( vx_vect, B.vx_vect, MA.vx_m0, MLU.vx_m0, Ind.vx_vect, MLU.u_len );
    }

/* SVD not yet implemented:
    inline void tVecObj::SVsolve( cmta MU, cmta MV, cvta W, cvta B )
    { // result may overwrite B only for "normal", not for over- or under-determined systems
        if( MU.u_ht < MU.u_len || MU.u_len != MV.u_len || MV.u_len != MV.u_ht ||
            B.ui_size != ui_size || W.ui_size != MV.u_len || ui_size != MU.u_ht )
            throw OptiVec_SizeErr();
        MCFSVsolve( vx_vect, MU.vx_m0, MV.vx_m0, W.vx_vect, B.vx_vect, (unsigned)B.ui_size, MU.u_len );
    }
*/
    inline int tVecObj::solve( cmta MA, cvta B )
    {
        if( ui_size != MA.u_len || ui_size != MA.u_ht ||
            ui_size != B.ui_size ) throw OptiVec_SizeErr();
        return MCFsolve( vx_vect, MA.vx_m0, B.vx_vect, MA.u_len );
    }

/*    inline int tVecObj::solveBySVD( cmta MA, cvta B )
    {
        if(ui_size != MA.u_len || (B.ui_size != MA.u_ht && B.vx_vect != vx_vect)) throw OptiVec_SizeErr();
        return MCFsolveBySVD( vx_vect, MA.vx_m0, B.vx_vect, MA.u_ht, MA.u_len );
    }

    inline int tVecObj::safeSolve( cmta MA, cvta B )
    {
        if( ui_size != MA.u_len || ui_size != MA.u_ht ||
            ui_size != B.ui_size ) throw OptiVec_SizeErr();
        return MCFsafeSolve( vx_vect, MA.vx_m0, B.vx_vect, MA.u_len );
    }
*/

#ifdef CDMATOBJ_H
	inline void cfMatObj::CDtoCF(const cdMatObj& X)
	{
		enforceSize(X.u_ht, X.u_len);
		V_CDtoCF(vx_m0,X.vx_m0,ui_size);
	}

	inline void cdMatObj::CFtoCD(const cfMatObj& X)
	{
		enforceSize(X.u_ht, X.u_len);
		V_CFtoCD(vx_m0,X.vx_m0,ui_size);
	}
#endif //CDMATOBJ_H

#ifdef CEMATOBJ_H
#ifdef __BORLANDC__ //80-bit numbers defined
	inline void cfMatObj::CEtoCF(const ceMatObj& X)
	{
		enforceSize(X.u_ht, X.u_len);
		V_CEtoCF(vx_m0,X.vx_m0,ui_size);
	}

	inline void ceMatObj::CFtoCE(const cfMatObj& X)
	{
		enforceSize(X.u_ht, X.u_len);
		V_CFtoCE(vx_m0,X.vx_m0,ui_size);
	}
#endif //__BORLANDC__
#endif //CEMATOBJ_H

#ifdef _MSC_VER
}  // end of namespace OptiVec
#endif

#undef cvta
#undef cmta
#undef cua
#undef cuint
#undef cta

#undef vta
#undef mta
#undef ta

#undef vt
#undef mt
#undef t
#undef tVector
#undef tMatrix
#undef tVecObj
#undef tMatObj

#undef vtbase
#undef tbase

#undef tVecObj
#undef tMatObj
#ifdef __BORLANDC__
#pragma warn .inl
#endif

#define CFMATOBJ_H
#endif //CFMATOBJ_H
