/************************************************************

	class CISImage

	For ImgSource v2.x !!!

	This is a wrapper class for the ISource functions. As only
	a few of the ISource functions are used, it is intended 
	to be used as a starting point for more complex classes.

	Sample use (load a BMP, reduce by 1/2, save as JPG) :

		CISImage image;
		image.LoadFromFile("C:\\input.bmp");

		CSize csSize = image.GetSize();
		csSize.cx/=2; csSize.cy/=2;
		image.Resize(csSize);

		image.SaveToFile("C:\\output.jpg", eFmtJPG);


	This class may be used and modified without restriction.

   You should not assume this code is bug-free. Be sure to test
   it in your own application!!

	ImgSource and CISImage are copyright 2000, Smaller Animals Software.

	All questions to : smallest@smalleranimals.com

************************************************************/

// ISImage.cpp: implementation of the CISImage class.
//
//////////////////////////////////////////////////////////////////////

#include "stdafx.h"
#include "ISImage.h"

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CISImage::CISImage()
{
	// this is currently only used to alert people that they may have
	// pointers which reference this image's data, but that this image
	// is being destructed...
   m_uReferenceCount=0;

	// the pixels
   m_hRGB=NULL;

	// how big is this?
   m_czPix = CSize(0,0);

   m_czDPI = CSize(DEF_DPIX, DEF_DPIY);
}

CISImage::~CISImage()
{
   Clear();
}     

//////////////////////////////////////////////////////////////////////
// Clear the image data
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::Clear()
{
   if (m_hRGB!=NULL)
   {

      // someone thinks they are still using a pointer to the image data!!
      // they are about to become mistaken...
      ASSERT(m_uReferenceCount==0);

      m_uReferenceCount=0;

      // clear it
      GlobalFree(m_hRGB);
      m_hRGB=NULL;

      m_czPix = CSize(0,0);
   }

   if (m_imagePal.GetSafeHandle())
      m_imagePal.DeleteObject();

   return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////
// allocate a blank, black image
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::AllocateBlank(const CSize &czPix)
{
   Clear();

	// RGB = w * 3 * h
   UINT32 uSizeBytes = czPix.cx * 3 * czPix.cy;

   m_hRGB = GlobalAlloc(GPTR, uSizeBytes);
   if (m_hRGB==NULL)
   {
      TRACE("Failed to allocate %u BYTEs\n", uSizeBytes);
      ASSERT(FALSE);
      return IS_ERR_MEM;
   }

   m_czPix = czPix;

   return IS_ERR_OK;   
}

//////////////////////////////////////////////////////////////////////
// Copy one image to another
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::Copy(CISImage &imgSource)
{
   Clear();

   UINT32 uRes = AllocateBlank(imgSource.GetSize());

   if (uRes!=IS_ERR_OK)
   {
      return uRes;
   }

   
   BYTE *pSource = imgSource.GetImagePointer();
   if (pSource==NULL)
   {
      imgSource.ReleaseImagePointer();
      Clear();
      return IS_ERR_MEM;
   }

   // this shouldn't fail; if it does, AllocateBlank is broken
   BYTE *pDest = GetImagePointer();
   ASSERT(pDest);

   UINT32 uSizeBytes = m_czPix.cx * 3 * m_czPix.cy;

   // copy it
   memcpy(pDest, pSource, uSizeBytes);

   ReleaseImagePointer();
   imgSource.ReleaseImagePointer();

   return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////
// Get a pointer to the RGB image data
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::CopyToCBitmap(CBitmap &cBmp, HPALETTE hPal)
{
	BYTE *pRGB = GetImagePointer();
	if (pRGB==NULL)
	{
		return IS_ERR_BADPARAM;
	}

	HPALETTE hUsePal = hPal;
	if (hUsePal==NULL)
   {
		hUsePal = (HPALETTE)m_imagePal.GetSafeHandle();
   }

   // get reference DC
   HDC hDC = GetDC(NULL);

	HBITMAP hBmp = ISrc(ISRGBToHBITMAP)(pRGB,
										(UINT)m_czPix.cx,
										(UINT)m_czPix.cy,
										hUsePal,
                              hDC);

   ReleaseDC(NULL, hDC);

	ReleaseImagePointer();

	if (hBmp)
	{
		if (cBmp.GetSafeHandle())
			cBmp.DeleteObject();

		cBmp.Attach(hBmp);
	}

	return ISrc(ISGetLastError)();
}


//////////////////////////////////////////////////////////////////////
// Get a pointer to the RGB image data
//////////////////////////////////////////////////////////////////////

BYTE *      CISImage::GetImagePointer()
{
   if (m_hRGB==NULL)
   {
      return NULL;
   }

   // how many times have we been examined?
   m_uReferenceCount++;

   return (BYTE *)m_hRGB;
}

//////////////////////////////////////////////////////////////////////
// Release the image pointer
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::ReleaseImagePointer()
{
   // how many times have we been examined?
   m_uReferenceCount--;

   return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////
// Load from file, guess file type from file
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::LoadFromFile(const char *pFileName)
{
   eFormat eFmt = GuessFileTypeFile(pFileName);
   if (eFmt==eFmtNone)
   {
      eFmt = GuessFileTypeFromExtension(pFileName);
   }

   return LoadFromFile(pFileName, eFmt);
}

//////////////////////////////////////////////////////////////////////
// read a file
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::LoadFromFile(const char *pFileName, eFormat eFmt)
{
   Clear();

   ASSERT (eFmt!=eFmtNone);

   UINT32 uRes = IS_ERR_OK;

   // metafiles can only be read from disk, so we have to bypass the
   // ISource source manager stuff
   if ((eFmt==eFmtWMF) || (eFmt==eFmtEMF))
   {
      UINT32 uWidth, uHeight;
      HGLOBAL hRGB = ISrc(ISReadMetafileToRGB)(pFileName, &uWidth, &uHeight, DEF_METAFILE_WIDTH, DEF_METAFILE_HEIGHT, DEF_METAFILE_BACKGROUND);

      uRes = ISrc(ISGetLastError)();
      if (uRes!=IS_ERR_OK)
      {
         TRACE("%s\n", ErrorString(uRes));
      }

      if (hRGB!=NULL)
      {
         m_hRGB = hRGB;
         m_czPix = CSize(uWidth, uHeight);

         // just in case we need this for display
         MakeImagePalette();
      }
   }
   else
   {
      HISSRC hSource = ISrc(ISOpenFileSource)(pFileName);
      if (hSource==NULL)
      {
         return ISrc(ISGetLastError)();
      }

      uRes = LoadFile(hSource, eFmt);
   
      ISrc(ISCloseSource)(hSource);
   }

   return uRes;
}

//////////////////////////////////////////////////////////////////////
// read a memory-based file
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::LoadFromMemory(const BYTE *pMem, const UINT32 uBufLen)
{
   return LoadFromMemory(pMem, uBufLen, GuessFileTypeMemory(pMem, uBufLen));
}

//////////////////////////////////////////////////////////////////////
// read a memory-based file
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::LoadFromMemory(const BYTE *pMem, const UINT32 uBufLen, eFormat eFmt)
{
   Clear();

   ASSERT (eFmt!=eFmtNone);
   ASSERT (eFmt!=eFmtWMF);		// no support
   ASSERT (eFmt!=eFmtEMF);		// no support
   
   HISSRC hSource = ISrc(ISOpenMemorySource)(pMem, uBufLen);
   if (hSource==NULL)
   {
      return ISrc(ISGetLastError)();
   }

   UINT32 uRes = LoadFile(hSource, eFmt);
   
   ISrc(ISCloseSource)(hSource);

   return uRes;
}

//////////////////////////////////////////////////////////////////////
// read a file
//////////////////////////////////////////////////////////////////////

UINT32 CISImage::LoadFile(HISSRC hSource, eFormat eFmt)
{
   UINT32 uWidth = 0, 
         uHeight = 0,
		 uDepth = 0;
   HGLOBAL hRGB = NULL;

   UINT32 uRes = IS_ERR_OK;

   // DPI
   ISDPIStruct dpi;

   dpi.uDPIUnits = 0;
   dpi.uDPIX = 0;
   dpi.uDPIY = 0;

   // set to defaults
   CalcDPI(eFmtNone, dpi.uDPIX, dpi.uDPIY, dpi.uDPIUnits);

   switch (eFmt)
   {
   case eFmtBMP:   
      hRGB = ISrc(ISReadBMP)(hSource, &uWidth, &uHeight, 24, NULL);
	  uRes = ISrc(ISGetLastError)();
      break;
   case eFmtJPG:
		{
			// get dims / DPI
			ISrc(ISGetJPGDims)(hSource, &uWidth, &uHeight, &uDepth, &dpi);

			// the get dims call moved the file pointer, let's go back to the start of the file
			ISrc(ISSeek)(hSource, 0, 0);

			CalcDPI(eFmtJPG, dpi.uDPIX, dpi.uDPIY, dpi.uDPIUnits);
			
			hRGB = ISrc(ISReadJPG)(hSource, &uWidth, &uHeight, 24);
			uRes = ISrc(ISGetLastError)();
			
			// get comment text
			UINT32 uTextCount = ISrc(ISGetJPGInputTextCount)();
			for (UINT32 i =0; i < uTextCount; i++)
			{
				const char * pText = (const char *)ISrc(ISGetJPGInputText)(i);
				if (pText)
				{
					TRACE("Read JPG comment : %s\n", pText);
					GlobalFree((HGLOBAL)pText);
				}
			}
			
			// get JPEG_APPx tags
			UINT32 uMarkerCount = ISrc(ISGetJPGInputMarkerCount)();
			for (i =0; i < uMarkerCount; i++)
			{
				HGLOBAL hMarker;
				UINT32 uDataLen, uDataType;
				ISrc(ISGetJPGInputMarker)(i, &hMarker, &uDataLen, &uDataType);
				TRACE("Read %d bytes in JPEG_APP%d tag : %s\n", 
					uDataLen,
					(uDataType - 0xE1), // offset from JPEG_APP0
					HexString((BYTE*)hMarker, uDataLen));
				GlobalFree(hMarker);
			}
			
		}
		break;
   case eFmtPNG:
	   {
			// get dims / DPI
		   UINT32 ct, it;

			ISrc(ISGetPNGDims)(hSource, &uWidth, &uHeight, &uDepth, &ct, &it, &dpi);

			CalcDPI(eFmtPNG, dpi.uDPIX, dpi.uDPIY, dpi.uDPIUnits);

			// the get dims call moved the file pointer, let's go back to the start of the file
			ISrc(ISSeek)(hSource, 0, 0);

			hRGB = ISrc(ISReadPNG)(hSource, &uWidth, &uHeight, 24, NULL);
		   
			uRes = ISrc(ISGetLastError)();
		   
		   // get comment text
		   UINT32 uTextCount = ISrc(ISGetPNGInputTextCount)();
		   for (UINT32 i =0; i < uTextCount; i++)
		   {
			   const char * pText = NULL;
			   const char * pKey = NULL;
			   UINT32 uCompression = 0;
			   ISrc(ISGetPNGInputText)(i, (HGLOBAL *)&pKey, (HGLOBAL *)&pText, &uCompression);
			   if (pKey)
			   {
				   TRACE("Read PNG comment : [%s]", pKey);
				   
				   GlobalFree((HGLOBAL)pKey);
				   if (pText)
				   {
					   TRACE(" %s %s", pText, uCompression ? "(compressed)" : "");
					   GlobalFree((HGLOBAL)pKey);
				   }
				   
				   TRACE("\n");
			   }
		   }
		   
	   }
	   break;
   case eFmtPCX:
	   hRGB = ISrc(ISReadPCX)(hSource, &uWidth, &uHeight, 24, NULL);
	   uRes = ISrc(ISGetLastError)();
	   break;
   case eFmtTIF:
	   ISrc(ISGetTIFFDims)(hSource, &uWidth, &uHeight, &uDepth, &dpi);

	   CalcDPI(eFmtTIF, dpi.uDPIX, dpi.uDPIY, dpi.uDPIUnits);
	   
	   // the get dims call moved the file pointer, let's go back to the start of the file
	   ISrc(ISSeek)(hSource, 0, 0);

	   hRGB = ISrc(ISReadTIFF)(hSource, &uWidth, &uHeight, 24, NULL, 0);
	   uRes = ISrc(ISGetLastError)();
	   break;
   case eFmtTGA:
	   hRGB = ISrc(ISReadTGA)(hSource, &uWidth, &uHeight, 24, NULL);
	   uRes = ISrc(ISGetLastError)();
	   break;
   case eFmtPSD:
	   hRGB = ISrc(ISReadPSD)(hSource, &uWidth, &uHeight, 24, NULL);
	   uRes = ISrc(ISGetLastError)();
	   break;
   default:
	   // unsupported format type
	   // (metafiles are handled in the LoadFromFile functions)
	   ASSERT(FALSE);
	   break;
   }
   
   
   if (uRes!=IS_ERR_OK)
   {
	   TRACE("** %s\n", ErrorString(uRes));
   }
   
   if (hRGB!=NULL)
   {                                                     
	   m_hRGB = hRGB;
	   m_czPix = CSize(uWidth, uHeight);
	   
	   // just in case we need this for display
	   MakeImagePalette();
   }
   
   return uRes;
}


//////////////////////////////////////////////////////////////////////
// write a file
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::SaveToFile(const char *pFileName, const eFormat eFmt)
{
	HISDEST hDest = ISrc(ISOpenFileDest)(pFileName);
	if (hDest==NULL)
	{
		return ISrc(ISGetLastError)();
	}
	
	UINT32 uBytesWritten = SaveImage(hDest, eFmt);
	
	ISrc(ISCloseDest)(hDest);
	
	return uBytesWritten;
}

//////////////////////////////////////////////////////////////////////
// write a file to memory
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::SaveToMemory(BYTE *pMem, const UINT32 uBufLen, const eFormat eFmt)
{
	HISDEST hDest = ISrc(ISOpenMemoryDest)(pMem, uBufLen);
	if (hDest==NULL)
	{
		return ISrc(ISGetLastError)();
	}
	
	UINT32 uBytesWritten = SaveImage(hDest, eFmt);
	
	ISrc(ISCloseDest)(hDest);
	
	return uBytesWritten;
}

//////////////////////////////////////////////////////////////////////
// write a file
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::SaveImage(HISDEST hDest, eFormat eFmt)
{
	BYTE *pRGB = GetImagePointer();
	if (pRGB==NULL)
	{
		return IS_ERR_BADPARAM;
	}
	
	UINT32 uBytesWritten = 0;
	
	switch (eFmt)
	{
	case eFmtBMP:
		uBytesWritten = ISrc(ISWriteBMP)(hDest, pRGB, m_czPix.cx, m_czPix.cy, 24, 1 << 24, NULL, FALSE);
		break;
	case eFmtJPG:
		uBytesWritten = ISrc(ISWriteJPG)(hDest, pRGB, m_czPix.cx, m_czPix.cy, DEF_JPG_QUALITY, TRUE, 24, NULL);
		break;
	case eFmtPCX:                                                           
		uBytesWritten = ISrc(ISWritePCX)(hDest, pRGB, m_czPix.cx, m_czPix.cy, 24, NULL);
		break;
	case eFmtPNG:                                          
		uBytesWritten = ISrc(ISWritePNG)(hDest, pRGB, m_czPix.cx, m_czPix.cy, m_czPix.cx * 3, 8, 1 << 24, NULL, 2, DEF_PNG_GAMMA, 0, NULL);
		break;
	case eFmtTIF:                                                             
		uBytesWritten = ISrc(ISWriteTIFF)(hDest, pRGB, m_czPix.cx, m_czPix.cy, 24, NULL, 1, NULL);
		break;
	case eFmtTGA:
		uBytesWritten = ISrc(ISWriteTGA)(hDest, pRGB, m_czPix.cx, m_czPix.cy, 24, NULL);
		break;
	case eFmtPSD:
		uBytesWritten = ISrc(ISWritePSD)(hDest, pRGB, m_czPix.cx, m_czPix.cy, 24, NULL);
		break;
	case eFmtWMF:
		uBytesWritten = ISrc(ISWriteRGBToWMF)(hDest, pRGB, m_czPix.cx, m_czPix.cy, TRUE);
		break;
	case eFmtEMF:
		uBytesWritten = ISrc(ISWriteRGBToEMF)(hDest, pRGB, m_czPix.cx, m_czPix.cy);
		break;
	default:
		uBytesWritten = Save8Bit(hDest, eFmt);
	}
	
	ReleaseImagePointer();
	
	UINT32 uRes = ISrc(ISGetLastError)();
	if (uRes!=IS_ERR_OK)
	{
		TRACE("** %s\n", ErrorString(uRes));
	}
	
	return uBytesWritten;
}

//////////////////////////////////////////////////////////////////////
// save 8 bit
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::Save8Bit(HISDEST hDest, eFormat eFmt)
{
	BYTE *pRGB = GetImagePointer();
	ASSERT(pRGB);
	
	UINT32 uBytesWritten = 0;
	
	RGBQUAD pal[256];
	
	// gray palette
	RGBQUAD grayPal[256];
	for (int i=0; i<256; i++) grayPal[i].rgbRed = grayPal[i].rgbGreen = grayPal[i].rgbBlue = i;
	
	
	BYTE *p8Bit = NULL;
	p8Bit = new BYTE [m_czPix.cx * m_czPix.cy];
	
	switch (eFmt)
	{
	case eFmtJPGGray:
		{
            // make a nice 8-bit grayscale image
            VERIFY(ISrc(ISRGBToGrayScale8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit));
			
            uBytesWritten = ISrc(ISWriteJPG)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, DEF_JPG_QUALITY, TRUE, 8, NULL);
		}
		break;
	case eFmtPNG8:
		{
            // make a nice 8-bit image
            VERIFY(ISrc(ISQuantizeRGBTo8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit, 256, pal, 0));
			
            uBytesWritten = ISrc(ISWritePNG)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, m_czPix.cx, 8, 256, pal, 3, DEF_PNG_GAMMA, 0, NULL);
		}
		break;
	case eFmtPNG8Gray:
		{
            // make a nice 8-bit grayscale image
            VERIFY(ISrc(ISRGBToGrayScale8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit));
            
            uBytesWritten = ISrc(ISWritePNG)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, m_czPix.cx, 8, 256, grayPal, 3, DEF_PNG_GAMMA, 0, NULL);
		}
		break;
	case eFmtBMP8:
		{
            // make a nice 8-bit image
            VERIFY(ISrc(ISQuantizeRGBTo8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit, 256, pal, 0));
			
            uBytesWritten = ISrc(ISWriteBMP)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, 8, 256, pal, FALSE);
		}
		break;
	case eFmtBMP8Gray:
		{
            // make a nice 8-bit grayscale image
            VERIFY(ISrc(ISRGBToGrayScale8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit));
			
            uBytesWritten = ISrc(ISWriteBMP)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, 8, 256, grayPal, FALSE);
		}
		break;
	case eFmtPCX8:
		{
            // make a nice 8-bit image
            VERIFY(ISrc(ISQuantizeRGBTo8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit, 256, pal, 0));
			
            uBytesWritten = ISrc(ISWritePCX)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, 8, pal);
		}
		break;
	case eFmtPCX8Gray:
		{
            // make a nice 8-bit grayscale image
            VERIFY(ISrc(ISRGBToGrayScale8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit));
			
            uBytesWritten = ISrc(ISWritePCX)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, 8, grayPal);
		}
		break;
	case eFmtTIF8:
		{
            // make a nice 8-bit image
            VERIFY(ISrc(ISQuantizeRGBTo8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit, 256, pal, 0));
			
            uBytesWritten = ISrc(ISWriteTIFF)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, 8, pal, 1, NULL);
		}
		break;
	case eFmtTIF8Gray:
        {
            // make a nice 8-bit grayscale image
            VERIFY(ISrc(ISRGBToGrayScale8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit));
			
            uBytesWritten = ISrc(ISWriteTIFF)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, 8, grayPal, 1, NULL);
		}
		break;
	case eFmtTGA8:
		{
            // make a nice 8-bit image
            VERIFY(ISrc(ISQuantizeRGBTo8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit, 256, pal, 0));
			
            uBytesWritten = ISrc(ISWriteTGA)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, 8, pal);
		}
		break;
	case eFmtTGA8Gray :
        {
            // make a nice 8-bit grayscale image
            VERIFY(ISrc(ISRGBToGrayScale8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit));
			
            uBytesWritten = ISrc(ISWriteTGA)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, 8, grayPal);
		}
		break;
	case eFmtPSD8:
		{
            // make a nice 8-bit image
            VERIFY(ISrc(ISQuantizeRGBTo8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit, 256, pal, 0));
			
            uBytesWritten = ISrc(ISWritePSD)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, 8, pal);
		}
		break;
	case eFmtPSD8Gray :
        {
            // make a nice 8-bit grayscale image
            VERIFY(ISrc(ISRGBToGrayScale8Bit)(pRGB, m_czPix.cx, m_czPix.cy, p8Bit));
			
            uBytesWritten = ISrc(ISWritePSD)(hDest, p8Bit, m_czPix.cx, m_czPix.cy, 8, grayPal);
		}
		break;
	}
	
	delete [] p8Bit;
	
	ReleaseImagePointer();
	
	return uBytesWritten;
}

//////////////////////////////////////////////////////////////////////
// draw image cropped
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::DrawCrop(const CDC *pDC, const CPoint & ptTopLeft, const CRect & displayRect, const HPALETTE hPal)
{
	BYTE *pRGB = GetImagePointer();
	if (pRGB==NULL)
	{
		return IS_ERR_BADPARAM;
	}
	
	BOOL ok = ISrc(ISDrawRGBCrop)(pDC->m_hDC, pRGB, m_czPix.cx, m_czPix.cy, ptTopLeft.x, ptTopLeft.y, displayRect.left, displayRect.top, displayRect.Width(), displayRect.Height(), hPal);
	
	ReleaseImagePointer();
	
	ASSERT(ok);
	
	return ISrc(ISGetLastError)();
	
}

//////////////////////////////////////////////////////////////////////
// draw image
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::Draw(const CDC *pDC, const CPoint & ptTopLeft, const HPALETTE hPal)
{
	BYTE *pRGB = GetImagePointer();
	if (pRGB==NULL)
	{
		return IS_ERR_BADPARAM;
	}
	
	BOOL ok = ISrc(ISDrawRGB)(pDC->m_hDC, pRGB, m_czPix.cx, m_czPix.cy, ptTopLeft.x, ptTopLeft.y, m_czPix.cx, m_czPix.cy, hPal);
	
	ReleaseImagePointer();
	
	ASSERT(ok);
	
	return ISrc(ISGetLastError)();
}

//////////////////////////////////////////////////////////////////////
// draw image
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::StretchDraw(const CDC *pDC, const CRect & outRect)
{
	BYTE *pRGB = GetImagePointer();
	if (pRGB==NULL)
	{
		return IS_ERR_BADPARAM;
	}
	
	BOOL ok = ISrc(ISStretchDrawRGB)(pDC->m_hDC, pRGB, m_czPix.cx, m_czPix.cy, outRect.left, outRect.top, outRect.Width(), outRect.Height());
	
	ReleaseImagePointer();
	
	ASSERT(ok);
	
	return ISrc(ISGetLastError)();
}

//////////////////////////////////////////////////////////////////////
// resize this image
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::Resize(const CSize &czPix)
{
	CISImage temp;
	
	UINT32 uRes = temp.CopyResized(czPix, *this);
	if (uRes!=IS_ERR_OK)
	{
		return uRes;
	}
	
	return Copy(temp);
}

//////////////////////////////////////////////////////////////////////
// copy a resized version of the source image to this 
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::CopyResized(const CSize & czPix, CISImage &imgSource)
{
	Clear();
	
	
	UINT32 uRes = AllocateBlank(czPix);
	
	if (uRes!=IS_ERR_OK)
	{
		return uRes;
	}
	
	BYTE *pDest = GetImagePointer();
	if (pDest==NULL)
	{
		ReleaseImagePointer();
		return IS_ERR_BADPARAM;
	}
	
	BYTE *pSrc = imgSource.GetImagePointer();
	if (pSrc==NULL)
	{
		ReleaseImagePointer();
		imgSource.ReleaseImagePointer();
		return IS_ERR_BADPARAM;
	}
	
	BOOL ok = ISrc(ISResizeImage)(pSrc, imgSource.m_czPix.cx, imgSource.m_czPix.cy, pDest, czPix.cx, czPix.cy, 3);
	
	ReleaseImagePointer();
	imgSource.ReleaseImagePointer();
	
	return ISrc(ISGetLastError)();
}

//////////////////////////////////////////////////////////////////////
// vertically flip the image
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::VerticalFlip()
{
	BYTE *pRGB = GetImagePointer();
	if (pRGB==NULL)
	{
		return IS_ERR_BADPARAM;
	}
	
	BOOL ok = ISrc(ISVertFlipBuf)(pRGB, m_czPix.cx * 3, m_czPix.cy);
	
	ReleaseImagePointer();
	
	return ISrc(ISGetLastError)();
}

//////////////////////////////////////////////////////////////////////
// horizontally flip the image
//////////////////////////////////////////////////////////////////////

UINT32      CISImage::HorizontalFlip()
{
	BYTE *pRGB = GetImagePointer();
	if (pRGB==NULL)
	{
		return IS_ERR_BADPARAM;
	}
	BOOL ok = ISrc(ISFlipHorizontalImage)(pRGB, m_czPix.cx, m_czPix.cy, 3);
	
	ReleaseImagePointer();
	
	return ISrc(ISGetLastError)();
}


//////////////////////////////////////////////////////////////////////
// file type guessing data
//////////////////////////////////////////////////////////////////////
typedef struct fmtSwitchTag
{
	CISImage::eFormat eFmt;
	UINT32            uFmt;
	const char *      pExt;
} fmtSwitch;

const fmtSwitch fmts [] = 
{
	CISImage::eFmtNone,  0,   "",
		CISImage::eFmtBMP,   1,   ".BMP",
		CISImage::eFmtJPG,   3,   ".JPG",
		CISImage::eFmtJPG,   3,   ".JPEG",
		CISImage::eFmtPNG,   4,   ".PNG",
		CISImage::eFmtPCX,   5,   ".PCX",
		CISImage::eFmtTIF,   6,   ".TIF",
		CISImage::eFmtTIF,   6,   ".TIFF",
		CISImage::eFmtWMF,   7,   ".WMF",
		CISImage::eFmtEMF,   8,   ".EMF",
		CISImage::eFmtPSD,   10,   ".PSD",
		CISImage::eFmtWBMP,   13,   ".WBMP",
		CISImage::eFmtTGA,   14,   ".TGA"
};

#define ARRAYSIZE(__a)    (sizeof(__a)/sizeof(__a[0]))



//////////////////////////////////////////////////////////////////////
// guess file type from header bytes
//////////////////////////////////////////////////////////////////////

CISImage::eFormat     CISImage::GuessFileTypeFile(const char *pFileName)
{
	HISSRC hSource = ISrc(ISOpenFileSource)(pFileName);
	if (hSource==NULL)
	{
		return eFmtNone;
	}
	
	UINT32 uType = ISrc(ISGuessFileType)(hSource);
	
	eFormat eFmt = eFmtNone;
	for (int i=0; i < ARRAYSIZE(fmts); i++)
	{
		if (fmts[i].uFmt == uType)
		{
			eFmt = fmts[i].eFmt;
			break;
		}
	}
	
	
	ISrc(ISCloseSource)(hSource);
	
	return eFmt;
}

//////////////////////////////////////////////////////////////////////
// guess file type from header bytes
//////////////////////////////////////////////////////////////////////

CISImage::eFormat     CISImage::GuessFileTypeMemory(const BYTE *pMem, const UINT32 uBufLen)
{
	HISSRC hSource = ISrc(ISOpenMemorySource)(pMem, uBufLen);
	if (hSource==NULL)
	{
		return eFmtNone;
	}
	
	UINT32 uType = ISrc(ISGuessFileType)(hSource);
	
	eFormat eFmt = eFmtNone;
	for (int i=0; i < ARRAYSIZE(fmts); i++)
	{
		if (fmts[i].uFmt == uType)
		{
			eFmt = fmts[i].eFmt;
			break;
		}
	}
	
	ISrc(ISCloseSource)(hSource);
	
	return eFmt;
}

//////////////////////////////////////////////////////////////////////
// guess file type from extension
//////////////////////////////////////////////////////////////////////

CISImage::eFormat    CISImage::GuessFileTypeFromExtension(const char *pFileName)
{
	// get input file extension
	CString csFullPath = pFileName;
	CString ext;
	int iDotPos = csFullPath.ReverseFind('.');
	if (iDotPos!=-1) 
	{
		ext = csFullPath.Mid(iDotPos);
	}
	ext.TrimRight();
	
	// guess
	eFormat eFmt = eFmtNone;
	for (int i=0; i < ARRAYSIZE(fmts); i++)
	{
		if (ext.CompareNoCase(fmts[i].pExt)==0)
		{
			eFmt = fmts[i].eFmt;
			break;
		}
	}
	
	return eFmt;
}

//////////////////////////////////////////////////////////////////////
// update our CPalette
//////////////////////////////////////////////////////////////////////

void     CISImage::MakeImagePalette()
{ 
	LPLOGPALETTE     lpPal;
	BYTE *           pLogPal;
	HPALETTE         hPal = NULL;
	int              i;
	
	try 
	{
		pLogPal = new BYTE [sizeof (LOGPALETTE) + (sizeof (PALETTEENTRY) * (256))];
	} 
	catch (CMemoryException *e) 
	{
		e->ReportError();
		e->Delete();
		pLogPal=NULL;
		return;
	}
	
	lpPal = (LPLOGPALETTE) pLogPal;
	lpPal->palVersion = 0x300;
	lpPal->palNumEntries = 256;
	
	RGBQUAD imgPal[256];
	GetPalette(imgPal);
	
	for (i=0;i<256;i++) 
	{
		lpPal->palPalEntry[i].peRed   = imgPal[i].rgbRed;
		lpPal->palPalEntry[i].peGreen = imgPal[i].rgbGreen;
		lpPal->palPalEntry[i].peBlue  = imgPal[i].rgbBlue;
		lpPal->palPalEntry[i].peFlags = 0;
	}
	
	if (m_imagePal.GetSafeHandle())
		m_imagePal.DeleteObject();
	
	BOOL ok = m_imagePal.CreatePalette (lpPal);
	
	delete [] pLogPal;
}

//////////////////////////////////////////////////////////////////////
// get an RGBQUAD palette from this image
//////////////////////////////////////////////////////////////////////

UINT32 CISImage::GetPalette(RGBQUAD *pPal)
{
	BYTE *pRGB = GetImagePointer();
	if (pRGB==NULL)
	{
		return IS_ERR_BADPARAM;
	}
	
	ISrc(ISGet8BitPaletteFromRGB)(pRGB,
		m_czPix.cx, 
		m_czPix.cy,
		256,
		pPal);
	
	ReleaseImagePointer();
	
	return IS_ERR_OK;
}

//////////////////////////////////////////////////////////////////////
// move all DPI results to D.P.I.
//////////////////////////////////////////////////////////////////////


void CISImage::CalcDPI(eFormat eFmt, UINT32 dx, UINT32 dy, UINT32 u)
{
	switch (eFmt)
	{
	default:
		m_czDPI = CSize(DEF_DPIX, DEF_DPIY);
		break;
	case eFmtJPG:
		switch(u)
		{
		default:
		case 0:
			m_czDPI = CSize(DEF_DPIX, DEF_DPIY);
			break;
		case 1:
			m_czDPI = CSize(dx, dy);
			break;
		case 2:
			m_czDPI = CSize(CMTOINCH(dx), CMTOINCH(dy));
			break;
		}
		break;
	case eFmtTIF:
		switch(u)
		{
		default:
		case 1:
			m_czDPI = CSize(DEF_DPIX, DEF_DPIY);
			break;
		case 2:
			m_czDPI = CSize(dx, dy);
			break;
		case 3:
			m_czDPI = CSize(CMTOINCH(dx), CMTOINCH(dy));
			break;
		}
			break;
	case eFmtPNG:
		switch(u)
		{
		default:
		case 0:
			m_czDPI = CSize(DEF_DPIX, DEF_DPIY);
			break;
		case 1:
			// PNG DPIs are based in dots per meter
			m_czDPI = CSize(CMTOINCH(dx / 100), CMTOINCH(dy / 100));
			break;
		}
		break;
		
	}
}

//////////////////////////////////////////////////////////////////////
// generate a nice error string
//////////////////////////////////////////////////////////////////////

CString CISImage::ErrorString(UINT32 uRes)
{
	
	CString csErr;
	csErr.Format("Error #%u!", uRes);
	
	switch (uRes)
	{
	case IS_ERR_TRIALVERSION:     	csErr = "The ImgSource library was not initialized with a registered key.\nAll images read or written will have a big red \"X\" placed on them."; break;
	case IS_ERR_OK:			   		csErr = "no err"; break;
	case IS_ERR_MEM:			      	csErr = "out of memory"; break;
	case IS_ERR_FILEOPEN:				csErr = "error on file open"; break;
	case IS_ERR_FILEREAD:				csErr = "error on file read"; break;
	case IS_ERR_FILEWRITE:				csErr = "error on file write"; break;
	case IS_ERR_BADPARAM:				csErr = "bad user param"; break;
	case IS_ERR_INVALIDBMP:		   	csErr = "bad BMP file"; break;
	case IS_ERR_BMPRLE:					csErr = "some RLE variations are not supported"; break;
	case IS_ERR_INVALIDJPG:		   	csErr = "bad JPG file"; break;
	case IS_ERR_DC:						csErr = "error with device context"; break;
	case IS_ERR_DIB:						csErr = "problem with a GetDIBits call"; break;
	case IS_ERR_NORESOURCE:		   	csErr = "resource not found"; break;
	case IS_ERR_CALLBACKCANCEL:		csErr = "callback returned FALSE - operation aborted"; break;
	case IS_ERR_INVALIDPNG:		   	csErr = "bad PNG file"; break;
	case IS_ERR_PNGCREATE:				csErr = "internal PNG lib behavior - contact smaller animals s.w."; break;
	case IS_ERR_INTERNAL:				csErr = "misc unexpected behavior error - contact smaller animals s.w."; break;
	case IS_ERR_FONT:				   	csErr = "trouble creating a font object"; break;
	case IS_ERR_INTTIFF:			   	csErr = "misc internal TIFF error"; break;
	case IS_ERR_INVALIDTIFF:			csErr = "invalid TIFF file"; break;
	case IS_ERR_NOTIFFLZW:				csErr = "this will not read TIFF-LZW images"; break;
	case IS_ERR_INVALIDPCX:		   	csErr = "invalid PCX image"; break;
	case IS_ERR_CREATEBMP:				csErr = "a call to the fn CreateCompatibleBitmap failed"; break;
	case IS_ERR_NOLINES:			   	csErr = "end of an image while using single-line de/compression"; break;
	case IS_ERR_GETDIB:					csErr = "error during a call to GetDIBits"; break;
	case IS_ERR_NODEVOP:			   	csErr = "device does not support an operation required by this function"; break;
	case IS_ERR_INVALIDWMF:		   	csErr = "invalid windows metafile"; break;
	case IS_ERR_DEPTHMISMATCH:	   	csErr = "the file was not of the requested bit-depth"; break;
	case IS_ERR_BITBLT:					csErr = "a call to BitBlt failed."; break;
	case IS_ERR_BUFTOOSMALL:			csErr = "output buffer is too small for this operation"; break;
	case IS_ERR_TOOMANYCOLORS:	   	csErr = "not enough room in the output palette to store the colors from this image"; break;
	case IS_ERR_INVALIDTGA:			   csErr = "bad TGA File"; break;
	case IS_ERR_CREATEDIB:				csErr = "a call to the fn CreateDIBitmap failed"; break;
	case IS_ERR_NOLZW:					csErr = "LZW de/compression is not permitted"; break;
	}
	
	return csErr;
}

CString CISImage::HexString(BYTE *pData, UINT32 uDataLen)
{
	CString out;
	
	// chop this...
	UINT32 uLocalDataLen = min(uDataLen, 20);
	
	for (UINT32 i=0; i < uLocalDataLen; i++)
	{
		CString f;
		
		if (pData[i]<16)
			f.Format("0%x", pData[i]);
		else
			f.Format("%x", pData[i]);
		
		out+=f;
	}
	
	if (uLocalDataLen < uDataLen)
		out+="...";
	
	return out;
}