/*************************************************************************
** interpcom-2.3   (command interpreter)                                 **
** mem.c :    Creation and destruction of arrays                         **
**                                                                       **
** Copyright (C) 2001  Jean-Marc Drezet                                  **
**                                                                       **
**  This library is free software; you can redistribute it and/or        **
**  modify it under the terms of the GNU Library General Public          **
**  License as published by the Free Software Foundation; either         **
**  version 2 of the License, or (at your option) any later version.     **
**                                                                       **
**  This library is distributed in the hope that it will be useful,      **
**  but WITHOUT ANY WARRANTY; without even the implied warranty of       **
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    **
**  Library General Public License for more details.                     **
**                                                                       **
**  You should have received a copy of the GNU Library General Public    **
**  License along with this library; if not, write to the Free           **
**  Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.   **
**                                                                       **
** Please mail any bug reports/fixes/enhancements to me at:              **
**      drezet@math.jussieu.fr                                           **
** or                                                                    **
**      Jean-Marc Drezet                                                 **
**      Institut de Mathematiques                                        **
**      Aile 45-55                                                       **
**      2, place Jussieu                                                 **
**      75251 Paris Cedex 05                                             **
**      France                                                           **
**                                                                       **
 *************************************************************************/

#include "interp.h"


/*--------------------------------------------------------------------
    Creation of an array with 'nbdim' dimensions d[0],..., of type
    'type'. The function returns the address of the associated
    'memm' structure.
     type = 0   : integers
            1   : real numbers in simple precision
            2   : real numbers in double precision
            3   : complex numbers in usual coordinates in simple
                  precision
            4   : complex numbers in usual coordinates in double
                  precision
            5   : complex numbers in polar coordinates in simple
                  precision
            6   : complex numbers in polar coordinates in double
                  precision
--------------------------------------------------------------------*/
memm
*memm_alloc(int* d, int nbdim, int type)
{
    int             i,
                    j,
                    p,
                    size,
                   *xi;
    float          *xf;
    double         *xd;
    fcomplex       *xfc;
    dcomplex       *xfd;
    fpolaire       *xfp;
    dpolaire       *xdp;
    memm           *M;
    char         ***e,
                  **c,
                   *g;

    M = (memm *) malloc((size_t) nbdim * sizeof(memm));
    size = 0;

    for (i = 0; i < nbdim; i++) {
        M[i].type = type;
        M[i].dim = d[i];
        M[i].prof = nbdim;
    }

    p = d[0] + 1;
    if (nbdim > 1) {
        e = (char ***) malloc((size_t) (p + 2) * sizeof(char **));
        M[0].ad = e + 1;
        c = (char **) e;
        g = (char *) M;
        c[0] = g;
    }

    for (i = 1; i < nbdim - 1; i++) {
        p *= d[i] + 1;
        (M[i - 1].ad)[0] = (char **) malloc((size_t) (p + 1) * sizeof(char *));
        M[i].ad = (char ***) (M[i - 1].ad)[0];
        for (j = 1; j <= p/(d[i] + 1); j++)
            (M[i - 1].ad)[j] = (M[i - 1].ad)[j - 1] + d[i] + 1;
    }

    switch(type) {
        case 0 :
            size = sizeof(int);
            break;
        case 1 :
            size = sizeof(float);
            break;
        case 2 :
            size = sizeof(double);
            break;
        case 3 :
            size = sizeof(fcomplex);
            break;
        case 4 :
            size = sizeof(dcomplex);
            break;
        case 5 :
            size = sizeof(fpolaire);
            break;
        case 6 :
            size = sizeof(dpolaire);
            break;
    }

    i = nbdim - 1;
    if (i > 0) {
        p *= d[i] + 1;
        (M[i - 1].ad)[0] = (char **) malloc((size_t) p * size);
        M[i].ad = (char ***) (M[i - 1].ad)[0];
        size /= sizeof(char **);
        for (j = 1; j <= p/(d[i] + 1); j++)
            (M[i - 1].ad)[j] = (M[i - 1].ad)[j - 1] + (d[i] + 1) * size;
    }

    if (nbdim == 1) {
        e = (char ***) malloc((size_t) (p + 2) * size);
        size /= sizeof(char **);
        M[0].ad = e + size;
        c = (char **) e;
        g = (char *) M;
        c[size - 1] = g;
    }

    switch(type) {
        case 0 :
            xi = (int *) M[i].ad;
            for (i = 0; i < p; i++)
                xi[i] = 0;
            break;
        case 1 :
            xf = (float *) M[i].ad;
            for (i = 0; i < p; i++)
                xf[i] = 0.0;
            break;
        case 2 :
            xd = (double *) M[i].ad;
            for (i = 0; i < p; i++)
                xd[i] = 0.0;
            break;
        case 3 :
            xfc = (fcomplex *) M[i].ad;
            for (i = 0; i < p; i++)
                xfc[i] = Complex(0.0, 0.0);
            break;
        case 4 :
            xfd = (dcomplex *) M[i].ad;
            for (i = 0; i < p; i++)
                xfd[i] = dComplex(0.0, 0.0);
            break;
        case 5 :
            xfp = (fpolaire *) M[i].ad;
            for (i = 0; i < p; i++)
                xfp[i] = Polaire(0.0, 0.0);
            break;
        case 6 :
            xdp = (dpolaire *) M[i].ad;
            for (i = 0; i < p; i++)
                xdp[i] = dPolaire(0.0, 0.0);
            break;
    }

    return M;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Returns the address of the 'memm' structure corresponding to the
    array whose address is 'c'
--------------------------------------------------------------------*/
memm *
_M(char **c)
{
    char           *d;

    d = c[-1];
    return (memm *) d;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Returns the address of the array corresponding to the 'memm'
    structure 'M'. Example : if 'M' corresponds to the 3-dimensional
    array A[i][j][k], the function returns A.
--------------------------------------------------------------------*/
char *
addr_eff(memm *M)
{
    return (char *) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Returns the address of the data stored in the array coressponding
    to the 'memm' structure M. Example : f 'M' corresponds to the
    3-dimensional array A[i][j][k], the function returns A[0][0].
--------------------------------------------------------------------*/
char *
addr_eff_b(memm *M)
{
    return (char *) M[M[0].prof - 1].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Returns the number of values stored in the array corresponding
    to 'M'
--------------------------------------------------------------------*/
int
longueur(memm *M)
{
    int             p,
                    i;

    p = 1;
    for (i = 0; i < M[0].prof; i++)
        p *= (M[i].dim + 1);
    return p;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Frees the memory occupied by the 'memm' structure 'M'
--------------------------------------------------------------------*/
void
Xfree(memm *M)
{
    int             i,
                    size;
    char         ***c;

    size = 0;
    for (i = M[0].prof - 1; i >= 1; i--)
        free((char *) M[i].ad);
    if (M[0].prof > 1)
        c = M[0].ad - 1;
    else {
        switch (M[0].type) {
            case 0 :
                size = sizeof(int);
                break;
            case 1 :
                size = sizeof(float);
                break;
            case 2 :
                size = sizeof(double);
                break;
            case 3 :
                size = sizeof(fcomplex);
                break;
            case 4 :
                size = sizeof(dcomplex);
                break;
            case 5 :
                size = sizeof(fpolaire);
                break;
            case 6 :
                size = sizeof(dpolaire);
                break;
        }

        c = M[0].ad - size / sizeof(char *);
    }
    free(c);
    free(M);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Frees the memory occupied by the 'memm' structure corresponding
    to the array whose address is 'c'. See the macro XFREE in
    interp.h
--------------------------------------------------------------------*/
void
Xfree_b(char **c)
{
    Xfree((memm *)(c[-1]));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Creates an array of integers with 1 dimension d1
--------------------------------------------------------------------*/
int *
int_alloc1(int d1)
{
    memm           *M;

    M = memm_alloc(&d1, 1, 0);
    return (int *) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Creates an array of integers with 2 dimensions d1, d2
--------------------------------------------------------------------*/
int **
int_alloc2(int d1, int d2)
{
    int             d[2];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    M = memm_alloc(d, 2, 0);
    return (int **) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Creates an array of integers with 3 dimensions d1, d2, d3
--------------------------------------------------------------------*/
int ***
int_alloc3(int d1, int d2, int d3)
{
    int             d[3];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    M = memm_alloc(d, 3, 0);
    return (int ***) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Creates an array of integers with 4 dimensions d1, d2, d3, d4
--------------------------------------------------------------------*/
int ****
int_alloc4(int d1, int d2, int d3, int d4)
{
    int             d[4];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    d[3] = d4;
    M = memm_alloc(d, 4, 0);
    return (int ****) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision real numbers with 1
     dimension d1
--------------------------------------------------------------------*/
float *
float_alloc1(int d1)
{
    memm           *M;

    M = memm_alloc(&d1, 1, 1);
    return (float *) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision real numbers with 2
     dimensions d1, d2
--------------------------------------------------------------------*/
float **
float_alloc2(int d1, int d2)
{
    int             d[2];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    M = memm_alloc(d, 2, 1);
    return (float **) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision real numbers with 3
     dimensions d1, d2, d3
--------------------------------------------------------------------*/
float ***
float_alloc3(int d1, int d2, int d3)
{
    int             d[3];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    M = memm_alloc(d, 3, 1);
    return (float ***) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision real numbers with 4
     dimensions d1, d2, d3, d4
--------------------------------------------------------------------*/
float ****
float_alloc4(int d1, int d2, int d3, int d4)
{
    int             d[4];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    d[3] = d4;
    M = memm_alloc(d, 4, 1);
    return (float ****) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision real numbers with 1
     dimension d1
--------------------------------------------------------------------*/
double *
double_alloc1(int d1)
{
    memm           *M;

    M = memm_alloc(&d1, 1, 2);
    return (double *) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision real numbers with 2
     dimensions d1, d2
--------------------------------------------------------------------*/
double **
double_alloc2(int d1, int d2)
{
    int             d[2];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    M = memm_alloc(d, 2, 2);
    return (double **) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision real numbers with 3
     dimensions d1, d2, d3
--------------------------------------------------------------------*/
double ***
double_alloc3(int d1, int d2, int d3)
{
    int             d[3];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    M = memm_alloc(d, 3, 2);
    return (double ***) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision real numbers with 4
     dimensions d1, d2, d3, d4
--------------------------------------------------------------------*/
double ****
double_alloc4(int d1, int d2, int d3, int d4)
{
    int             d[4];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    d[3] = d4;
    M = memm_alloc(d, 4, 2);
    return (double ****) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision complex numbers in usual
     coordinates with 1 dimension d1
--------------------------------------------------------------------*/
fcomplex *
fcomplex_alloc1(int d1)
{
    memm           *M;

    M = memm_alloc(&d1, 1, 3);
    return (fcomplex *) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision complex numbers in usual
     coordinates with 2 dimensions d1, d2
--------------------------------------------------------------------*/
fcomplex **
fcomplex_alloc2(int d1, int d2)
{
    int             d[2];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    M = memm_alloc(d, 2, 3);
    return (fcomplex **) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision complex numbers in usual
     coordinates with 3 dimensions d1, d2, d3
--------------------------------------------------------------------*/
fcomplex ***
fcomplex_alloc3(int d1, int d2, int d3)
{
    int             d[3];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    M = memm_alloc(d, 3, 3);
    return (fcomplex ***) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision complex numbers in usual
     coordinates with 4 dimensions d1, d2, d3, d4
--------------------------------------------------------------------*/
fcomplex ****
fcomplex_alloc4(int d1, int d2, int d3, int d4)
{
    int             d[4];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    d[3] = d4;
    M = memm_alloc(d, 4, 3);
    return (fcomplex ****) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision complex numbers in usual
     coordinates with 1 dimension d1
--------------------------------------------------------------------*/
dcomplex *
dcomplex_alloc1(int d1)
{
    memm           *M;

    M = memm_alloc(&d1, 1, 4);
    return (dcomplex *) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision complex numbers in usual
     coordinates with 2 dimensions d1, d2
--------------------------------------------------------------------*/
dcomplex **
dcomplex_alloc2(int d1, int d2)
{
    int             d[2];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    M = memm_alloc(d, 2, 4);
    return (dcomplex **) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision complex numbers in usual
     coordinates with 3 dimensions d1, d2, d3
--------------------------------------------------------------------*/
dcomplex ***
dcomplex_alloc3(int d1, int d2, int d3)
{
    int             d[3];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    M = memm_alloc(d, 3, 4);
    return (dcomplex ***) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision complex numbers in usual
     coordinates with 4 dimensions d1, d2, d3, d4
--------------------------------------------------------------------*/
dcomplex ****
dcomplex_alloc4(int d1, int d2, int d3, int d4)
{
    int             d[4];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    d[3] = d4;
    M = memm_alloc(d, 4, 4);
    return (dcomplex ****) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision complex numbers in polar
     coordinates with 1 dimension d1
--------------------------------------------------------------------*/
fpolaire *
fpolaire_alloc1(int d1)
{
    memm           *M;

    M = memm_alloc(&d1, 1, 5);
    return (fpolaire *) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision complex numbers in polar
     coordinates with 2 dimensions d1, d2
--------------------------------------------------------------------*/
fpolaire **
fpolaire_alloc2(int d1, int d2)
{
    int             d[2];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    M = memm_alloc(d, 2, 5);
    return (fpolaire **) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision complex numbers in polar
     coordinates with 3 dimensions d1, d2, d3
--------------------------------------------------------------------*/
fpolaire ***
fpolaire_alloc3(int d1, int d2, int d3)
{
    int             d[3];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    M = memm_alloc(d, 3, 5);
    return (fpolaire ***) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of simple precision complex numbers in polar
     coordinates with 4 dimensions d1, d2, d3, d4
--------------------------------------------------------------------*/
fpolaire ****
fpolaire_alloc4(int d1, int d2, int d3, int d4)
{
    int             d[4];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    d[3] = d4;
    M = memm_alloc(d, 4, 5);
    return (fpolaire ****) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision complex numbers in polar
     coordinates with 1 dimension d1
--------------------------------------------------------------------*/
dpolaire *
dpolaire_alloc1(int d1)
{
    memm           *M;

    M = memm_alloc(&d1, 1, 6);
    return (dpolaire *) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision complex numbers in polar
     coordinates with 2 dimensions d1, d2
--------------------------------------------------------------------*/
dpolaire **
dpolaire_alloc2(int d1, int d2)
{
    int             d[2];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    M = memm_alloc(d, 2, 6);
    return (dpolaire **) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision complex numbers in polar
     coordinates with 3 dimensions d1, d2, d3
--------------------------------------------------------------------*/
dpolaire ***
dpolaire_alloc3(int d1, int d2, int d3)
{
    int             d[3];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    M = memm_alloc(d, 3, 6);
    return (dpolaire ***) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
     Creates an array of double precision complex numbers in polar
     coordinates with 4 dimensions d1, d2, d3, d4
--------------------------------------------------------------------*/
dpolaire ****
dpolaire_alloc4(int d1, int d2, int d3, int d4)
{
    int             d[4];
    memm           *M;

    d[0] = d1;
    d[1] = d2;
    d[2] = d3;
    d[3] = d4;
    M = memm_alloc(d, 4, 6);
    return (dpolaire ****) M[0].ad;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Function that returns 0 if the two 'memm' structures c1, c2
    are not well defined or of different type or dimensions, and
    1 otherwise
--------------------------------------------------------------------*/
int
compare_tab(memm *c1, memm *c2)
{
    if (compare_dim(c1, c2) == 0)
        return 0;
    if (c1[0].type != c2[0].type)
        return 0;
    return 1;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Function that returns 0 if the two 'memm' structures c1, c2
    have the same dimensions, and 0 otherwise
--------------------------------------------------------------------*/
int
compare_dim(memm *c1, memm *c2)
{
    int             i;

    if (c1 == NULL || c2 == NULL)
        return 0;
    if (c1[0].prof != c2[0].prof)
        return 0;
    for (i = 0; i < c1[0].prof; i++)
        if (c1[i].dim != c2[i].dim)
            return 0;
    return 1;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Function performing operations on arrays
    cas =  0 :   (c1, c2)  -------------------------> c3
    cas =  1 :   (c1, real number) -----------------> c2
    cas =  2 :   c1 --------------------------------> c2
    cas =  3 :   (complex c1, real number) ---------> complex c2
    cas =  4 :   (complex c1, complex number) ------> complex c2
    cas =  5 :   real number -----------------------> c1
    cas =  6 :   nothing
    cas =  7 :   complex number --------------------> complex c1
    cas =  8 :   complex c1 ------------------------> complex c1
    cas =  9 :   (real c1, real c2) ----------------> complex c3
    cas = 10 :   complex c1 ------------------------> real c2
--------------------------------------------------------------------*/
void
manip1_tab(memm *c1, memm *c2, memm *c3, T_fonc *Tw, RT_fonc *RTw,
    T_foncC *TCw, T_fonc1 *T1w, T_foncX *TXw, T_foncRC *TRCw,
    T_foncCR *TCRw, T_foncXd *TXdw, double x, dcomplex z ,int cas, CC_fonc *F)
{
    int             len,
                    i,
                   *ai1,
                   *ai2,
                   *ai3;
    float          *af1,
                   *af2,
                   *af3;
    double         *ad1,
                   *ad2,
                   *ad3;
    fcomplex       *afc1,
                   *afc2,
                   *afc3;
    dcomplex       *adc1,
                   *adc2,
                   *adc3;
    fpolaire       *afp1,
                   *afp2,
                   *afp3;
    dpolaire       *adp1,
                   *adp2,
                   *adp3;

    if (cas == 0)
        if (compare_tab(c1, c2) == 0 || compare_tab(c1, c3) == 0)
            goto xxx;
    if ((cas >= 1 && cas <= 4) || cas == 8) {
        if (compare_tab(c1, c2) == 0)
            goto xxx;
    }
    if (cas == 9) {
        if (compare_tab(c1, c2) == 0)
            goto xxx;
        if (c1[0].type != 1 && c1[0].type != 2)
            goto xxx;
        if (compare_dim(c1, c3) == 0)
            goto xxx;
        if (c1[0].type == 1 && c3[0].type != 3 && c3[0].type != 5)
            goto xxx;
        if (c1[0].type == 2 && c3[0].type != 4 && c3[0].type != 6)
            goto xxx;
    }
    if (cas == 10) {
        if (compare_dim(c1, c2) == 0)
            goto xxx;
        if (c1[0].type < 3)
            goto xxx;
        if ((c1[0].type == 3 || c1[0].type == 5) && c2[0].type != 1)
            goto xxx;
        if ((c1[0].type == 4 || c1[0].type == 6) && c2[0].type != 2)
            goto xxx;
    }

    len = longueur(c1);

    switch(c1[0].type) {
/*------------------------------------------------------
    Case of integer variables
-------------------------------------------------------*/
        case  0 :
            ai1 = (int *) addr_eff_b(c1);

            switch(cas)  {
                case 0 :
                    ai2 = (int *) addr_eff_b(c2);
                    ai3 = (int *) addr_eff_b(c3);
                    for (i = 0; i < len; i++)
                        ai3[i] = (*Tw).F_i(ai2[i], ai1[i]);
                    break;

                case 1 :
                    ai2 = (int *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        ai2[i] = (*RTw).F_i(x, ai1[i]);
                    break;

                case 2 :
                    ai2 = (int *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        ai2[i] = (*T1w).F_i(ai1[i]);
                    break;

                case 5 :
                    for (i = 0; i < len; i++)
                        ai1[i] = (*TXw).F_i(x);
                    break;
            }

            break;
/*----------------------------------------------------*/


/*------------------------------------------------------
    Case of simple precision real variables
-------------------------------------------------------*/
        case 1 :
            af1 = (float *) addr_eff_b(c1);

            switch(cas) {
                case 0 :
                    af2 = (float *) addr_eff_b(c2);
                    af3 = (float *) addr_eff_b(c3);
                    for (i = 0; i < len; i++)
                        af3[i] = (float) (*Tw).F_d(af2[i], af1[i]);
                    break;

                case 1 :
                    af2 = (float *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        af2[i] = (float) (*RTw).F_d(x, af1[i]);
                    break;

                case 2 :
                    af2 = (float *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        af2[i] = (float) (*T1w).F_d(af1[i]);
                    break;

                case 5 :
                    for (i = 0; i < len; i++)
                        af1[i] = (float) (*TXw).F_d(x);
                    break;

                case 9 :
                    af1 = (float *) addr_eff_b(c1);
                    af2 = (float *) addr_eff_b(c2);
                    if (c3[0].type == 3) {
                        afc1 = (fcomplex *) addr_eff_b(c3);
                        for (i = 0; i < len; i++)
                            afc1[i] = (*TCRw).F_fc((double) af1[i],
                    (double) af2[i]);
                    }
                    else {
                       afp1 = (fpolaire *) addr_eff_b(c3);
                        for (i = 0; i < len; i++)
                            afp1[i] = (*TCRw).F_fp((double) af1[i],
                    (double) af2[i]);
                    }
                    break;
            }

            break;
/*----------------------------------------------------*/


/*------------------------------------------------------
    Case of double precision real variables
-------------------------------------------------------*/
        case 2 :
            ad1 = (double *) addr_eff_b(c1);

            switch(cas) {
                case 0 :
                    ad2 = (double *) addr_eff_b(c2);
                    ad3 = (double *) addr_eff_b(c3);
                    for (i = 0; i < len; i++)
                        ad3[i] = (*Tw).F_d(ad2[i], ad1[i]);
                    break;

                case 1 :
                    ad2 = (double *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        ad2[i] = (*RTw).F_d(x, ad1[i]);
                    break;

                case  2 :
                    ad2 = (double *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        ad2[i] = (*T1w).F_d(ad1[i]);
                    break;

                case  5 :
                    for (i = 0; i < len; i++)
                        ad1[i] = (*TXw).F_d(x);
                    break;

                case 9 :
                    ad1 = (double *) addr_eff_b(c1);
                    ad2 = (double *) addr_eff_b(c2);
                    if (c3[0].type == 4) {
                        adc1 = (dcomplex *) addr_eff_b(c3);
                        for (i = 0; i < len; i++)
                            adc1[i] = (*TCRw).F_dc(ad1[i], ad2[i]);
                    }
                    else {
                        adp1 = (dpolaire *) addr_eff_b(c3);
                        for (i = 0; i < len; i++)
                            adp1[i] = (*TCRw).F_dp(ad1[i], ad2[i]);
                    }
                    break;

            }

            break;
/*----------------------------------------------------*/


/*------------------------------------------------------
    Case of simple precision complex variables
    in usual coordinates
-------------------------------------------------------*/
        case 3 :
            afc1 = (fcomplex *) addr_eff_b(c1);

            switch(cas) {
                case 0 :
                    afc2 = (fcomplex *) addr_eff_b(c2);
                    afc3 = (fcomplex *) addr_eff_b(c3);
                    for (i = 0; i < len; i++)
                        afc3[i] = (*Tw).F_fc(afc2[i], afc1[i]);
                    break;

                case 1 :
                    afc2 = (fcomplex *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        afc2[i] = (*RTw).F_fc(x, afc1[i]);
                    break;

                case 2 :
                    afc2 = (fcomplex *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        afc2[i] = (*T1w).F_fc(afc1[i]);
                    break;

                case 3 :
                    afc2 = (fcomplex *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        afc2[i] = (*TRCw).F_fc(x, afc1[i]);
                    break;

                case 4 :
                    afc2 = (fcomplex *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        afc2[i] = (*TCw).F_fc(z, afc1[i]);
                    break;

                case 5 :
                    for (i = 0; i < len; i++)
                        afc1[i] = (*TXw).F_fc(x);
                    break;

                case 7 :
                    for (i = 0; i < len; i++)
                        afc1[i] = (*TXdw).F_fc(z);
                    break;

                case 8 :
                    for (i = 0; i < len; i++)
                        afc1[i] = (*T1w).F_fc(afc1[i]);
                    break;

                case 10 :
                    af2 = (float *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        af2[i] = (float) (*F).CF_d(dComplex(
                            (double) afc1[i].r, (double) afc1[i].i));
                    break;
            }

        break;
/*----------------------------------------------------*/


/*------------------------------------------------------
    Case of double precision complex variables
    in usual coordinates
-------------------------------------------------------*/
        case 4 :
            adc1 = (dcomplex *) addr_eff_b(c1);

            switch(cas) {
                case 0 :
                    adc2 = (dcomplex *) addr_eff_b(c2);
                    adc3 = (dcomplex *) addr_eff_b(c3);
                    for (i = 0; i < len; i++)
                        adc3[i] = (*Tw).F_dc(adc2[i], adc1[i]);
                    break;

                case 1 :
                    adc2 = (dcomplex *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        adc2[i] = (*RTw).F_dc(x, adc1[i]);
                    break;

                case 2 :
                    adc2 = (dcomplex *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        adc2[i] = (*T1w).F_dc(adc1[i]);
                    break;

                case 3 :
                    adc2 = (dcomplex *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        adc2[i] = (*TRCw).F_dc(x, adc1[i]);
                    break;

                case 4 :
                    adc2 = (dcomplex *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        adc2[i] = (*TCw).F_dc(z, adc1[i]);
                    break;

                case 5 :
                    for (i = 0; i < len; i++)
                        adc1[i] = (*TXw).F_dc(x);
                    break;

                case 7 :
                    for (i = 0; i < len; i++)
                        adc1[i] = (*TXdw).F_dc(z);
                    break;

                case 8 :
                    for (i = 0; i < len; i++)
                        adc1[i] = (*T1w).F_dc(adc1[i]);
                    break;

                case 10 :
                    ad2 = (double *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        ad2[i] = (*F).CF_d(adc1[i]);
                    break;
            }

        break;
/*----------------------------------------------------*/


/*------------------------------------------------------
    Case of simple precision complex variables
    in polar coordinates
-------------------------------------------------------*/
        case 5 :
            afp1 = (fpolaire *) addr_eff_b(c1);

            switch(cas) {
                case 0 :
                    afp2 = (fpolaire *) addr_eff_b(c2);
                    afp3 = (fpolaire *) addr_eff_b(c3);
                    for (i = 0; i < len; i++)
                        afp3[i] = (*Tw).F_fp(afp2[i], afp1[i]);
                    break;

                case 1 :
                    afp2 = (fpolaire *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        afp2[i] = (*RTw).F_fp(x, afp1[i]);
                    break;

                case 2 :
                    afp2 = (fpolaire *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        afp2[i] = (*T1w).F_fp(afp1[i]);
                    break;

                case 3 :
                    afp2 = (fpolaire *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        afp2[i] = (*TRCw).F_fp(x, afp1[i]);
                    break;

                case 4 :
                    afp2 = (fpolaire *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        afp2[i] = (*TCw).F_fp(z, afp1[i]);
                    break;

                case 5 :
                    for (i = 0; i < len; i++)
                        afp1[i] = (*TXw).F_fp(x);
                    break;

                case 7 :
                    for (i = 0; i < len; i++)
                        afp1[i] = (*TXdw).F_fp(z);
                    break;

                case 8 :
                    for (i = 0; i < len; i++)
                        afp1[i] = (*T1w).F_fp(afp1[i]);
                    break;

                case 9 :
                    af1 = (float *) addr_eff_b(c1);
                    af2 = (float *) addr_eff_b(c2);
                    afp1 = (fpolaire *) addr_eff_b(c3);
                    for (i = 0; i < len; i++)
                        afp1[i] = (*TCRw).F_fp(af1[i], af2[i]);
                    break;

                case 10 :
                    af2 = (float *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        af2[i] = (float) (*F).PF_d(dPolaire(
                                (double) afp1[i].rm, (double) afp1[i].th));
                    break;
            }

        break;
/*----------------------------------------------------*/


/*------------------------------------------------------
    Case of double precision complex variables
    in polar coordinates
-------------------------------------------------------*/
        case 6 :
            adp1 = (dpolaire *) addr_eff_b(c1);

            switch(cas) {
                case 0 :
                    adp2 = (dpolaire *) addr_eff_b(c2);
                    adp3 = (dpolaire *) addr_eff_b(c3);
                    for (i = 0; i < len; i++)
                        adp3[i] = (*Tw).F_dp(adp2[i], adp1[i]);
                    break;

                case 1 :
                    adp2 = (dpolaire *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        adp2[i] = (*RTw).F_dp(x, adp1[i]);
                    break;

                case 2 :
                    adp2 = (dpolaire *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        adp2[i] = (*T1w).F_dp(adp1[i]);
                    break;

                case 3 :
                    adp2 = (dpolaire *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        adp2[i] = (*TRCw).F_dp(x, adp1[i]);
                    break;

                case 4 :
                    adp2 = (dpolaire *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        adp2[i] = (*TCw).F_dp(z, adp1[i]);
                    break;

                case 5 :
                    for (i = 0; i < len; i++)
                        adp1[i] = (*TXw).F_dp(x);
                    break;

                case 7 :
                    for (i = 0; i < len; i++)
                        adp1[i] = (*TXdw).F_dp(z);
                    break;

                case 8 :
                    for (i = 0; i < len; i++)
                        adp1[i] = (*T1w).F_dp(adp1[i]);
                    break;

                case 9 :
                    ad1 = (double *) addr_eff_b(c1);
                    ad2 = (double *) addr_eff_b(c2);
                    adp1 = (dpolaire *) addr_eff_b(c3);
                    for (i = 0; i < len; i++)
                        adp1[i] = (*TCRw).F_dp(ad1[i], ad2[i]);
                    break;

                case 10 :
                    ad2 = (double *) addr_eff_b(c2);
                    for (i = 0; i < len; i++)
                        ad2[i] = (*F).PF_d(adp1[i]);
                    break;
            }

        break;
/*----------------------------------------------------*/
    }

    return;

#ifdef _ENG_LANG
xxx: printf("Incorrect arrays\n");
#else
xxx: printf("Tableaux incorrects\n");
#endif
    return;
}

void
ajoute_tab(memm *c1, memm *c2, memm *c3)
{
    manip1_tab(c1, c2, c3,
        &T_add, NULL, NULL, NULL,
        NULL, NULL, NULL, NULL,
        0.0, dComplex(0.0, 0.0), 0, NULL);
}

void
soustrait_tab(memm *c1, memm *c2, memm *c3)
{
    manip1_tab(c1, c2, c3,
        &T_sub, NULL, NULL, NULL,
        NULL, NULL, NULL, NULL,
        0.0, dComplex(0.0, 0.0), 0, NULL);
}

void
multiplie_tab(memm *c1, memm *c2, memm *c3)
{
    manip1_tab(c1, c2, c3,
        &T_mult,NULL, NULL, NULL,
        NULL, NULL, NULL, NULL,
        0.0, dComplex(0.0, 0.0), 0, NULL);
}

void
copie_tab(memm *c1, memm *c2)
{
    manip1_tab(c1, c2, NULL,
        NULL, NULL, NULL, &T_id,
        NULL, NULL, NULL, NULL,
        0.0, dComplex(0.0, 0.0), 2, NULL);
}

void
Rmul_tab(memm *c1, double x, memm *c2)
{
    manip1_tab(c1, c2, NULL,
        NULL, &T_Rmul, NULL, NULL,
        NULL, NULL, NULL, NULL,
        x, dComplex(0.0, 0.0), 1, NULL);
}

void
Cmul_tab(memm *c1, dcomplex z, memm *c2)
{
    manip1_tab(c1, c2, NULL,
        NULL, NULL, &C_mul,  NULL,
        NULL, NULL, NULL, NULL,
        0.0, z, 4, NULL);
}

void
Conjg_tab(memm *c1, memm *c2)
{
    manip1_tab(c1, c2, NULL,
        NULL, NULL, NULL, &C_conjg,
        NULL, NULL, NULL, NULL,
        0.0, dComplex(0.0, 0.0), 8, NULL);
}

void
Module_tab(memm *c1, memm *c2)
{
    manip1_tab(c1, c2, NULL,
        NULL, NULL, NULL, NULL,
        NULL, NULL, NULL, NULL,
        0.0, dComplex(0.0, 0.0), 10, &CC_R);
}

void
Phase_tab(memm *c1, memm *c2)
{
    manip1_tab(c1, c2, NULL,
        NULL, NULL, NULL, NULL,
        NULL, NULL, NULL, NULL,
        0.0, dComplex(0.0, 0.0), 10, &CC_Th);
}

void
reel_tab(memm *c1, memm *c2)
{
    manip1_tab(c1, c2, NULL,
        NULL, NULL, NULL, NULL,
        NULL, NULL, NULL, NULL,
        0.0, dComplex(0.0, 0.0), 10, &CC_Reel);
}

void
imag_tab(memm *c1, memm *c2)
{
    manip1_tab(c1, c2, NULL,
        NULL, NULL, NULL, NULL,
        NULL, NULL, NULL, NULL,
        0.0, dComplex(0.0, 0.0), 10, &CC_Imag);
}

void
Polaire_tab(memm *c1, memm *c2, memm *c3)
{
    manip1_tab(c1, c2, c3,
        NULL, NULL, NULL, NULL,
        NULL, NULL, &C_Polaire,  NULL,
        0.0, dComplex(0.0, 0.0), 9, NULL);
}

void
Complex_tab(memm *c1, memm *c2, memm *c3)
{
    manip1_tab(c1, c2, c3,
        NULL, NULL, NULL, NULL,
        NULL, NULL, &C_Complex, NULL,
        0.0, dComplex(0.0, 0.0), 9, NULL);
}

void
cnst_tab(memm *c1, double x)
{
    manip1_tab(c1, NULL, NULL,
        NULL, NULL, NULL, NULL,
        &R_cnst, NULL, NULL, NULL,
        x, dComplex(0.0, 0.0), 5, NULL);
}

void
Ccnst_tab(memm *c1, dcomplex z)
{
    manip1_tab(c1, NULL, NULL,
        NULL, NULL, NULL, NULL,
        NULL, NULL, NULL, &C_cnst,
        0.0, z, 7, NULL);
}

void
RPolcnst_tab(memm *c1, double x)
{
    manip1_tab(c1, c1, NULL,
        NULL, NULL, NULL, NULL,
        NULL, &Rpol_cnst, NULL, NULL,
        x, dComplex(0.0, 0.0), 3, NULL);
}

void
ThPolcnst_tab(memm *c1, double x)
{
    manip1_tab(c1, c1, NULL,
        NULL, NULL, NULL, NULL,
        NULL, &Thpol_cnst, NULL, NULL,
        x, dComplex(0.0, 0.0), 3, NULL);
}


/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    If cas=0 this function stores the array corresponding to M
    in file s, and otherwise reads it. s must have been opened
    before
--------------------------------------------------------------------*/
size_t
fwriteB(void * c, size_t i, size_t j, FILE *s)
{
    return fwrite(c, i, j, s);
}
/*------------------------------------------------------------------*/





/*--------------------------------------------------------------------
    If cas=0 this function stores the array corresponding to M
    in file s, and otherwise reads it. s must have been opened
    before
--------------------------------------------------------------------*/
int
svg_tab(FILE *s, char *c, int cas)
{
    int             len,
                    i,
                    j,
                   *ix;
    float          *fx;
    double         *dx;
    fcomplex       *fcx;
    dcomplex       *dcx;
    fpolaire       *fpx;
    dpolaire       *dpx;
    memm           *M;
    size_t        (*F)(void *, size_t, size_t, FILE *);

    if (cas == 0)
        F = fwriteB;
    else
        F = fread;
    M = _M((char **) c);
    len = longueur(M);
    if (cas == 0) {
        F(&M[0].type, sizeof(int), 1, s);
        F(&M[0].prof, sizeof(int), 1, s);
        for (i = 0; i < M[0].prof; i++)
            F(&M[i].dim, sizeof(int), 1, s);
    }
    else {
        i = 0;
        F(&i, sizeof(int), 1, s);
        if (i != M[0].type) {
#ifdef _ENG_LANG
            printf("Incorrect stored array\n");
#else
            printf("Tableau enregistre incorrect\n");
#endif
            return 0;
        }
        F(&i, sizeof(int), 1, s);
        if (i != M[0].prof) {
#ifdef _ENG_LANG
            printf("Incorrect stored array\n");
#else
            printf("Tableau enregistre incorrect\n");
#endif
            return 0;
        }

        for (i = 0; i < M[0].prof; i++) {
            F(&j, sizeof(int), 1, s);
            if (j != M[i].dim)  {
#ifdef _ENG_LANG
            printf("Incorrect stored array\n");
#else
            printf("Tableau enregistre incorrect\n");
#endif
            return 0;
            }
        }
    }


    switch(M[0].type) {
        case 0 :
            ix = (int *) addr_eff_b(M);
            for (i = 0; i < len; i++)
                F(&ix[i], sizeof(int), 1, s);
            break;
        case 1 :
            fx = (float *) addr_eff_b(M);
            for (i = 0; i < len; i++)
                F(&fx[i], sizeof(float), 1, s);
            break;
        case 2 :
            dx = (double *) addr_eff_b(M);
            for (i = 0; i < len; i++)
                F(&dx[i], sizeof(double), 1, s);
            break;
        case 3 :
            fcx = (fcomplex *) addr_eff_b(M);
            for (i = 0; i < len; i++)
                F(&fcx[i], sizeof(fcomplex), 1, s);
            break;
        case 4 :
            dcx = (dcomplex *) addr_eff_b(M);
            for (i = 0; i < len; i++)
                F(&dcx[i], sizeof(dcomplex), 1, s);
            break;
        case 5 :
            fpx = (fpolaire *) addr_eff_b(M);
            for (i = 0; i < len; i++)
                F(&fpx[i], sizeof(fpolaire), 1, s);
            break;
        case 6 :
            dpx = (dpolaire *) addr_eff_b(M);
            for (i = 0; i < len; i++)
                F(&dpx[i], sizeof(dpolaire), 1, s);
            break;
    }

    return 1;
}
/*-------------------------------------------------------------------*/





/*--------------------------------------------------------------------
 (1) Addition of arrays
     (the four last functions are in complex.c)
--------------------------------------------------------------------*/
T_fonc T_add = {
    iadd,
    dadd,
    Cadd,
    dCadd,
    Padd,
    dPadd
};

int
iadd(int a, int b)
{
    return (a + b);
}

double
dadd(double x, double y)
{
    return (x + y);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (2) Substraction of arrays
     (the four last functions are in complex.c)
--------------------------------------------------------------------*/
T_fonc T_sub = {
    isub,
    dsub,
    Csub,
    dCsub,
    Psub,
    dPsub
};

int
isub(int a, int b)
{
    return (a - b);
}

double
dsub(double x, double y)
{
    return (x - y);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (3) Multiplication of arrays by real numbers
--------------------------------------------------------------------*/
RT_fonc T_Rmul = {
    i_Rmul,
    d_mult,
    RCmulb,
    dRCmul,   /* in complex.c */
    RPmulb,
    dRPmul    /* in complex.c */
};

int
i_Rmul(double x, int a)
{
    return (a * (int) x);
}

double
d_mult(double x, double y)
{
    return (x * y);
}

fcomplex
RCmulb(double x, fcomplex u)
{
    return RCmul((float) x, u);
}

fpolaire
RPmulb(double x, fpolaire u)
{
    return RPmul((float) x, u);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (4) Multiplication of two arrays
     (the four last functions are in complex.c, d_mult is in (3))
--------------------------------------------------------------------*/
T_fonc T_mult = {
    i_mult,
    d_mult,
    Cmul,
    dCmul,
    Pmul,
    dPmul
};

int
i_mult(int a, int b)
{
    return (a * b);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (5) Inversion of an array
     (the last four functions are in complex.c)
--------------------------------------------------------------------*/
T_fonc1 T_inv = {
    i_inv,
    d_inv,
    Cinv,
    xdCinv,
    Pinv,
    dPinv
};

int
i_inv(int i)
{
    if (i == 0)
        return 0;
    else
        return (1 / i);
}

double
d_inv(double x)
{
    if (x == 0.0)
        return 0.0;
    else
        return (1. / x);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (6) Constant functions (real)
--------------------------------------------------------------------*/
T_foncX R_cnst = {
    i_cnst,
    d_cnst,
    fc_cnst,
    dc_cnst,
    fp_cnst,
    dp_cnst
};

int
i_cnst(double a)
{
    return ((int) a);
}

double
d_cnst(double a)
{
    return a;
}

fcomplex
fc_cnst(double a)
{
    return Complex((float) a, 0.0);
}

dcomplex
dc_cnst(double a)
{
    return dComplex(a, 0.0);
}

fpolaire
fp_cnst(double a)
{
    fpolaire        c;

    c.rm = (float) fabs(a);
    if (a >= 0.)
        c.th = 0.;
    else
        c.th = 3.141593;
    return c;
}

dpolaire
dp_cnst(double a)
{
    dpolaire        c;

    c.rm = fabs(a);
    if (a >= 0.)
        c.th = 0.;
    else
        c.th = 3.141592653589793;
    return c;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (7) Constant functions (complex)
--------------------------------------------------------------------*/
T_foncXd C_cnst =
{
    Ccnst,
    dCcnst,
    Pcnst,
    dPcnst
};

fcomplex
Ccnst(dcomplex a)
{
    return Complex((float)a.r, (float)a.i);
}

dcomplex
dCcnst(dcomplex a)
{
    return a;
}

fpolaire
Pcnst(dcomplex a)
{
    return Conv_ctp(Complex((float)a.r, (float)a.i));
}

dpolaire
dPcnst(dcomplex a)
{
    return dConv_ctp(a);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (8) Identity functions
--------------------------------------------------------------------*/
T_fonc1 T_id =
{
    icnst,
    dcnst,
    Ccnst_id,
    dCcnst,      /* in (7) */
    Pcnst_id,
    dPcnst_id
};

int
icnst(int a)
{
    return a;
}

double
dcnst(double a)
{
    return a;
}

fcomplex
Ccnst_id(fcomplex a)
{
    return a;
}

fpolaire
Pcnst_id(fpolaire a)
{
    return a;
}


dpolaire
dPcnst_id(dpolaire a)
{
    return a;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (9) Multiplication of arrays by a complex number
--------------------------------------------------------------------*/
T_foncC C_mul =
{
    Cmul_x,
    dCmul,
    Pmul_x,
    dPmul_x
};

fcomplex
Cmul_x(dcomplex a, fcomplex b)
{
    return Cmul(Complex((float) a.r, (float) a.i), b);
}

fpolaire
Pmul_x(dcomplex a, fpolaire b)
{
    return Pmul(Conv_ctp(Complex((float) a.r, (float) a.i)), b);
}

dpolaire
dPmul_x(dcomplex a, dpolaire b)
{
    return dPmul(dConv_ctp(a), b);
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (10) fixing the radius of a complex number
--------------------------------------------------------------------*/
T_foncRC Rpol_cnst =
{
    fRPolcnst,
    dRPolcnst,
    PRPolcnst,
    dPRPolcnst
};

fcomplex
fRPolcnst(double r, fcomplex z)
{
    return Conv_ptc(Polaire((float) r, Conv_ctp(z).th));
}

dcomplex
dRPolcnst(double r, dcomplex z)
{
    return dConv_ptc(dPolaire(r, dConv_ctp(z).th));
}

fpolaire
PRPolcnst(double r, fpolaire z)
{
    fpolaire        u;

    u.rm = (float) r;
    u.th = z.th;
    return u;
}

dpolaire
dPRPolcnst(double r, dpolaire z)
{
    dpolaire        u;

    u.rm = r;
    u.th = z.th;
    return u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (11) fixing the phase of a complex numbers
--------------------------------------------------------------------*/
T_foncRC Thpol_cnst =
{
    fThPolcnst,
    dThPolcnst,
    PThPolcnst,
    dPThPolcnst
};

fcomplex
fThPolcnst(double r, fcomplex z)
{
    return Conv_ptc(Polaire(Conv_ctp(z).rm, (float) r));
}

dcomplex
dThPolcnst(double r, dcomplex z)
{
    return dConv_ptc(dPolaire(dConv_ctp(z).rm, r));
}

fpolaire
PThPolcnst(double r, fpolaire z)
{
    fpolaire        u;

    u.th = (float) r;
    u.rm = z.rm;
    return u;
}

dpolaire
dPThPolcnst(double r, dpolaire z)
{
    dpolaire        u;

    u.th = r;
    u.rm = z.rm;
    return u;
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (11) Conjugation of complex numbers
      (the functions are in complex.c)
--------------------------------------------------------------------*/
T_fonc1 C_conjg =
{
    NULL,
    NULL,
    Conjg,
    dConjg,
    Pconjg,
    dPconjg
};
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (12) Complex arrays from real arrays (real and imag. parts)
--------------------------------------------------------------------*/
fcomplex
Complexw(double a, double b)
{
    fcomplex        z;

    z.r = (float) a;
    z.i = (float) b;
    return z;
}

T_foncCR C_Complex =
{
    Complexw,    /* in complex.c */
    dComplex,    /* in complex.c */
    PComplex,
    dPComplex,
};

fpolaire
PComplex(double x, double y)
{
    return Conv_ctp(Complex((float) x, (float) y));
}

dpolaire
dPComplex(double x, double y)
{
    return dConv_ctp(dComplex(x, y));
}
/*------------------------------------------------------------------*/




/*--------------------------------------------------------------------
 (13) Complex arrays from real arrays (radius and phase)
--------------------------------------------------------------------*/
fpolaire
Polairew(double r, double t)
{
    return Polaire((float)r, (float)t);
}

T_foncCR C_Polaire =
{
    CPolaire,
    dCPolaire,
    Polairew,     /* in complex.c */
    dPolaire      /* in complex.c */
};

fcomplex
CPolaire(double r, double t)
{
    return Conv_ptc(Polaire((float) r, (float) t));
}

dcomplex
dCPolaire(double r, double t)
{
    return dConv_ptc(dPolaire(r, t));
}
/*--------------------------------------------------------------------
--------------------------------------------------------------------*/




/*--------------------------------------------------------------------
    Complex -----> real
--------------------------------------------------------------------*/

/*---- real part ----*/
CC_fonc CC_Reel =
{
    Reel_dC,
    Reel_dP
};

double
Reel_dC(dcomplex z)
{
    return z.r;
}

double
Reel_dP(dpolaire z)
{
    return dConv_ptc(z).r;
}


/*---- imaginary part ----*/
CC_fonc CC_Imag =
{
    Imag_dC,
    Imag_dP
};

double
Imag_dC(dcomplex z)
{
    return z.i;
}

double
Imag_dP(dpolaire z)
{
    return dConv_ptc(z).i;
}


/*---- radius ----*/
CC_fonc CC_R =
{
    R_dC,
    R_dP
};

double
R_dC(dcomplex z)
{
    return dConv_ctp(z).rm;
}

double
R_dP(dpolaire z)
{
    return z.rm;
}


/*---- phase ----*/
CC_fonc CC_Th =
{
    Th_dC,
    Th_dP
};

double
Th_dC(dcomplex z)
{
    return dConv_ctp(z).th;
}

double
Th_dP(dpolaire z)
{
    return z.th;
}
/*------------------------------------------------------------------*/
