/******************************************************************************
 * cl.cpp: "Proxy CL" STL Error Filter Driver for VC++5/6/7
 *
 * Framework by Leor Zolman
 *				BD Software
 *				(leor@bdsoft.com)
 *
 * Win32 piping logic by Thomas Becker
 *
 * Copyright (c) 2001 Leor Zolman
 * Permission is granted to use this code without restriction
 * provided this copyright notice is included in the code.
 *
 * Release 2.21 - 9/28/2001
 *
 * Calls native CL (renamed to CL2.EXE) to perform compilation; for
 * files with applicable extensions, filters output through a Perl
 * script to simplify cryptic STL messages
 *
 * Options: /silent   Disables STL filtering status message and auto-
 *       or -silent      matically adds /nologo option to NATIVE_CL command line
 *
 *          /iter:x	  Sets iterator policy to x (S, M or L)
 *			/alloc:x  Sets allocator policy to x (S or L) [for .NET only]
 *          /func:x   Sets "common functor" policy to x (S or L) [for .NET only]
 *
 * If the file designated by the CONFIG_FILE symbol (PROXY-CL.INI as
 * distributed) is present in the Windows directory upon startup, CL reads
 * configuration information from the file to override like-named defaults
 * as #defined later in this source file. The allowed configuration options,
 * in their respective sections (with default settings shown) are:
 *
 *      [common]
 *		FILTER_SCRIPT=c:\STLFilt.pl
 *		TOGGLE_FILE_DIR=c:\
 *		NATIVE_CL=cl2
 *		PERL_EXE=c:\perl\bin\perl
 *		
 *      [proxy.cl]
 *      SILENT=false
 *		DEBUG=false
 *		DEBUG_DIR=c:\
 *		OUTPUT_WIDTH=0
 *		ITERATOR_POLICY=M
 *		ALLOCATOR_POLICY=S
 *      FUNCTOR_POLICY=S
 *
 * For general information, see README.txt.
 * The Quick Start guide is QUICKSTART.txt.
 *
 * Error filtering is toggled by the presence (or absence) of
 * the file FILTERING.ON in the TOGGLE_FILE_DIR directory.
 * Toggling the existence of this file is managed either by the STLFilt.BAT
 * batch file or the STLTask.EXE taskbar icon app.
 *
 * Compile: cl /Ox /GX cl.cpp
 *
 * Note: Be careful not to compile this cl.cpp file using any version
 *       of CL.EXE resident in the current directory, or the linker won't
 *		 be able to overwrite it to create the new CL.EXE...
 *****************************************************************************/

#define CL_ID		"BD Software Proxy CL v2.21"

#include <iostream>
#include <fstream>
#include <string>
#include <process.h>				// for _spawnvp()
#include <windows.h>				// needed to create processes

////////////////////////////////////////////////////////////////////////////////////////////
//
// User-configurable settings: The next four #defines specify default settings for each
// of the given options. They may be overriden by settings in the configuation file.
//
// Note: Regardless of whether or not any of the pathnames defined below contain embedded
//		 spaces, NEVER use "extra" quotes (as required in older versions of CL.CPP)
//
//		 Remember to use double-backslashes (\\) for directory delimiters in pathnames!
//

#define PERL_EXE		"c:\\perl\\bin\\perl"		// Your Perl interpreter
#define FILTER_SCRIPT	"c:\\STLFilt.pl"			// Perl script pathname

#define TOGGLE_FILE_DIR	"c:\\"						// filtering toggle file directory
													//	(see companion variable FILT_BASE
													//   in STLFILT.BAT)

#define NATIVE_CL		"cl2"						// name of standard CL.EXE, must
													//		reside in a system PATH dir

#define CONFIG_FILE		"PROXY-CL.INI"				// if full path not specified, default
													// is Windows directory

#define SILENT			false						// true to surpress sign-on message (when
													//	filtering C++ source files)

#define DEBUG			false						// if true, logging enabled by default
#define DEBUG_CAPABLE	1							// 1 to compile with debugging features
#define DEBUG_DIR		"C:\\"						// Directory to log debug info to
#define DEBUG_LOGFILE	"cl-dblog.txt"				// main debugging log file
#define DEBUG_ATFILE	"atfile.txt"				// project @file copied to this file
#define OUTPUT_WIDTH	"0"							// column to wrap output at (0: no wrap)
#define ITERATOR_POLICY	"M"							// default iterator policy
#define	ALLOCATOR_POLICY "S"						// default allocator policy (.NET only)
#define FUNCTOR_POLICY	"S"							// default functor policy (.NET only)

const std::string filterExt[] =						// Filter only output from compiling
{													// files with these extensions
	".cpp",
	".cxx"
};

const std::string objExt[] =						// If we see these extensions on any
{													// CL arguments at the end of the line,
	".lib",											// keep searching for some other
	".obj"											// extension to qualify for filtering...
};

// END User-configurable settings
////////////////////////////////////////////////////////////////////////////////////////////

const size_t nfilter = sizeof(filterExt) / sizeof(std::string);
const size_t nobj = sizeof(objExt) / sizeof(std::string);

BOOL WriteEofToPipe(HANDLE hPipe);
bool hasExt(const std::string &filename, const std::string extensions[], int);
int doWin32Stuff(char *full_filt_cmd, char *cmdline);
void chompSlash(std::string &pathname);				// remove trailing slash, if any

#if DEBUG_CAPABLE
void logit(const std::string &msg);
std::string toString(int n);
#define log(s) logit(s)
#else
#define log(s)
#endif

std::string native_cl;
std::string debug_dir = "";
std::string debug_log = "";
bool debug = false;

int main(int argc, char *argv[])
{
	using namespace std;
	const int MAXLINE = 200;						// max length of config file line
	char linebuf[MAXLINE] = "";
	FILE *fp;
	bool filtering = false;							// not filtering by default
	bool silent = false;							// not silent by default
	string full_filt_cmd;
	string filename;								// file to compile
	string perl_exe;
	string filter_script;
	string toggle_file;
	string toggle_file_dir;
	int output_width = 0;
	string iterator_policy;
	string allocator_policy;
	string functor_policy;
	int i;
	
	if (argc == 1)
	{
		cerr << "Proxy CL: Invoked without any arguments. Giving up..." << endl;
		exit(1);
	}
	
	string configFile = CONFIG_FILE;

	GetPrivateProfileString( "proxy.cl", "debug_dir", DEBUG_DIR,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	debug_dir = linebuf;
	chompSlash(debug_dir);
	
	GetPrivateProfileString( "proxy.cl", "debug", DEBUG ? "true" : "false",
							 linebuf, sizeof(linebuf), configFile.c_str() );
	char c;
	debug = (c = tolower(linebuf[0])) == 't' || c == 'y' || c == '1';

	if (debug)
#if DEBUG_CAPABLE
	{
		debug_log = debug_dir + "\\" + DEBUG_LOGFILE;
		log("-------------------------------------------------------------------");
		log(string("debug = ") + (debug ? "true" : "false"));
		log("debug_log = " + debug_log);
	}
#else
		cerr << "Proxy CL: WARNING: debug option specified, but Proxy CL not debug capable!" << endl;
#endif
		
	log("configFile set to: " + configFile);

	GetPrivateProfileString( "common", "perl_exe", PERL_EXE,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	perl_exe = linebuf;
	log("perl_exe = " + perl_exe);

	GetPrivateProfileString( "common", "filter_script", FILTER_SCRIPT,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	filter_script = linebuf;
	log("filter_script = " + filter_script);

	GetPrivateProfileString( "common", "toggle_file_dir", TOGGLE_FILE_DIR,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	toggle_file_dir = linebuf;
	chompSlash(toggle_file_dir);
	log("toggle_file_dir = " + toggle_file_dir);

	GetPrivateProfileString( "common", "native_cl", NATIVE_CL,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	native_cl = linebuf;
	log("native_cl = " + native_cl);

	GetPrivateProfileString( "proxy.cl", "iterator_policy", ITERATOR_POLICY,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	iterator_policy = linebuf;
	log("iterator_policy = " + iterator_policy);

	GetPrivateProfileString( "proxy.cl", "allocator_policy", ALLOCATOR_POLICY,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	allocator_policy = linebuf;
	log("allocator_policy = " + allocator_policy);

	GetPrivateProfileString( "proxy.cl", "functor_policy", FUNCTOR_POLICY,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	functor_policy = linebuf;
	log("functor_policy = " + functor_policy);

	GetPrivateProfileString( "proxy.cl", "silent", SILENT ? "true" : "false",
							 linebuf, sizeof(linebuf), configFile.c_str() );
	silent = (c = tolower(linebuf[0])) == 't' || c == 'y' || c == '1';
	log(string("silent = ") + (silent ? "true" : "false"));

	GetPrivateProfileString( "proxy.cl", "output_width", OUTPUT_WIDTH,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	output_width = atoi(linebuf);
	log("output_width = " + output_width);

	full_filt_cmd = "\"" + perl_exe + "\" \"" + filter_script + "\"";
	log("full_filt_cmd = " + full_filt_cmd);

	toggle_file = toggle_file_dir + "\\filtering.on";
	log("toggle_file = " + toggle_file);

#if DEBUG_CAPABLE
	for (i = 1; i < argc; i++)
		log("Arg #" + toString(i) + " is '" + argv[i] + "'");
#endif
	
	log("-----------");

	if (argv[1][0] == '@')							// indirect command line?
	{
		log(string("Indirect command line detected:") + argv[1] + "\nContents:");

#if DEBUG_CAPABLE			// replicate the @file produced by VC, since it is deleted immediately:
		if (debug)
			system((string("copy ") + &argv[1][1] + " " + debug_dir +
											"\\" + DEBUG_ATFILE).c_str());
#endif

		ifstream fin(&argv[1][1]);
		if (!fin)
		{
			string msg = string("Proxy CL: Bad @ filename? First Command line arg:") +
							 argv[1];
			log(msg);
			cerr << msg << endl;
			exit(1);
		}

		string linebuf, lastline;
		while (getline(fin, linebuf))
		{
			lastline = linebuf;
			log(linebuf);
		}
		fin.close();

		log("The complete final line from the indirect file:\n" + lastline);

		bool seenquote = false;
		for (i = lastline.length() - 1; i >= 0; i--)
			if (lastline[i] == '"')
				if (!seenquote)
				{
					seenquote = true;
					lastline.resize(i);
				}
				else
					break;
			else if (lastline[i] == ' ' && !seenquote)
				break;
		
		filename = lastline.substr(i + 1);
		log("filename detected indirectly:" + filename);
	}
	else
		for (i = argc - 1; i > 0; i--)
		{
			if (hasExt(argv[i], objExt, nobj ))		// ignore libraries, obj files
				continue;
			if (strchr(argv[i], '.') == NULL)		// ignore options (no extensions)
				continue;
			filename = argv[i];						// first non-lib/obj/option is the filename
			break;
		}
	
	if (filename.length() == 0 || !hasExt(filename, filterExt, nfilter )) // if extension of filename is not in the
		silent = true;								// approved list, never filter it
	else
	{
		log("about to open toggle_file ("+toggle_file+")");
		if((fp = fopen(toggle_file.c_str(), "r")) != NULL)
		{												// otherwise, filter iff
			filtering = true;							// toggle_file exists
			fclose(fp);
		}
	}
	
	string args = " ";								// replicate CL command line args

	char **new_argv = new char *[argc + 1];			// also create a new "argv" with each arg in
	size_t new_argc = 0;							// quotes in case we're not filtering and
													// have to use _spawnvp()
	for (i = 0; i < argc; i++)
	{
		if (!stricmp(argv[i],"/silent") || !stricmp(argv[i],"-silent"))
		{
			string nologo = "/nologo";
			silent = true;
			args += nologo;
			new_argv[new_argc] = new char[nologo.length()];
			strcpy(new_argv[new_argc++], nologo.c_str());
		}
		else if (!strnicmp(argv[i], "/iter:", 6)  || !strnicmp(argv[i], "-iter:", 6))
		{
			iterator_policy = &argv[i][6];
		}
		else if (!strnicmp(argv[i], "/alloc:", 7)  || !strnicmp(argv[i], "-alloc:", 7))
		{
			allocator_policy = &argv[i][7];
		}
		else if (!strnicmp(argv[i], "/func:", 6)  || !strnicmp(argv[i], "-func:", 6))
		{
			functor_policy = &argv[i][6];
		}
		else
		{
			// surround each arg with double quotes to avoid long filename problems
			string nextarg = "\"";			// insert opening quote
											
			char *b = argv[i];
			char *p = strchr(b, '"');		// find next double quote char
			while (p != 0)					// while there is another double quote char...
			{
				*p = '\0';					// copy everything up to (not incl.) the quote
 
				nextarg += b;
				nextarg += "\\\"";			// escape the quote
				b = p + 1;					// keep looking for quotes...
				p = strchr(b, '"');
			}

			nextarg += b;					// copy remainder of the arg text

			int n = nextarg.length() - 1;	// if arg ends with pattern [^\]\,
			if (nextarg[n-1] != '\\' && nextarg[n] == '\\')	// add another backslash so we don't
				nextarg += "\\";			// end up with an escaped " at the end!
			nextarg += "\"";				// insert closing quote
			if (i > 0)						// if not command name, add to args list for filtering
				args += nextarg;
			new_argv[new_argc] = new char[nextarg.length() + 1];	// and build new_argv in case
			strcpy(new_argv[new_argc++], nextarg.c_str());			// we end up spawning CL
		}

		args += " ";
	}

	new_argv[new_argc] = NULL;

	if (output_width != 0)
	{
		full_filt_cmd +=  " ";
		full_filt_cmd += "/width:" + toString(output_width);
	}

	full_filt_cmd +=  " ";
	full_filt_cmd += "/iter:" + iterator_policy;

	full_filt_cmd +=  " ";
	full_filt_cmd += "/alloc:" + allocator_policy;

	full_filt_cmd +=  " ";
	full_filt_cmd += "/func:" + functor_policy;

	int status;

	if (filtering)						
	{
		if (!silent)
			cout << "  ****** {"CL_ID"} STL Message Decryption is ON! ******" << endl;

		// Create some "C" strings to pass to the doWin32Stuff function...

		char *full_filt_cmd_cstr = new char[full_filt_cmd.length() + 1];
		strcpy(full_filt_cmd_cstr, full_filt_cmd.c_str());
												// Create replacement command line
		string cmdline = native_cl + args;	// start with native CL.EXE and append
		log(cmdline);							//		whatever args were passed

		char *cmdline_cstr = new char[cmdline.length() + 1];
		strcpy(cmdline_cstr, cmdline.c_str());

		status = doWin32Stuff(full_filt_cmd_cstr, cmdline_cstr);

		delete [] full_filt_cmd_cstr;
		delete [] cmdline_cstr;

	}
	else
	{
		if (!silent)
			cout << "     ****** {"CL_ID"} STL Message Decryption is Off ******" << endl;

#if DEBUG_CAPABLE
		for (i = 1; i < new_argc; i++)
			log("new_argv[" + toString(i) + "] = '" + new_argv[i] + "'");
#endif

												// if not filtering, use native CL:
		status = _spawnvp(_P_WAIT, native_cl.c_str(), new_argv);

		if (status == -1)
		{
			cerr << "CL.EXE: Failure to start " << native_cl << " (does your PATH include it?)" << endl;
			status = 1;
		}
	}

	for(i = 0; i < new_argc; i++)					// release the "new_argv" dynamic memory
		delete[] new_argv[i];
	delete [] new_argv;

	return status;
}


// This has to be in its own function, since we can't mix __try / __finally
// and string objects in main()...Geez Louise...

int doWin32Stuff(char *full_filt_cmd, char *cmdline)
{
	using namespace std;
    // Get std device handles
    HANDLE hStdIn = GetStdHandle(STD_INPUT_HANDLE);
    HANDLE hStdOut = GetStdHandle(STD_OUTPUT_HANDLE);
    HANDLE hStdError = GetStdHandle(STD_ERROR_HANDLE);
    
    // Create an anonymous pipe for communication between native_cl and Perl
    //
    HANDLE hPipeWrite = NULL;
    HANDLE hPipeRead = NULL;
    SECURITY_ATTRIBUTES sa;
    sa.nLength = sizeof(sa); 
    sa.lpSecurityDescriptor = NULL; 
    sa.bInheritHandle = TRUE;
    if( ! ::CreatePipe(
      &hPipeRead,					// address of variable for read handle 
      &hPipeWrite,					// address of variable for write handle  
      &sa,							// pointer to security attributes 
      0								// number of bytes reserved for pipe 
      ) )
    {
		cerr << "CL.EXE: Failure to create pipe between " << native_cl << " and Perl" << endl;
      return 1;
    }
    
    // Start perl
	//
    PROCESS_INFORMATION pi_perl;
    pi_perl.hProcess = NULL;
    pi_perl.hThread = NULL;
    STARTUPINFO si_perl;
    memset(&si_perl, '\0', sizeof(si_perl));
    si_perl.cb = sizeof(si_perl);
    si_perl.dwFlags = STARTF_USESTDHANDLES;
    si_perl.hStdInput = hPipeRead;	// read side of pipe
    si_perl.hStdOutput = hStdOut;	// stdout of parent
    si_perl.hStdError = hStdOut;	// stderr of parent
    if( ! ::CreateProcess(
      NULL,
	  full_filt_cmd,				// something like: perl stlfilt.pl [/iter:n]
      NULL,							// pointer to process security attributes 
      NULL,							// pointer to thread security attributes 
      TRUE,							// handle inheritance flag 
      0,							// creation flags 
      NULL,							// pointer to new environment block 
      NULL,							// pointer to current directory name 
      &si_perl,						// pointer to STARTUPINFO 
      &pi_perl						// pointer to PROCESS_INFORMATION  
      ) )
    {
      cerr << "CL.EXE: Failure to start perl.exe (check pathname in CL.CPP)" << endl;
      return 1;
    }
    
    // Try-finally block for closing handles and sending EOF to perl.
    //
    PROCESS_INFORMATION pi_cl2 ;
    pi_cl2.hProcess = NULL;
    pi_cl2.hThread = NULL;
    //
    __try
    {
      // Start native_cl:
      //
      STARTUPINFO si_cl2;
      memset(&si_cl2, '\0', sizeof(si_cl2));
      si_cl2.cb = sizeof(si_cl2);
      si_cl2.dwFlags = STARTF_USESTDHANDLES;
      si_cl2.hStdInput = hStdIn;		// stdin of parent
      si_cl2.hStdOutput = hPipeWrite;	// write side of child-to-parent
      si_cl2.hStdError = hStdError;		// stderr of parent
      if( ! ::CreateProcess(NULL,
        cmdline,
        NULL,							// pointer to process security attributes 
        NULL,							// pointer to thread security attributes 
        TRUE,							// handle inheritance flag 
        0,								// creation flags 
        NULL,							// pointer to new environment block 
        NULL,							// pointer to current directory name 
        &si_cl2,						// pointer to STARTUPINFO 
        &pi_cl2							// pointer to PROCESS_INFORMATION  
        ) )
      {
        cerr << "CL.EXE: Failure to start " << native_cl << " (does your PATH include it?)" << endl;
        return 1;
      }
      

      // Wait for native_cl to finish.
      if( WAIT_FAILED == ::WaitForSingleObject(pi_cl2.hProcess, INFINITE) )
      {
        cerr << "CL.EXE: Failure waiting for " << native_cl << " to terminate" << endl;
        return 1;
      }
    }
    __finally
    {
      
      // Send EOF to perl to make it end processing and exit.
      if( ! WriteEofToPipe(hPipeWrite) )
      {
        cerr << "CL.EXE: Failure to send EOF to perl.exe" << endl;
        return 1;
      }
    }
    
    // Wait for perl.exe to finish.
    if( WAIT_FAILED == ::WaitForSingleObject(pi_perl.hProcess, INFINITE) )
    {
      cerr << "CL.EXE: Failure waiting for perl.exe to terminate" << endl;
      return 1;
    }
    
    // Get exit code for native_cl. This is possible because we have not closed
    // the process handle yet.
    //
    DWORD dwExitCode;
    if( ! GetExitCodeProcess(
      pi_cl2.hProcess,		// handle to the process
      &dwExitCode			// address to receive termination status
      ) )
    {
	  cerr << "CL.EXE: Could not get exit code for " << native_cl << endl;
      return 1;
    }
    
    // Close all handles, just to make the point.
    //
    ::CloseHandle(hPipeWrite);
    ::CloseHandle(hPipeRead);
    ::CloseHandle(pi_perl.hProcess);
    ::CloseHandle(pi_perl.hThread);
    ::CloseHandle(pi_cl2.hProcess);
    ::CloseHandle(pi_cl2.hThread);

	return dwExitCode;
}    

BOOL WriteEofToPipe(HANDLE hPipe)
{
	SYSTEM_INFO info;		// Check for > 1 processor...
	GetSystemInfo(&info);
	if (info.dwNumberOfProcessors > 1) 
		Sleep(100);		// Fixes obscure timing problem on dual PIII system

	DWORD dwNumBytesWritten = 0 ;
	BOOL bWriteRetVal = WriteFile(
	  hPipe,				// handle to file to write to 
	  "\032",				// pointer to data to write to file 
	  1,					// number of bytes to write 
	  &dwNumBytesWritten,	// pointer to number of bytes written 
	  NULL) ;				// addr. of structure needed for overlapped I/O  
	if( ! bWriteRetVal || 1 != dwNumBytesWritten )
		  return FALSE;

    ::FlushFileBuffers(hPipe);
    return TRUE;
}

bool hasExt(const std::string &filenam, const std::string extensions[], int nexts)
{
	using namespace std;
	string filename = filenam;
	int length;
	
	int i;

	if (filename.length() < 2)
		return false;
	
					// get rid of trailing whitespace on the filename...
	for (i = filename.length() - 1; i >= 0; i--)
		if (!isspace(filename[i]))
			break;
	filename.resize(length = i + 1);
	
	log(string("hasExt() testing: \"") + filename + "\"");

	for (i = 0; i < nexts; i++)
	{
		int extLen = extensions[i].length();
		log("testing extension:" + extensions[i]);
		if (length >= extLen + 1 &&
			!stricmp(filename.substr(length - extLen).c_str(), extensions[i].c_str()))
		{
			log(string("hasExt() found extension: ") + extensions[i]);
			return true;
		}
	}
	log("hasExt() did not find a matching extension.\n");
	return false;
}
			

// remove trailing slash, if any, from string:
void chompSlash(std::string &pathname)
{
	char c;
	if ((c = pathname[pathname.length() - 1]) == '/' || c == '\\')
		pathname.resize(pathname.length() - 1);
}

#if DEBUG_CAPABLE
void logit(const std::string &msg)
{
	if (!debug)
		return;
	
	std::ofstream out(debug_log.c_str(), std::ios_base::app);
	if (!out) {
		std::cerr << "Can't append to " << debug_log << ":" << std::endl;
		std::cerr << "Offending message: " << msg << std::endl;
	}
	else
	{
		out << msg << std::endl;
		out.close();
	}
}

std::string toString(int n)
{
	char buf[20];
	sprintf(buf, "%d", n);
	return buf;
}
#endif
