/////////////////////////////////////////////////////////////////////////////////////////
// Name:        STLTask.cpp
// Purpose:     Control STL Error Message Filtering from taskbar icon
// Author:      Leor Zolman, adapted from wxTaskBarIcon demo by Julian Smart
// Created:     05/02/01
// Modified:	9/14/01 v2.03
//
// Usage:
//		stltask [/q] [init-file] 
//
//		-q or /q: supress initial sign-on dialog (useful for Startup folder shortcut)
//
//      init-file, if specified, may contain the following configuration lines:
//
//				[common]
//				FILTER_SCRIPT=c:\stlfilt.pl
//				TOGGLE_FILE_DIR=c:\								
//				NATIVE_CL=cl2.exe
//				PERL_EXE=c:\perl\bin\perl.exe
//
//				[stltask.exe]
//				CL_DIR=c:\program files\microsoft visual studio\vc98\bin
//				CLIPBOARD_ONLY=false
//				PROXY_CL_UNINSTALLED_FNAME=cl.stl
//				CHECK_FOR_PREVIOUS_INSTANCE=true
//
//		If init-file is not specified, STLTask will look for the file Proxy-CL.INI
//		in your Windows directory by default. See the "Configuration" section of
//		STLTask-README.txt for a full description of the config file format.
//
// To compile, make sure the wxWindows environment variable WXWIN is set correctly
// and the NON-debug library is built, then say:
//
//		nmake FINAL=1 -f makefile.vc
//
// If you compile in debug mode (without the FINAL=1), it crashes if still active when
// you log out of / shutdown Windows (98, anyway). wxWindows bug???
//
/////////////////////////////////////////////////////////////////////////////////////////

#define STLTASK_ID "                 STLTask v2.03 -- 9/14/2001"

/////////////////////////////////////////////////////////////////////////////////////////
//
// USER CONFIGURABLE SYMBOLS:
//

#define CONFIG_FILE		"Proxy-CL.INI"

// Default values for all user-selectable dir/file names (config file overrides):
#define CLIPBOARD_ONLY		false				// if true, Proxy CL features are disabled
												// (so that STLTask is only usable for
												// its "filter clipboard" feature)

#define CLIPBOARD_WARNING_THRESHOLD 50			// warn if clipboard contents to filter
												//		are shorter than this

#define FILTER_SCRIPT	"c:\\stlfilt.pl"		// The filtering Perl script
												// (Do NOT use "extra" quotes, even if
												// the pathname contains spaces)

												// the following only matter when
												//		CLIPBOARD_ONLY is false:

												// below: MSVC's bin directory
#define CL_DIR				"c:\\program files\\microsoft visual studio\\vc98\\bin"
#define NATIVE_CL			"CL2.EXE"			// Copy of MSVC's native CL.EXE
#define PROXY_CL_UNINSTALLED_FNAME	"CL.STL"	// Uninstalled Proxy CL filename
#define PERL_EXE			"c:\\perl\\bin\\perl.exe" // Perl command
#define TOGGLE_FILE_DIR		"c:\\"				// Toggle file directory
#define CHECK_FOR_PREVIOUS_INSTANCE	 true		// true to use sentinel file (to detect
												//		previous instance)
#define LOCKFILE			"STLTASK.LOK"

// Debugging:
#define DEBUG 0									 // If 1, use the filename specified by DEBUG_LOG
#define DEBUG_LOG "d:\\src\\cl\\task\\debug.out" // to log debugging messages via log(...)

//
// END USER-CONFIGURABLE SYMBOLS
////////////////////////////////////////////////////////////////////////////////////

// Filenames for standard and proxy CL commands when active and inactive (do NOT edit):
#define CL_EXE	"CL.EXE"						// active standard CL

#if DEBUG
#define log(s) logit(s)
#else
#define log(s)
#endif

// For compilers that support precompilation, includes "wx.h".
#include <fstream>
#include "wx/msgdlg.h"
#include "wx/clipbrd.h"
#include "wx/msw/taskbar.h"

#ifdef __BORLANDC__
#pragma hdrstop
#endif

#ifndef WX_PRECOMP
#include <wx/wx.h>
#endif

#include <string>
using namespace std;

#include "sys/stat.h"
#include "STLtask.h"

enum action {		// possible paramters to examine() function
	INSTALLING,
	UNINSTALLING,
	NEITHER
};


bool exists(const string &filename);
bool copyfile(const string &src, const string &dest);
bool sameSize(const string &file1, const string &file2);
void message(const string &msg);
void emessage(const string &msg);
void message(const string &msg, const string &title);
void logit(const string &s);
void logit(const char *msg);
string tostring(int n);
bool examine(action mode);
void deleteSentinel();
void chompSlash(string &pathname);	// remove trailing slash, if any

//#include "config.h"
#include <wtypes.h>

// Declare two frame
MyDialog   *dialog = NULL;

string toggle_file_dir;			// directory where filtering control toggle file resides
string toggle_file_basename;	// name (without extension) of the toggle file (FILTERING)
string cl_dir;					// MSVC's bin directory
string cl_exe;					// full pathname of CL.EXE (in MSVC's bin directory)
string native_cl;				// filename only of native CL (CL2.EXE)
string std_cl2;					// full pathname of CL2.EXE
string proxy_cl_uninstalled_fname;	// filename only of Proxy CL's uninstalled filename
string proxy_cl_uninstalled;	// full pathname of Proxy CL's uninstalled filename
string filter_script;			// full path of the Perl script (STLFilt.pl)
string stltask_sentinel_file;	// created upon startup, deleted up exit
string perl_exe;				// full pathname of Perl interpreter
bool clipboard_only;			// if true, STLTask does nothing but clipboard filtering
bool check_for_previous_instance;  // true to create/delete a sentinel file in cl_dir

IMPLEMENT_APP(MyApp)

bool MyApp::OnInit(void)
{
	FILE *fp;
	const int MAXLINE = 200;
	char linebuf[MAXLINE];
	bool quiet_mode = false;
	
    wxIcon icon_on("STLTask");
	onp = &icon_on;

	// Test for presence of config file name on the command line. If present,
	// allow config file settings to override default setting for the 6
	// configurable values:

	string configFile = CONFIG_FILE;

	for (int i = 1; i < argc; i++)
	{
		if (!strcmpi(argv[i], "-q") || !strcmpi(argv[i], "/q"))
		{
				quiet_mode = true;
				continue;
		}
		else if ((fp = fopen(argv[i], "r")) != NULL)
		{
			fclose(fp);
			configFile=argv[i];
		}
		else
		{
			emessage(string("Unrecognized command line argument: ") + argv[i]);
			exit(1);
		}
	}
		
	log("----------------------------------------------------------");
	log("configFile set to: " + configFile);

	GetPrivateProfileString( "common", "filter_script", FILTER_SCRIPT,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	filter_script = linebuf;
	log("filter_script = " + filter_script);

	GetPrivateProfileString( "common", "toggle_file_dir", TOGGLE_FILE_DIR,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	toggle_file_dir = linebuf;
	chompSlash(toggle_file_dir);
	log("toggle_file_dir = " + toggle_file_dir);

	GetPrivateProfileString( "common", "native_cl", NATIVE_CL,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	native_cl = linebuf;
	log("native_cl = " + native_cl);

	GetPrivateProfileString( "common", "perl_exe", PERL_EXE,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	perl_exe = linebuf;
	log("perl_exe = " + perl_exe);

	GetPrivateProfileString( "stltask.exe", "cl_dir", CL_DIR,
							 linebuf, sizeof(linebuf), configFile.c_str() );
	cl_dir = linebuf;
	chompSlash(cl_dir);
	log("cl_dir = " + cl_dir);

	GetPrivateProfileString( "stltask.exe", "proxy_cl_uninstalled_fname",
							 PROXY_CL_UNINSTALLED_FNAME, linebuf, sizeof(linebuf),
							 configFile.c_str() );
	proxy_cl_uninstalled_fname = linebuf;
	log("proxy_cl_uninstalled_fname = " + proxy_cl_uninstalled_fname);

	GetPrivateProfileString( "stltask.exe", "clipboard_only",
							 CLIPBOARD_ONLY ? "true" : "false",
							 linebuf, sizeof(linebuf), configFile.c_str() );
	char c;
	clipboard_only = (c = tolower(linebuf[0])) == 't' || c == 'y' || c == '1';
	log(string("clipboard_only = ") + (clipboard_only ? "true" : "false"));

	GetPrivateProfileString( "stltask.exe", "check_for_previous_instance",
							 CHECK_FOR_PREVIOUS_INSTANCE ? "true" : "false",
							 linebuf, sizeof(linebuf), configFile.c_str() );
	check_for_previous_instance = (c = tolower(linebuf[0])) == 't' || c == 'y' || c == '1';
	log(string("check_for_previous_instance = ") + (check_for_previous_instance ? "true" : "false"));

	//
	// Now set the four filenames that will actually be used:
	//
	cl_exe = cl_dir + "\\" + CL_EXE;
	std_cl2 = cl_dir + "\\" + native_cl;
	proxy_cl_uninstalled = cl_dir + "\\" + proxy_cl_uninstalled_fname;
	toggle_file_basename = toggle_file_dir + "\\filtering";
	stltask_sentinel_file = cl_dir + "\\" + LOCKFILE;

	log("cl_exe set to: " + cl_exe);
	log("std_cl2 set to: " + std_cl2);
	log("proxy_cl_uninstalled set to: " + proxy_cl_uninstalled);
	log("toggle_file_basename set to: " + toggle_file_basename);
	log("stltask_sentinel_file set to: " + stltask_sentinel_file);

    // Create the main frame window
    dialog = new MyDialog(NULL, -1, "STLTask Message Decryptor Control Utility",
			wxPoint(-1, -1), wxSize(400, 235),
			wxDIALOG_MODELESS|wxSTAY_ON_TOP|wxTHICK_FRAME);

	//
	// Check for previous instance:
	//

	if (check_for_previous_instance)
	{
		if ((fp = fopen(stltask_sentinel_file.c_str(), "r")) != NULL)
		{
			fclose(fp);

			if (!quiet_mode)
			{
				wxString msg = wxString("A previous instance of STLTask.EXE may already be running. ") +
						   "Are you SURE you want to start a new instance?";
				wxMessageDialog *d = new wxMessageDialog(NULL, msg,
					"STL Filter Taskbar Monitor", wxYES_NO | wxNO_DEFAULT | wxICON_QUESTION );
				int goOn = d->ShowModal();
				delete d;

				if (goOn == wxID_NO)
					exit(0);
			}
		}
		else if ((fp = fopen(stltask_sentinel_file.c_str(), "w")) == NULL)
		{
			emessage("ERROR: Can't create " + stltask_sentinel_file);
			exit(0);
		}
		else
			fclose(fp);
	}
	
    if (!m_taskBarIcon.SetIcon(icon_on, "STLTask Message Decryptor Utility"))
        wxMessageBox("Could not set icon.");

	if (!clipboard_only)
	{
		// Make sure everything is consistent when we start up....
		examine(NEITHER);
		dialog->update("                  STLTask loaded successfully. Current Status:");
	}
	
	if (!quiet_mode)
		dialog->Show(TRUE);

    return TRUE;
}

void deleteSentinel()
{
	
	if (check_for_previous_instance && exists(stltask_sentinel_file))
		if (remove(stltask_sentinel_file.c_str()))
			emessage("ERROR: Can't delete " + stltask_sentinel_file);
}

MyApp::~MyApp()
{
	deleteSentinel();
}

void MyApp::OnEndSession(wxCloseEvent& event)
{
	deleteSentinel();
	m_taskBarIcon.RemoveIcon();
}

// return value is true if we end up installed, false if we don't:
bool examine(action mode)
{
	if (!exists(std_cl2))
	{
		if (exists(proxy_cl_uninstalled) && exists(cl_exe))
		{
			if (sameSize(proxy_cl_uninstalled, cl_exe))
			{
				emessage("Corrupt installation. No native " CL_EXE " located. Bailing out.");
				deleteSentinel();
				exit(1);
			}
			else
				if (copyfile(cl_exe, std_cl2))
				{
					MyDialog::startupWarning =
							"                             Copied  " CL_EXE " to " +
							native_cl;
					if (mode == INSTALLING)
					{
						if (copyfile(proxy_cl_uninstalled, cl_exe))
						{
							MyDialog::startupWarning2 = "                              Copied "
								+ proxy_cl_uninstalled_fname + " to " CL_EXE;
							return true;
						}
						else
						{
							MyDialog::startupWarning2 = "                         Error copying " +
								proxy_cl_uninstalled_fname + " to " CL_EXE;
							return false;
						}
					}
					return true;
				}
				else
				{
					emessage("Couldn't copy " CL_EXE " to " + native_cl);
					deleteSentinel();
					exit(1);
				}
		}
		else if (exists(proxy_cl_uninstalled))
		{
			emessage("Corrupt configuration -- No native " CL_EXE " detected. Bailing out.");
			deleteSentinel();
			exit(1);
		}
		else if (exists(cl_exe))
		{
			if (copyfile(cl_exe, std_cl2))
			{
				MyDialog::startupWarning =
						"                                Copied " CL_EXE " to " + native_cl;
				return false;
			}
			else
			{
				emessage("Error copying " CL_EXE " to " + native_cl +
						"; Bailing out.");
				deleteSentinel();
				exit(1);
			}
		}
		else
		{
			emessage("No compiler components located at all. Bailing out.");
			deleteSentinel();
			exit(1);
		}			
	}
	else	// std_cl2 does exist...
	{
		if (exists(proxy_cl_uninstalled) && exists(cl_exe))
		{
			if (sameSize(proxy_cl_uninstalled, cl_exe))
			{
				if (mode == UNINSTALLING)
				{
					if (copyfile(std_cl2, cl_exe))
					{
						return false;
					}
					else
					{
						emessage("Error copying " +  native_cl + " to " CL_EXE "; Bailing out.");
						deleteSentinel();
						exit(1);
					}
				}
				return true;
			}
			else if (sameSize(std_cl2, cl_exe))
			{
				if (mode == INSTALLING)
				{
					if (copyfile(proxy_cl_uninstalled, cl_exe))
					{
						return true;
					}
					else
					{
						emessage("Error copying " +  proxy_cl_uninstalled +
								 " to " CL_EXE "; Bailing out.");
						deleteSentinel();
						exit(1);
					}
				}
				return false;
			}
			else
			{
				emessage("STLTask has detected a corrupt configuation of files:\n" +
						 cl_exe +"'s size matches neither " + native_cl + " nor " +
						 proxy_cl_uninstalled + ".\n"
						 "If you've recently updated " + cl_exe + ", try deleting\n"
						 + proxy_cl_uninstalled + " and running STLTask again.");
				deleteSentinel();
				exit(1);
			}
		}
		else if (exists(cl_exe))
		{
			if (sameSize(std_cl2, cl_exe))
			{
				emessage("Warning: Proxy CL missing from MSVC's bin directory");
				return false;
			}
			else
			{
				if (copyfile(cl_exe, proxy_cl_uninstalled))
				{
					MyDialog::startupWarning =
						"                              Copied " CL_EXE " to " +
						proxy_cl_uninstalled_fname;
					return true;
				}
				else
				{
					emessage("Error copying " CL_EXE " to " +
							 proxy_cl_uninstalled + "; Bailing out.");
					deleteSentinel();
					exit(1);
				}
			}				
		}
		else if (exists(proxy_cl_uninstalled))
		{
			if (mode == INSTALLING)
			{
				if (copyfile(proxy_cl_uninstalled, cl_exe))
				{
					MyDialog::startupWarning =
						"                    No " CL_EXE " found; Restoring from " +
						proxy_cl_uninstalled_fname;
					return true;
				}
				else
				{
					emessage("Error copying " +  proxy_cl_uninstalled +
							 " to " CL_EXE "; Bailing out.");
					deleteSentinel();
					exit(1);
				}
			}
			else
			{
				if (copyfile(std_cl2, cl_exe))
				{
					MyDialog::startupWarning =
							"                  No " CL_EXE
							" found; Restoring from " + native_cl;
					return false;
				}
				else
				{
					emessage("Error copying " +  std_cl2 + " to " CL_EXE "; Bailing out.");
					deleteSentinel();
					exit(1);
				}
			}
		}
		else
		{
			if (mode == INSTALLING)
			{
				emessage("No Proxy CL files detected. Cannot install. Bailing out.");
				deleteSentinel();
				exit(1);
			}
			if (copyfile(std_cl2, cl_exe))
			{
				MyDialog::startupWarning =
					"             No " CL_EXE " at all detected in MSVC's bin directory";
				MyDialog::startupWarning2 =
					"                         Restored " CL_EXE "from " + native_cl;
				return false;
			}
			else
			{
				emessage("Error copying " +  std_cl2 + " to " CL_EXE "; Bailing out.");
				deleteSentinel();
				exit(1);
			}
		}
	}
	emessage("INTERNAL ERROR: end of examine() -- shouldn't be here!");
	return false;
}


BEGIN_EVENT_TABLE(MyDialog, wxDialog)
    EVT_BUTTON(wxID_OK, MyDialog::OnOK)
    EVT_BUTTON(wxID_EXIT, MyDialog::OnExit)
    EVT_CLOSE(MyDialog::OnCloseWindow)
END_EVENT_TABLE()

BEGIN_EVENT_TABLE(MyApp, wxApp)
	EVT_END_SESSION(MyApp::OnEndSession)
END_EVENT_TABLE()


MyDialog::MyDialog(wxWindow* parent, const wxWindowID id, const wxString& title,
    const wxPoint& pos, const wxSize& size, const long windowStyle):
  wxDialog(parent, id, title, pos, size, windowStyle)
{
    Init();
}

void MyDialog::OnOK(wxCommandEvent& event)
{
    Show(FALSE);
}

void MyDialog::OnExit(wxCommandEvent& event)
{
	Close(TRUE);
}

void MyDialog::OnCloseWindow(wxCloseEvent& event)
{
    Destroy();
}

void MyDialog::update()
{
	update("");
}

void MyDialog::update(const string &msg)
{
	update(msg.c_str());
}

string MyDialog::startupWarning = "";
string MyDialog::startupWarning2 = "";

void MyDialog::update(const char *errMsg)
{
  bool filtering = false, installed;
  static bool firstTime = true;
  
  static wxStaticText * errStat = 0;
  static wxStaticText * warning2Stat = 0;
  static wxStaticText * filtStat = 0;
  static wxStaticText * instStat = 0;

  static wxStaticText * id = new wxStaticText(this, -1,
					STLTASK_ID, wxPoint(50,170));
  static wxStaticText * website = new wxStaticText(this, -1,
					"www.bdsoft.com", wxPoint(130,185));

  if (errStat)
	  delete errStat;

  if (warning2Stat)
	  delete warning2Stat;
  
  if (filtStat)
	  delete filtStat;
  
  if (instStat)
	  delete instStat;

  if (installed = sameSize(cl_exe, proxy_cl_uninstalled))
	filtering = exists(toggle_file_basename + ".on");

  if (startupWarning != "")
	  errMsg = startupWarning.c_str();
  
  errStat = new wxStaticText(this, -1, errMsg ? errMsg :
					"                                 ", wxPoint(10, 10));
  startupWarning = "";

  warning2Stat = new wxStaticText(this, -1,
				(startupWarning2 != "") ? startupWarning2.c_str() :
					"                                 ", wxPoint(10, 25));
  startupWarning2 = "";

  char *installed_status = installed ? "           Proxy CL is Installed      " :
											 "       Proxy CL is NOT Installed";
  if (clipboard_only)
	  installed_status = "";
  
  char *filt_status = filtering ? "**** Filtering is ON ****" :
										"      Filtering is OFF        ";
  instStat = new wxStaticText(this, -1, installed_status, wxPoint(88, 40));
 
  if (!clipboard_only)
	  filtStat = new wxStaticText(this, -1,
				installed ? filt_status : "                              ",
				wxPoint(115, 55));
  else
	  filtStat = new wxStaticText(this, -1,
				"STLTask is in Clipboard Filtering ONLY mode",
				wxPoint(50, 55));

  firstTime = false;
}

void MyDialog::Init(void)
{
  int dialogWidth = 365;
  int dialogHeight = 150;

  update();

  wxStaticText* stat1 = new wxStaticText(this, -1,
						"Press OK to put me back in the taskbar, Exit to quit.",
		wxPoint(45, 90));

  wxButton *okButton = new wxButton(this, wxID_OK, "OK",
												wxPoint(85, 120), wxSize(80, 25));
  wxButton *exitButton = new wxButton(this, wxID_EXIT, "Exit",
												wxPoint(170, 120), wxSize(80, 25));
  okButton->SetDefault();
  this->Centre(wxBOTH);
}


enum {
    PU_SHOWSTATUS = 10001,
	PU_ENABLE = 10002,
	PU_DISABLE = 10003,
	PU_INSTALL = 10004,
	PU_UNINSTALL = 10005,
	PU_PREPARE_FOR_SERVICE_PACK = 10006,
	PU_DOCLIPBOARD = 10007,
    PU_EXIT
};


BEGIN_EVENT_TABLE(StlTask, wxTaskBarIcon)
    EVT_MENU(PU_SHOWSTATUS, StlTask::OnMenuRestore)
    EVT_MENU(PU_ENABLE,    StlTask::Enable)
    EVT_MENU(PU_DISABLE,    StlTask::Disable)
    EVT_MENU(PU_INSTALL,    StlTask::Install)
    EVT_MENU(PU_UNINSTALL,    StlTask::Uninstall)
	EVT_MENU(PU_PREPARE_FOR_SERVICE_PACK, StlTask::WipeOut)
    EVT_MENU(PU_DOCLIPBOARD,    StlTask::DoClipBoard)
    EVT_MENU(PU_EXIT,    StlTask::OnMenuExit)
END_EVENT_TABLE()

void StlTask::OnMenuRestore(wxCommandEvent& )
{
	dialog->update("                      Current STL Message Decryptor Status:");
    dialog->Show(TRUE);
}

void StlTask::DoClipBoard(wxCommandEvent& )
{
  size_t i;
  if (wxTheClipboard->Open())
  {
	log("Clipboard opened.");
    if (wxTheClipboard->IsSupported( wxDF_TEXT )) 
    {
	  wxArrayString output;
	  wxArrayString errors;
      wxTextDataObject data;
      wxTheClipboard->GetData( data );		// get clipboard data

	  wxString errText = data.GetText();

	  if (errText.length() < CLIPBOARD_WARNING_THRESHOLD)	// confirm filtering if message text real short...
	  {
		  string msg(string("Your clipboard contents:\n     \"") + errText.c_str() + "\"\nare only " +
					tostring(errText.length()) +
					" characters long. Are you SURE you want them filtered?");
		  wxMessageDialog *d = new wxMessageDialog(NULL, msg.c_str(),
			  "STL Filter Taskbar Monitor", wxYES_NO | wxNO_DEFAULT | wxICON_EXCLAMATION );
		  int goOn = d->ShowModal();
		  delete d;

		  if (goOn == wxID_NO)
			  return;
	  }

	  int column = 0;							  // strip newlines introduced by

	  for (i = 0; i < errText.length(); i++)	  // copying into the clipboard...
	  {
		  if (errText[i] == '\n')
		  {
			  if (column == 250)
			  {
				  errText.Remove(i-1,2);			// remove the CR-LF
				  i -= 2;							// if it is a long, wrapped line
			  }
			  else
			  {
				  errText.Remove(i-1,1);			// remove just the CR
				  i--;								// if it is a shorter line
			  }
			  column = 0;
			  continue;
		  }
		  column++;
	  }
	  
	  char *tmpFile = wxGetTempFileName("STL");
	  ofstream hold(tmpFile);
	  hold << errText.c_str();
	  hold.close();

	  string command = ("\"" + perl_exe + "\" \"" + filter_script + "\" " + tmpFile);

	  long status =
			::wxExecute(wxString(command.c_str()), output, errors);

	  string result;
	  for (i = 0; i < output.Count(); i++)
		  result = result + output[i].c_str() + "\n";

      message(result, "Decryption Results");
	  ::wxRemoveFile(tmpFile);		// remove the tmp file
	  delete tmpFile;				// release the dynamic tmp file name string 
    }
	else
	{
      emessage("Nothing in the clipboard to process!");
	}		
    wxTheClipboard->Close();
  }
  else
      emessage("Nothing in the clipboard to process!");
}



void StlTask::Enable(wxCommandEvent& )
{
	FILE *fp;

	if (!sameSize(cl_exe, proxy_cl_uninstalled))
	{
		emessage("Filter toggling requires Proxy CL.\nSelect \"Install Proxy CL\" first.");
		return;
	}
		
	if (exists(toggle_file_basename + ".on"))
	{
		dialog->update("                    STL message filtering was already enabled!");
		dialog->Show(TRUE);
		return;
	}

	if (exists(toggle_file_basename + ".off"))
	{
		if (remove((toggle_file_basename + ".off").c_str()))
		{
			dialog->update("              ERROR: Can't delete " +
							toggle_file_basename + ".off");
			dialog->Show(TRUE);
			return;
		}
	}
	if ((fp = fopen((toggle_file_basename + ".on").c_str(), "w")) != NULL)
	{
		fclose(fp);
		dialog->update("                    STL message filtering has been enabled.");
	}
	else
		dialog->update("              ERROR: Can't create " + toggle_file_basename + ".on");

	dialog->Show(TRUE);
}

void StlTask::Disable(wxCommandEvent& )
{
	FILE *fp;
	
	if (!sameSize(cl_exe, proxy_cl_uninstalled))
	{
		emessage("Filter toggling requires Proxy CL.\nSelect \"Install Proxy CL\" first.");
		return;
	}

	if (exists(toggle_file_basename + ".off"))
	{
		dialog->update("                    STL message filtering was already disabled!");
		dialog->Show(TRUE);
		return;
	}

	if (exists(toggle_file_basename + ".on"))
	{
		if (remove((toggle_file_basename + ".on").c_str()))
		{
			dialog->update("              ERROR: Can't delete " +
									   toggle_file_basename + ".on");
			dialog->Show(TRUE);
			return;
		}
	}
	 
	if ((fp = fopen((toggle_file_basename + ".off").c_str(), "w")) != NULL)
	{
		fclose(fp);
		dialog->update("                    STL message filtering has been disabled.");
	}
	else
		dialog->update("            ERROR: Can't create " + toggle_file_basename + ".off");

	dialog->Show(TRUE);

}

// Prepare for applying a Service Pack
void StlTask::WipeOut(wxCommandEvent& ce)
{
	string msg(string("This will remove ") + native_cl + " and terminate STLTask\n"
			"(STLTask will reconfigure things next time it loads)\n"
			"        Are you SURE you want to do this?");
	wxMessageDialog *d = new wxMessageDialog(NULL, msg.c_str(),
			"STL Filter Taskbar Monitor", wxYES_NO | wxNO_DEFAULT | wxICON_EXCLAMATION );
	int goOn = d->ShowModal();
	delete d;
	if (goOn == wxID_NO)
		return;

	// make sure we're uninstalled:
	if (sameSize(cl_exe, proxy_cl_uninstalled))
	{
		if (examine(UNINSTALLING))
		{
			emessage("Error occurred uninstalling.");
			return;
		}
	}

	if (remove(std_cl2.c_str()))
	{
		emessage("ERROR: Can't delete " + std_cl2);
		return;
	}

	message("It is now safe to apply a Visual Studio 6 Service Pack\n"
		    "STLTask will terminate when you press the OK button.");
	deleteSentinel();
	exit(1);
}
	

// copy Proxy CL (CL.STL) to CL.EXE:
void StlTask::Install(wxCommandEvent &c)
{
	if (sameSize(cl_exe, proxy_cl_uninstalled))
	{
		dialog->update("                               Proxy CL is already installed!");
		dialog->Show(TRUE);
		return;
	}

	bool installed = examine(INSTALLING);
	if (installed)
	{		
		dialog->update("                           Proxy CL successfully installed.");
		dialog->Show(TRUE);
	}
	else
		emessage("Error installing Proxy CL.");
}

// Copy inactive native CL (CL2.EXE) to CL.STL:
void StlTask::Uninstall(wxCommandEvent& )
{
	if (!sameSize(cl_exe, proxy_cl_uninstalled))
	{
		dialog->update("         Proxy CL has to be INstalled before it can be UNinstalled...");
		dialog->Show(TRUE);
		return;
	}

	bool installed = examine(UNINSTALLING);
	if (!installed)
	{
		dialog->update("                    Native MSVC CL.EXE has been restored.");
		dialog->Show(TRUE);
	}
}


void StlTask::OnMenuExit(wxCommandEvent& )
{
	dialog->Close(TRUE);
	deleteSentinel();
    // Nudge wxWindows into destroying the dialog, since
    // with a hidden window no messages will get sent to put
    // it into idle processing.
    wxGetApp().ProcessIdle();
}


// Overridables
void StlTask::OnMouseMove(wxEvent&)
{
}

void StlTask::OnLButtonDown(wxEvent&)
{
}

void StlTask::OnLButtonUp(wxEvent& e)
{
//	OnRButtonUp(e);
}

void StlTask::OnRButtonDown(wxEvent&)
{
}

void StlTask::OnRButtonUp(wxEvent&)
{
    wxMenu      menu("STLTask Message Decryptor Control Utility");

    menu.Append(PU_SHOWSTATUS, "&Show status");
	menu.AppendSeparator();

	if (!clipboard_only)
	{
		menu.Append(PU_ENABLE, "&Enable filtering");
		menu.Append(PU_DISABLE, "&Disable filtering");
		menu.AppendSeparator();
		menu.Append(PU_INSTALL, "&Install proxy CL");
		menu.Append(PU_UNINSTALL, "&Uninstall proxy CL (restore native)");
		menu.AppendSeparator();
	}

	menu.Append(PU_DOCLIPBOARD, "Filter &Clipboard");
	menu.AppendSeparator();

	if (!clipboard_only)
	{
		menu.Append(PU_PREPARE_FOR_SERVICE_PACK, "&Prepare to apply Visual Studio service pack");
		menu.AppendSeparator();
	}
		
    menu.Append(PU_EXIT,    "E&xit");
    PopupMenu(&menu);
}

void StlTask::OnLButtonDClick(wxEvent&)
{
	dialog->update("                      Current STL Message Decryptor Status:");
    dialog->Show(TRUE);
}

void StlTask::OnRButtonDClick(wxEvent&)
{
}


// return true if both files have same size
bool sameSize(const string &file1, const string &file2)
{
   struct _stat buf;
   long size1;
   int result;

   result = _stat(file1.c_str(), &buf);

   if( result != 0 )
   {
      emessage("Missing file: " + file1
#if DEBUG
			   + " (2nd arg to sameSize: " + file2
#endif
			  );
	  
	  deleteSentinel();
	  exit(0);
   }
   size1 = buf.st_size;
   
   result = _stat(file2.c_str(), &buf);

   if( result != 0 )
   {
      emessage("Missing file: " + file2
#if DEBUG
			   + " (1st arg to sameSize: " + file1
#endif

			  );
	  deleteSentinel();
	  exit (0);
   }

   return size1 == buf.st_size;
}

bool exists(const string &filename)
{
	FILE *fp;

	if ((fp = fopen(filename.c_str(), "r")) != NULL)
		fclose(fp);

	return fp != NULL;
}

bool copyfile(const string &src, const string &dest)
{
	FILE *fp_src, *fp_dest;
	int c;

	if ((fp_src = fopen(src.c_str(), "rb")) == NULL)
		return false;

	if ((fp_dest = fopen(dest.c_str(), "wb")) == NULL)
		return false;

	while ((c = getc(fp_src)) != EOF)
		putc(c, fp_dest);

	fclose(fp_src);
	fclose(fp_dest);
	return true;
}

// Display non-Error message:
void message(const string &msg)
{
	wxMessageDialog *d = new wxMessageDialog(
		NULL, msg.c_str(), "STL Filter Taskbar Monitor", wxOK);
	d->ShowModal();
	delete d;
}

// Display non-Error message with custom title:
void message(const string &msg, const string &title)
{
	wxMessageDialog *d = new wxMessageDialog(
		NULL, msg.c_str(), title.c_str(), wxOK);
	d->ShowModal();
	delete d;
}


// Display Error message:
void emessage(const string &msg)
{
	wxMessageDialog *d = new wxMessageDialog(
		NULL, msg.c_str(), "STL Filter Taskbar Monitor", wxOK | wxICON_EXCLAMATION );
	d->ShowModal();
	delete d;
}

void logit(const string &s)
{
	logit(s.c_str());
}

// remove trailing slash, if any, from string:
void chompSlash(string &pathname)
{
	char c;
	if ((c = pathname[pathname.length() - 1]) == '/' || c == '\\')
		pathname.resize(pathname.length() - 1);
}

// For debugging, append msg to log file
void logit(const char *msg)
{
	FILE *fp;
	
	if ((fp = fopen(DEBUG_LOG, "a")) == NULL)
		message("Can't append to debug.out.");
	else {
		fputs(msg, fp);
		fputs("\n", fp);
		fclose(fp);
	}
}
	
string tostring(int n)
{
	char buf[20];
	sprintf(buf, "%d", n);
	return buf;
}
	
