/*  Tidelib  Station index and locator map functions used by XTide and Tide.
    Last modified 1998-08-08

    This program uses the harmonic method to compute tide levels.
    All of the data and constants are read in from the harmonics file.
    Please refer to README for more information.

    Copyright (C) 1997  David Flater.
    Also starring:  Jef Poskanzer; Rob Miracle; Geoff Kuenning;
    Dale DePriest;

    This program addition is the contribution of Mike Hopper.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The tide prediction algorithm used in this program was developed
    with United States Government funding, so no proprietary rights
    can be attached to it.  For more information, refer to the
    following publications:

    Manual of Harmonic Analysis and Prediction of Tides.  Special
    Publication No. 98, Revised (1940) Edition.  United States
    Government Printing Office, 1941.

    Computer Applications to Tides in the National Ocean Survey.
    Supplement to Manual of Harmonic Analysis and Prediction of Tides
    (Special Publication No. 98).  National Ocean Service, National
    Oceanic and Atmospheric Administration, U.S. Department of
    Commerce, January 1982.

    The station locator index with subordinate station processing,
    region/country/state sorting and locator map were contributed by
    Mike Hopper.
*/
#include "config.h"

#include <windows.h>
#include <windowsx.h>  // <- nothing to do with Xwindows ;-) just some handy macros
#include <commdlg.h>

#include "everythi.h"
#include "arglib.h"
#include "loclib.h"
#include "wxtidres.h"

#define RAD2DEG 57.2958279088L
#define DEG2RAD 0.0174532778L
#define MAX_MAP_PNTS 10000

/* External Variables and routines */
extern char hfile_name[], location[], indexfile_name[], HelpFileName[];
extern int  list, curonly;
extern HINSTANCE g_hinst;
extern COLORREF fgmapdot_color;
extern char *youwant;
extern time_t tm2gmt (struct tm *t);
extern int slackcmp (char *a, char *b);

/* Exported variables */
int       window_list;
int       have_index=0, had_map=0;
char      IDX_type, IDX_zone[40], IDX_station_name[1024], IDX_reference_name[90];
double    IDX_lon, IDX_lat;
float     IDX_ht_mpy, IDX_ht_off, IDX_lt_mpy, IDX_lt_off;
short int IDX_rec_num=0, IDX_time_zone, IDX_sta_num, IDX_ref_file_num;
short int IDX_ht_time_off, IDX_lt_time_off, have_user_offsets = 0;

void XFillRectangle (HDC display, HWND window, COLORREF color, int x, int y,int  w,int h);
#ifdef WIN32
LRESULT CALLBACK TidesMapProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);
#endif
void CreateMapWindow(HWND hwndParent);

/* Local variables */
static char index_line[1024];
static int  index_in_memory=0, levelonly, using_lon_lat;
static int  cklevels=1, ckcurrent=1, ckrefsta=1, cksubsta=1;
static int  sortalpha=0, sortnearest=0, sortdistance=5000, num_listed;

static HWND hwndMap = NULL;

/* -----------------10/13/97 2:44PM------------------
         Variables for station indexer
 --------------------------------------------------*/
typedef struct {
   int   type;
   char *short_s;
   char *long_s;
} abbreviation_entry;

static  FILE *IndexFile;
#define REGION 1
#define COUNTRY 2
#define STATE 3

typedef struct {
   void     *IDX_next;
   short int IDX_rec_num;           // Keeps track of multiple entries w/same name
   char      IDX_type;              // Entry "TCtcI" identifier
   char      IDX_zone[40];          // Alpha region/country/state ID
   char      IDX_station_name[90];  // Name of station
   double    IDX_lon;               // Longitude (+East)
   double    IDX_lat;               // Latitude (+North)
   short int IDX_time_zone;         // Minutes offset from UTC
   short int IDX_ht_time_off;       // High tide offset in minutes
   float     IDX_ht_mpy;            // High tide multiplier (nom 1.0)
   float     IDX_ht_off;            // High tide level offset (feet?)
   short int IDX_lt_time_off;       // Low tide offset in minutes
   float     IDX_lt_mpy;            // Low tide multiplier (nom 1.0)
   float     IDX_lt_off;            // Low tide level offset (feet?)
   short int IDX_sta_num;           // Subordinate station number
   short int IDX_ref_file_num;      // # of reference file where reference station is
   char      IDX_reference_name[90];// Name of reference station
} IDX_entry;

typedef struct {
   void     *next;
   short int rec_start;
   char     *name;
} harmonic_file_entry;

static abbreviation_entry **abbreviation_list = NULL;
static IDX_entry *pIDX_first = NULL, IDX;
static char      original_name[200];
static short int original_rec_num=0;
static double    sortlon, sortlat;
static harmonic_file_entry *harmonic_file_list=NULL;

void no_mem_msg() {
printf("Could not allocate memory for index!\n");
}

/* -----------------10/13/97 2:48PM------------------
   clear string of leading and trailing white space
 --------------------------------------------------*/
void clean_string(char *str) {
   while ((str[0] <= ' ') && (str[0] != 0))
         memmove(str, str+1, strlen(str)); // Strip leading blanks
   while (strlen(str) && str[strlen(str)-1] <= ' ')
         str[strlen(str)-1] = '\0';  // Strip trailing blanks and controls
}

/* -----------------10/13/97 2:45PM------------------
   Free arrays allocated for harmonic file list
 --------------------------------------------------*/
void free_harmonic_file_list() {
harmonic_file_entry *pHarmonic, *pHarmonic_next;

   if (pHarmonic=harmonic_file_list) {
      while (pHarmonic->next) {
         pHarmonic_next = pHarmonic->next;
         free(pHarmonic->name);
         free(pHarmonic);
         pHarmonic=pHarmonic_next;
      }
      harmonic_file_list = NULL;
   }
}

/* -----------------10/13/97 2:45PM------------------
   Free arrays allocated for abbreviation list
 --------------------------------------------------*/
void free_abbreviation_list() {
int i, done;

   if (abbreviation_list) {
      done = FALSE;
      for (i=0; abbreviation_list[i] && !done; i++) {
         if (abbreviation_list[i]->type) {
            free( abbreviation_list[i]->short_s);
            free( abbreviation_list[i]->long_s);
         }
         else done = TRUE;
         free( abbreviation_list[i] );
      }
      free(abbreviation_list); // and free master pointer
      abbreviation_list = NULL;
   }
   have_index = FALSE;
}

/* -----------------10/13/97 2:45PM------------------
   Free arrays allocated for station index
 --------------------------------------------------*/
void free_station_index() {
IDX_entry *pIDX, *pIDX_prev;

   if (pIDX_first) {
      pIDX = pIDX_first;
      while (!(pIDX_prev=pIDX->IDX_next)) {
         free(pIDX);
         pIDX = pIDX_prev;
      }
      free( pIDX_first);
      pIDX_first = NULL;
   }
   index_in_memory = FALSE;
}

/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *

   Allocate space and copy string

 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
int allocate_copy_string(char **dst, char *string) {
   *dst=(char *)malloc( (int)(strlen(string)) +1);
   if (dst) {
      strcpy(*dst,string);
      return(0);
   } else return (-1);
}

/* -----------------10/20/97 12:06PM-----------------
   Decode an index data line into an IDX_entry structure.
 --------------------------------------------------*/
int build_IDX_entry(IDX_entry *pIDX ) {
int TZHr, TZMin;
harmonic_file_entry *pHarmonic;

   if (7 != sscanf( index_line, "%c%s%lf%lf%d:%d%*c%[^\n]",
         &pIDX->IDX_type,&pIDX->IDX_zone,&pIDX->IDX_lon,&pIDX->IDX_lat,&TZHr,&TZMin,
         &pIDX->IDX_station_name)) return(1);
   pIDX->IDX_time_zone = TZHr*60 + TZMin;
//if (!TZHr) printf("%d,%s",TZHr,index_line);
   if (strchr("tc",index_line[0])) { // Substation so get second line of info
      fgets( index_line, 1024, IndexFile);
      if (9 != sscanf(index_line, "%*c%d %f %f %d %f %f %d %d%*c%[^\n]",
         &pIDX->IDX_ht_time_off, &pIDX->IDX_ht_mpy, &pIDX->IDX_ht_off,
         &pIDX->IDX_lt_time_off, &pIDX->IDX_lt_mpy, &pIDX->IDX_lt_off,
         &pIDX->IDX_sta_num, &pIDX->IDX_ref_file_num, pIDX->IDX_reference_name))
         return(1);
   }
   else { // Reference stations have no offsets
      pIDX->IDX_ht_time_off = pIDX->IDX_lt_time_off = 0;
      pIDX->IDX_ht_mpy      = pIDX->IDX_lt_mpy = 1.0;
      pIDX->IDX_ht_off      = pIDX->IDX_lt_off = 0.0;
      pIDX->IDX_sta_num     = 0;
      strcpy(pIDX->IDX_reference_name, pIDX->IDX_station_name);

      pIDX->IDX_ref_file_num= 0;
      pHarmonic = harmonic_file_list;
      while (pHarmonic && (pHarmonic->rec_start <= pIDX->IDX_rec_num)) {
         pHarmonic = pHarmonic->next;
         pIDX->IDX_ref_file_num++;
      }
   }
   return(0);
}

BOOL CALLBACK BusyProcDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam) {
   if (msg == WM_INITDIALOG) {
      SetDlgItemText(hwnd, ID_TEXT, (char *) lParam);
//      SendDlgItemMessage(hwnd,ID_TEXT,WM_SETTEXT,0,lParam);
      return(1);
   }
   else return(0);
}

/* -----------------10/13/97 3:29PM------------------
   Initialize index file.
   Generate local copy of abbreviation list then make
   a linked list of decoded station entries.
 --------------------------------------------------*/
int init_index_file(int load_index, HWND hwnd) {
char s1[80], s2[80], s3[80];
long int xref_start=0;
int doing_xref=0, num_abv=0, num_IDX=0, i;
IDX_entry *pIDX, *pIDX_prev;
harmonic_file_entry *pHarmonic, *pHarmonic_prev;
HWND hwndBusy=NULL;

   if (load_index)
      hwndBusy=CreateDialogParam(g_hinst,"BUSY",hwnd,BusyProcDlg,
               (LPARAM)(LPSTR) "\r\nLoading Station Index...");
   free_harmonic_file_list();
   free_abbreviation_list();
   free_station_index();
   have_index = index_in_memory = 0;
   pIDX_first = pIDX_prev = NULL;
   if ((IndexFile = fopen( indexfile_name, "rt")) != NULL) {
      while (fgets( index_line, 1024, IndexFile) != NULL) {
         if ((index_line[0] != '#') && (index_line[0] > ' ')) {
            if (!have_index && !xref_start) {
               if (!strncmp(index_line, "XREF", 4))
                  xref_start = ftell(IndexFile);
            }
            else if (!have_index && !strncmp(index_line, "*END*", 5)) {
               if (num_abv == 0) {
                  fclose( IndexFile );
                  return(FALSE); // missing at least some data so no valid index
               }
               if (doing_xref++ == 0) { // First pass through, flag for second pass
                  fseek( IndexFile, xref_start, SEEK_SET ); // Position back to start of index
                  // Allocate memory for the array
                  if (NULL==(abbreviation_list=
                     (abbreviation_entry **)malloc((num_abv+1)*sizeof(abbreviation_entry *)))) {
                     fclose( IndexFile );
                     no_mem_msg();
                     return(FALSE);
                  }
                  else {
                     for (i=0; i<=num_abv; i++)
                        if (NULL==(abbreviation_list[i] = (abbreviation_entry *)
                                 malloc(sizeof(abbreviation_entry)))) { // If we can't allocate..
                           free_abbreviation_list();
                           fclose( IndexFile );
                           no_mem_msg();
                           return(FALSE);
                        }
                     abbreviation_list[num_abv]->type    = 0;   // Flag final entry
                     abbreviation_list[num_abv]->short_s = NULL;
                     abbreviation_list[num_abv]->long_s  = NULL;
                     num_abv = 0;
                  }
               }
                 // We're done (and no errors)
               else have_index = 1;
            } // found *END* of cross reference

            else if (!have_index && xref_start) {
               sscanf( index_line, "%s%s%[^\n]", s1, s2, s3 );
               clean_string( s3 );
               if (  (!strncmp(s1,"REGION",6))  ||
                     (!strncmp(s1,"COUNTRY",7)) ||
                     (!strncmp(s1,"STATE",5))) {
                  if (doing_xref) {
                     if (allocate_copy_string(&abbreviation_list[num_abv]->short_s, s2) ||
                        (allocate_copy_string(&abbreviation_list[num_abv]->long_s, s3))) {
                        free_abbreviation_list();
                        fclose( IndexFile );
                        no_mem_msg();
                        return(FALSE);
                     }
                     if      (!strncmp(s1,"REGION",6))  abbreviation_list[num_abv]->type = REGION;
                     else if (!strncmp(s1,"COUNTRY",7)) abbreviation_list[num_abv]->type = COUNTRY;
                     else                               abbreviation_list[num_abv]->type = STATE;
                  }
                  num_abv++;
               }
            }
            else if (have_index && (strchr("TtCcI", index_line[0]))) {
// All done with abbreviation list.
// Load index file data into memory (assuming we can).
               num_IDX++; // Keep counting entries for harmonic file stuff
               if (load_index) {
                  if (NULL!=(pIDX = malloc(sizeof(IDX_entry)))) {
                     index_in_memory = TRUE;
                     pIDX->IDX_next = NULL;
                     pIDX->IDX_rec_num = num_IDX;
                     if (build_IDX_entry(pIDX))
                        printf("Index file error at entry %d!\n", num_IDX);
                     if (pIDX_first == NULL)
                        pIDX_first = pIDX;
                     else pIDX_prev->IDX_next = pIDX;
                     pIDX_prev = pIDX;
                  }
                  else {  // Could not allocate memory for index, do long way
// We couldn't allocate memory somewhere along the line, so free all we have so far.
                     no_mem_msg();
                     free_station_index(); // Free any we have allocated so far
                  }
               }
            }
            else if (have_index && (index_line[0] == 'H')) {
// This is a new harmonic file name.
               sscanf(index_line, "Harmonic %s", s1);
               pHarmonic = harmonic_file_list;
               while (pHarmonic && pHarmonic->next)
                  pHarmonic = pHarmonic->next;
               pHarmonic_prev = pHarmonic;
               if (!(pHarmonic = malloc(sizeof(harmonic_file_entry)))) {
                  no_mem_msg();
                  free_harmonic_file_list();
               }
               else {
                  if (!harmonic_file_list)
                         harmonic_file_list = pHarmonic;
                  else pHarmonic_prev->next = pHarmonic;
                  pHarmonic->next = NULL;
                  pHarmonic->rec_start = num_IDX;
                  if (allocate_copy_string(&pHarmonic->name,s1)) {
                     no_mem_msg();
                     free_harmonic_file_list();
                  }
               }
            }
         } // if (not comment)
      } // while (more file)
      if (index_in_memory) fclose(IndexFile); // All done with file
   } // index file can't be opened
   if (hwndBusy) DestroyWindow(hwndBusy);
   return( have_index );
} // init_index_file()

/* -----------------10/13/97 3:18PM------------------
   get data from index file
   If the data has been loaded into memory (by init_index) then
   all we have to do is pass the next pointer.
   Otherwise, we have to read the next line(s) from the index file.

   rec_num parameter takes the following values:
   <0: Next record
   =0: rewind to first record
   >0: return absolute record number
 --------------------------------------------------*/
IDX_entry *get_index_data( short int rec_num ) {
static IDX_entry *pIDX;
static int looking_end, rewound=0;

   if (rec_num >= 0) {
      if (index_in_memory) {
         if (rec_num == 0) rewound = TRUE;
         else {
            rewound = FALSE;
            pIDX = pIDX_first;
            while ((--rec_num > 0) && (pIDX != NULL)) pIDX=pIDX->IDX_next;
            return (pIDX );
         }
      }
      else {
         IDX_rec_num = 0;
         fseek(IndexFile, 0l, SEEK_SET);
         looking_end = TRUE;
         while (looking_end && fgets( index_line, 1024, IndexFile) != NULL) {
            if (!strncmp(index_line, "*END*", 5)) // Stop looking when *END* found
               looking_end = FALSE;
         }
         if (!looking_end) {
            while ((rec_num > 0) && (fgets( index_line, 1024, IndexFile) != NULL))
               if (strchr("TtCcI",index_line[0])) {
                  rec_num--;
                  IDX_rec_num++;
               }
            if (rec_num < 0) {
               pIDX = &IDX;
               if (build_IDX_entry(pIDX))
                  printf("Index file error at entry %d!\n", IDX_rec_num);
               pIDX->IDX_rec_num = IDX_rec_num;
               return (pIDX);
            }
         }
      }
   }
   else {
      if (index_in_memory) {
         if (rewound) {
            rewound = FALSE;
            return( pIDX = pIDX_first);
         }
         if (pIDX != NULL) return(pIDX = pIDX->IDX_next);
      }
      else {
         while (fgets( index_line, 1024, IndexFile) != NULL) {
            if (strchr("TtCcI",index_line[0])) {
               IDX_rec_num++;
               pIDX = &IDX;
               build_IDX_entry(pIDX);
               pIDX->IDX_rec_num = IDX_rec_num;
               return (pIDX);
            }
         }
      }
   }
   return( NULL );
}

#define ALL_MSG "(All)"
HWND loc_hwnd = NULL;
static long int *pmap_list = NULL;

/* -----------------10/23/97 1:07PM------------------
   Put station location to a list to send to the map.
   List is terminated and sent when Lat/Lon illegal.
 --------------------------------------------------*/
long int *put_map( double lon, double lat ) {
   if (hwndMap != NULL) {      // Only if map is running...
      if (pmap_list == NULL) { // No map point list allocated (yet)
         if (pmap_list=malloc( MAX_MAP_PNTS * sizeof(long))) // Allocate a chunk of memory
            pmap_list[0] = 0;         // Initialize the point count
         else no_mem_msg();
      }
      if (pmap_list != NULL) {
         if (pmap_list[0] < 0) pmap_list[0] = 0;  // Turn off map display of points while updating
         if ((fabs(lon) <= 180.0) && (fabs(lat) <= 90.0)) {
            if (pmap_list[0] < MAX_MAP_PNTS) pmap_list[++pmap_list[0]] =
               ((long int)lat << 16) | ((long int)lon & 0x0000ffff);
         }
         else if ((fabs(lon) > 280.0) || (fabs(lat) > 190.0)) {
            pmap_list[0] = 0-pmap_list[0]; // Negative count = ready to output
            PostMessage(hwndMap, WM_COMMAND, IDM_MAP_PAINT_DOT, (LPARAM)pmap_list);
         }
      }
   }
   return( pmap_list );
}

/* -----------------10/18/97 3:11PM------------------
   Put a station entry into the station list.
 --------------------------------------------------*/
int put_location( char *loc, int rec_num ) {
int nItemIndex;
   if (loc_hwnd != NULL) {
      if (!have_index) {               // No index so append station type manually
         if (strstr(loc, "Current")) {
            if (levelonly) return(0); // Only showing levels
            strcat(loc, " (C)");
         }
         else {
            if (curonly) return(0); // Only showing currents
            strcat(loc, " (T)");
         }
      }
      if (loc[0] != ' ') num_listed++; // Count number of non-info stations
      if (sortalpha)                   // Let list box sort entries
         nItemIndex=SendDlgItemMessage(loc_hwnd, IDL_STATION, LB_ADDSTRING, 0, (LPARAM) ((LPSTR) loc));
      else                             // No sort, just add to end of list
         nItemIndex=SendDlgItemMessage(loc_hwnd, IDL_STATION, LB_INSERTSTRING,-1,(LPARAM) ((LPSTR) loc));
      // Set data item for this entry to rec_num
      SendDlgItemMessage(loc_hwnd, IDL_STATION, LB_SETITEMDATA, nItemIndex, MAKELONG(rec_num,0));
   }
   return( nItemIndex );
}

#define MAX_NEAREST 101
#define MAX_DISTANCE 10000.0

typedef struct {
   double distance;
   double lon;
   double lat;
   int  rec_num;
   char name[80];
} nearest_entry;

int put_nearest( IDX_entry *pIDX, char *name ) {
static int i, j, nItemIndex;
double dLat, dLon, deltNM;
static nearest_entry nearest1, nearest2, **pnear=NULL;

   nItemIndex = 0;
   if (pIDX) {
      if (!pnear) {
// We have to allocate the nearest array
         if (pnear=(nearest_entry **)malloc(MAX_NEAREST * sizeof(nearest_entry *))) {
            for (i=0; i<MAX_NEAREST; i++)
               if (!(pnear[i]=(nearest_entry *)malloc(sizeof(nearest_entry)))) {
// Oops, couldn't allocate
                  while (i>0) free( pnear[--i] );
                  free( pnear );
                  pnear = NULL;
                  no_mem_msg();
                  break;
               }
         if (pnear) pnear[0]->distance = MAX_DISTANCE;
         }
         else no_mem_msg();
      }
      if (pnear) {
         dLat = sortlat - pIDX->IDX_lat;
         dLon = sortlon - pIDX->IDX_lon;
         if      (dLat >  90.0) dLat =  180.0 - dLat;
         else if (dLat < -90.0) dLat = -180.0 - dLat;
         if      (dLon > 180.0) dLon =  360.0 - dLon;
         else if (dLon <-180.0) dLon = -360.0 - dLon;

         if (fabs(sortlat) > fabs(pIDX->IDX_lat))  // Use greater LAT for delta LON
              dLon *= cos(      sortlat*DEG2RAD);
         else dLon *= cos(pIDX->IDX_lat*DEG2RAD);

         deltNM= sqrt((dLat*dLat) + (dLon*dLon)) * 60.0;  // 60 NM per degree of Latitude

         if ((deltNM < sortdistance) && (deltNM < MAX_DISTANCE)) {
            strcpy(nearest1.name, name);
            nearest1.distance = deltNM;
            nearest1.lon = pIDX->IDX_lon;
            nearest1.lat = pIDX->IDX_lat;
            nearest1.rec_num = pIDX->IDX_rec_num;

            i = 0;
            while (deltNM > pnear[i]->distance) i++;  // Find where to put the data

            while ((i < (MAX_NEAREST-1)) && (pnear[i]->distance < MAX_DISTANCE)) {
               memcpy(&nearest2,  pnear[i], sizeof(nearest_entry)); // save current entry
               memcpy( pnear[i], &nearest1, sizeof(nearest_entry)); // copy new one in
               memcpy(&nearest1, &nearest2, sizeof(nearest_entry)); // make next entry ready
               i++;
            }
            if (i < (MAX_NEAREST-1))
               memcpy( pnear[i++],&nearest1,sizeof(nearest_entry));// copy last one in if room

            pnear[i]->distance = MAX_DISTANCE; // List always terminates with this
         }
      }
   }

   else if (pnear) {
      i = 0;
      while ((i<MAX_NEAREST) && (pnear[i]->distance < MAX_DISTANCE)) {
         put_map( pnear[i]->lon, pnear[i]->lat );
         j = put_location( pnear[i]->name, pnear[i]->rec_num );
         if (pnear[i++]->rec_num == IDX_rec_num)
            nItemIndex = j;             // Tell caller where selected entry is
      }
      put_map( 1000, 1000 );            // Send data to map

      for (i=0; i<MAX_NEAREST; i++) free( pnear[i] );
      free( pnear );
      pnear = NULL;
   }
   return( nItemIndex );
}
/* -----------------10/13/97 3:16PM------------------
   Find an entry for a given station and load it's info.
   If rec_num is a valid index record number, that info is
   loaded, otherwise a station name search is performed.
 --------------------------------------------------*/
int load_location_info(char *station_name, int rec_num) {
char type, temp_station[90];
int s, found, sloppy=TRUE;
IDX_entry *pIDX;

// First extract (then remove) any station type ("TtCc") appended to the station name.
   type = '\0';
   strncpy( temp_station, station_name, sizeof(temp_station)-1 );
   temp_station[sizeof(temp_station)-1] = '\0'; // If station name is LOOOONNNGGGG, clip it
   if (strlen(temp_station) > 4) {
      s = strlen(temp_station) - 3;
      if ((temp_station[s] == '(') && (temp_station[s+2] == ')')) {
         type = temp_station[s+1];
         if (strchr("TtCc", type)) {
            sloppy = FALSE;
            temp_station[s-1] = '\0';
         }
      }
   }
// Now search for that station name and type code
   clean_string( temp_station );
   if (have_index) {
      if ((rec_num <= 0) || ((rec_num>0) && (!(pIDX=get_index_data( rec_num ))))) {
         get_index_data( 0 ); // Have to search for it
         found = FALSE;
         while (!found && (NULL!=(pIDX=(get_index_data( -1 )))))
            if (((type == '\0') || (type == pIDX->IDX_type)) &&
               (!strcmp(pIDX->IDX_station_name,temp_station) ||
                (sloppy && !slackcmp(pIDX->IDX_station_name,temp_station))) )
               found = TRUE;
      }
      if (pIDX) {
// We found the entry!
//printf("IDX_type=%c, IDX_lon=%f, IDX_lat=%f \n",pIDX->IDX_type, pIDX->IDX_lon, pIDX->IDX_lat);
//printf("IDX_station_name=[%s]\n",pIDX->IDX_station_name);
//printf("IDX_ht_time_off=%d, IDX_ht_mpy=%f, IDX_ht_off=%f\n",pIDX->IDX_ht_time_off,pIDX->IDX_ht_mpy,pIDX->IDX_ht_off);
//printf("IDX_lt_time_off=%d, IDX_lt_mpy=%f, IDX_lt_off=%f\n",pIDX->IDX_lt_time_off,pIDX->IDX_lt_mpy,pIDX->IDX_lt_off);
//printf("IDX_sta_num=%d,IDX_ref_file_num=%d,IDX_reference_name=%s\n",pIDX->IDX_sta_num,pIDX->IDX_ref_file_num,pIDX->IDX_station_name);
         IDX_rec_num     = pIDX->IDX_rec_num;
         IDX_type        = pIDX->IDX_type;
         strcpy(IDX_zone, pIDX->IDX_zone);
         IDX_lon         = pIDX->IDX_lon;
         IDX_lat         = pIDX->IDX_lat;
         IDX_time_zone   = pIDX->IDX_time_zone;
         strcpy( IDX_station_name, pIDX->IDX_station_name);
         IDX_ht_time_off = pIDX->IDX_ht_time_off;
         IDX_ht_mpy      = pIDX->IDX_ht_mpy;
         IDX_ht_off      = pIDX->IDX_ht_off;
         IDX_lt_time_off = pIDX->IDX_lt_time_off;
         IDX_lt_mpy      = pIDX->IDX_lt_mpy;
         IDX_lt_off      = pIDX->IDX_lt_off;
         IDX_sta_num     = pIDX->IDX_sta_num;
         IDX_ref_file_num= pIDX->IDX_ref_file_num;
         strcpy(IDX_reference_name, pIDX->IDX_reference_name);
         return( TRUE ); //  "We found it"
      }
   }
   else {  // We don't have an index so fake it
      strcpy( IDX_station_name,   temp_station );
      strcpy( IDX_reference_name, temp_station );
      if (strstr(station_name, "Current"))
           IDX_type = 'C';
      else IDX_type = 'T';
      strcpy(IDX_zone, ":::");
      IDX_rec_num     = 0;
      IDX_lon         = IDX_lat = 0.0;
      IDX_ht_mpy      = IDX_lt_mpy = 1.0;
      IDX_ht_off      = IDX_lt_off = 0.0;
      IDX_ht_time_off = IDX_lt_time_off = 0;
      IDX_time_zone   = 0;
      IDX_sta_num     = 0;
      IDX_ref_file_num= 0;
      return ( TRUE ); // "We found it"
   }
   return( FALSE ); // "We didn't find it"
}

/* -----------------10/13/97 3:16PM------------------
   Load data for a given location and start the tide process.
 --------------------------------------------------*/
int load_location_data(char *station_name, int rec_num) {
harmonic_file_entry *pHarmonic;
HCURSOR hcurSave;
int i, opened;

   opened=(!index_in_memory && (ftell(IndexFile)==-1L) &&
          ((IndexFile = fopen( indexfile_name, "rt")) != NULL));
   if (load_location_info( station_name, rec_num )) {
      if ((rec_num==0) && strchr("TC",IDX_type) &&
          ((Ihttimeoff) || (Ilttimeoff) ||
          (Ihlevelmult  != 1.0) ||
          (Ihtleveloff  != 0.0) ||
          (Illevelmult  != 1.0) ||
          (Iltleveloff  != 0.0) ||
            have_user_offsets)) {
// Is first time, is reference station, and have user define offsets
         have_user_offsets = IDX_rec_num;
         IDX_rec_num = 0;
         httimeoff = Ihttimeoff;
         lttimeoff = Ilttimeoff;
         hlevelmult= Ihlevelmult;
         htleveloff= Ihtleveloff;
         llevelmult= Illevelmult;
         ltleveloff= Iltleveloff;
      }
      else {
         have_user_offsets = 0;
/* -----------------3/15/98 10:34AM------------------
 * Changed following to use alternate value if <unk>
 * --------------------------------------------------*/
         if (labs(IDX_ht_time_off) != 1111)
              httimeoff = (long int)IDX_ht_time_off * 60;
         else if (labs(IDX_lt_time_off) != 1111)
              httimeoff = (long int)IDX_lt_time_off * 60;
         else httimeoff = 0;

         if (labs(IDX_lt_time_off) != 1111)
              lttimeoff = (long int)IDX_lt_time_off * 60;
         else lttimeoff = httimeoff;

         if ((IDX_ht_mpy > 0.1) && (IDX_ht_mpy < 10.0))
              hlevelmult = IDX_ht_mpy;
         else if ((IDX_lt_mpy > 0.1) && (IDX_lt_mpy < 10.0))
              hlevelmult = IDX_lt_mpy;
         else hlevelmult = 1.0;

         if ((IDX_lt_mpy > 0.1) && (IDX_lt_mpy < 10.0))
              llevelmult = IDX_lt_mpy;
         else llevelmult = hlevelmult;

         if (fabs(IDX_ht_off) < 100.0)
              htleveloff = IDX_ht_off;
         else if (fabs(IDX_lt_off) < 100.0)
              htleveloff = IDX_lt_off;
         else htleveloff = 0;

         if (fabs(IDX_lt_off) < 100.0)
              ltleveloff = IDX_lt_off;
         else ltleveloff = htleveloff;
      }
      strcpy(location, IDX_reference_name);
      if (IDX_ref_file_num && (pHarmonic=harmonic_file_list)) {
         i = 1;
         while (pHarmonic && (i < IDX_ref_file_num)) {
            pHarmonic = pHarmonic->next;
            i++;
         }
         if (pHarmonic)
            check_file_path(hfile_name, pHarmonic->name);
//            strcpy(hfile_name, pHarmonic->name);
      }
   }
   else {
      if (opened) fclose(IndexFile);
      return(FALSE); // No got it!
   }
   if (opened) fclose(IndexFile);
   hcurSave = SetCursor(LoadCursor(NULL, IDC_WAIT)); // Load hourglass cursor
   load_data(); //Load data and start tide'ing
   fudge_constituents(youwant);
   amplitude = 0.0;                // Force multiplier re-compute
   happy_new_year (yearoftimet (time(NULL)));//Force new multipliers
   SetCursor(hcurSave);                              // Back to normal cursor
   if (loctz)
      change_time_zone (tzfile); /* Moof! */
   if (!have_user_offsets || !custom_name) {
      if (custom_name) free(custom_name);
      if (have_user_offsets)
           custom_name = stradoop(location);
      else custom_name = stradoop(IDX_station_name);
   }
   return(TRUE); // Got it!
}

void convert_special(char *str) {
int i;
   for (i=0; i<strlen(str); i++) {
         if (str[i] == 0x81) str[i]=0xfc;
         if (str[i] == 0x94) str[i]=0xf6;
   }
}

/* -----------------10/15/97 7:06AM------------------
   Generate a multi-line string describing the station last selected.
 --------------------------------------------------*/
char *describe_station(char * description) {
int lat_deg, lon_deg;
long int i_lon, i_lat;
char lon_ch, lat_ch, ht_hr_ch, lt_hr_ch, *s_type;
char s_region[80], s_country[80], s_state[80], station_name[256];
char s_htt[10], s_ltt[10], s_hto[40], s_lto[40], s_dist[40];
double lat_min, lon_min;
double dLon, dLat, deltNM;
int dir, i;
harmonic_file_entry *pHarmonic;

   if (have_user_offsets && (have_user_offsets == IDX_rec_num)) {
      s_type = "User offsets";
      if (strcmp(location, custom_name))
          sprintf(station_name,"%s\r\n%s", custom_name, location);
      else strcpy(station_name, location);
   }
   else {
      strcpy(station_name, IDX_station_name);
      convert_special(station_name);
           if (IDX_type == 'C') s_type = "Current reference";
      else if (IDX_type == 'c') s_type = "Current substation";
      else if (IDX_type == 'T') s_type = "Tidal reference";
      else if (IDX_type == 't') s_type = "Tidal substation";
   }

   if (have_index) {
      i_lon = fabs(IDX_lon) * 1000.0;
      i_lat = fabs(IDX_lat) * 1000.0;
      lon_deg = i_lon / 1000;
      lon_min = ((float)(i_lon % 1000) * 0.06);
      lat_deg = i_lat / 1000;
      lat_min = ((float)(i_lat % 1000) * 0.06);
      if (IDX_lon < 0.0) lon_ch = 'W';
      else               lon_ch = 'E';
      if (IDX_lat < 0.0) lat_ch = 'S';
      else               lat_ch = 'N';

      dLat = sortlat - IDX_lat;
      dLon = sortlon - IDX_lon;
      if      (dLat >  90.0) dLat =  180.0 - dLat;
      else if (dLat < -90.0) dLat = -180.0 - dLat;
      if      (dLon > 180.0) dLon =  360.0 - dLon;
      else if (dLon <-180.0) dLon = -360.0 - dLon;
      if (fabs(sortlat) > fabs(IDX_lat))
           dLon *= cos(sortlat*DEG2RAD);
      else dLon *= cos(IDX_lat*DEG2RAD);
      deltNM= sqrt((dLat*dLat) + (dLon*dLon)) * 60.0;  // 60 NM per degree of Latitude
      if (deltNM < 99.5)
           sprintf(s_dist, "%0.1lf", deltNM);
      else sprintf(s_dist, "%0.0lf", deltNM);

      if      ((dLon == 0) && (dLat >=0)) dir = 0;
      else if ((dLon == 0) && (dLat < 0)) dir = 180;
      else dir = (atan2( dLon, dLat )*RAD2DEG) + 180;

      sprintf(description,"%s\r\n"
            "Position:%0d %0.1f' %c, %0d %0.1f' %c  Type:%s\r\n"
            "%sNM at %d from %s. Timezone:utc%+d:%02d\r\n",
//            "%0.1lfNM at %d from present station. Timezone:utc%+d:%02d\r\n",
            station_name, lon_deg, lon_min, lon_ch, lat_deg, lat_min, lat_ch, s_type,
            s_dist, dir, using_lon_lat?"Lon/Lat":"present station",
            (IDX_time_zone / 60), (IDX_time_zone % 60));

      if (strstr(s_type, "sub")) { // Substation type
         if (IDX_ht_time_off < 0) ht_hr_ch = '-';
         else                     ht_hr_ch = '+';
         if (IDX_lt_time_off < 0) lt_hr_ch = '-';
         else                     lt_hr_ch = '+';

         if (abs(IDX_ht_time_off) >= 1111)
              strcpy(s_htt, "<unk>");
         else sprintf(s_htt,"%c%0d:%02d", ht_hr_ch,
               abs(IDX_ht_time_off)/60, abs(IDX_ht_time_off)%60);

         if (abs(IDX_lt_time_off) >= 1111)
              strcpy(s_ltt, "<unk>");
         else sprintf(s_ltt,"%c%0d:%02d", lt_hr_ch,
               abs(IDX_lt_time_off)/60, abs(IDX_lt_time_off)%60);

         if (fabs(IDX_ht_off) >= 111.0)
              sprintf(s_hto, "*%.2f offset +<unk>", IDX_ht_mpy);
         else sprintf(s_hto, "*%.2f offset %+.1lf", IDX_ht_mpy, IDX_ht_off);

         if (fabs(IDX_lt_off) >= 111.0)
              sprintf(s_lto, "*%.2f  offset +<unk>", IDX_lt_mpy);
         else sprintf(s_lto, "*%.2f  offset %+.1lf", IDX_lt_mpy, IDX_lt_off);

         sprintf(description+strlen(description),
            "Reference: %s\r\n"
//            "Station identifier: %d\r\n"
            "High tide time: %s,  level multiply %s\r\n"
            "Low tide time: %s,  level multiply %s\r\n",
            IDX_reference_name, //IDX_sta_num,
            s_htt, s_hto,
            s_ltt, s_lto);
      }
      else { // Reference station type
         pHarmonic = harmonic_file_list;
         i = 1;
         while (pHarmonic && (i < IDX_ref_file_num)) {
            pHarmonic = pHarmonic->next;
            i++;
         }
         if (pHarmonic && strlen(pHarmonic->name))
            sprintf(description+strlen(description),
               "Defined in harmonic constituent file: %s\r\n", pHarmonic->name);
      }

      sscanf( IDX_zone, "%[^:]:%[^:]:%[^:]:", s_region, s_country, s_state);
      for (i=0; abbreviation_list[i]->type && ((abbreviation_list[i]->type != REGION) ||
         strcmp(abbreviation_list[i]->short_s, s_region)); i++);
      if (abbreviation_list[i]->long_s != NULL)
         sprintf( description+strlen(description), "Region: %s\r\n", abbreviation_list[i]->long_s);

      for (i=0; abbreviation_list[i]->type && ((abbreviation_list[i]->type != COUNTRY) ||
         strcmp(abbreviation_list[i]->short_s, s_country)); i++);
      if (abbreviation_list[i]->long_s != NULL)
         sprintf( description+strlen(description), "Country: %s      ", abbreviation_list[i]->long_s);

      for (i=0; abbreviation_list[i]->type && ((abbreviation_list[i]->type != STATE) ||
         strcmp(abbreviation_list[i]->short_s, s_state)); i++);
      if (abbreviation_list[i]->long_s != NULL)
         sprintf( description+strlen(description), "State: %s\r\n", abbreviation_list[i]->long_s);
   }
   else
      sprintf( description, "%s\r\n"
          "Type:%s\r\n"
          "No additional information available without index file\r\n",
          IDX_station_name, s_type );
   return( description );
}

/* -----------------9/7/98 10:37AM-------------------
 * Build test ID
 * --------------------------------------------------*/
char *build_test_ID() {
DWORD dwIndex;
int i;
char l_region[80], l_country[80], l_state[80];
char *r, *c, *s;
static char test_ID[80];

   dwIndex = SendDlgItemMessage(loc_hwnd, IDL_REGION, CB_GETCURSEL, 0, 0L);
   if (dwIndex != CB_ERR)
      SendDlgItemMessage(loc_hwnd, IDL_REGION,CB_GETLBTEXT,
         (WPARAM) dwIndex, (LPARAM) ((LPSTR) l_region));
   for (i=0; abbreviation_list[i]->type && ((abbreviation_list[i]->type != REGION) ||
       strcmp(abbreviation_list[i]->long_s, l_region)); i++);
   r = abbreviation_list[i]->short_s;

   dwIndex = SendDlgItemMessage(loc_hwnd, IDL_COUNTRY, CB_GETCURSEL, 0, 0L);
   if (dwIndex != CB_ERR)
      SendDlgItemMessage(loc_hwnd, IDL_COUNTRY,CB_GETLBTEXT,
         (WPARAM) dwIndex, (LPARAM) ((LPSTR) l_country));
   for (i=0; abbreviation_list[i]->type && ((abbreviation_list[i]->type != COUNTRY) ||
       strcmp(abbreviation_list[i]->long_s, l_country)); i++);
   c = abbreviation_list[i]->short_s;

   dwIndex = SendDlgItemMessage(loc_hwnd, IDL_STATE, CB_GETCURSEL, 0, 0L);
   if (dwIndex != CB_ERR)
      SendDlgItemMessage(loc_hwnd, IDL_STATE,CB_GETLBTEXT,
         (WPARAM) dwIndex, (LPARAM) ((LPSTR) l_state));
   for (i=0; abbreviation_list[i]->type && ((abbreviation_list[i]->type != STATE) ||
       strcmp(abbreviation_list[i]->long_s, l_state)); i++);
   s = abbreviation_list[i]->short_s;

   if (!r) r = "";
   if (!c) c = "";
   if (!s) s = "";
   sprintf(test_ID,"%s:%s:%s:",r,c,s); // Build test ID template
   return(test_ID);
}

/* -----------------9/7/98 10:49AM-------------------
 * Match test ID with current data.
 * --------------------------------------------------*/
int it_matches_test_ID(IDX_entry *pIDX, char *test_ID) {
char *k;
int i, len_test;

   len_test = strlen(test_ID);
   k = pIDX->IDX_zone;
   for (i=0; i<len_test; i++)  // Now check ID
      if ( test_ID[i] != *k++ ) {
         if (test_ID[i] != ':') break;
         else if ((i!=0) && (test_ID[i-1] != ':')) break;
         else { // Skip to next field in zone
            if (NULL==(k = strchr(k-1, ':'))) break;
            else k++;
         }
      }
   return(i == len_test);
}

/* -----------------10/19/97 5:37AM------------------
   Load a list of stations matching defined criteria
   into the location list box.
 --------------------------------------------------*/
int load_station_list() {
int cursave, levelsave, rec_found, t_rec, startIDX;
char loc_name[90], first_ch[6], *test_ID;
IDX_entry *pIDX, *matching_pIDX;
char *k;
int i, len_test;

   matching_pIDX = NULL;
   rec_found = 0;
   if (!have_index) {
// No index so simulate "-list" command and let load_data handle it.
      list = window_list = 1;
      cursave   = curonly;
      levelsave = levelonly;
      curonly   = (!cklevels &&  ckcurrent);
      levelonly = ( cklevels && !ckcurrent);
      load_data();   // Load the list of stations matching criteria
      levelonly = levelsave;
      curonly   = cursave;
      list = window_list = 0;
      if (LB_ERR!=(t_rec=(SendDlgItemMessage(loc_hwnd, IDL_STATION,
         LB_FINDSTRING, -1, (LPARAM) ((LPSTR) location)))))
         rec_found = t_rec;
   }

   else { // We have an index file, phew!
// Build a zone ID match template for region, country, state
      test_ID = build_test_ID();
// Now build station type string
      strcpy(first_ch,"");
      if (ckrefsta) {
         if (cklevels)  strcat(first_ch, "T"); // Tidal reference
         if (ckcurrent) strcat(first_ch, "C"); // Current reference
      }
      if (cksubsta) {
         if (cklevels)  strcat(first_ch, "t"); // Tidal substations
         if (ckcurrent) strcat(first_ch, "c"); // Current substations
      }
      if (!sortalpha && !sortnearest)
         strcat(first_ch, "I"); // No sorting of stations so include info lines
//printf("test_ID=[%s], first_ch=[%s]\n",test_ID,first_ch);
// Now search for stations matching criteria
      len_test = strlen(test_ID);
      get_index_data( 0 ); // Rewind index file
      while (NULL!=(pIDX=(get_index_data( -1 )))) {
         if (strchr(first_ch, pIDX->IDX_type)) { // Matched station type
//            len_test = strlen(test_ID);
            k = pIDX->IDX_zone;
            for (i=0; i<len_test; i++)  // Now check ID
               if ( test_ID[i] != *k++ ) {
                  if (test_ID[i] != ':') break;
                  else if ((i!=0) && (test_ID[i-1] != ':')) break;
                  else { // Skip to next field in zone
                     if (NULL==(k = strchr(k-1, ':'))) break;
                     else k++;
                  }
               }
            if (i == len_test) {
//            if (it_matches_test_ID(pIDX, test_ID)) {
// ding! ding! ding! we found a matching entry!
               strcpy(loc_name, pIDX->IDX_station_name);
               convert_special(loc_name);
               if (pIDX->IDX_type != 'I') { // Append station type to non-info lines
                  sprintf(loc_name+strlen(loc_name), " (%c)", pIDX->IDX_type);
                  if (sortnearest)
                     put_nearest( pIDX, loc_name );
                  else
                     put_map( pIDX->IDX_lon, pIDX->IDX_lat );
               }
// Don't send data if sorting by distance or Info line and only reference stations
               if (!sortnearest && (!(!cksubsta && (pIDX->IDX_type == 'I')))) {
                  t_rec = put_location( loc_name, pIDX->IDX_rec_num );
                  if (pIDX->IDX_rec_num == IDX_rec_num) {
                     matching_pIDX = pIDX;
                     rec_found = t_rec;
                  }
               }
            }
         }
//         else printf("missed test_ID=[%s], first_ch=[%s]\n",test_ID,first_ch);
      }
// We've now sent all the stations matching sort parameters to the list
      if (sortnearest) rec_found = put_nearest( NULL, NULL );
      else {
         put_map( 1000, 1000 ); // Send list to map
         if (sortalpha) {
// Oh, dear.  When we let windows sort the list, we don't know where in the *hhhad* the selected
// entry ends up so we're going to have to search through the list for the station name and match
// record numbers until we find it (or we reach the end of the list).
            rec_found = -1;
            if (matching_pIDX != NULL) {  // If NULL, the station we want is NOT in the list!
               strcpy(loc_name, IDX_station_name);
               convert_special(loc_name);
               startIDX = -1; // Start from beginning of list
               while ((rec_found < 0) &&
                     (LB_ERR!=(t_rec=(SendDlgItemMessage(loc_hwnd, IDL_STATION,
                        LB_FINDSTRING, startIDX, (LPARAM) ((LPSTR) loc_name)))))) {
                  if (t_rec != LB_ERR) {
                     if (IDX_rec_num == LOWORD(SendDlgItemMessage( loc_hwnd, IDL_STATION,
                                        LB_GETITEMDATA, t_rec, 0L)))
                        rec_found = t_rec; // We found it!
                     else if (t_rec > startIDX) startIDX = t_rec; // Next time start search from here
                     else rec_found = 0; // We looped around so give up
                  }
               }
            if (rec_found < 0) rec_found = 0;
            }
         }
      }
   }
   return (rec_found );
}

/* -----------------10/31/97 6:07PM------------------
   Convert DDD.MMSSS format to decimal degrees.
 --------------------------------------------------*/
int cvt_DMM_2_deg(double *dst, char *str) {
int ret;
long int t_deg, t_min;
double t_pos, result;

   sscanf(str, "%lf", &t_pos);
   t_deg = t_pos * 1000;
   t_min = t_deg % 1000;
   result = (long int)(t_deg / 1000) + ((double)t_min / 600.0);
   ret = result != *dst;
   if ((abs(t_deg) < 180000) && (abs(t_min) < 600))
      *dst = result;
   return(ret);
}

/* -----------------10/31/97 6:07PM------------------
   Convert decimal degrees to DDD.MMSSS format.
 --------------------------------------------------*/
void cvt_deg_2_DMM(char *str, double pos) {
long int t_deg, t_min;
double t_pos;

   t_deg = pos * 1000;
   t_min = (t_deg % 1000);
   t_pos = (double)(t_deg / 1000) + ((double)t_min * .0006);
   sprintf(str, "%.3lf", t_pos);
}

/* -----------------10/31/97 6:06PM------------------
   Check if current station list entry is an info line.
 --------------------------------------------------*/
int is_info_line( HWND hwnd, DWORD dwIndex ) {
char name[80];

   if (LB_ERR!=SendDlgItemMessage(hwnd, IDL_STATION,LB_GETTEXT,
      (WPARAM) dwIndex, (LPARAM) ((LPSTR) name))) {
      if (name[0] == ' ') return( 0 );  // Return "Yes, entry is info"
      else return( 1 );                 // Return "No, entry is not info"
   }
   return( -1 );                        // Return "Error"
}

#define DIR_UP   ( 1)
#define DIR_DOWN (-1)
DWORD prev_sel;

/* -----------------10/31/97 6:09PM------------------
   Find the next non-info line in station list UP or DOWN
   from current station list entry.
 --------------------------------------------------*/
DWORD find_non_info_line( HWND hwnd, int dir ) {
DWORD dwIndex;
int is_info;

   if (LB_ERR!=(dwIndex=SendDlgItemMessage(hwnd, IDL_STATION, LB_GETCURSEL, 0, 0L))) {// Get current entry
      while (!(is_info=is_info_line(hwnd, dwIndex))) dwIndex += dir;
      if ( is_info == 1 ) return ( dwIndex );
   }
   return( LB_ERR );
}

/* -----------------9/7/98 11:16AM-------------------
 *
 * --------------------------------------------------*/
int any_zone_match(char *test_ID) {
char *k;
int i, len_test;
IDX_entry *pIDX;

   len_test = strlen(test_ID);
   get_index_data( 0 ); // Rewind index file
   while (NULL!=(pIDX=(get_index_data( -1 )))) {
      k = pIDX->IDX_zone;
      for (i=0; i<len_test; i++)  // Now check ID
         if ( test_ID[i] != *k++ ) {
            if (test_ID[i] != ':') break;
            else if ((i!=0) && (test_ID[i-1] != ':')) break;
            else { // Skip to next field in zone
               if (NULL==(k = strchr(k-1, ':'))) break;
               else k++;
            }
         }
      if (i == len_test) {
         return(TRUE);
      }
   }
   return(FALSE);
}

/* -----------------9/7/98 10:02AM-------------------
 * Changes region/country/state buttons
 * --------------------------------------------------*/
void CheckRegionCountryState(HWND hwnd, WORD type) {
char test_ID[80], region[80], country[80], state[80];

   strcpy(test_ID, build_test_ID());
   if (!any_zone_match(test_ID)) {
      sscanf( test_ID, "%[^:]:%[^:]:%[^:]:", region, country, state);
      if ( type == IDL_REGION) {
         sprintf(test_ID,"%s:%s::",region,country);
         if (!any_zone_match(test_ID))
            SendDlgItemMessage(hwnd, IDL_COUNTRY,CB_SETCURSEL, 0, 0L);
         sprintf(test_ID,"%s::%s:",region,state);
         if (!any_zone_match(test_ID))
            SendDlgItemMessage(hwnd, IDL_STATE,  CB_SETCURSEL, 0, 0L);
      }
      else if ( type == IDL_COUNTRY) {
         sprintf(test_ID,"%s:%s::",region,country);
         if (!any_zone_match(test_ID))
            SendDlgItemMessage(hwnd, IDL_REGION, CB_SETCURSEL, 0, 0L);
         sprintf(test_ID,":%s:%s:",country,state);
         if (!any_zone_match(test_ID))
            SendDlgItemMessage(hwnd, IDL_STATE,  CB_SETCURSEL, 0, 0L);
      }
      else {
         sprintf(test_ID,"%s::%s:",region,state);
         if (!any_zone_match(test_ID))
            SendDlgItemMessage(hwnd, IDL_REGION, CB_SETCURSEL, 0, 0L);
         sprintf(test_ID,":%s:%s:",country,state);
         if (!any_zone_match(test_ID))
            SendDlgItemMessage(hwnd, IDL_COUNTRY,CB_SETCURSEL, 0, 0L);
      }
   }
}

#define UPDATE_INIT    1
#define UPDATE_LIST    2
#define UPDATE_STATION 3

/* -----------------10/31/97 6:10PM------------------
   This is one of the biggies.
   Does ALL updates to location dialog based on update_type.
 --------------------------------------------------*/
void new_location_dlg (HWND hwnd, int update_type) {
char loc_str[1024];
int i, nItemIndex;
char region[50], country[50], state[50];
HCURSOR hcurSave;

   loc_hwnd = hwnd;
   if (!ckcurrent && !cklevels) ckcurrent = cklevels = TRUE;
   CheckDlgButton(hwnd,IDL_FLAG_CURRENT,ckcurrent);
   CheckDlgButton(hwnd,IDL_FLAG_LEVELS, cklevels);

   CheckDlgButton(hwnd,IDL_NOSORT, (!sortalpha && !sortnearest));
   CheckDlgButton(hwnd,IDL_ALPHABET, sortalpha);
   CheckDlgButton(hwnd,IDL_NEAREST,  sortnearest);

   if (!have_index) {
// No station index so disable most of the controls.
      CheckDlgButton(hwnd,IDL_FLAG_REFERENCE,  TRUE);
      CheckDlgButton(hwnd,IDL_FLAG_SUBSTATION, FALSE);
      SendDlgItemMessage(hwnd,IDL_REGION, CB_RESETCONTENT, 0,0L);
      SendDlgItemMessage(hwnd,IDL_REGION ,CB_ADDSTRING, 0, (LPARAM) ((LPSTR) "* No index file *"));
      SendDlgItemMessage(hwnd,IDL_REGION, CB_SETCURSEL, 0, 0L);

      SendDlgItemMessage(hwnd,IDL_COUNTRY,CB_RESETCONTENT, 0,0L);
      SendDlgItemMessage(hwnd,IDL_COUNTRY,CB_ADDSTRING, 0, (LPARAM) ((LPSTR) "* No index file *"));
      SendDlgItemMessage(hwnd,IDL_COUNTRY,CB_SETCURSEL, 0, 0L);

      SendDlgItemMessage(hwnd,IDL_STATE,  CB_RESETCONTENT, 0,0L);
      SendDlgItemMessage(hwnd,IDL_STATE,  CB_ADDSTRING, 0, (LPARAM) ((LPSTR) "* No index file *"));
      SendDlgItemMessage(hwnd,IDL_STATE,  CB_SETCURSEL, 0, 0L);

      SendDlgItemMessage(hwnd,IDL_REGION, EM_SETREADONLY, TRUE, 0L);
      SendDlgItemMessage(hwnd,IDL_COUNTRY,EM_SETREADONLY, TRUE, 0L);
      SendDlgItemMessage(hwnd,IDL_STATE,  EM_SETREADONLY, TRUE, 0L);
      SendDlgItemMessage(hwnd,IDL_FLAG_SUBSTATION,EM_SETREADONLY, TRUE, 0L);
      SendDlgItemMessage(hwnd,IDL_LON,    EM_SETREADONLY, TRUE, 0L);
      SendDlgItemMessage(hwnd,IDL_LAT,    EM_SETREADONLY, TRUE, 0L);
   }

   else {
// We have an index
      if (!cksubsta && !ckrefsta) ckrefsta = cksubsta = TRUE;
      CheckDlgButton(hwnd,IDL_FLAG_REFERENCE,  ckrefsta);
      CheckDlgButton(hwnd,IDL_FLAG_SUBSTATION, cksubsta);
      if (update_type != UPDATE_LIST) {
         cvt_deg_2_DMM( loc_str, sortlon);
         SetDlgItemText(hwnd,IDL_LON, loc_str);
         cvt_deg_2_DMM( loc_str, sortlat);
         SetDlgItemText(hwnd,IDL_LAT, loc_str);
      }
//      SendDlgItemMessage(hwnd, IDL_LON, EM_SETREADONLY, !sortnearest, 0L);
//      SendDlgItemMessage(hwnd, IDL_LAT, EM_SETREADONLY, !sortnearest, 0L);

      if (update_type == UPDATE_INIT) {
         SendDlgItemMessage(hwnd,IDL_REGION, CB_RESETCONTENT, 0,0L);
         SendDlgItemMessage(hwnd,IDL_REGION, CB_ADDSTRING, 0, (LPARAM) ((LPSTR) ALL_MSG));
         SendDlgItemMessage(hwnd,IDL_REGION, CB_SETCURSEL, 0, 0L);

         SendDlgItemMessage(hwnd,IDL_COUNTRY,CB_RESETCONTENT, 0,0L);
         SendDlgItemMessage(hwnd,IDL_COUNTRY,CB_ADDSTRING, 0, (LPARAM) ((LPSTR) ALL_MSG));
         SendDlgItemMessage(hwnd,IDL_COUNTRY,CB_SETCURSEL, 0, 0L);

         SendDlgItemMessage(hwnd,IDL_STATE,  CB_RESETCONTENT, 0,0L);
         SendDlgItemMessage(hwnd,IDL_STATE,  CB_ADDSTRING, 0, (LPARAM) ((LPSTR) ALL_MSG));
         SendDlgItemMessage(hwnd,IDL_STATE,  CB_SETCURSEL, 0, 0L);
// Load region, country, and state combo box entries
         for (i=0; abbreviation_list[i]->type; i++) {
            if      (abbreviation_list[i]->type == REGION)
               SendDlgItemMessage(hwnd, IDL_REGION, CB_ADDSTRING, 0,
                     (LPARAM) ((LPSTR) abbreviation_list[i]->long_s ));

            else if (abbreviation_list[i]->type == COUNTRY)
               SendDlgItemMessage(hwnd, IDL_COUNTRY, CB_ADDSTRING, 0,
                     (LPARAM) ((LPSTR) abbreviation_list[i]->long_s ));

            else if (abbreviation_list[i]->type == STATE)
               SendDlgItemMessage(hwnd, IDL_STATE, CB_ADDSTRING, 0,
                     (LPARAM) ((LPSTR) abbreviation_list[i]->long_s ));
         }
// Select the region, country, and state that matches current location
         sscanf(IDX_zone, "%[^:]:%[^:]:%[^:]", region, country, state);
         for (i=0; abbreviation_list[i]->type; i++) {
            if      ((abbreviation_list[i]->type == REGION) &&
                  (!strcmp(abbreviation_list[i]->short_s, region)))
               SendDlgItemMessage(hwnd, IDL_REGION, CB_SELECTSTRING, 0, (LPARAM)
                        ((LPSTR) abbreviation_list[i]->long_s));

            else if ((abbreviation_list[i]->type == COUNTRY) &&
                  (!strcmp(abbreviation_list[i]->short_s, country)))
               SendDlgItemMessage(hwnd, IDL_COUNTRY, CB_SELECTSTRING, 0, (LPARAM)
                        ((LPSTR) abbreviation_list[i]->long_s));

            else if ((abbreviation_list[i]->type == STATE) &&
                  (!strcmp(abbreviation_list[i]->short_s, state)))
               SendDlgItemMessage(hwnd, IDL_STATE, CB_SELECTSTRING, 0, (LPARAM)
                        ((LPSTR) abbreviation_list[i]->long_s));
         }
      }
   }
// Check the region, country, state search boxes if current selection is not "(All)"
   CheckDlgButton(hwnd,IDL_FLAG_REGION,
       (SendDlgItemMessage(loc_hwnd,IDL_REGION, CB_GETCURSEL, 0, 0L)));
   CheckDlgButton(hwnd,IDL_FLAG_COUNTRY,
       (SendDlgItemMessage(loc_hwnd,IDL_COUNTRY,CB_GETCURSEL, 0, 0L)));
   CheckDlgButton(hwnd,IDL_FLAG_STATE,
       (SendDlgItemMessage(loc_hwnd,IDL_STATE,  CB_GETCURSEL, 0, 0L)));

   CheckDlgButton(hwnd,IDL_FLAG_MAP, hwndMap!=NULL);

   if (update_type != UPDATE_STATION) {
// Not simple station info update, we gotta load a new station list
      SendDlgItemMessage(hwnd, IDL_STATION, LB_RESETCONTENT, 0,0L);
      num_listed = 0;
      hcurSave = SetCursor(LoadCursor(NULL, IDC_WAIT)); // Load hourglass cursor
      nItemIndex = load_station_list(); // Load the list of stations
      SetCursor(hcurSave);                              // Back to normal cursor
//Try to select present location, if not in list, or sort nearest, select 1st entry
      if ((sortnearest) // Sort nearest ALWAYS selects first (nearest) entry
         || ( nItemIndex && (LB_ERR==(SendDlgItemMessage(hwnd, IDL_STATION, LB_SETCURSEL,
            nItemIndex, 0L)))) // try found item first
         || (!nItemIndex && (LB_ERR==(SendDlgItemMessage(hwnd, IDL_STATION, LB_SELECTSTRING,
            0, (LPARAM) ((LPSTR) IDX_station_name))))) ) // else try string matching station name
         SendDlgItemMessage(hwnd, IDL_STATION, LB_SETCURSEL, 0, 0L);// Select 1st entry

      prev_sel = SendDlgItemMessage(hwnd, IDL_STATION, LB_GETCURSEL, 0, 0L);
      if (prev_sel != LB_ERR)
         SendDlgItemMessage(hwnd, IDL_STATION,LB_GETTEXT,
            (WPARAM) prev_sel, (LPARAM) ((LPSTR) loc_str)); //get new location name
      if (loc_str[0] == ' ') { // Info line
         if (LB_ERR==(prev_sel=find_non_info_line( hwnd, DIR_UP ))) { // Only info lines
            SendDlgItemMessage(hwnd, IDL_STATION, LB_RESETCONTENT, 0,0L); // Clear all data
            loc_str[0]='\0';
         }
         else {
            SendDlgItemMessage(hwnd, IDL_STATION, LB_SETCURSEL, prev_sel,0L);
            SendDlgItemMessage(hwnd, IDL_STATION,LB_GETTEXT,
               (WPARAM) prev_sel, (LPARAM) ((LPSTR) loc_str)); //get new location name
         }
      }
// Load info new *new* current station
      if (strlen(loc_str) > 3)
         load_location_info( loc_str,
            LOWORD(SendDlgItemMessage(hwnd, IDL_STATION, LB_GETITEMDATA, prev_sel, 0L)) );
   }
// Back to common code
   describe_station(loc_str);
   SetDlgItemText(hwnd,IDL_LOCATION_INFO,loc_str);
   sprintf(loc_str, "%d", num_listed);
   SetDlgItemText(hwnd,IDL_LISTED,loc_str);

   if (update_type == UPDATE_STATION) {
      put_map( IDX_lon, IDX_lat );      // Set point on map for this station
      put_map( 1000, 1000 );            // Send on it's way
   }

   if ((GetActiveWindow() == hwnd) || (update_type == UPDATE_INIT)) // Move focus to station list
      PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, IDL_STATION), -1);
}

#ifdef WIN32
BOOL CALLBACK LocationDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
#else
BOOL CALLBACK __export LocationDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
#endif
{
DWORD dwIndex;
static char selected_name[80];

    switch(msg) {
      case WM_INITDIALOG:
        if (!index_in_memory) {
           init_index_file(TRUE, hwnd); // Load full index
           load_location_info( location, IDX_rec_num );
        }
        sprintf(original_name, "%s (%c)", location, IDX_type);
        if (have_user_offsets)
             original_rec_num = have_user_offsets;
        else original_rec_num = IDX_rec_num;
        sortlon = IDX_lon;
        sortlat = IDX_lat;
        using_lon_lat = FALSE;
        hwndMap = NULL;
        if (have_index && had_map)
           CreateMapWindow(hwnd);
        new_location_dlg(hwnd, UPDATE_INIT);
        prev_sel = SendDlgItemMessage(hwnd, IDL_STATION, LB_GETCURSEL, 0, 0L);
        break;

      case WM_ACTIVATE:
         if (wParam && !HIWORD(lParam))
            PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, IDL_STATION), -1); // Focus on station list
         break;

      case WM_COMMAND:
        switch(LOWORD(wParam)) {

          case IDL_STATION:
            if ((HIWORD(wParam)) == LBN_SELCHANGE) {
               dwIndex = SendDlgItemMessage(hwnd, IDL_STATION, LB_GETCURSEL, 0, 0L);
               if (dwIndex != LB_ERR)
                  SendDlgItemMessage(hwnd, IDL_STATION,LB_GETTEXT,
                     (WPARAM) dwIndex, (LPARAM) ((LPSTR) selected_name));
               if (selected_name[0] == ' ') { // An information line has been selected, find another
                  if (dwIndex < prev_sel) { // Selection was moving toward lower number entry
                     if (LB_ERR==(dwIndex=find_non_info_line(hwnd, DIR_DOWN))) // No more down
                        dwIndex=find_non_info_line(hwnd, DIR_UP);              // ..so try up
                  }
                  else { // Selection was moving toward higher number entry
                     if (LB_ERR==(dwIndex=find_non_info_line(hwnd, DIR_UP))) // No more up
                        dwIndex=find_non_info_line(hwnd, DIR_DOWN);          // ..so try down
                  }
                  if (dwIndex != LB_ERR)
                     SendDlgItemMessage(hwnd, IDL_STATION, LB_SETCURSEL, dwIndex, 0L);
                     SendDlgItemMessage(hwnd, IDL_STATION,LB_GETTEXT,
                        (WPARAM) dwIndex, (LPARAM) ((LPSTR) selected_name));
               }
               prev_sel = dwIndex;
               load_location_info( selected_name,
                  LOWORD(SendDlgItemMessage(hwnd, IDL_STATION, LB_GETITEMDATA, dwIndex, 0L )));
               new_location_dlg(hwnd, UPDATE_STATION);
            }
            if ((HIWORD(wParam)) != LBN_DBLCLK)
               break;
// NOTE: Double clicking a station entry falls through to IDOK code

          case IDOK:
            if (have_user_offsets &&
               (IDYES != MessageBox(hwnd,
                    "Custom station offsets are active and will\r\n"
                    "be disabled when the new station is loaded.\r\n\r\n"
                    "Load new station data?",
                    "Open new station", MB_ICONQUESTION|MB_YESNO)))
               break;
            dwIndex = SendDlgItemMessage(hwnd, IDL_STATION, LB_GETCURSEL, 0, 0L);
            if (dwIndex != LB_ERR)
              SendDlgItemMessage(hwnd, IDL_STATION,LB_GETTEXT,
                 (WPARAM) dwIndex, (LPARAM) ((LPSTR) location)); //set new location name
            loc_hwnd = NULL; // So put_location will NOT
            if ((had_map=hwndMap!=NULL))
               PostMessage(hwndMap,WM_CLOSE,0,0L); // Close map window down
            if (pmap_list != NULL) free(pmap_list);
            pmap_list = NULL;
            load_location_data( location,
               LOWORD(SendDlgItemMessage(hwnd, IDL_STATION,LB_GETITEMDATA, dwIndex, 0L)) );
            if (!keep_index) {
//               free_abbreviation_list();
               free_station_index();
            }
            EndDialog(hwnd,TRUE);
            break;

          case IDCANCEL:
            load_location_info ( original_name, original_rec_num ); // So it will be ready next time
            loc_hwnd = NULL; // So put_location won't
            if ((had_map=hwndMap!=NULL))
               PostMessage(hwndMap,WM_CLOSE,0,0L); // Close map window down
            if (pmap_list != NULL) free(pmap_list);
            pmap_list = NULL;
            if (!keep_index) {
//               free_abbreviation_list();
               free_station_index();
            }
            EndDialog(hwnd,FALSE);//!have_index);
            break;

          case IDHELP:
            WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Station Locator Menu"));
            break;

          case IDL_REGION:
          case IDL_COUNTRY:
          case IDL_STATE:
            if (have_index && (HIWORD(wParam)) == CBN_CLOSEUP) {
               CheckRegionCountryState(hwnd, LOWORD(wParam));
               new_location_dlg(hwnd, UPDATE_LIST);
            }
            break;

          case IDL_FLAG_REFERENCE:
          case IDL_FLAG_SUBSTATION:
            if (!have_index) {
               CheckDlgButton(hwnd,IDL_FLAG_REFERENCE,  TRUE);
               CheckDlgButton(hwnd,IDL_FLAG_SUBSTATION, FALSE);
               new_location_dlg(hwnd, UPDATE_STATION);
               break;
            }
// Fall through to do new location

          case IDL_FLAG_LEVELS:
          case IDL_FLAG_CURRENT:
            ckrefsta  = IsDlgButtonChecked(hwnd, IDL_FLAG_REFERENCE);
            cksubsta  = IsDlgButtonChecked(hwnd, IDL_FLAG_SUBSTATION);
            ckcurrent = IsDlgButtonChecked(hwnd, IDL_FLAG_CURRENT);
            cklevels  = IsDlgButtonChecked(hwnd, IDL_FLAG_LEVELS);
            new_location_dlg(hwnd, UPDATE_LIST);
            break;

          case IDL_FLAG_REGION:
            if (!have_index) {
               CheckDlgButton(hwnd,IDL_FLAG_REGION, FALSE);
               new_location_dlg(hwnd, UPDATE_STATION);
            }
            else if (!IsDlgButtonChecked(hwnd, IDL_FLAG_REGION)) {
               SendDlgItemMessage(hwnd, IDL_REGION, CB_SETCURSEL, 0, 0L);// Select 1st entry
               new_location_dlg(hwnd, UPDATE_LIST);
            }
            else new_location_dlg(hwnd, UPDATE_STATION);
            break;

          case IDL_FLAG_COUNTRY:
            if (!have_index) {
               CheckDlgButton(hwnd,IDL_FLAG_COUNTRY, FALSE);
               new_location_dlg(hwnd, UPDATE_STATION);
            }
            else if (!IsDlgButtonChecked(hwnd, IDL_FLAG_COUNTRY)) {
               SendDlgItemMessage(hwnd, IDL_COUNTRY, CB_SETCURSEL, 0, 0L);// Select 1st entry
               new_location_dlg(hwnd, UPDATE_LIST);
            }
            else new_location_dlg(hwnd, UPDATE_STATION);
            break;

          case IDL_FLAG_STATE:
            if (!have_index) {
               CheckDlgButton(hwnd,IDL_FLAG_STATE, FALSE);
               new_location_dlg(hwnd, UPDATE_STATION);
            }
            else if (!IsDlgButtonChecked(hwnd, IDL_FLAG_STATE)) {
               SendDlgItemMessage(hwnd, IDL_STATE, CB_SETCURSEL, 0, 0L);// Select 1st entry
               new_location_dlg(hwnd, UPDATE_LIST);
            }
            else new_location_dlg(hwnd, UPDATE_STATION);
            break;

          case IDL_NOSORT:
//            if (sortalpha || sortnearest) {
               sortalpha   = FALSE;
               sortnearest = FALSE;
               new_location_dlg(hwnd, UPDATE_LIST);
//            }
//            else new_location_dlg(hwnd, UPDATE_STATION);
            break;

          case IDL_ALPHABET:
//            if (!sortalpha || sortnearest) {
               sortalpha   = TRUE;
               sortnearest = FALSE;
               new_location_dlg(hwnd, UPDATE_LIST);
//            }
//            else new_location_dlg(hwnd, UPDATE_STATION);
            break;

          case IDM_MAP_LBDOWN:
            using_lon_lat = TRUE;
            sortlon = (double)(short int)LOWORD(lParam);
            sortlat = (double)(short int)HIWORD(lParam);
            cvt_deg_2_DMM(selected_name, sortlon );
            SetDlgItemText(hwnd,IDL_LON, selected_name);
            cvt_deg_2_DMM(selected_name, sortlat );
            SetDlgItemText(hwnd,IDL_LAT, selected_name);
            SendDlgItemMessage(hwnd, IDL_REGION, CB_SETCURSEL, 0, 0L);// Select 1st entry
            SendDlgItemMessage(hwnd, IDL_COUNTRY,CB_SETCURSEL, 0, 0L);// Select 1st entry
            SendDlgItemMessage(hwnd, IDL_STATE,  CB_SETCURSEL, 0, 0L);// Select 1st entry
// Fall through to do sort nearest

          case IDL_NEAREST:
have_nearest:
            if (have_index) {
               sortalpha   = FALSE;
               sortnearest = TRUE;
               new_location_dlg(hwnd, UPDATE_LIST);
            }
            else {
               CheckDlgButton(hwnd,IDL_NOSORT,  !sortalpha);
               CheckDlgButton(hwnd,IDL_ALPHABET, sortalpha);
               CheckDlgButton(hwnd,IDL_NEAREST,  FALSE);
               new_location_dlg(hwnd, UPDATE_STATION);
            }
            break;

          case IDL_LON:
          case IDL_LAT:
            if (have_index) {
               int chng;
//               if ((HIWORD(wParam)) == EN_CHANGE) {
               if ((HIWORD(wParam)) == EN_KILLFOCUS) {
                  GetDlgItemText(hwnd,IDL_LON, selected_name, sizeof(selected_name)-1);
//                  if (strchr(selected_name, '\r'))
                  chng = cvt_DMM_2_deg(&sortlon, selected_name);
                  GetDlgItemText(hwnd,IDL_LAT, selected_name, sizeof(selected_name)-1);
//                  if (strchr(selected_name, '\r'))
                  chng |= cvt_DMM_2_deg(&sortlat, selected_name);
                  using_lon_lat |= chng;
                  if (chng) {
                     new_location_dlg(hwnd, UPDATE_LIST);
                     goto have_nearest;
                  }
               }
            }
            else new_location_dlg(hwnd, UPDATE_STATION);
            break;

          case IDL_FLAG_MAP:
            if (hwndMap != NULL) {
               PostMessage(hwndMap,WM_CLOSE,0,0L);
               new_params = TRUE;
            }
            else if (have_index) {
               CreateMapWindow(hwnd);
               new_location_dlg(hwnd, UPDATE_STATION);
               new_params = TRUE;
            }
            else
               CheckDlgButton(hwnd, IDL_FLAG_MAP, FALSE);
               new_location_dlg(hwnd, UPDATE_STATION);
            break;

          case IDM_MAP_CHANGE:
            new_location_dlg(hwnd, UPDATE_STATION);
            break;
        }
    }
    return FALSE;
}

/* -----------------10/29/97 5:47PM------------------
   Start of map window and functions
 --------------------------------------------------*/
#define MAP_SIZE_X 380
#define MAP_SIZE_Y 236
static char szMapName[] = "WorldMap";
static HBITMAP hBitmap, hBmpOld;
static WNDCLASS wcm;
//static HINSTANCE m_hinst;
static int MapWinX, MapWinY, MapWinH, MapWinW;
static int mapmpy, map_cx, map_cy;
static BITMAP bm;

typedef struct {
   int x;
   int y;
   int lon;
   int lat;
} map_fix_entry;

/* -----------------10/22/97 8:56AM------------------
Map point conversion routines.
 The following routines are used to convert to and from
 map coordinates (y,x) and longitude, latitude.
 Since any map we will use will be a flat earth map, any
 lat/lon point has a linear relationship with map coords.
 The map however, will always be slewed usually with emphasis
 on the northern hemisphere.
 So, what we do is define conversions for the four corners
 of the map (and world), then define control points for
 known locations on the map.  For any given point, we
 perform linear interpolation between points nearest
 less than and greater than the point of interest.
 --------------------------------------------------*/

static map_fix_entry map_fix[]= {
//   x   y   lon lat
   {  0,  0,-179, 89}, // Upper left
   {380,235,+179,-89}, // Lower right
   { 23, 75,-150, 61}, // Anchorage, AK
   { 53,116,-122, 38}, // San Fran
   { 67,133,-110, 23}, // Cabo San Lucas
   { 98,150, -80,  9}, // Balboa
   { 96,131, -82, 25}, // Key West
   {122,221, -58,-52}, // Stanley, Falkland Islands
   {202,195,  18,-34}, // Capetown, Africa
   {181, 96,   0, 51}, // Brighton, England
   {129,159, -51,  0}, // Amazon River
   {343,193, 153,-27}, // Brisbane, Aus
   {-1,-1,-1,-1}
};

int interpolate(int p, int ple, int pge, int rle, int rge) {
   if (ple == pge) return (rle);
   else return(((long int)(p - ple) * (long int)(rge - rle)) / (long int)(pge - ple) + rle);
}

/* -----------------10/31/97 6:23PM------------------
   Convert a map (x,y) coordinate to LON/LAT coordinate.
 --------------------------------------------------*/
LPARAM cvt_map_2_lonlat( int map_x, int map_y ) {
int i, xle, xge, yle, yge, ilon, ilat;

   if (map_x > bm.bmWidth)  map_x = bm.bmWidth;  // Clip to edges
   if (map_x < 0) map_x = bm.bmWidth - map_x;
   if (map_y > bm.bmHeight) map_y = bm.bmHeight; // Clip to edges
   if (map_y < 0) map_y = bm.bmHeight - map_y;
   xle = yle = 0; // Entry 0 has upper left corner
   xge = yge = 1; // Entry 1 has lower left corner
   for (i=0; map_fix[i].x >= 0; i++) {
      if ((map_fix[i].x <= map_x)&&(map_fix[xle].x < map_fix[i].x)) xle=i;// Find closest X <=
      if ((map_fix[i].x >= map_x)&&(map_fix[xge].x > map_fix[i].x)) xge=i;// Find closest X >=
      if ((map_fix[i].y <= map_y)&&(map_fix[yle].y < map_fix[i].y)) yle=i;// Find closest Y <=
      if ((map_fix[i].y >= map_y)&&(map_fix[yge].y > map_fix[i].y)) yge=i;// Find closest Y >=
   }
   ilon = interpolate(map_x,
         map_fix[xle].x, map_fix[xge].x, map_fix[xle].lon, map_fix[xge].lon);
   ilat = interpolate(map_y,
         map_fix[yle].y, map_fix[yge].y, map_fix[yle].lat, map_fix[yge].lat);
   return ( MAKELONG(ilon, ilat) );
//   return (((unsigned long)ilat << 16) | (ilon & 0x0000ffff));
}

/* -----------------10/31/97 6:24PM------------------
   Convert a LON/LAT coordinate to map (x,y) coordinate.
 --------------------------------------------------*/
void cvt_lonlat_2_map( int *x1, int *y1, LPARAM point ) {
int i, ilon, ilat, xle, xge, yle, yge;

   ilon = (short int)LOWORD(point);
   ilat = (short int)HIWORD(point);
   xle = yge = 0;
   xge = yle = 1;
   for (i=0; map_fix[i].x >= 0; i++) {
      if ((map_fix[i].lon <= ilon)&&(map_fix[xle].lon < map_fix[i].lon)) xle=i;// Find closest X <=
      if ((map_fix[i].lon >= ilon)&&(map_fix[xge].lon > map_fix[i].lon)) xge=i;// Find closest X >=
      if ((map_fix[i].lat <= ilat)&&(map_fix[yle].lat < map_fix[i].lat)) yle=i;// Find closest Y <=
      if ((map_fix[i].lat >= ilat)&&(map_fix[yge].lat > map_fix[i].lat)) yge=i;// Find closest Y >=
   }
   *x1 = interpolate(ilon,
         map_fix[xle].lon, map_fix[xge].lon, map_fix[xle].x, map_fix[xge].x);
   *y1 = interpolate(ilat,
         map_fix[yle].lat, map_fix[yge].lat, map_fix[yle].y, map_fix[yge].y);
   return;
}

/* -----------------10/27/97 7:35AM------------------
   Draw map in map window, centered around a given point with zooming.
   Since the map is cylindrical, the hard part is filling in the left
   and right sides with residue from the right and left sides of the
   bitmap. ARG!
 --------------------------------------------------*/
void DrawMap( HDC mapDC, HDC bmpmemDC, int map_cx, int map_cy, int mapmpy ) {
int map_lx, map_ty, map_win_lx, map_win_ty;
int half_map_w, half_map_h;
int map_w, map_h, map_win_w, map_win_h, map_ox, map_rx;

   half_map_w = bm.bmWidth  / ((mapmpy+1)*2);
   half_map_h = bm.bmHeight / ((mapmpy+1)*2);

// Find the left side offsets
   map_ox = map_lx = (map_cx - half_map_w);
   if (map_lx < 0) {
      map_win_lx = (-map_lx * (mapmpy+1) * MapWinW) / bm.bmWidth;
      map_lx = 0;
   }
   else map_win_lx = 0;

// Find the top offsets
   map_ty = (map_cy - half_map_h);
   if (map_ty < 0) {
      map_win_ty = (-map_ty * (mapmpy+1) * MapWinH) / bm.bmHeight;
      map_ty = 0;
   }
   else map_win_ty = 0;

// Find the right side of the image
   map_rx = map_w = bm.bmWidth  / (mapmpy+1);
   if ((map_lx + map_w) > bm.bmWidth)  {
      map_w = bm.bmWidth - map_lx;
      map_win_w = (map_w * (mapmpy+1) * MapWinW) / bm.bmWidth;
   }
   else {
      map_win_w = MapWinW;
      map_rx = 0;
   }

// Find the bottom of the image
   map_h = bm.bmHeight / (mapmpy+1);
   if ((map_ty + map_h) > bm.bmHeight) {
      map_h = bm.bmHeight - map_ty;
      map_win_h = (map_h * (mapmpy+1) * MapWinH) / bm.bmHeight;
   }
   else map_win_h = MapWinH;

   hBmpOld=SelectObject(bmpmemDC, hBitmap);

   SetBkColor(  mapDC, RGB(255,255,255)); // Background is WHITE
   SetTextColor(mapDC, RGB(128,128,128)); // Black dots are GREY

// If there is a hole on the left, fill it in with the right side of the bitmap.
   if (map_win_lx)
      StretchBlt(mapDC, 0, map_win_ty, map_win_lx, map_win_h, bmpmemDC,
          bm.bmWidth+map_ox, map_ty, (-map_ox), map_h, SRCCOPY);

// Show the image centered on the point
   StretchBlt(mapDC, map_win_lx, map_win_ty, map_win_w, map_win_h,
           bmpmemDC, map_lx,     map_ty,     map_w,     map_h, SRCCOPY);

// If there is a hole on the right, fill it in with the left side of the bitmap.
   if (map_rx)
      StretchBlt(mapDC, map_win_w, map_win_ty, MapWinW-map_win_w, map_win_h, bmpmemDC,
          0, map_ty, (map_rx-map_w), map_h, SRCCOPY);

   SelectObject(bmpmemDC, hBmpOld);
}

/* -----------------10/29/97 5:51PM------------------
   Draw a dot on the map. In this case we draw a square
   2 units wide and high.  At the highest zoom factor,
   this leaves a little daylight between integer Lat/Lons.
 --------------------------------------------------*/
void DrawMapDot( HDC mapDC, HWND hwnd, COLORREF color, LPARAM point) {
int x1, y1, halfx, halfy, relx, rely;

   cvt_lonlat_2_map( &x1, &y1, point );
   relx = x1 - map_cx;
   rely = y1 - map_cy;
   if ((relx) >  (bm.bmWidth/2)) relx = bm.bmWidth - relx;
   if ((relx) < -(bm.bmWidth/2)) relx = bm.bmWidth + relx;
   x1 = ((relx * (mapmpy+1) * MapWinW) / bm.bmWidth) + (MapWinW/2);
   y1 = ((rely * (mapmpy+1) * MapWinH) / bm.bmHeight)+ (MapWinH/2);

   halfx = halfy = 2;
   if ((x1 > -halfx) && (x1 < halfx)) x1 = halfx;// Make all points JUST visible,
   if ((y1 > -halfy) && (y1 < halfy)) y1 = halfy;// .. ALL visible
   if ((x1 > MapWinW-halfx) && (x1 < MapWinW+halfx)) x1 = MapWinW-halfx;
   if ((y1 > MapWinH-halfy) && (y1 < MapWinH+halfy)) x1 = MapWinH-halfy;

   if ((x1 >= halfx) && (x1 <= (MapWinW-halfx))  && // Only display points in view
       (y1 >= halfy) && (y1 <= (MapWinH-halfy)) ) {
   XFillRectangle (mapDC, hwnd, color, x1-halfx, y1-halfy, halfx*2, halfy*2);
   }
}

/* -----------------10/29/97 5:53PM------------------
   Create a window for the map.
 --------------------------------------------------*/
static int hold_map_window;
extern RECT WinDimMap;
void CreateMapWindow(HWND hwndParent) {
int cxscreen, cyscreen, MapWinX, MapWinY;

   hold_map_window = TRUE;
   hBitmap = LoadBitmap(g_hinst, szMapName);
   GetObject(hBitmap, sizeof(BITMAP), &bm);

   cxscreen = GetSystemMetrics(SM_CXSCREEN);
   cyscreen = GetSystemMetrics(SM_CYSCREEN);
   if ((WinDimMap.left+8   >= 0) &&
       (WinDimMap.right  < cxscreen) &&
       (WinDimMap.top+24    >= 0) &&
       (WinDimMap.bottom < cyscreen) &&
       (WinDimMap.right > WinDimMap.left) &&
       (WinDimMap.bottom > WinDimMap.top) &&
       (WinDimMap.left|WinDimMap.right|WinDimMap.top|WinDimMap.bottom) ) {
      MapWinX = WinDimMap.left;
      MapWinY = WinDimMap.top;
      MapWinW = WinDimMap.right - WinDimMap.left - 8;
      MapWinH = WinDimMap.bottom - WinDimMap.top - 24;
   }
   else {
      MapWinW = bm.bmWidth;
      MapWinH = bm.bmHeight;
      MapWinX = CW_USEDEFAULT;
      MapWinY = CW_USEDEFAULT;
   }

   wcm.lpfnWndProc   = TidesMapProc;
   wcm.hIcon         = LoadIcon(g_hinst, "WXTide");
   wcm.hCursor       = LoadCursor(NULL, IDC_CROSS);
   wcm.hInstance     = g_hinst;
   wcm.lpszClassName = szMapName;
   wcm.style         = CS_BYTEALIGNWINDOW;// |CS_DBLCLKS;
   wcm.lpszMenuName  = NULL;
   wcm.cbClsExtra    = 0;
   wcm.cbWndExtra    = 0;
   wcm.hbrBackground = GetStockObject(WHITE_BRUSH);

   RegisterClass (&wcm);
   hwndMap = CreateWindow(
       szMapName,
       "Station map LCLK:zoom+nearest  RCLK:-zoom",
       WS_POPUPWINDOW |WS_CAPTION |WS_SIZEBOX,
       MapWinX, MapWinY,
//       CW_USEDEFAULT, // X coord - let windows decide
//       CW_USEDEFAULT, // Y coord - let windows decide
       MapWinW+ 8,    // width
       MapWinH+24,    // height
       NULL,//       hwndParent,    // hwndParent, who owns us
       NULL,          // No menu
       g_hinst,       // Handle of this instance of the program
       NULL           // No additional arguments
       );

// Display the window
   ShowWindow(hwndMap, SW_SHOW);
   UpdateWindow(hwndMap);
   hold_map_window = FALSE;
}

/* -----------------10/31/97 6:27PM------------------
   Message processing for the map.
 --------------------------------------------------*/
#ifdef WIN32
LRESULT CALLBACK TidesMapProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
#endif
{
PAINTSTRUCT ps;
static HDC mapDC, bmpmemDC;
static int num_pnts=0;
static long int *pmap_list = NULL;

   switch(msg) {
     case WM_CREATE:
          mapmpy = 0;
          map_cx = bm.bmWidth  / 2;
          map_cy = bm.bmHeight / 2;
          SetFocus(loc_hwnd);
          return FALSE;

     case WM_SIZE:
          MapWinW=LOWORD(lParam);
          MapWinH=HIWORD(lParam);
          InvalidateRect(hwnd,NULL,TRUE);
          if (!hold_map_window) {
            if (save_windows) new_params = TRUE;
          }
          else
            SetFocus(loc_hwnd);
          return FALSE;

     case WM_MOVE:
          MapWinX=LOWORD(lParam);
          MapWinY=HIWORD(lParam);
          if (!hold_map_window) {
            if (save_windows) new_params = TRUE;
          }
          else
            SetFocus(loc_hwnd);
          return FALSE;

     case WM_COMMAND:
       switch(wParam) {
         case IDM_MAP_PAINT_DOT:
            pmap_list = (long int *)lParam;
              break;
       }
       InvalidateRect(hwnd,NULL,FALSE);
       UpdateWindow(hwnd);
//       SetFocus(loc_hwnd);
       return FALSE;

     case WM_PAINT:
          BeginPaint(hwnd,&ps);
          mapDC=GetDC(hwnd);
          bmpmemDC=CreateCompatibleDC(mapDC);

          DrawMap( mapDC, bmpmemDC, map_cx, map_cy, mapmpy );

          if (pmap_list != NULL) {
            num_pnts = -pmap_list[0];
            while (num_pnts-- > 0)
               DrawMapDot( mapDC, hwnd, fgmapdot_color, (LPARAM)pmap_list[num_pnts+1]);
//            free(pmap_list);
//            pmap_list = NULL;
          }
          DeleteDC(bmpmemDC);
          ReleaseDC(hwnd, mapDC);
          EndPaint(hwnd,&ps);
          if (hold_map_window) SetFocus(loc_hwnd);
          return FALSE;

     case WM_CLOSE:
          GetWindowRect(hwndMap,&WinDimMap);
          pmap_list = NULL;
          DeleteObject(hBitmap);
          hwndMap = NULL;
          PostMessage(loc_hwnd, WM_COMMAND, IDM_MAP_CHANGE, 0L);
          DestroyWindow(hwnd);
          return FALSE;

     case WM_RBUTTONDOWN:
          if (mapmpy > 0) {
             mapmpy--;
             if (!mapmpy) {
                map_cx = bm.bmWidth / 2;
                map_cy = bm.bmHeight / 2;
             }
          }
          InvalidateRect(hwnd,NULL,TRUE);
          UpdateWindow(hwnd);
//          SetFocus(loc_hwnd);
          return FALSE;

     case WM_LBUTTONDOWN:
          map_cx += ((LOWORD(lParam)-(MapWinW/2)) * bm.bmWidth ) / (MapWinW*(mapmpy+1));
          map_cy += ((HIWORD(lParam)-(MapWinH/2)) * bm.bmHeight) / (MapWinH*(mapmpy+1));

          if (map_cx < 0) map_cx += bm.bmWidth;
          if (map_cy < 0) map_cy += bm.bmHeight;
          if (map_cx > bm.bmWidth ) map_cx -= bm.bmWidth;
          if (map_cy > bm.bmHeight) map_cy -= bm.bmHeight;

          if (mapmpy < 4) {
            mapmpy++;
            SetCursorPos(MapWinX+(MapWinW/2), MapWinY+(MapWinH/2)); // Center cursor
          }

          PostMessage(loc_hwnd, WM_COMMAND, IDM_MAP_LBDOWN, cvt_map_2_lonlat(map_cx, map_cy));
          InvalidateRect(hwnd,NULL,TRUE);
          UpdateWindow(hwnd);
//          SetFocus(loc_hwnd);
          return FALSE;

//     case WM_LBUTTONDBLCLK:
//          printf("Double click at (%d,%d)\n",map_cx, map_cy);
//          SetFocus(loc_hwnd);
//          return FALSE;
   }
   return DefWindowProc(hwnd,msg,wParam,lParam);
}

// skycal.cc -- Functions for sunrise, sunset, phase of moon.
// Last modified 1998-01-25

// This source file began its life as skycalendar.c and skycalc.c in
// John Thorstensen's skycal distribution (version 4.1, 1994-09) at
// ftp://iraf.noao.edu/contrib/skycal.tar.Z.  Those portions that are
// unchanged from the original sources are covered by the original
// license statement, included below.  The new portions and "value
// added" by David Flater are covered under the GNU General Public
// License:

/*
    Copyright (C) 1998  David Flater.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

// The original skycal comments and license statement are as follows:

/*
   This is a self-contained c-language program to print a nighttime
   astronomical calendar for use in planning observations.
   It prints to standard output (usually the terminal); the
   operator should capture this output (e. g., using redirection
   in UNIX or the /out= switch in VMS) and then print it on an
   appropriate output device.  The table which is printed is some
   125 columns wide, so a wide device is required (either a line
   printer or a laserprinter in LANDSCAPE mode.)  It is assumed that
   the ASCII form-feed character will actually begin a new page.
   The original program was to run on VMS, but it should be very
   transportable.  Non-vms users will probably want to change
   'unixio.h' to 'stdio.h' in the first line.
   An explanatory text is printed at the beginning of the output, which
   includes the appropriate CAUTIONS regarding accuracy and applicability.

   A number of 'canned site' parameters have been included.  Be
   careful of time zones, DST etc. for foreign sites.
   To customize to your own site, install an option in the
   routine 'load_site'.  The code is very straightforward; just do
   it exactly the same as the others.  You might also want to erase
   some seldom-used choices.  One can also specify new site parameters
   at run time.

   This program may be used freely by anyone for scientific or educational
   purposes.  If you use it for profit, I want a cut, and claim
   a copyright herewith.  In any case please acknowledge the source:

			John Thorstensen
			Dept. of Physics and Astronomy
			Dartmouth College
			Hanover, NH 03755
			John.Thorstensen@dartmouth.edu

			May 26, 1993.
*/

//#include "common.hh"

#define SUN  1
#define MOON 2
#define DEG_IN_RADIAN     57.2957795130823
#define HRS_IN_RADIAN     3.819718634
#define SEC_IN_DAY        86400.

// This affects the sunrise/sunset predictions.  Normally you would
// need to adjust it for the evelation of the location, but since this
// is a tide prediction program, we can assume that we are always at
// sea level, and actually be RIGHT :-)
#define sunrise_altitude -0.83

// It is important to know one's limitations.
// If time_t is a signed 32-bit int, you get:
//    time_t 0x00000000 = 1970-01-01 00:00:00Z
//    time_t 0x7FFFFFFF = 2038-01-19 03:14:07Z
//    time_t 0x80000000 = 1901-12-13 20:45:52Z

// The beginning of time (1970-01-01 00:00:00) as a Julian date.
#define beginning_of_time 2440587.5

// Create a Timestamp for the specified Julian date.
time_t julian2time_t (double jd) {
  return((time_t)((jd - beginning_of_time) * DAYSECONDS));
}

// Convert to Julian date.
double time_t2julian(time_t *t) {
  return (((double)*t / (double)DAYSECONDS) + beginning_of_time);
}

static double atan_circ(double x, double y)
{
  /* returns radian angle 0 to 2pi for coords x, y */
  double theta;
  if(x == 0) {
    if(y > 0.) theta = M_PI/2.;
    else if(y < 0.) theta = 3.*M_PI/2.;
    else theta = 0.;   /* x and y zero */
  }
  else theta = atan(y/x);
  if(x < 0.) theta = theta + M_PI;
  if(theta < 0.) theta = theta + 2.*M_PI;
  return(theta);
}

static double altit (double dec, double ha, double lat)
/* dec deg, dec hrs, dec deg */
{
  double x;
  dec = dec / DEG_IN_RADIAN;
  ha = ha / HRS_IN_RADIAN;
  lat = lat / DEG_IN_RADIAN;  /* thank heavens for pass-by-value */
  x = DEG_IN_RADIAN * asin(cos(dec)*cos(ha)*cos(lat) + sin(dec)*sin(lat));
  return(x);
}

static double lst(double jd, double longit)

{
	/* returns the local MEAN sidereal time (dec hrs) at julian date jd
	   at west longitude long (decimal hours).  Follows
           definitions in 1992 Astronomical Almanac, pp. B7 and L2.
           Expression for GMST at 0h ut referenced to Aoki et al, A&A 105,
	   p.359, 1982. */

	double t, ut, jdmid, jdint, jdfrac, sid_g;
	double jdnoon2000jan1 = 2451545.;
	long jdin, sid_int;

	jdin = (long)jd;         /* fossil code from earlier package which
			split jd into integer and fractional parts ... */
	jdint = jdin;
	jdfrac = jd - jdint;
	if(jdfrac < 0.5) {
		jdmid = jdint - 0.5;
		ut = jdfrac + 0.5;
	}
	else {
		jdmid = jdint + 0.5;
		ut = jdfrac - 0.5;
	}
	t = (jdmid - jdnoon2000jan1)/36525;
	sid_g = (24110.54841+8640184.812866*t+0.093104*t*t-6.2e-6*t*t*t)/86400.;
	sid_int = (long)sid_g;
	sid_g = sid_g - (double) sid_int;
	sid_g = sid_g + 1.0027379093 * ut - longit/24.;
	sid_int = (long)sid_g;
	sid_g = (sid_g - (double) sid_int) * 24.;
	if(sid_g < 0.) sid_g = sid_g + 24.;
	return(sid_g);
}

void lpmoon(double jd,double lat,double sid,double *ra,double *dec)
      //,double *dist)

/* implements "low precision" moon algorithms from
   Astronomical Almanac (p. D46 in 1992 version).  Does
   apply the topocentric correction.
Units are as follows
jd,lat, sid;   decimal hours
*ra, *dec,   decimal hours, degrees
	*dist;      earth radii */
{

	double T, lambda, beta, pie, l, m, n, x, y, z, alpha, delta,
		rad_lat, rad_lst, distance, topo_dist;

   T = (jd - 2451545.0) / 36525.;  /* jul cent. since J2000.0 */

	lambda = 218.32 + 481267.883 * T
	   + 6.29 * sin((134.9 + 477198.85 * T) / DEG_IN_RADIAN)
	   - 1.27 * sin((259.2 - 413335.38 * T) / DEG_IN_RADIAN)
	   + 0.66 * sin((235.7 + 890534.23 * T) / DEG_IN_RADIAN)
	   + 0.21 * sin((269.9 + 954397.70 * T) / DEG_IN_RADIAN)
	   - 0.19 * sin((357.5 + 35999.05 * T) / DEG_IN_RADIAN)
	   - 0.11 * sin((186.6 + 966404.05 * T) / DEG_IN_RADIAN);
	lambda = lambda / DEG_IN_RADIAN;
	beta = 5.13 * sin((93.3 + 483202.03 * T) / DEG_IN_RADIAN)
	   + 0.28 * sin((228.2 + 960400.87 * T) / DEG_IN_RADIAN)
	   - 0.28 * sin((318.3 + 6003.18 * T) / DEG_IN_RADIAN)
	   - 0.17 * sin((217.6 - 407332.20 * T) / DEG_IN_RADIAN);
	beta = beta / DEG_IN_RADIAN;
	pie = 0.9508
	   + 0.0518 * cos((134.9 + 477198.85 * T) / DEG_IN_RADIAN)
	   + 0.0095 * cos((259.2 - 413335.38 * T) / DEG_IN_RADIAN)
	   + 0.0078 * cos((235.7 + 890534.23 * T) / DEG_IN_RADIAN)
	   + 0.0028 * cos((269.9 + 954397.70 * T) / DEG_IN_RADIAN);
	pie = pie / DEG_IN_RADIAN;
	distance = 1 / sin(pie);

	l = cos(beta) * cos(lambda);
	m = 0.9175 * cos(beta) * sin(lambda) - 0.3978 * sin(beta);
	n = 0.3978 * cos(beta) * sin(lambda) + 0.9175 * sin(beta);

	x = l * distance;
	y = m * distance;
	z = n * distance;  /* for topocentric correction */
	/* lat isn't passed right on some IBM systems unless you do this
	   or something like it! */
   rad_lat = lat / DEG_IN_RADIAN;
	rad_lst = sid / HRS_IN_RADIAN;
	x = x - cos(rad_lat) * cos(rad_lst);
	y = y - cos(rad_lat) * sin(rad_lst);
	z = z - sin(rad_lat);

	topo_dist = sqrt(x * x + y * y + z * z);

	l = x / topo_dist;
	m = y / topo_dist;
	n = z / topo_dist;

	alpha = atan_circ(l,m);
	delta = asin(n);
	*ra = alpha * HRS_IN_RADIAN;
	*dec = delta * DEG_IN_RADIAN;
//   *dist = topo_dist;
}

static void
lpsun(double jd, double *ra, double *dec)

/* Low precision formulae for the sun, from Almanac p. C24 (1990) */
/* ra and dec are returned as decimal hours and decimal degrees. */

{
	double n, L, g, lambda,epsilon,x,y,z;

	n = jd - 2451545.0;
	L = 280.460 + 0.9856474 * n;
	g = (357.528 + 0.9856003 * n)/DEG_IN_RADIAN;
	lambda = (L + 1.915 * sin(g) + 0.020 * sin(2. * g))/DEG_IN_RADIAN;
	epsilon = (23.439 - 0.0000004 * n)/DEG_IN_RADIAN;

	x = cos(lambda);
	y = cos(epsilon) * sin(lambda);
	z = sin(epsilon)*sin(lambda);

	*ra = (atan_circ(x,y))*HRS_IN_RADIAN;
	*dec = (asin(z))*DEG_IN_RADIAN;
}

// This function combines a few steps that are always used together to
// find the altitude of the sun.
static double
body_altitude (double jd, double lat, double longit, int body) {
  double ra, dec;
  if (body == SUN)
     lpsun(jd, &ra, &dec);
  else
     lpmoon(jd, lat, lst(jd,longit), &ra, &dec);
  return altit(dec, lst(jd,longit)-ra, lat);

}

// I messed with this a good bit.
//
//   *  It now converges to within 1 minute.
//   *  It converges better from bad initial guesses.  (Deriv is now
//      updated inside of the loop.)
//   *  It won't roam more than half a day in either direction.
//   *  Max iterations chosen conservatively.
//   *  It finishes with a check to determine what it found.

static double jd_sun_alt(double alt, double jdorig, double lat, double longit,
int *is_sunrise, int body)
{
	/* returns jd at which sun is at a given
		altitude, given jdguess as a starting point. */

        double jdguess = jdorig;
	double jdout, adj = 1.0;
	double deriv, err, del = 0.002;
	double alt2,alt3;
	short i = 0;

        // Convergence when new diff is less than 1/4 minute.
        double convrg = 0.25 / (24.0 * 60.0);

	/* first guess */

   alt2 = body_altitude (jdguess, lat, longit, body);
	jdguess = jdguess + del;
   alt3 = body_altitude (jdguess, lat, longit, body);
	err = alt3 - alt;
	deriv = (alt3 - alt2) / del;
        if (deriv == 0.0) {
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
          cerr << "jd_sun_alt got nuked!" << endl;
#endif
          return (-1.0e10);
        }
        adj = -err/deriv;
	while(fabs(adj) >= convrg) {
	  if (i++ == 12) {
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
            cerr << "jd_sun_alt exceeded max iterations" << endl;
#endif
            return(-1.0e10); /* bad status flag */
  	  }
	  jdguess += adj;
          if (fabs (jdguess - jdorig) > 0.5) {
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
            cerr << "jd_sun_alt ran outside of its box" << endl;
#endif
            return (-1.0e10);
          }
          alt2 = alt3;
     alt3 = body_altitude (jdguess, lat, longit, body);
	  err = alt3 - alt;
  	  deriv = (alt3 - alt2) / adj;
          if (deriv == 0.0) {
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
            cerr << "jd_sun_alt got nuked!" << endl;
#endif
            return (-1.0e10);
          }
          adj = -err/deriv;
	}
	jdout = jdguess;

        // Figure out whether this is a sunrise or a sunset by shifting
        // by 1 second.
	{
          jdguess -= 1.0 / SEC_IN_DAY;
     alt2 = body_altitude (jdguess, lat, longit, body);
          *is_sunrise = (alt2 < alt3);
	}

#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
        cerr << "jd_sun_alt converged in " << i << " iterations" << endl;
#endif
	return(jdout);
}

static void flmoon(int n, int nph, double *jdout)

/* Gives jd (+- 2 min) of phase nph on lunation n; replaces
less accurate Numerical Recipes routine.  This routine
implements formulae found in Jean Meeus' *Astronomical Formulae
for Calculators*, 2nd edition, Willman-Bell.  A very useful
book!! */

{
  double jd, cor;
  double M, Mpr, F;
  double T;
  double lun;

  lun = (double) n + (double) nph / 4.;
  T = lun / 1236.85;
  jd = 2415020.75933 + 29.53058868 * lun
	  + 0.0001178 * T * T
	  - 0.000000155 * T * T * T
	  + 0.00033 * sin((166.56 + 132.87 * T - 0.009173 * T * T)/DEG_IN_RADIAN);
  M = 359.2242 + 29.10535608 * lun - 0.0000333 * T * T - 0.00000347 * T * T * T;
  M = M / DEG_IN_RADIAN;
  Mpr = 306.0253 + 385.81691806 * lun + 0.0107306 * T * T + 0.00001236 * T * T * T;
  Mpr = Mpr / DEG_IN_RADIAN;
  F = 21.2964 + 390.67050646 * lun - 0.0016528 * T * T - 0.00000239 * T * T * T;
  F = F / DEG_IN_RADIAN;
  if((nph == 0) || (nph == 2)) {/* new or full */
	  cor =   (0.1734 - 0.000393*T) * sin(M)
		  + 0.0021 * sin(2*M)
		  - 0.4068 * sin(Mpr)
		  + 0.0161 * sin(2*Mpr)
		  - 0.0004 * sin(3*Mpr)
		  + 0.0104 * sin(2*F)
		  - 0.0051 * sin(M + Mpr)
		  - 0.0074 * sin(M - Mpr)
		  + 0.0004 * sin(2*F+M)
		  - 0.0004 * sin(2*F-M)
		  - 0.0006 * sin(2*F+Mpr)
		  + 0.0010 * sin(2*F-Mpr)
		  + 0.0005 * sin(M+2*Mpr);
	  jd = jd + cor;
  }
  else {
	  cor = (0.1721 - 0.0004*T) * sin(M)
		  + 0.0021 * sin(2 * M)
		  - 0.6280 * sin(Mpr)
		  + 0.0089 * sin(2 * Mpr)
		  - 0.0004 * sin(3 * Mpr)
		  + 0.0079 * sin(2*F)
		  - 0.0119 * sin(M + Mpr)
		  - 0.0047 * sin(M - Mpr)
		  + 0.0003 * sin(2 * F + M)
		  - 0.0004 * sin(2 * F - M)
		  - 0.0006 * sin(2 * F + Mpr)
		  + 0.0021 * sin(2 * F - Mpr)
		  + 0.0003 * sin(M + 2 * Mpr)
		  + 0.0004 * sin(M - 2 * Mpr)
		  - 0.0003 * sin(2*M + Mpr);
	  if(nph == 1) cor = cor + 0.0028 -
			  0.0004 * cos(M) + 0.0003 * cos(Mpr);
	  if(nph == 3) cor = cor - 0.0028 +
			  0.0004 * cos(M) - 0.0003 * cos(Mpr);
	  jd = jd + cor;

  }
  *jdout = jd;
}

// This began as print_phase in skycalc.c.
// Set direction = 1 to go forward, -1 to go backward.
static void
find_next_moon_phase (double *jd, int *phase, int direction) {
  double newjd, lastnewjd, nextjd;
  short kount=0;
  int nlast;
  assert (direction == 1 || direction == -1);

  // Originally, there was no problem with getting snagged, but since
  // I introduced the roundoff error going back and forth with Timestamp,
  // now it's a problem.
  // Move ahead by 1 second to avoid snagging.
  *jd += (double)direction / SEC_IN_DAY;

  // Find current lunation.  I have doubled the safety margin since
  // it seemed biased for forwards search.
  nlast = (int)((*jd - 2415020.5) / 29.5307 - 2*direction);

  flmoon(nlast,0,&lastnewjd);
  nlast += direction;
  flmoon(nlast,0,&newjd);
  while ((direction == 1 && newjd <= *jd) ||
         (direction == -1 && newjd >= *jd)) {
    lastnewjd = newjd;
    nlast += direction;
    flmoon(nlast,0,&newjd);
    assert (kount++ < 5); // Original limit was 35 (!)
  }
#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
  cerr << "Finding lunation took " << kount << " iterations." << endl;
#endif

  // We might save some work here by estimating, i.e.:
  //   x = jd - lastnewjd;
  //   noctiles = (int)(x / 3.69134);  /* 3.69134 = 1/8 month; truncate. */
  // However....

  if (direction == 1) {
    assert (lastnewjd <= *jd && newjd > *jd);
    *phase = 1;
    nlast--; // Lunation is lastnewjd's lunation
    flmoon (nlast, *phase, &nextjd);       // Phase = 1
    if (nextjd <= *jd) {
      flmoon (nlast, ++(*phase), &nextjd);   // Phase = 2
      if (nextjd <= *jd) {
        flmoon (nlast, ++(*phase), &nextjd); // Phase = 3
        if (nextjd <= *jd) {
          *phase = 0;
          nextjd = newjd;
        }
      }
    }
  } else {
    assert (lastnewjd >= *jd && newjd < *jd);
    *phase = 3;
    // Lunation is newjd's lunation
    flmoon (nlast, *phase, &nextjd);       // Phase = 3
    if (nextjd >= *jd) {
      flmoon (nlast, --(*phase), &nextjd);   // Phase = 2
      if (nextjd >= *jd) {
        flmoon (nlast, --(*phase), &nextjd); // Phase = 1
        if (nextjd >= *jd) {
          *phase = 0;
          nextjd = newjd;
        }
      }
    }
  }
  *jd = nextjd;
}

// Front end with XTide data types
// 0=new moon, 1=first quarter, 2=fullmoon, 3=last quarter
int next_moon_phase (time_t *t, int dir) {
  int phase;
  double jd = time_t2julian(t);
  find_next_moon_phase (&jd, &phase, dir);
  *t = julian2time_t(jd);
  return (phase);
}

int moon_phase (time_t *moontime, int *phase, time_t nowtime) {
  int this_phase = -1, now_day;
  now_day = tmtime(nowtime)->tm_mday;

  if (now_day == tmtime(*moontime)->tm_mday)
//  if ((*moontime / DAYSECONDS) == (nowtime / DAYSECONDS)) // same day
     this_phase = *phase;
  if (nowtime > *moontime)
     *phase = next_moon_phase(moontime, 1);
  if (now_day == tmtime(*moontime)->tm_mday)
//  if ((*moontime / DAYSECONDS) == (nowtime / DAYSECONDS))
     this_phase = *phase;
  return(this_phase >= 0);
}

// Here's another opportunity for Jeff Dairiki to write a better root
// finder :-)
//
// jd_sun_alt needed good initial guesses to find sunrises and
// sunsets.  This was not a problem since good guesses were easy to
// come by.  The original skycalendar did this with estimates based on
// the local midnight:
//
//    jd = date_to_jd(date); /* local midnight */
//    jdmid = jd + zone(use_dst,stdz,jd,jdbdst,jdedst) / 24.;  /* corresponding ut */
//    stmid = lst(jdmid,longit);
//    lpsun(jdmid,&rasun,&decsun);
//    hasunset = ha_alt(decsun,lat,-(0.83+horiz));
//    jdsunset = jdmid + adj_time(rasun+hasunset-stmid)/24.; /* initial guess */
//    jdsunset = jd_sun_alt(-(0.83+horiz), jdsunset,lat,longit); /* refinement */
//    jdsunrise = jdmid + adj_time(rasun-hasunset-stmid)/24.;
//    jdsunrise = jd_sun_alt(-(0.83+horiz),jdsunrise,lat,longit);
//
// While efficient, this is an inconvenient way to go about it when
// I'm looking for the next event from time t, and don't even know
// when midnight is.  So I messed with jd_sun_alt to make it converge
// better from bad initial guesses, and substituted three bad guesses
// for one good one.  Normally, two would suffice, but I wanted to
// add a safety margin in case one of them happens to land at a point
// that nukes jd_sun_alt.

// Set direction = 1 to go forward, -1 to go backward.
static void
find_next_body_event (double *jd, double lat, double longit, int *is_sunrise,
int direction, int body) {
  int its = 0, looking_for;
  double jdorig, inc, jdlooper;
  assert (direction == 1 || direction == -1);

  // Move ahead by 1 minute to avoid snagging.
  *jd += (double)direction / (24.0 * 60.0);

  jdorig = *jd;
  inc = (double)direction / 6.0; // 4 hours

  // First we want to know what we are looking for.
  looking_for = (body_altitude (jdorig, lat, longit, body) < sunrise_altitude);
  if (direction == -1)
    looking_for = !looking_for;

  // Now give it a decent try.  Because jd_sun_alt is so unpredictable,
  // we can even find things out of order (which is one reason we need
  // to know what we're looking for).
  jdlooper = jdorig;
  do {
    its++;
    *jd = jd_sun_alt (sunrise_altitude, jdlooper, lat, longit, is_sunrise, body);
    jdlooper += inc;
  // Loop either on error return (which is a negative number), or if we
  // found an event in the wrong direction, or the wrong kind of event.
  } while ((*jd < 0.0) ||
           (direction == 1 && *jd <= jdorig) ||
           (direction == -1 && *jd >= jdorig) ||
           (*is_sunrise != looking_for));

//#ifdef SUPER_ULTRA_VERBOSE_DEBUGGING
//  printf("find_next_sun_event converged in %d iterations at %lf\n",its,*jd);
//#endif
}

// Front end with XTide data types
int next_sun_event(time_t *t, double lat, double lon, int dir) {
  int issunrise;
  double jd = time_t2julian(t);
  // skycal "longit" is measured in HOURS WEST, not degrees east.
  // (lat is unchanged)
  find_next_body_event (&jd, lat, -lon/15.0, &issunrise, dir, SUN);
  *t = julian2time_t(jd);
  return( issunrise );
}

// Front end with XTide data types
int next_moon_event(time_t *t, double lat, double lon, int dir) {
  int ismoonrise;
  double jd = time_t2julian(t);
  // skycal "longit" is measured in HOURS WEST, not degrees east.
  // (lat is unchanged)
  find_next_body_event (&jd, lat, -lon/15.0, &ismoonrise, dir, MOON);
  *t = julian2time_t(jd);
  return( ismoonrise );
}

/* -----------------3/21/98 12:39PM------------------
 * Make string for Sunrise/Sunset
 * --------------------------------------------------*/
void s_sunrise_set(char *dst, time_t tm) {
char srise[64], sset[64];
time_t sun_time, trise, tset;

   sun_time = prev_day(tm); // Start at midnight
   if (!next_sun_event(&sun_time, IDX_lat, IDX_lon, 1))
      // We found sunset!
      next_sun_event(&sun_time, IDX_lat, IDX_lon, 1);
   trise = sun_time;
   next_sun_event(&sun_time, IDX_lat, IDX_lon, 1);
   tset  = sun_time;
   do_timestamp( srise, tmtime(trise) );
   do_timestamp( sset , tmtime(tset ) );
   sprintf(dst, "Sunrise %s, Sunset %s", srise, sset);
}

/* -----------------3/21/98 12:39PM------------------
 * Make string for Sunrise/Sunset
 * --------------------------------------------------*/
void s_moonrise_set(char *dst, time_t tm) {
char m1[64], m2[64], *m1_type, *m2_type;
time_t moon_time, tm1, tm2;

   moon_time = prev_day(tm); // Start at midnight
   if (!next_moon_event(&moon_time, IDX_lat, IDX_lon, 1)) {
      m1_type = "set";
      m2_type = "rise";
   }
   else {
      m2_type = "set";
      m1_type = "rise";
   }
   tm1 = moon_time;
   next_moon_event(&moon_time, IDX_lat, IDX_lon, 1);
   tm2 = moon_time;
   next_moon_event(&moon_time, IDX_lat, IDX_lon, 1);
   do_timestamp( m1, tmtime(tm1) );
   do_timestamp( m2, tmtime(tm2) );
   sprintf(dst, "Moon%s %s, Moon%s %s", m1_type, m1, m2_type, m2);
}