/*  WXTide32 Harmonic tide clock and tide predictor
    Last modified 2002-02-10 by Mike Hopper for version 3.0

    This program uses the harmonic method to display tide levels
    either as a tide clock (showing the current level) or as a graph.
    All of the data and constants are read in from the harmonics file.
    Please refer to README for more information.

    Copyright (C) 1998  Michael Hopper
    Based on XTide.c Copyright (C) 1997  David Flater.
    ..and WinTide.c from 2/15/1996 by Paul C. Roberts
    Also starring:  Jef Poskanzer; Jack Greenbaum; Stan Uno; Dean Pentcheff.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


    The tide prediction algorithm used in this program was developed
    with United States Government funding, so no proprietary rights
    can be attached to it.  For more information, refer to the
    following publications:

    Manual of Harmonic Analysis and Prediction of Tides.  Special
    Publication No. 98, Revised (1940) Edition.  United States
    Government Printing Office, 1941.

    Computer Applications to Tides in the National Ocean Survey.
    Supplement to Manual of Harmonic Analysis and Prediction of Tides
    (Special Publication No. 98).  National Ocean Service, National
    Oceanic and Atmospheric Administration, U.S. Department of
    Commerce, January 1982.

    Parts of this program were carried over from XGrav, which was
    based on XSwarm, which was based on something else, ad infinitum.
*/

#include <windows.h>
#include <windowsx.h>  // <- nothing to do with Xwindows ;-) just some handy macros
#include <commdlg.h>
#include <shellapi.h>
#include <stdio.h>

#include "everythi.h" // (Mostly) everything defined
#include "loclib.h"   // Station locator stuff
#include "wxtidres.h" // Windows resource definitions

#define WINWIDTH 175      /* default window width */ //mgh was 84
#define GRAPHWIDTH 640
#define SKINNYWIDTH 30
#define WINHEIGHT 300     /* default window height */

/* How much water at lowest tide; how much sky at highest tide. */
#define margin 0.05

/* External variables */

extern double next_ht_amplitude, next_lt_amplitude;
extern int       have_index;
extern char      IDX_type, IDX_zone[40], IDX_station_name[80], IDX_reference_name[80], index_line[1024];
extern double    IDX_lon, IDX_lat;
extern float     IDX_ht_mpy, IDX_ht_off, IDX_lt_mpy, IDX_lt_off;
extern short int IDX_rec_num, IDX_time_zone, IDX_sta_num, IDX_ref_file_num;
extern short int IDX_ht_time_off, IDX_lt_time_off;
extern int       first_year, num_epochs;
extern char *tz_names[][2];

void set_local_tz();

/* Exported variables */
HINSTANCE g_hinst;
COLORREF fgmapdot_color;
int  keep_index=0, new_params=0, auto_save=0, save_windows=1,
     had_user_offsets=0, sun_moon=0, num_days=5, increment_step=60, incremental_tides=0 ;
char indexfile_name[MAXARGLEN]; // Index file name
char *fgmapdot_color_arg=NULL;  // Color of dots on station locator map
char *youwant = NULL;
char *custom_name = NULL;
int  have_BOGUS = 0, convert_BOGUS = 0;
RECT WinDimMain = {0,0,0,0},
     WinDimText = {0,0,0,0},
     WinDimMap  = {0,0,0,0};

/* Local variables */
#define DISPLAY_TEXT 1
#define DISPLAY_GRAPH 2

static char *szAccum=NULL, *output_filename=NULL;
static int nLenAccum=0, usescrollbar=0, notnowtime = 0;
static BOOL done;
static COLORREF fgrise_color, fgfall_color, fgtext_color,
 fgmark_color, fgmllw_color, fgmiddle_color, bgday_color, bgnite_color;
static HPEN fgtext_color_gc,fgmark_color_gc,fgmiddle_color_gc,fgmllw_color_gc,
          fgrise_color_gc,fgfall_color_gc,fgmapdot_color_gc,
          bgday_color_gc, bgnite_color_gc;
static char *bgday_color_arg = NULL, *bgnite_color_arg = NULL;
static int winX, winY, testmode = 0, testspeed = 0, rising = -1,
  hairycalibrate = 0, tmax, tmin, norename = 0,
  display_doesnt_suck = 1, appendWXTide32 = 0;
static int event_type, showampl=0, top_text_bottom, show_moons=0;
static unsigned int winW, winH;
static HWND window,hwndText=NULL,popup=NULL,scrollbar=NULL,nowbutton=NULL;
static HINSTANCE t_hinst;
static HDC display=NULL;
static HFONT gfont=NULL, tfont=NULL;
static double prev_tide, this_tide;
time_t gstart_time=0, window_left_time;
static int nowarn = 0, have_new_custom = 0, have_params = 0, hold_main_window = TRUE;
HFILE hf;
char szProgramName[256], szProgramPath[256], szAlternatePath[256], szConfigPath[256];
char HelpFileName[256];
OPENFILENAME ofn;
static FILE *textLFN=NULL;
static int gFontSize= 8;
static char gFontName[LF_FACESIZE+1]="MS Sans Serif";
static int tFontSize= 10;
static char tFontName[LF_FACESIZE+1]="Courier";
static double tide2wl (double tide);
static char MRU[10][100];
static int max_mru=9;
//#define MAX_MRU 4
static char tadjust_last[256];

LRESULT CALLBACK TidesWndProc(HWND,UINT,WPARAM,LPARAM);
BOOL CALLBACK TextTidesDlgProc(HWND hwndDlg,UINT msg,WPARAM wParam,LPARAM lParam);
LRESULT CALLBACK TidesMapProc(HWND,UINT,WPARAM,LPARAM);
static void NameWindow( HWND hwnd, char * name );

#define MoveTo(hdc,x,y) MoveToEx(hdc,x,y,NULL)
#define SetWindowOrg(a,b,c) SetWindowOrgEx((a),(b),(c),NULL)
#define SetWindowExt(a,b,c) SetWindowExtEx((a),(b),(c),NULL)
#define SetViewportExt(a,b,c) SetViewportExtEx((a),(b),(c),NULL)

DWORD GetTextExtent(LPSTR text, int cbLen) {
    DWORD dwReturn;
    SIZE sz;

    GetTextExtentPoint32(display, text, cbLen, &sz);
    dwReturn = MAKELONG(sz.cx, sz.cy);

    return dwReturn;
}

//
//some Xfunctions can be replaced by inline fns
//
void XFillRectangle (HDC display, HWND window, COLORREF color, int x, int y,int  w,int h) {
     RECT rc;
     HBRUSH hbr;
     rc.left=x;
     rc.top=y;
     rc.right=x+w;
     rc.bottom=y+h;
     hbr=CreateSolidBrush(color);

     FillRect(display,&rc,hbr);
     DeleteBrush(hbr);
}

void XDrawLine (HDC display, HPEN hpen, int x1,int y1,int x2,int y2) {
     hpen=SelectPen(display,hpen);
     MoveTo(display,x1,y1);
     LineTo(display,x2,y2);
     SelectPen(display,hpen);
}

// colour is totally different under windows. most of the
// probs under X don't arise. You never get denied a color
// (if it isn't in the colormap (palette in windows) you just get
// given the nearest.

///* Used by GetColor. */

///* If a color cannot be allocated, the following function assumes a
//   256-color colormap and searches for the closest match. */

static long GetColor (char *color) {
  int r,g,b;

  if (color) {
    if (sscanf (color, "rgb:%x/%x/%x", &r, &g, &b) != 3)
      barf (BADCOLORSPEC);
  }
  if( ! ( (r >= 0 && r <= 255) &&
          (g >= 0 && g <= 255) &&
          (b >= 0 && b <= 255) ) )
    barf (BADCOLORSPEC);

  return RGB(r,g,b);
}

// don't know how relevant this is lose all the X calls for now
static void die () {
  DeletePen(fgrise_color_gc);
  DeletePen(fgfall_color_gc);
  DeletePen(bgday_color_gc);
  DeletePen(bgnite_color_gc);
  DeletePen(fgtext_color_gc);
  DeletePen(fgmark_color_gc);
  DeletePen(fgmiddle_color_gc);
  DeletePen(fgmllw_color_gc);
  DeleteFont(gfont);
}


/*-----------------10/2/2002 6:59AM-----------------
* Add station name to Most Recently Used list
* * --------------------------------------------------*/
void add_mru( char *station ) {
  char name[256], *p, found_c;
  int i, found;

  strcpy(name, "0 ");
  strcat(name, station);
  if (strlen(name) >= sizeof(MRU[0]) - 1)
      name[sizeof(MRU[0])-1] = '/0';

  found = -1;
  for (i=0; i<10; i++) {
    p = MRU[i];
    if (isdigit(*p) && !strcmp(name+1, p+1)) found = i;
  }

  if (found == -1) { // This is a new entry
    for (i=0; i<10; i++) {
      p = MRU[i];
      if (isdigit(*p)) *p += 1;
      if (found==-1 && !isdigit(*p)) {
        strcpy(p, name);
        found = 0;
      }
    }
  }

  else { // This is an existing entry
    found_c = MRU[found][0];
    for (i=0; i<10; i++) {
      p = MRU[i];
      if (i == found)
          strcpy(p, name);
      else if (isdigit(*p) && *p < found_c )
          *p += 1;
    }
  }
}

/*-----------------10/2/2002 7:29AM-----------------
* Get MRU entry, called by write config and ID_MRU#
* mru is 0..9
* --------------------------------------------------*/
char *get_mru_entry( int mru ) {
  char *p, mru_id;
  int i;

  if (mru > max_mru) return( 0 );
  mru_id = mru + '0';
  for (i=0; i<10; i++) {
    p = MRU[i];
    if (*p == mru_id)
      return( p );
  }
  return( 0 );
}

void load_mru( int mru ) {
  load_location_data(get_mru_entry(mru)+2, 0);
  NameWindow(window, custom_name);
}


void draw_moon(int moonX, int phase) {
HDC moonDC, phaseDC;
HBITMAP hMask, hPhase, hBmpOld;
BITMAP bm;
COLORREF oldbg,oldfg;
int maskW, maskH, moonDCX, moonDCY;
DWORD maskOP;
char *szMoonMask="MOONMASK",*szMoonPhase;

   display = display;
   hMask = LoadBitmap(g_hinst, szMoonMask);
   GetObject(hMask, sizeof(BITMAP), &bm);
   moonDC=CreateCompatibleDC(display);
   hBmpOld=SelectObject(moonDC, hMask);

   oldbg = SetBkColor(  display, RGB(255,255,255)); // Background is WHITE
   oldfg = SetTextColor(display, RGB(  0,  0,  0)); // Black dots are BLACK
   maskW = bm.bmWidth;
   moonDCX = moonX-(maskW/2);
   maskH = bm.bmHeight;
   moonDCY = top_text_bottom;
   if ((moonDCX > -maskW) && (moonDCX < winW) &&
       (moonDCY > 0) && ((moonDCY+maskH) < winH)) {
      if (phase == 2) { // Full moon
// Full moon, Use invert of mask to make a white area on the screen.
      BitBlt(display,moonDCX,moonDCY,maskW,maskH,moonDC,0,0,MERGEPAINT);
      }
      else {
// First use mask to make a black area on the screen.
      BitBlt(display,moonDCX,moonDCY,maskW,maskH,moonDC,0,0,SRCAND);

      if (phase == 0) { // New moon
         szMoonPhase = "MOONNEW";
         maskOP      = SRCERASE;
      }
      else if (phase == 1) { // First quarter moon
         szMoonPhase = "MOONHALF";
         maskOP      = SRCERASE;
      }
      else  { // Third quarter moon
         szMoonPhase = "MOONHALF";
         maskOP      = NOTSRCERASE;
      }
      hPhase = LoadBitmap(g_hinst, szMoonPhase);
      GetObject(hPhase, sizeof(BITMAP), &bm);
      phaseDC=CreateCompatibleDC(display);
      SelectObject(phaseDC, hPhase);
// Set up masking template
      BitBlt(moonDC, 0,0,maskW,maskH,phaseDC,0,0,maskOP);
// Now merge the template with the display
      BitBlt(display,moonDCX,moonDCY,maskW,maskH,moonDC,0,0,SRCPAINT);

      DeleteDC(phaseDC);
      DeleteObject(hPhase);
      }
   }
   SelectObject(moonDC, hBmpOld);
   DeleteDC(moonDC);
   SetBkColor(  display, oldbg);
   SetTextColor(display, oldfg);
   DeleteObject(hMask);
}

static void draw_moons() {
time_t moon_time;
int moon;
   if (show_moons) {
      moon_time = window_left_time - (DAYSECONDS / 2); // Start back 1/2 day
      moon=next_moon_phase(&moon_time, 1);
      while (moon_time < (window_left_time + (tstep*winW))) {
         draw_moon((moon_time-window_left_time)/tstep, moon);
         moon=next_moon_phase(&moon_time, 1);
      }
   }
}


static void close_popup() {
  if (popup) {
     DestroyWindow(popup);
     popup = NULL;
  }
}

static void open_popup(DWORD mouseXY, char *text) {
int mouseX, mouseY, cxscreen, cyscreen, startX, startY, side;
PAINTSTRUCT ps;
RECT purc = {0,0,0,0};
HFONT oldfont;
#define POPUP_BORDER 1

  close_popup();
// Open window with dummy position and size to get device context
  popup = CreateWindow("STATIC", "", WS_POPUP|WS_VISIBLE,//|WS_BORDER,
                10,10,250,50, window, NULL, t_hinst, NULL);

  BeginPaint(popup,&ps);
  oldfont = SelectFont(ps.hdc,gfont);
// Get extents for this text
  DrawText(ps.hdc, text, strlen(text), &purc, DT_CALCRECT);
  purc.right  += POPUP_BORDER*2;
  purc.bottom += POPUP_BORDER*2;

  mouseX = LOWORD(mouseXY);
  mouseY = HIWORD(mouseXY);
  cxscreen = GetSystemMetrics(SM_CXSCREEN);
//  cyscreen = GetSystemMetrics(SM_CYFULLSCREEN);
  if (side=(cxscreen > (mouseX + winX + purc.right)))
       startX = mouseX + winX;
  else startX = mouseX + winX - purc.right;
  if (0 < (mouseY + winY - purc.bottom))
       startY = mouseY + winY - purc.bottom;
  else startY = mouseY + winY;
// Now make window the right size and in the right place
  MoveWindow(popup,startX,startY,purc.right,purc.bottom,FALSE);
//  SetTextColor(ps.hdc,fgtext_color);
//  SetBkColor(ps.hdc,RGB(255,255,255));
  FillRect( ps.hdc, &purc, GetStockObject(WHITE_BRUSH));
  FrameRect(ps.hdc, &purc, GetStockObject(BLACK_BRUSH));
  purc.left   += POPUP_BORDER;
  purc.top    += POPUP_BORDER;
  purc.right  -= POPUP_BORDER;
  purc.bottom -= POPUP_BORDER;
  DrawText(ps.hdc,text,strlen(text),&purc,
         DT_WORDBREAK|(side? DT_LEFT : DT_RIGHT));
  SelectFont(ps.hdc,oldfont);
  EndPaint(popup,&ps);
}


time_t CrosshairTime=0;
int CrosshairEnable = FALSE, CrosshairDrawn=0, UseCrosshairPopup=1;

static void DrawCrosshair(time_t CrossTime, HDC h) {
int ropsave, CrosshairX, CrosshairY;
HDC hdc;
   if (CrossTime > 0) {
      if (h) hdc = h;
      else   hdc = GetDC(window);
      ropsave= SetROP2(hdc, R2_NOT);
      CrosshairX = (CrossTime-window_left_time)/tstep;
      CrosshairY = tide2wl ((time2asecondary(CrossTime) - fakedatum) / fakeamplitude);
      MoveTo(hdc, CrosshairX, 0);
      LineTo(hdc, CrosshairX, winH);
      MoveTo(hdc, 0,    CrosshairY);
      LineTo(hdc, winW, CrosshairY);
      SetROP2(hdc, ropsave);
      if (!h) ReleaseDC(hdc, window);
      if (UseCrosshairPopup && CrosshairDrawn == 0) {
         char s[100], stime[64];
         int skinny_save = skinny;
         skinny=1;
         do_timestamp( stime, tmtime(CrossTime) );
         sprintf(s,"%s %1.2lf%s",stime, time2asecondary(CrossTime), units_abbrv);
         open_popup(MAKELONG(CrosshairX,0), s);
         skinny=skinny_save;
         SetFocus(window);
      }
      CrosshairDrawn = CrossTime;
   }
}

static void UndrawCrosshair() {
   if (CrosshairEnable && CrosshairDrawn > 0) {
      if (UseCrosshairPopup) close_popup();
      DrawCrosshair( CrosshairDrawn, NULL );
   }
   CrosshairDrawn = 0;
//   UseCrosshairPopup = 0;
}

static void DoCrosshair(HDC hdc) {
   if (UseCrosshairPopup && CrosshairDrawn != 0)
     close_popup();
   CrosshairDrawn = 0;
   if (CrosshairTime <= 0) {
     CrosshairTime = time( NULL );
   }
   else if (CrosshairTime <= window_left_time) {
     CrosshairTime = window_left_time + HOURSECONDS;
     CrosshairEnable = FALSE;
   }
   else if (CrosshairTime >= window_left_time + winW*tstep) {
     CrosshairTime = window_left_time + winW*tstep - HOURSECONDS;
     CrosshairEnable = FALSE;
   }
   if (CrosshairEnable)
     DrawCrosshair(CrosshairTime, hdc);
   else {
     CrosshairDrawn = 0;
//     UseCrosshairPopup = 0;
   }
}

static char *astro_info(time_t mousetime) {
static char s[512],stime[64],sdate[64],smdate[64],s_sun[80],s_moon[80];
time_t moon_time;
int moon, now_day;
struct tm *tmmoon;
char *phase_names[4] = {
   {"New moon, first quarter"},
   {"First quarter moon, full"},
   {"Full moon, last quarter"},
   {"Last quarter moon, new"} };

   do_timestamp( stime, tmtime(mousetime) );
   do_datestamp( sdate, tmtime(mousetime) );

   s_sunrise_set(s_sun, mousetime);
   s_moonrise_set(s_moon, mousetime);

   moon_time = mousetime;
   moon = next_moon_phase(&moon_time, -1);
   next_moon_phase(&moon_time, 1);
   now_day = tmtime(mousetime)->tm_mday;
   tmmoon = tmtime(moon_time);
   if (now_day == tmmoon->tm_mday) { // It's today!
      strcpy(smdate, "at ");
      do_timestamp( smdate+strlen(smdate), tmmoon);
   }
   else {
      strcpy(smdate, "on ");
      do_datestamp( smdate+strlen(smdate), tmmoon);
   }
   sprintf(s,"%s  %s  %01.2lf %s\r\n"
       "%s\r\n%s\r\n%s %s",
       stime, sdate, time2asecondary(mousetime), units,
       s_sun, s_moon, phase_names[moon], smdate);
   return(s);
}

#define SCROLL_DAY_RANGE 30
#define SCROLL_MON_RANGE 12
static int    scroll_range, scroll_pos;
static time_t scroll_start_time;

static time_t scroll_bar_2_time_t(int pos) {
struct tm *tm;
int offset = pos - scroll_pos;

   tm = tzlocaltime( &scroll_start_time );
   tm->tm_hour = tm->tm_min = tm->tm_sec = 0;
   if (abs(offset) <= SCROLL_DAY_RANGE)
      tm->tm_mday += offset;
   else if (abs(offset) <= (SCROLL_DAY_RANGE + SCROLL_MON_RANGE)) {
      if (offset < 0)
           tm->tm_mon += (offset + SCROLL_DAY_RANGE);
      else tm->tm_mon += (offset - SCROLL_DAY_RANGE);
      }
   else {
      if (offset < 0)
           tm->tm_year += (offset + (SCROLL_DAY_RANGE + SCROLL_MON_RANGE));
      else tm->tm_year += (offset - (SCROLL_DAY_RANGE + SCROLL_MON_RANGE));
      }
   return( mktime( tm ) );
}

static void DoScrollBar(int adj) {
int  fmoff, lmoff, rpad, lpad, nowW, nowH = GetSystemMetrics(SM_CYHSCROLL);
time_t first_day, last_day, now_day;
struct tm *tm;
RECT clientRC;

  if (scrollbar) DestroyWindow(scrollbar);
  if (nowbutton) DestroyWindow(nowbutton);
  scrollbar = nowbutton = NULL;

  nowW = notnowtime? 35 : 0;
  if (usescrollbar && graphmode) {
     if (adj) winH -= nowH;
     now_day = time(NULL);
     tm = tzlocaltime( &now_day ); // Just to set the pointer!
//Save first date in data set
     tm->tm_year = first_year - 1900;
     tm->tm_mon = tm->tm_hour = tm->tm_min = tm->tm_sec = 0;
     tm->tm_mday = 3;
     first_day = mktime( tm ) / DAYSECONDS;
//Save last date in data set
     tm->tm_year = first_year+num_epochs-1 - 1900;
     tm->tm_mon = tm->tm_hour = tm->tm_min = tm->tm_sec = 0;
     tm->tm_mon = 11;
     tm->tm_mday = 29;
     last_day = mktime( tm ) / DAYSECONDS;
//Get current (or last forced) date
     scroll_start_time = gstart_time? gstart_time : time(NULL);
     tm = tzlocaltime( &scroll_start_time );
     now_day = scroll_start_time / DAYSECONDS;
//Find months and padding needed from start of data to now
     fmoff=(tm->tm_year+1900 - first_year)*12 + tm->tm_mon;
     if (fmoff >= SCROLL_MON_RANGE)
          lpad = SCROLL_MON_RANGE;
     else lpad = fmoff;
//Find months and padding needed from now to end of data
     lmoff=(first_year+num_epochs-1 - tm->tm_year-1900)*12 + (11- tm->tm_mon);
     if (lmoff >= SCROLL_MON_RANGE)
          rpad = SCROLL_MON_RANGE;
     else rpad = lmoff;
//Adjust for days before/after
     if ((now_day - first_day) >= SCROLL_DAY_RANGE)
          lpad += SCROLL_DAY_RANGE;
     else lpad += (now_day - first_day);

     if ((last_day - now_day) >= SCROLL_DAY_RANGE)
          rpad += SCROLL_DAY_RANGE;
     else rpad += (last_day - now_day);

     scroll_range = num_epochs-1 + lpad + rpad;
     scroll_pos = tm->tm_year+1900 - first_year + lpad;
//Do it!
     GetClientRect(window,&clientRC);
     scrollbar=CreateWindow("SCROLLBAR", "", WS_CHILD|WS_VISIBLE|SBS_HORZ|SBS_BOTTOMALIGN,
              nowW, 0, clientRC.right-nowW, clientRC.bottom, window, 0, g_hinst, NULL);
     SetScrollRange(scrollbar,SB_CTL,0,scroll_range,FALSE);
     SetScrollPos(scrollbar,SB_CTL,scroll_pos,TRUE);
     if (notnowtime) nowbutton =
        CreateWindow("BUTTON", "Now", WS_CHILD|WS_VISIBLE|BS_PUSHBUTTON,
        0, clientRC.bottom-nowH, nowW, nowH, window, (HMENU)ID_NOW, g_hinst, NULL);
   }
  else if (adj) winH += nowH;
}

static void Init_Application(HINSTANCE hinst) {
  WNDCLASS wc;

  wc.style         = 0;//CS_HREDRAW|CS_VREDRAW;
  wc.lpfnWndProc   = TidesWndProc;
  wc.cbClsExtra    = 0;
  wc.cbWndExtra    = 0;
  wc.hInstance     = hinst;
  wc.hIcon         = LoadIcon(hinst, "WXTide");
  wc.hCursor       = LoadCursor(NULL, IDC_ARROW);
  wc.hbrBackground = NULL; //CreateSolidBrush(bgday_color);
  wc.lpszMenuName  = "WXTide";
  wc.lpszClassName = "WXTide";

  RegisterClass(&wc);
}

static void NameWindow( HWND hwnd, char * name ) {
static char *PROG_NAME = {" - WXTide32"};
char *win_name;
  if ( appendWXTide32 ) {
    win_name = malloc(strlen(name) + strlen(PROG_NAME) + 1);
    strncpy(win_name, name, strlen(name)+1);
    strcat(win_name, PROG_NAME);
    SetWindowText(hwnd, win_name);
    free(win_name);
  }
  else
    SetWindowText(hwnd, name);
}

/* Create and setup the window. */
static void Create_Window (/*char *geometry*/HINSTANCE hinst) {
int cxscreen, cyscreen;

  cxscreen = GetSystemMetrics(SM_CXSCREEN);
  cyscreen = GetSystemMetrics(SM_CYSCREEN);

  if ((WinDimMain.left   >= 0) &&
      (WinDimMain.right  <= cxscreen) &&
      (WinDimMain.top    >= 0) &&
      (WinDimMain.bottom <= cyscreen) &&
      (WinDimMain.right  > WinDimMain.left) &&
      (WinDimMain.bottom > WinDimMain.top) &&
      (WinDimMain.left|WinDimMain.right|WinDimMain.top|WinDimMain.bottom) ) {
     winX = WinDimMain.left;
     winY = WinDimMain.top;
     winW = WinDimMain.right - WinDimMain.left;
     winH = WinDimMain.bottom - WinDimMain.top;
  }
  else {
     winW = (graphmode ? GRAPHWIDTH :
              (skinny ? SKINNYWIDTH : WINWIDTH));
     winH = WINHEIGHT;

     if(geometry) {
           if(sscanf(geometry,"%dx%d",&winW,&winH)!=2)
               barf(BADGEOMETRY);
     }
     winX=((GetSystemMetrics(SM_CXSCREEN)-winW)>>1);
     winY=((GetSystemMetrics(SM_CYSCREEN)-winH)>>1);
  }

  window=CreateWindow("WXTide", "", WS_OVERLAPPEDWINDOW|WS_CLIPCHILDREN,
                winX, winY, winW, winH, NULL, 0, hinst, NULL);

#define GRAPH_TITLE " Graph"
  {
      char *win_name;

      if (graphmode) {
       win_name = malloc(strlen(custom_name) + strlen(GRAPH_TITLE) + 1);
       strncpy(win_name, custom_name, strlen(custom_name)+1);
       strcat(win_name, GRAPH_TITLE);
      }
      else win_name = custom_name;

      NameWindow(window, win_name);
//      SetWindowText(window,win_name);

      if (graphmode) free(win_name);
  }
//    gfont=CreateFont(15,0,0,0,0,0,0,0,0,OUT_TT_ONLY_PRECIS,0,0,VARIABLE_PITCH|FF_SWISS,NULL);
}

/* Convert a normalized tide to a Y coordinate in the window. */
static double
tide2wl (double tide)
{
  return (((double)winH * (1.0 - 2.0 * margin)) * (-tide) * 0.5
    + (double)winH * 0.5);
}

/* Inverse. */
static double
wl2tide (int y)
{
  return -2.0*(y - winH * 0.5) / ((double)winH * (1.0 - 2.0 * margin));
}

/* Text is centered around x.  Will break horribly if the font changes.
Most of this function has to do with making the window wider if the text
didn't fit. */
static void
center_text (int x, int y, char *text)
{
  RECT r;
  int l = strlen (text);
  DWORD dwExtents;
  r.top = r.left=0;
  r.bottom=winH;
  r.right=winW;
  dwExtents=GetTextExtent(text,l);
  ExtTextOut(display,x-(LOWORD(dwExtents)>>1),y-HIWORD(dwExtents)+1,ETO_CLIPPED,&r,text,l,NULL);
}

/* Draw text telling when the high tide is.  Now kluged to do low tide as
   well, and currents, and everything.... */
static void
write_high_tide (int x)
{
  char temp[80], next_hta_text[80], next_lta_text[80], now_text[80];
  int  bottom = winH-1;
  char longest[80], mediumest[80], units[80];

  if (strcmp (units_abbrv, "unknown") && (skinny != 2))
       strcpy(units, units_abbrv);
  else strcpy(units, "");

  do_timestamp (now_text, tmtime (time (NULL)));
  if (showampl) {
    sprintf(next_hta_text, "%s %01.1f%s", next_ht_text, next_ht_amplitude, units);
    sprintf(next_lta_text, "%s %01.1f%s", next_lt_text, next_lt_amplitude, units);
    sprintf(now_text+strlen(now_text)," %01.1f%s",time2asecondary(time(NULL)),units);
  }
  else {
    strcpy(next_hta_text, next_ht_text);
    strcpy(next_lta_text, next_lt_text);
  }

  if (graphmode) {
    center_text (x, 12, next_ht_date);
    if (event_type &1) {                   //mgh added
       if (showampl)
         sprintf(next_hta_text,"%s %01.1f%s", next_ht_text, next_lt_amplitude, units);
       else strcpy(next_hta_text, next_ht_text);
    }
    center_text (x, 22, next_hta_text);
    top_text_bottom = 22;
    return;
  }

  if (hairy && lines) bottom -= 9;
  if (tinc)           bottom -= 12;

  sprintf(longest,  "Next High Tide %s", next_hta_text);
  sprintf(mediumest,"High Tide %s",      next_hta_text);

  if (winW >= 6+(LOWORD(GetTextExtent(longest, lstrlen(longest))))) {
// Longest fits on one line
    if (iscurrent) {
       sprintf (temp, "Next Max Flood %s", next_hta_text);
       center_text (x, 12, temp);
       sprintf (temp, "Next Max Ebb %s",   next_lta_text);
       center_text (x, bottom, temp);
       top_text_bottom = 12;
    }
    else {
       sprintf (temp, "Next High Tide %s", next_hta_text);
       center_text (x, 12, temp);
       sprintf (temp, "Next Low Tide %s",  next_lta_text);
       center_text (x, bottom, temp);
       top_text_bottom = 12;
    }
    if (now) {
       sprintf(temp, "Time Now %s", now_text);
       center_text (x, 23, temp);
       top_text_bottom = 23;
    }
  }

  else if (winW >= 6+(LOWORD(GetTextExtent(mediumest, lstrlen(mediumest))))) {
// Medium line fits
    if (iscurrent) {
       sprintf (temp, "Max Flood %s", next_hta_text);
       center_text (x, 12, temp);
       sprintf (temp, "Max Ebb %s",   next_lta_text);
       center_text (x, bottom, temp);
       top_text_bottom = 12;
    }
    else {
       sprintf (temp, "High Tide %s", next_hta_text);
       center_text (x, 12, temp);
       sprintf (temp, "Low Tide %s",  next_lta_text);
       center_text (x, bottom, temp);
       top_text_bottom = 12;
    }
    if (now) {
       sprintf(temp, "Time Now %s", now_text);
       center_text (x, 23, temp);
       top_text_bottom = 23;
    }
  }

  else { // Nothing fits, use two lines per
    if (iscurrent) {
      center_text (x, 12, "Max Flood");
      center_text (x, 22, next_hta_text);
      center_text (x, bottom-12, "Max Ebb");
      center_text (x, bottom, next_lta_text);
      top_text_bottom = 22;
    }
    else {
      center_text (x, 12, "High Tide");
      center_text (x, 23, next_hta_text);
      center_text (x, bottom-12, "Low Tide");
      center_text (x, bottom, next_lta_text);
      top_text_bottom = 23;
    }
    if (now) {
      center_text (x, 34, "Time Now");
      center_text (x, 45, now_text);
      top_text_bottom = 45;
    }
  }
}

/* Draw text telling when the mark transition occurred.  Only called
   by graph mode. */
static void
write_trans_time (int x)
{
  int bottom = winH - 10;
  if (tinc)
    bottom -= 12;
  center_text (x, bottom-12, next_ht_date);
  center_text (x, bottom, next_ht_text);
  return;
}

static void find_tmin_tmax() {
  /* This silly walk is to determine the range within which to draw
     tick marks. */
  tmin = 30;
  if (now && !graphmode && !ppm)
    tmin += 24;
  if (graphmode) {
    tmax = winH - 14;
    if (mark)
      tmax -= 24;
    if (tinc)
      tmax -= 12;
  }

  else {
    tmax = winH - 29;
    if (tinc)
      tmax -= 12;
    if (hairy)
      tmax -= 9;
  }
  tmin = (int) (wl2tide (tmin) * fakeamplitude + fakedatum);
  tmax = (int) (ceil (wl2tide (tmax) * fakeamplitude + fakedatum));
}

/* The unit lines are now calibrated against MLLW and try not to
clobber text.  Jack Greenbaum submitted the original calibrated tick
mark code, which I promptly munged beyond recognition. */
static void
draw_unit_lines ()
{
  int a, b, len, firstflag=1;
  if (!lines)
    return;
  if (graphmode)
    len = winW-1;
  else
    len = 6;

  find_tmin_tmax();

  /* Now get on with it! */
  SelectPen(display,fgtext_color_gc);
  for (a=tmin;a>=tmax;a--) {
    if (fakeamplitude > 30.0 && (a % 10))
      continue;
    b = tide2wl (((double)a - fakedatum) / fakeamplitude);
    if (hinc) {
      if (!(a % hinc)) {
        char temp[20+MAXARGLEN];
        int l;
        DWORD dwExtents;

        make_depth_caption (&firstflag, a, temp);
        l = strlen (temp);
        dwExtents=GetTextExtent(temp,l);
        if (graphmode) {
            MoveTo(display,0,b);
            LineTo(display,len-(LOWORD(dwExtents)>>1),b);
            TextOut(display,winW-LOWORD(dwExtents),b-(HIWORD(dwExtents)>>1),temp,l);
        }
        else {
            MoveTo(display,0,b);
            LineTo(display,len,b);
            TextOut(display,len+3,b-(HIWORD(dwExtents)>>1),temp,l);
        }
        continue;
      }
    }
    XDrawLine (display, fgtext_color_gc, 0, b, len, b);
  }
}

/* In graph mode, the numbers labeling the depth lines get clobbered by
   the graph.  This puts them back.  Tmax and tmin are assumed
   to have been set up by draw_unit_lines. */
static void
graph_relabel_unit_lines ()
{
  int a, l, b, firstflag=1;
  char temp[20+MAXARGLEN];
  DWORD dwExtents;

  if (!hinc)
    return;
  for (a=tmin;a>=tmax;a--) {
    if (fakeamplitude > 30.0 && (a % 10))
      continue;
    if (!(a % hinc)) {
      b = tide2wl (((double)a - fakedatum) / fakeamplitude);
      make_depth_caption (&firstflag, a, temp);
      l = strlen (temp);
      dwExtents=GetTextExtent(temp,l);
      TextOut(display,winW-LOWORD(dwExtents),b-(HIWORD(dwExtents)>>1),temp,l);
    }
  }
}

/* Middle and mark and mllw options. */
static void
draw_extra_lines ()
{
  int b;
  if (mark) {
    b = tide2wl ((marklev - fakedatum) / fakeamplitude);
    XDrawLine (display, fgmark_color_gc, 0, b, winW, b);
  }
  if (middle) {
    b = tide2wl (0.0);
    XDrawLine (display, fgmiddle_color_gc, 0, b, winW, b);
  }
  if (mllw) {
    b = tide2wl (-fakedatum / fakeamplitude);
    XDrawLine (display, fgmllw_color_gc, 0, b, winW, b);
  }
  if (iscurrent && !mark) {
    b = tide2wl (-fakedatum / fakeamplitude);
    XDrawLine (display, fgtext_color_gc, 0, b, winW, b);
  }
}

/* Tick marks for hours along X axis */
static void
draw_tick_marks (time_t start)
{
  if (!lines)
    return;
  if ((!graphmode) && (!hairy))
    return;
  if (hairy && !hairycalibrate && !graphmode && !ppm && !gif) {
    /* Center tick marks around current time; ignore start. */
    int a, b;
    b = winW>>1;
    XDrawLine (display, fgtext_color_gc, b, winH-7, b, winH-1);
    for (a=1;a<=(int)((winW>>1) * tstep / 3600);a++) {
      b = (winW>>1) + a * 3600 / tstep;
      XDrawLine (display, fgtext_color_gc, b, winH-7, b, winH-1);
      b = (winW>>1) - a * 3600 / tstep;
      XDrawLine (display, fgtext_color_gc, b, winH-7, b, winH-1);
    }
  } else {
    /* Calibrate tick marks with top of the hour */
    int x, hour_mod, ihour, hour_len;
    time_t hour;
    DWORD dwExtents=GetTextExtent("24",2);
    hour_len = HOURSECONDS / tstep;
    if (LOWORD(dwExtents) < hour_len)
         hour_mod = 1;
    else if (LOWORD(dwExtents) < hour_len*2)
         hour_mod = 2;
    else if (LOWORD(dwExtents) < hour_len*4)
         hour_mod = 4;
    else hour_mod = 8;
    for (hour=prev_hour(start);hour<=start+winW*tstep+3600;
         hour=increment_hour(hour)) {
      x = (hour - start) / tstep;
      if (x < winW) {
        XDrawLine (display, fgtext_color_gc, x, winH-7, x, winH-1);
        if (tinc) {
          struct tm *foo;
          foo = tmtime (hour);
          if (!((foo->tm_hour) % tinc)) {
            char buf[20];
            do_timestamp (buf, foo);
            if (buf[1] == ':')
                 buf[1] = '\0';
            else buf[2] = '\0';
            ihour = atoi(buf);
            if (!(ihour % hour_mod))
               center_text (x, winH-10, buf);
          }
        }
      }
    }
    /* Make tick marks for day boundaries thicker */
    /* This is not guaranteed to coincide with an hour transition! */
    for (hour=prev_day(start);hour<=start+winW*tstep+3600;
    hour=increment_day(hour)) {
      x = (hour - start) / tstep;
      XDrawLine (display, fgtext_color_gc, x-1, winH-7, x-1, winH-1);
      XDrawLine (display, fgtext_color_gc, x, winH-7, x, winH-1);
      XDrawLine (display, fgtext_color_gc, x+1, winH-7, x+1, winH-1);
    }
  }
}

/* Handle ugliness relating to high and low tide updates.  This is
   only difficult because (1) I want the timestamps to update at the
   same time that the water changes color, and (2) the change of epoch
   causes special problems. */
static void
watch_tides (time_t this_time)
{
  prev_tide = this_tide;
  this_tide = time2secondary (this_time);
  if (rising == -1) {
    /* prev_tide is unreliable on startup */
    if (this_tide < time2secondary (this_time+1))
      rising = 1;
    else
      rising = 0;
  } else {
    if (prev_tide < this_tide) {
      /* Extra effort to get the tide to update at the same time that the
         water changes color.  The 70 instead of 60 is intentional. */
      if ((!rising) && (this_time > prev_ht_adj - 70))
        update_high_tide ();
      rising = 1;
    }
    if (prev_tide > this_tide) {
      /* See above. */
      if ((rising) && (this_time > prev_ht_adj - 70)) {
        update_high_tide ();
//        set_icon_name ();
      }
      rising = 0;
    }
  }
  /* Clean up any tides that were missed by the above code.  (It happens.) */
  while (this_time >= prev_ht_adj + 70)
    update_high_tide ();
}

/* Redraw the window for the normal tide clock. */
static void
set_water_level ()
{
  int nwl;
  time_t this_time, next_sun_time;
  COLORREF background;
  if (testmode)
    this_time = faketime;
  else
    this_time = time (NULL);
  window_left_time = this_time;
  watch_tides (this_time);
  nwl = tide2wl (this_tide);
  next_sun_time = this_tide;
  background = next_sun_event(&next_sun_time, IDX_lat, IDX_lon, -1)?
               bgnite_color : bgday_color;
  if (iscurrent) {
    int midwl = tide2wl (-fakedatum / fakeamplitude);
    if (nwl < midwl) {
      XFillRectangle (display, window, background, 0, 0, winW, nwl);
      XFillRectangle (display, window, background, 0, midwl+1, winW,
      winH-midwl);
      XFillRectangle (display, window, fgrise_color, 0, nwl, winW,
      midwl-nwl);
    } else if (nwl > midwl) {
      XFillRectangle (display, window, background, 0, 0, winW, midwl);
      XFillRectangle (display, window, background, 0, nwl+1, winW,
      winH-nwl);
      XFillRectangle (display, window, fgfall_color, 0, midwl+1, winW,
      nwl-midwl);
    } else {
      XFillRectangle (display, window, background, 0, 0, winW, winH);
    }
  } else {
    XFillRectangle (display, window, background, 0, 0, winW, nwl);
    if (rising)
      XFillRectangle (display, window, fgrise_color, 0, nwl, winW,
      winH-nwl);
    else
      XFillRectangle (display, window, fgfall_color, 0, nwl, winW,
      winH-nwl);
  }
  write_high_tide (winW>>1);
  draw_unit_lines();
  draw_extra_lines();
}

/* Support functions to abstract out the ugly decision tree for */
/* graphing currents. */

static void
x_wl_to_midwl (double wl, int midwl, int looper, int clearflag, HPEN bg_color)
{
  if ((int)wl < midwl) {
    if (display_doesnt_suck && !linegraph) {
      if ((int)wl+1 <= midwl-1)
        XDrawLine (display, fgrise_color_gc, looper, (int)wl+1, looper, midwl);
      XDrawLine (display, fgrise_color_gc, looper, (int)wl, looper, (int)wl+1);
    } else
        XDrawLine (display, fgrise_color_gc, looper, (int)wl, looper, midwl-1);
    if (clearflag) {
      XDrawLine (display, bg_color, looper, 0, looper, (int)wl);
      XDrawLine (display, bg_color, looper, midwl+1, looper, winH-1);
    }
  } else if ((int)wl > midwl) {
    if (display_doesnt_suck && !linegraph) {
      if ((int)wl >= midwl+1)
        XDrawLine (display, fgfall_color_gc, looper, midwl+1, looper, (int)wl);
      XDrawLine (display, fgfall_color_gc, looper, (int)wl, looper, (int)wl+1);
      if (clearflag) {
        XDrawLine (display, bg_color, looper, 0 , looper, midwl);
        XDrawLine (display, bg_color, looper, (int)wl+1, looper, winH-1);
      }
    } else {
      XDrawLine (display, fgfall_color_gc, looper, midwl+1, looper, (int)wl);
      if (clearflag) {
        XDrawLine (display, bg_color, looper, 0, looper, midwl-1);
        XDrawLine (display, bg_color, looper, (int)wl+1, looper, winH-1);
      }
    }
  } else {
    if (clearflag)
      XDrawLine (display, bg_color, looper, 0, looper, winH-1);
  }
}

static void
x_current_graph (double nwl, double owl, int midwl, int looper, int clearflag, HPEN bg_color)
{
  if (linegraph) {
    if (clearflag)
      XDrawLine (display, bg_color, looper, 0, looper, winH-1);
    if (nwl < midwl) {
      if (owl < midwl)
        XDrawLine (display, fgrise_color_gc, looper-1, (int)owl, looper, (int)nwl);
      else {
        x_wl_to_midwl (owl, midwl, looper-1, 0, bg_color);
        x_wl_to_midwl (nwl, midwl, looper,   0, bg_color);
      }
    } else if (nwl > midwl) {
      if (owl > midwl)
        XDrawLine (display, fgfall_color_gc, looper-1, (int)owl, looper, (int)nwl);
      else {
        x_wl_to_midwl (owl, midwl, looper-1, 0, bg_color);
        x_wl_to_midwl (nwl, midwl, looper,   0, bg_color);
      }
    } else {
      x_wl_to_midwl (owl, midwl, looper-1, 0, bg_color);
    }
  } else {
    x_wl_to_midwl (nwl, midwl, looper, clearflag, bg_color);
  }
}

/* Similar for normal tides */
static void
x_tide_graph (double nwl, double owl, int looper, double prev_g_tide, double
this_g_tide, int clearflag, HPEN bg_color)
{
  int prevx, prevy;
  if (clearflag)
    XDrawLine (display, bg_color, looper, 0, looper, winH-1);
  if (linegraph) {
    prevx = looper - 1;
    prevy = owl;
  } else {
    prevx = looper;
    prevy = winH - 1;
  }
  if (/*!hairy ||*/ this_g_tide > prev_g_tide) {
    if (display_doesnt_suck && !linegraph)
      XDrawLine (display, fgrise_color_gc, prevx, prevy, looper, (int)nwl+1);
    else
      XDrawLine (display, fgrise_color_gc, prevx, prevy, looper, (int)nwl);
  } else {
    if (display_doesnt_suck && !linegraph)
      XDrawLine (display, fgfall_color_gc, prevx, prevy, looper, (int)nwl+1);
    else
      XDrawLine (display, fgfall_color_gc, prevx, prevy, looper, (int)nwl);
  }
}

/* BOGUS units conversion - Added mgh
 * For stations with knots^2 values, converts a normalized graphics tide
 * level to match the units lines. */
void check_BOGUS(double *norm) {
double off,n,sqr;
   if (have_BOGUS && !have_offsets && convert_BOGUS) {
      off = -fakedatum / fakeamplitude;
      n   = (*norm - off) * amplitude;      // n = original amplitude
      sqr = sqrt(fabs(n)) / fakeamplitude;  // convert to fake units
      if (n >= 0.0)
           *norm = sqr + off;
      else *norm = off - sqr;
   }
   return;
}

/* Draw graph; don't return. */
static void
draw_graph ()
{
  int a, /*event_type,*/ midwl, sun_is_up, beg, end;
  double prev_g_tide, this_g_tide, nwl, owl;
  time_t start = faketime, this_time = time(NULL), finish, next_sun_time;
  HPEN bg_color;
  COLORREF background;
  /* Initialize the amplitude. */
  happy_new_year (yearoftimet (start));
  midwl = tide2wl (-fakedatum / fakeamplitude);
  faketime = start;
  next_sun_time = faketime;
  next_sun_event(&next_sun_time, IDX_lat, IDX_lon, -1);
//  beg = end = 0;
//  do {
//     sun_is_up = next_sun_event(&next_sun_time, IDX_lat, IDX_lon, 1);
//     background  = sun_is_up? bgnite_color : bgday_color;
//     end = (next_sun_time - faketime) / tstep;
//     if (end > winW)
//         end = winW;
//     XFillRectangle (display, window, background, beg, 0, end, winH);
//     beg = end;
//  }
//  while (end < winW);

  faketime = start;
  next_sun_time = window_left_time = faketime;
  next_sun_event(&next_sun_time, IDX_lat, IDX_lon, -1);
//  if (!toplines)
//    draw_unit_lines();
  find_tmin_tmax();

  prev_g_tide = time2secondary (faketime-1);
  check_BOGUS(&prev_g_tide);
  owl = tide2wl (prev_g_tide);
  for (a=0;a<winW;a++) {
    if (faketime > next_sun_time) {
       sun_is_up = next_sun_event(&next_sun_time, IDX_lat, IDX_lon, 1);
       bg_color  = sun_is_up? bgnite_color_gc : bgday_color_gc;
    }
    this_g_tide = time2secondary (faketime);
    check_BOGUS(&this_g_tide);
    nwl = tide2wl (this_g_tide);
    if (iscurrent)
      x_current_graph (nwl, owl, midwl, a, 1, bg_color);
    else
      x_tide_graph (nwl, owl, a, prev_g_tide, this_g_tide, 1, bg_color);
    prev_g_tide = this_g_tide;
    owl = nwl;
    faketime += tstep;
  }
  if (toplines)
    draw_unit_lines();
  else
    graph_relabel_unit_lines ();
  draw_extra_lines();
  /* Do this first so red marks will go on top. */
  draw_tick_marks(start);
  if (1) {// Always make an X// (now) {
    /* X marks the spot */
    int x, y;
    x = (int)((this_time - start) / tstep);
    this_g_tide = time2secondary (this_time);
    check_BOGUS(&this_g_tide);
    y = tide2wl (this_g_tide);
    XDrawLine (display, fgtext_color_gc, x, y-4, x, y+4);
    XDrawLine (display, fgtext_color_gc, x-4, y, x+4, y);
  }
  next_ht = start - MAX(abs(httimeoff),abs(lttimeoff));
  finish = start + winW*tstep + MAX(abs(httimeoff),abs(lttimeoff));
  event_type = update_high_tide ();
  while (next_ht < finish) {
    int x = (int)((next_ht_adj - start) / tstep);
    if (event_type & 3)
      write_high_tide (x);
    if (event_type & 12) {
      write_trans_time (x);
      XDrawLine (display, fgmark_color_gc, x, winH-7, x, winH-1);
    }
    event_type = update_high_tide ();
  }
  draw_moons();
}

/* Redraw the window for the hairy tide clock. */
static void
set_hairy_water_level ()
{
  int looper, w2, midwl, sun_is_up;
  time_t this_time, step_time, next_sun_time, finish;
  double prev_g_tide, this_g_tide, nwl, owl;
  HPEN bg_color;
  w2 = winW>>1;
  if (testmode)
    this_time = faketime;
  else
    this_time = time (NULL);
  next_sun_time = window_left_time = this_time - tstep * (w2+1);
  next_sun_event(&next_sun_time, IDX_lat, IDX_lon,-1);
  watch_tides (this_time);
  prev_g_tide = time2secondary (this_time - tstep * (w2+1));
  check_BOGUS(&prev_g_tide);
  owl = tide2wl (prev_g_tide);
  midwl = tide2wl (-fakedatum / fakeamplitude);
  for (looper=0;looper<winW;looper++) {
    step_time = this_time - tstep * (w2-looper);
    this_g_tide = time2secondary (step_time);
    if (step_time > next_sun_time) {
       sun_is_up = next_sun_event(&next_sun_time, IDX_lat, IDX_lon, 1);
       bg_color  = sun_is_up? bgnite_color_gc : bgday_color_gc;
    }
    check_BOGUS(&this_g_tide);
    nwl = tide2wl (this_g_tide);
    if (iscurrent)
      x_current_graph (nwl, owl, midwl, looper, 1, bg_color);
    else
      x_tide_graph (nwl, owl, looper, prev_g_tide, this_g_tide, 1, bg_color);
    prev_g_tide = this_g_tide;
    owl = nwl;
  }
  next_ht = this_time - MAX(abs(httimeoff),abs(lttimeoff));
  do update_high_tide();
  while (next_ht_adj < this_time); // Find first "next" tide
  update_high_tide ();//mgh+ Get next 2 tides from NOW
  write_high_tide (winW>>1);
  draw_unit_lines();
  draw_extra_lines();
  draw_tick_marks(this_time - tstep * w2);
  /* X marks the spot */
  this_g_tide = time2secondary (this_time);
  check_BOGUS(&this_g_tide);
  nwl = tide2wl (this_g_tide);
  XDrawLine (display, fgtext_color_gc, w2, (int)nwl-4, w2, (int)nwl+4);
  XDrawLine (display, fgtext_color_gc, w2-4, (int)nwl, w2+4, (int)nwl);

  if (mark && lines) {
     next_ht = window_left_time - MAX(abs(httimeoff),abs(lttimeoff));
     finish = window_left_time + winW*tstep + MAX(abs(httimeoff),abs(lttimeoff));
     graphmode = 1; // Allow mark processing
     event_type = update_high_tide ();
     while (next_ht < finish) {
       if (event_type & 12) {
         int x = (int)((next_ht_adj - window_left_time) / tstep);
         XDrawLine (display, fgmark_color_gc, x, winH-7, x, winH-1);
       }
       event_type = update_high_tide ();
     }
     graphmode = 0;
  }
  draw_moons();
}

/*-----------------10/3/2002 6:44PM-----------------
 *
 * --------------------------------------------------*/
void NextEventAdjusted( time_t *ttm, int direction ) {
time_t told, start;
  start = *ttm;
  next_ht = start - MAX(abs(httimeoff),abs(lttimeoff));
  if (direction > 0) {
    do update_high_tide ();
    while (next_ht_adj <= start);
    *ttm = next_ht_adj;
  }
  else {
    next_ht -= DAYSECONDS*2;
    do {
      told = next_ht_adj;
      update_high_tide();
    } while (next_ht_adj < start);
    *ttm = told;
  }
}

/* Redraw the window for the hairy tide clock. */

DWORD WINAPI MyMessageLoop(LPVOID lpv) {
    MSG msg;

    while(GetMessage(&msg,NULL,0,0)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
    }
    return 0;
}

void crash(void) {
    if(szAccum)
        MessageBox(NULL,szAccum,"Tidelib Fatal Error:",MB_ICONSTOP);
}

void list_switches() {
   printf (
"Usage:\n\
wxtide32 [-24]                  Use 24-hour time, not AM / PM\n\
         [-banner]              Sideways ASCII tide graph\n\
         [-bg ...]              Background color\n\
         [-calen]               Make 1 month tide calendar\n\
         [-calib]               (Hairy)  Tick marks at top of the hour\n\
         [-check YYYY]          Check for errors in equilibrium arguments\n\
         [-config {filename | -}]  Read configuration file / stdin\n\
         [-cur]                 Restrict -location search to currents\n\
         [-display ...]         X display\n\
         [-fgfall ...]          Color of ebbing water\n\
         [-fgmark ...]          Color of mark line\n\
         [-fgmiddle ...]        Color of middle line\n\
         [-fgmllw ...]          Color of mllw line\n\
         [-fgrise ...]          Color of flooding water\n\
         [-fgtext ...]          Color of text and lines\n");
   printf (
#ifdef DEANGIF
"         [-geometry ...]        X, PPM, or GIF geometry\n\
         [-gif {filename | -}]  Write graph as GIF to file / stdout\n");
#else
"         [-geometry ...]        X or PPM geometry\n");
#endif
   printf (
"         [-graph]               Graph 2 days of tides (modulo gstretch)\n\
         [-gstart YYYY:MM:DD:HH:MM]  Time at which to start graph or text\n\
         [-gstretch N.NN]       Adjust graph detail vs. time span\n\
         [-hairy]               Tide graph in clock / 2-color graph\n\
         [-help]                Send usage to standard output\n\
         [-hfile ...]           Location and name of harmonics file\n\
         [-hinc [N]]            Label depth marks in increments of N\n\
         [-hloff [{-|x}]N.NN]   Tide level offset for high tide\n\
         [-htoff [-]HH:MM]      Time offset for high tide\n\
         [-list]                List all locations in harmonics file\n\
         [-listtz]              List with meridians and time zone info\n\
         [-lloff [{-|x}]N.NN]   Tide level offset for low tide\n\
         [-location \"Name\"]     Location to show tides for\n\
         [-loctz]               Try to use time zone of location (Moof!)\n\
         [-ltoff [-]HH:MM]      Time offset for low tide\n\
         [-mark [-]N.NN]        Draw line at specified tide level\n\
         [-middle]              Draw line near mean tide level\n\
         [-mllw]                Draw line at mean lower low water level\n\
         [-nofill]              Line graph instead of filled-in graph\n\
         [-nolines]             Suppress tick marks / depth lines\n\
         [-norename]            Don't set icon name to time of high tide\n\
         [-now]                 Show current time / mark place on graph\n\
         [-nowarn]              Suppress big ugly warning message\n\
         [-ppm {filename | -}]  Write graph as PPM to file / stdout\n\
         [-ps]                  Generate PostScript[tm] output\n\
         [-raw YYYY:MM:DD:HH:MM [HH:MM]] Raw output from gstart to this\n\
                                time, with optional step (default 1:00)\n\
         [-skinny]              Trim the fat, make window tall and skinny\n\
         [-stats YYYY:MM:DD:HH:MM]  Find stats from gstart to this time\n\
         [-test [N]]            Test mode (mostly useless)\n\
         [-text [N]]            List N tides / select ASCII graph mode\n\
         [-thin]                Leave out year and time zone\n\
         [-tinc [N]]            Label each N hours\n\
         [-toplines]            (Graph)  Depth lines on top of graph\n\
         [-tz [-]HH:MM]         Fixed time zone offset for timestamps\n\
         [-units {ft|m}]        Convert units where applicable\n\
         [-utc]                 Show timestamps in UTC\n\
         [-uutc]                User's times (gstart, etc.) are in UTC\n\
         [-version]             Print WXTide32 version\n\
         [-weekday]             Show day of week in printed dates\n\
\n\
Other flags added by mgh\n\
         [-autokt]              Auto change knots^2 to knots\n\
         [-autosave]            Save changed options and window positions on exit\n\
         [-custname \"name\"]     Name of custom station, only used with custom\n\
                                station offsets (-htoff, -hloff, -ltoff, -htoff)\n\
         [-fgmapdot ...]        Color of dots on locator map\n\
         [-incremental [minutes]] Show incremental tide levels\n\
         [-indexfile {filename} Name of index file\n\
         [-keepindex]           Keeps station index in memory (faster,\n\
                                locate), otherwise dumps arrays when done\n\
         [-moon]                Show moon phases on graphs\n\
         [-scrollbar]           Show time scrollbar in graph mode\n\
         [-showlevels]          Show tide levels on graphs and clocks\n\
         [-sunmoon]             Show sun/moon rise/fall on text tides\n\
         [-textfile filepath]   Redirect text output to file, -nowarn inhibits msg\n\
         [-append]              Append - WXTide32 to end of window name\n\
\n\
Time format example -- half past midnight, June 1, 1995:  1995:06:01:00:30\n");
   printf (
#ifdef DEANGIF
"In PPM/GIF mode, colors must be specified as rgb:hh/hh/hh (where hh is a\n\
2-digit hexadecimal number) and geometry must be specified as NxN.\n");
#else
"In PPM mode, colors must be specified as rgb:hh/hh/hh (where hh is a 2-digit\n\
hexadecimal number) and geometry must be specified as NxN.\n");
#endif
}

/* -----------------2/12/98 7:07AM-------------------
 * File management stuff added for drag and drop.
 * --------------------------------------------------*/

/* get_path_only transfers the path only to a destination */
void get_path_only(char *dst, char *filename) {
int i;
char ch;

   strcpy(dst, filename);
   for (i=strlen(dst)-2; (i>0) && // Strip program name to just leave path
       ((ch=dst[i]) != '\\') && (ch != ':') ; i--) dst[i] = '\0';
}

/* get_name_only transfers the name only to a destination */
void get_name_only(char *dst, char *filename, int bufsize) {
WIN32_FIND_DATA ffbuf;
HANDLE hfind;
   if((hfind = FindFirstFile( filename, &ffbuf)) != INVALID_HANDLE_VALUE) {
      strcpy(dst, ffbuf.cFileName);
      FindClose(hfind);
   }
   else
      GetFileTitle( filename, dst, bufsize);
}

/* check_file_path finds a file (if possible) by using the program path and
 * any alternate path.  When a file is found, the alternate path is set */
void make_full_path_name(char *name) {
   WIN32_FIND_DATA ffbuf;
   HANDLE hfind;
printf("trying FindFirstFile for %s\n",name);
   if((hfind = FindFirstFile( name, &ffbuf)) != INVALID_HANDLE_VALUE) {
printf("got FindFirstFile for %s\n",ffbuf.cFileName);
      strcpy(name, ffbuf.cFileName);
      FindClose(hfind);
   }
}

int check_file_path(char *dst, char*filename) {
OFSTRUCT ofn;
   if (HFILE_ERROR != OpenFile(filename, &ofn, OF_EXIST)) {
      strcpy(dst, ofn.szPathName);
      if (!strlen(szAlternatePath)) // Only set this once!
         get_path_only(szAlternatePath, ofn.szPathName);//filename);
      return(1);
   }
   sprintf(dst, "%s%s", szProgramPath, filename);
   if (HFILE_ERROR != OpenFile(filename, &ofn, OF_EXIST)) {
      strcpy(dst, ofn.szPathName);
      return(1);
   }
   sprintf(dst, "%s%s", szAlternatePath, filename);
   if (HFILE_ERROR != OpenFile(filename, &ofn, OF_EXIST)) {
      strcpy(dst, ofn.szPathName);
      return(1);
   }
   else
      return(0);
}

/* -----------------2/12/98 9:33AM-------------------
 * init_OPENFILENAME will initialize an OPENFILENAME structure
 * for file open or save as.
 * --------------------------------------------------*/
void init_OPENFILENAME(int flags, char *title) {
static int i = -10;
static char szFile[256], szFileTitle[256], szFilter[256];

   strcpy(szFilter,"WXTide32 config files (*.wxt;*.cfg)|*.wxt;*.cfg|");
   for ( i=0; szFilter[i] != '\0'; i++)
      if (szFilter[i] == '|') szFilter[i] = '\0';
   if (have_user_offsets && have_new_custom)
      get_name_only( szFile, custom_name, sizeof(szFile) );
   else if (strlen(szConfigPath))
      get_name_only( szFile, szConfigPath, sizeof(szFile) );
   else if (i == -10) szFile[0] = '\0';
   if (!strlen(szAlternatePath)) // Make sure AlternatePath points somewhere
      strcpy(szAlternatePath, szProgramPath);

   memset (&ofn, 0, sizeof(OPENFILENAME));
   ofn.lStructSize    = sizeof(OPENFILENAME);
   ofn.hwndOwner      = window;
   ofn.lpstrFilter    = szFilter;
   ofn.nFilterIndex    = 1;
   ofn.lpstrFile      = szFile;
   ofn.nMaxFile       = sizeof(szFile);
   ofn.lpstrFileTitle = szFileTitle;
   ofn.nMaxFileTitle  = sizeof(szFileTitle);
   ofn.lpstrTitle     = title;
   ofn.lpstrInitialDir= szAlternatePath;
   ofn.Flags = flags | OFN_NOCHANGEDIR | OFN_HIDEREADONLY;
   ofn.lpstrDefExt = "wxt";
}

/* Convert level (ft/m) value to/from current units */
//void convert_level_units(double *level, int tofeet) {
//   if (units && (tolower(units[0]) == 'm')) {
//      if (tofeet) *level *= 0.3048;
//      else        *level /= 0.3048;
//   }
//}

/* -----------------2/12/98 10:32AM------------------
 * Routine to write current configuration to a file
 * --------------------------------------------------*/
void write_sw(HFILE hf, char *sw, int nocomment) {
char s[128], comment;
   if (nocomment)
        comment = ' ';
   else comment = '#';
   sprintf(s, "%c-%s\r\n", comment, sw);
   _lwrite(hf,s,strlen(s));
}

void write_sw_str(HFILE hf, char *sw, char *str, int nocomment) {
char s[128], comment;
   if (nocomment)
        comment = ' ';
   else comment = '#';
   sprintf(s, "%c-%s %s\r\n", comment, sw, str);
   _lwrite(hf,s,strlen(s));
}

void write_sw_quote(HFILE hf, char *sw, char *str) {
char s[128];
   sprintf(s, " -%s \"%s\"\r\n", sw, str);
   _lwrite(hf,s,strlen(s));
}

void write_sw_rgb(HFILE hf, char *sw, long int color) {
unsigned char s[128], r,g,b;
   r = GetRValue(color);
   g = GetGValue(color);
   b = GetBValue(color);
   sprintf(s, " -%s rgb:%02x%/%02x%/%02x\r\n", sw, r,g,b);
   _lwrite(hf,s,strlen(s));
}

void write_sw_hhmm(HFILE hf, char *sw, long int sec, int nocomment) {
char s[128], sign, comment;
int hr, min;

   if (nocomment)
        comment = ' ';
   else comment = '#';
      if (sec < 0)
           sign = '-';
      else sign = '+';
      hr = abs(sec) / 3600;
      min= (abs(sec)/60) % 60;
      sprintf(s, "%c-%s %c%1d:%02d\r\n", comment, sw, sign, hr, min);
      _lwrite(hf,s,strlen(s));
}

void write_sw_win(HFILE hf, char *sw, RECT win) {
char s[128], comment;
//   if (save_windows)
        comment = ' ';
//   else comment = '#';
   sprintf(s, "%c-%s %d %d %d %d\r\n",comment,sw,win.left,win.right,win.top,win.bottom);
   _lwrite(hf,s,strlen(s));
}

void write_config_file(char *filename, int notsaveas) {
HFILE hf;
double dbl;
char s[256], fn[256], *p;
int i;

   if (have_params && notsaveas) {
      get_name_only(fn, filename, sizeof(fn));
      sprintf(s,"Merge command line switches into \"%s\"?\r\n"
                "Selecting No will cancel the save.", fn);
      if (IDYES != MessageBox(window, s, "Configuration File Update",
            MB_ICONQUESTION|MB_YESNO))
          return;
   }
   if (!have_user_offsets && had_user_offsets && notsaveas) {
      get_name_only(fn, filename, sizeof(fn));
      sprintf(s,"Overwrite custom station offsets in \"%s\"?\r\n"
                "Selecting No will cancel the save.", fn);
      if (IDYES != MessageBox(window, s, "Configuration File Update",
            MB_ICONQUESTION|MB_YESNO))
          return;
   }
   if (HFILE_ERROR == (hf=_lcreat(filename, 0))) {
      sprintf(s,"Could not open \"%s\" for write.",filename);
      MessageBox(window, s,
            "Configuration File Write Error", MB_ICONEXCLAMATION|MB_OK);
   }
   else {
      have_params = FALSE;
      new_params = FALSE;
//      dbl = Ihtleveloff;
//      convert_level_units(&dbl, 1);
      sprintf(s, "*%0.2lf %0.2lf", Ihlevelmult, Ihtleveloff);
      write_sw_str(hf, "hloff", s, have_user_offsets);

//      dbl = Iltleveloff;
//      convert_level_units(&dbl, 1);
      sprintf(s, "*%0.2lf %0.2lf", Illevelmult, Iltleveloff);
      write_sw_str(hf, "lloff", s, have_user_offsets);

      write_sw_hhmm(hf,"htoff", Ihttimeoff, have_user_offsets);
      write_sw_hhmm(hf,"ltoff", Ilttimeoff, have_user_offsets);

      if (have_user_offsets) {
         write_sw_quote(hf, "location", IDX_reference_name);
         write_sw_quote(hf, "custname", custom_name);
      }
      else {
         sprintf(s,"%s (%c)",IDX_station_name,IDX_type);
//         write_sw_quote(hf, "location", IDX_station_name);
         write_sw_quote(hf, "location", s);
         write_sw(hf, "custname \"name\"", FALSE);
      }

      write_sw(hf,     "24", noampm);
      write_sw(hf,     "autokt",convert_BOGUS);
      write_sw(hf,     "autosave",auto_save);
//      write_sw_rgb(hf, "bg", bg_color);
      write_sw_rgb(hf, "bgday", bgday_color);
      write_sw_rgb(hf, "bgnite", bgnite_color);
      write_sw(hf,     "calib", hairycalibrate);
      write_sw(hf,     "cur", curonly);
      write_sw_rgb(hf, "fgfall", fgfall_color);
      write_sw_rgb(hf, "fgmapdot", fgmapdot_color);
      write_sw_rgb(hf, "fgmark", fgmark_color);
      write_sw_rgb(hf, "fgmiddle", fgmiddle_color);
      write_sw_rgb(hf, "fgmllw", fgmllw_color);
      write_sw_rgb(hf, "fgrise", fgrise_color);
      write_sw_rgb(hf, "fgtext", fgtext_color);
      write_sw(hf,     "graph", graphmode);
      sprintf(s, "%0.2lf", (180.0 / tstep));
      write_sw_str(hf, "gstretch", s, (tstep != 180));
      write_sw(hf,     "hairy", hairy);
      write_sw(hf,     "hinc", hinc);

      sprintf(s, "%d", increment_step);
      write_sw_str(hf, "incrementstep", s, TRUE);

      write_sw_quote(hf, "indexfile", indexfile_name);
//      write_sw(hf,     "keepindex", keep_index);
      write_sw(hf,     "loctz", loctz);
      write_sw(hf,     "locmap", had_map);
      sprintf(s, "%0.2lf", marklev);
      write_sw_str(hf, "mark", s, mark);
      write_sw_str(hf, "marklev", s, TRUE);
      write_sw(hf,     "middle", middle);
      write_sw(hf,     "mllw", mllw);
      write_sw(hf,     "moon", show_moons);
      write_sw(hf,     "nofill", linegraph);
      write_sw(hf,     "nolines", !lines);
      write_sw(hf,     "now", now);
      write_sw(hf,     "nowarn", TRUE);// nowarn);

      sprintf(s, "%d", num_days);
      write_sw_str(hf, "numdays", s, TRUE);

      write_sw(hf,     "scrollbar", usescrollbar);
      write_sw(hf,     "showlevels", showampl);
      write_sw(hf,     "skinny", (skinny==2));
      write_sw(hf,     "sunmoon", sun_moon);
      if (Usetadjust)
         write_sw_str(hf, "tz", tadjust_last, TRUE);
      else
         write_sw_str(hf, "tzsave", tadjust_last, TRUE);
      write_sw(hf,     "thin", (skinny==1));
      write_sw(hf,     "tinc", tinc);
      write_sw(hf,     "toplines", toplines);
      write_sw_str(hf, "units", youwant, (youwant!=NULL));
      write_sw(hf,     "utc", (utc && !Usetadjust && !loctz));
      write_sw(hf,     "weekday", weekday);

      sprintf(s,"\n# Switches that can only be used on a command line\n");
      _lwrite(hf,s,strlen(s));
      write_sw(hf,     "banner", FALSE);
      write_sw(hf,     "calen", FALSE);
      write_sw(hf,     "gstart", FALSE);
      write_sw(hf,     "incremental [minutes]", FALSE);
      write_sw(hf,     "list", FALSE);
      write_sw(hf,     "listtz", FALSE);
      write_sw(hf,     "ps", FALSE);
      write_sw(hf,     "raw", FALSE);
      write_sw(hf,     "stats", FALSE);
      write_sw(hf,     "test", FALSE);
      write_sw(hf,     "text", FALSE);
//      write_sw(hf,     "tz", FALSE);
      write_sw(hf,     "uutc", FALSE);

      sprintf(s,"\n# Switches used only internally to WXTide32.\n");
      _lwrite(hf,s,strlen(s));
      write_sw_win(hf, "WinDimMain", WinDimMain);
      write_sw_win(hf, "WinDimText", WinDimText);
      write_sw_win(hf, "WinDimMap",  WinDimMap);
      sprintf(s, "%d \"%s\"", gFontSize, gFontName);
      write_sw_str(hf, "gfont", s, TRUE);
      sprintf(s, "%d \"%s\"", tFontSize, tFontName);
      write_sw_str(hf, "tfont", s, TRUE);
//      write_sw(hf,     "append", appendWXTide32);
      for (i = 9; i>=0; i--) {
        p = get_mru_entry( i );
        if (p) write_sw_quote(hf, "mru", p+2);
      }
      sprintf(s, "%d", max_mru);
      write_sw_str(hf, "maxmru", s, TRUE);
      _lclose(hf);
   }
}

// all our command line handling is going to have to change a bit cos windows
// doesn't really work the same way.
// we will try and allow normal handling of command lines but
// .xtiderc will be replaced by 'wintide.ini' in the windows dir

#define getarg if (!(arg = poparg())) barf (WANTMOREARGS)
#define getint(a) if (sscanf (arg, "%d", &a) != 1) barf (BADINTEGER)
#define getfloat(a) if (sscanf (arg, "%lf", &a) != 1) barf (BADFLOAT)

//int main(int argc, char **argv) {
int FAR PASCAL WinMain(HINSTANCE hinst,HINSTANCE hprev,LPSTR lpCmdLine,int nCmdShow) {
  MSG msg;
  char szFileName[_MAX_PATH], szTempCmdLine[256];
  char szPathName[_MAX_PATH];
  char *arg, *pCmdLine;
  int argl=1, rawstep = 0, i;
  time_t stoptime = 0;
  HACCEL hAccel;

  g_hinst=hinst;
  atexit(crash);
  set_local_tz(); // LCC doesn't seed current timezone info

  /* From config.h */
  assert ((int)(strlen (deflocation)) <= MAXARGLEN);
  strcpy (location, deflocation);

  assert ((int)(strlen (hfile)) <= MAXARGLEN);
  strcpy (hfile_name, hfile);

  assert ((int)(strlen (indexfile)) <= MAXARGLEN);
  strcpy (indexfile_name, indexfile);

  /* Set program path in case we need it */
  GetModuleFileName(g_hinst,szProgramName, sizeof(szFileName));
  get_path_only(szProgramPath, szProgramName);
  szAlternatePath[0] = '\0';

  /* Check environment */
  if ((arg = getenv ("LOCATION"))) {
    assert ((int)(strlen (arg)) <= MAXARGLEN);
    strcpy (location, arg);
  }

  if ((arg = getenv ("HFILE"))) {
    assert ((int)(strlen (arg)) <= MAXARGLEN);
    strcpy (hfile_name, arg);
  }

  /* See if we were passed a file name */
  szConfigPath[0] = '\0';
// Cure for being passed a garbage command line.....
  pCmdLine = GetCommandLine();
  while (*pCmdLine == ' ') pCmdLine++; //Strip leading spaces
  if (*pCmdLine != '"')
     while (*pCmdLine != ' ' && *pCmdLine) pCmdLine++;  // Find first space
  else {
     do pCmdLine++;
     while (*pCmdLine != '"' && *pCmdLine); // Find ending quote
     if (*pCmdLine == '"') pCmdLine++;
  }
  strcpy(szTempCmdLine, pCmdLine);

  while (szTempCmdLine[0] == ' ')
        strcpy(szTempCmdLine, szTempCmdLine+1); // Strip leading blanks
  for (i=strlen(szTempCmdLine)-1; (i>0) && (szTempCmdLine[i] <= ' '); i--)
        szTempCmdLine[i] = '\0';             // Strip trailing blanks

  if ((szTempCmdLine[0] == '"') || (szTempCmdLine[0] == '\\') ||
      (szTempCmdLine[0] == '.') || isalpha(szTempCmdLine[0])) {
     if (szTempCmdLine[0] == '"') {
        for (i=1; (i<strlen(szTempCmdLine) && (szTempCmdLine[i]!='"')); i++) {
            szFileName[i-1] = szTempCmdLine[i];
            szFileName[i  ] = '\0';
        }
     }
     else {
        for (i=0; (i<strlen(szTempCmdLine) && (szTempCmdLine[i]!=' ')); i++) {
            szFileName[i  ] = szTempCmdLine[i];
            szFileName[i+1] = '\0';
        }
     }
     if (i < strlen(szTempCmdLine)) i++; // Skip over terminator
     strcpy(szTempCmdLine, szTempCmdLine+i); // Strip file name from command line

     if (check_file_path(szPathName, szFileName)) { // File of that name exists
        for (i=0; i<4; i++) szFileName[i] = tolower(szPathName[strlen(szPathName)-4+i]);
        if (!strncmp(szFileName, ".wxt", 4) || (!strncmp(szFileName, ".cfg", 4))) {
     // File exists and has an extention of "wxt" or "cfg" so load it.
           if (push_config_file (szPathName)) {
              argl = 0;
              strcpy( szConfigPath, szPathName );
           }
        }
     }
     if (argl) {
        printf("Ignoring improper command line format: [%s]\n",lpCmdLine);
        strcpy(szTempCmdLine,"");
        nowarn = TRUE;
     }
  }

  have_params = (strlen(szTempCmdLine) && strchr(szTempCmdLine,'-'));

  if (have_params) {

// BAD Kludge
// I'm going to stuff the command line into a file so it can be read
// this is cos windows doesn't handle cmd lines properly

  GetTempPath(sizeof szPathName, szPathName);
  GetTempFileName(szPathName,"tide",0,szFileName);
     hf=_lcreat(szFileName,0);
     assert(hf!=HFILE_ERROR);
     _lwrite(hf,szTempCmdLine,strlen(szTempCmdLine));
     _lclose(hf);
     push_config_file(szFileName);
     remove(szFileName);
  }

  /* Try to read "WXTide32.cfg */
  if (argl) {
      char tfile[MAXARGLEN+1], tpath[256];
      sprintf (tfile, "%s", winconfigFN);
      if (check_file_path(tpath, tfile)) { // File of that name exists
         if (push_config_file (tpath)) {
            argl = 0;
            strcpy( szConfigPath, tpath );
         }
      }
  }

  /* If failed, try to read system config file */
  while ((arg = poparg())) {
    if ((!strcmp (arg, "-geometry"))||(!strcmp (arg, "-ppmgeom"))) {
      getarg;
      if (geometry)
        free (geometry);
      geometry = stradoop (arg);
    } else if (!strcmp (arg, "-ppm")) {
      getarg;
      if (ppm)
        free (ppm);
      ppm = stradoop (arg);
    } else if (!strcmp (arg, "-gif")) {
#ifdef DEANGIF
      getarg;
      if (gif)
        free (gif);
      gif = stradoop (arg);
#else
      barf (NODEANGIF);
#endif
    } else if (!strcmp (arg, "-config")) {
      getarg;
      if (!(push_config_file (arg)))
        barf (CONFIGFAIL);
      else
        strcpy( szConfigPath, szPathName );
    } else if (!strcmp (arg, "-gfont")) {
      getarg;
      getint (gFontSize);
      getarg;
      strcpy(gFontName, arg);
    } else if (!strcmp (arg, "-tfont")) {
      getarg;
      getint (tFontSize);
      getarg;
      strcpy(tFontName, arg);
    } else if (!strcmp (arg, "-fgrise")) {
      getarg;
      if (fgrise_color_arg)
        free (fgrise_color_arg);
      fgrise_color_arg = stradoop(arg);
    } else if (!strcmp (arg, "-fgfall")) {
      getarg;
      if (fgfall_color_arg)
        free (fgfall_color_arg);
      fgfall_color_arg = stradoop(arg);
    } else if (!strcmp (arg, "-fgtext")) {
      getarg;
      if (fgtext_color_arg)
        free (fgtext_color_arg);
      fgtext_color_arg = stradoop(arg);
    } else if (!strcmp (arg, "-fgmark")) {
      getarg;
      if (fgmark_color_arg)
        free (fgmark_color_arg);
      fgmark_color_arg = stradoop(arg);
    } else if (!strcmp (arg, "-fgmllw")) {
      getarg;
      if (fgmllw_color_arg)
        free (fgmllw_color_arg);
      fgmllw_color_arg = stradoop(arg);
    } else if (!strcmp (arg, "-fgmiddle")) {
      getarg;
      if (fgmiddle_color_arg)
        free (fgmiddle_color_arg);
      fgmiddle_color_arg = stradoop(arg);
    } else if (!strcmp (arg, "-hfile")) {
      getarg;
      check_file_path(hfile_name, arg);
    } else if (!strcmp (arg, "-bg")) {
      getarg;
      if (bgday_color_arg)
        free (bgday_color_arg);
      bgday_color_arg = stradoop(arg);
      if (bgnite_color_arg)
        free (bgnite_color_arg);
      bgnite_color_arg = stradoop(arg);
    } else if (!strcmp (arg, "-bgday")) {
      getarg;
      if (bgday_color_arg)
        free (bgday_color_arg);
      bgday_color_arg = stradoop(arg);
    } else if (!strcmp (arg, "-bgnite")) {
      getarg;
      if (bgnite_color_arg)
        free (bgnite_color_arg);
      bgnite_color_arg = stradoop(arg);
    } else if (!strcmp (arg, "-check")) {
      getarg;
      getint (checkyear);
    } else if (!strcmp (arg, "-gstart")) {
      getarg;
      faketime = parse_time_string (arg);
      gstart_time=faketime;
    } else if (!strcmp (arg, "-stats")) {
      getarg;
      stoptime = parse_time_string (arg);
    } else if (!strcmp (arg, "-raw")) {
      getarg;
      stoptime = parse_time_string (arg);
      if (dataarg ()) {
        getarg;
        rawstep = hhmm2seconds (arg);
      }
      if (!rawstep)
        rawstep = 3600;
    } else if (!strcmp (arg, "-location")) {
      getarg;
      strcpy (location, arg);
    } else if (!strcmp (arg, "-test")) {
      testmode = 1;
      if (dataarg ()) {
        getarg;
        getint (testspeed);
      }
    } else if (!strcmp (arg, "-nowarn")) {
      nowarn = 1;
    } else if (!strcmp (arg, "-text")) {
      text = -1;
      if (dataarg ()) {
        getarg;
        getint (text);
      }
    } else if (!strcmp (arg, "-incremental")) {
      incremental_tides = -1;
      if (dataarg ()) {
        getarg;
        getint (incremental_tides);
      }
    } else if (!strcmp (arg, "-hinc")) {
      if (dataarg ()) {
        getarg;
        getint (hinc);
      }
      if (hinc < 1)
        hincmagic = 1;
    } else if (!strcmp (arg, "-tinc")) {
      if (dataarg ()) {
        getarg;
        getint (tinc);
      }
      if (tinc < 1)
        tinc = 1;
    } else if (!strcmp (arg, "-tz")) {
      getarg;
      if (isalpha(arg[0])) {
        strcpy(tadjust_last, arg);
        strcpy(tadjust_tzname, arg);
        Usetadjust = 2;
        utc = loctz = 0;
      }
      else {
        strcpy(tadjust_last, arg);
        tz_time2sec( arg, &Itadjust );
        if (Itadjust) {
          tadjust = Itadjust;
          Usetadjust = 1;
          utc = 1;
          loctz = 0;
        }
      }
    } else if (!strcmp (arg, "-tzsave")) {
      getarg;
      if (isalpha(arg[0])) {
        strcpy(tadjust_last, arg);
        strcpy(tadjust_tzname, arg);
      }
      else {
        strcpy(tadjust_last, arg);
        tz_time2sec( arg, &Itadjust );
      }
    } else if (!strcmp (arg, "-nolines")) {
      lines = 0;
    } else if (!strcmp (arg, "-norename")) {
      norename = 1;
    } else if (!strcmp (arg, "-list")) {
      list = 1;
    } else if (!strcmp (arg, "-listtz")) {
      list = 2;
    } else if (!strcmp (arg, "-loctz")) {
      loctz = 1;
      utc = Usetadjust = tadjust = 0;
    } else if (!strcmp (arg, "-cur")) {
      curonly = 1;
    } else if (!strcmp (arg, "-calib")) {
      hairycalibrate = 1;
    } else if (!strcmp (arg, "-graph")) {
      graphmode = 1; hairy = 0;
    } else if (!strcmp (arg, "-nofill")) {
      linegraph = 1;
    } else if (!strcmp (arg, "-toplines")) {
      toplines = 1;
    } else if (!strcmp (arg, "-gstretch")) {
      double gstretch;
      getarg;
      getfloat (gstretch);
      if (gstretch <= 0.0)
        barf (STRETCHTOOSMALL);
      tstep = (int)(180.0 / gstretch);
      if (tstep <= 0)
        barf (STRETCHTOOBIG);
    } else if (!strcmp (arg, "-hairy")) {
      hairy = 1; graphmode = 0;
    } else if (!strcmp (arg, "-skinny")) {
      skinny = 2;
    } else if (!strcmp (arg, "-thin")) {
      skinny = 1;
    } else if (!strcmp (arg, "-weekday")) {
      weekday = 1;
    } else if (!strcmp (arg, "-middle")) {
      middle = 1;
    } else if (!strcmp (arg, "-banner")) {
      banner = 1;
    } else if (!strcmp (arg, "-mllw")) {
      mllw = 1;
    } else if (!strcmp (arg, "-java")) {
//      java = 1;
    } else if (!strcmp (arg, "-javanh")) {
//      java = javanh = 1;
    } else if (!strcmp (arg, "-24")) {
      noampm = 1;
    } else if (!strcmp (arg, "-now")) {
      now = 1;
    } else if (!strcmp (arg, "-calen")) {
      calendar = 1;
    } else if (!strcmp (arg, "-mark")) {
      mark = 1;
      getarg;
      getfloat (marklev);
    } else if (!strcmp (arg, "-marklev")) {
      getarg;
      getfloat (marklev);
    } else if (!strcmp (arg, "-utc")) {
      utc = 1;
      Usetadjust = tadjust = loctz = 0;
    } else if (!strcmp (arg, "-units")) {
      getarg;
      if (youwant)
        free (youwant);
      youwant = stradoop (arg);
    } else if (!strcmp (arg, "-uutc")) {
      uutc = 1;
    } else if (!strcmp (arg, "-ps")) {
      ps = 1;
    } else if (!strcmp (arg, "-hloff")) {
      Ihlevelmult = 1.0;
      Ihtleveloff = 0.0;
      do {
         getarg;
         if (arg[0] == '*' || arg[0] == 'x') {
            arg[0] = ' ';
            getfloat (Ihlevelmult);
            if (Ihlevelmult <= 0.0)
               barf (BADMULT);
         }
         else
            getfloat (Ihtleveloff);
      }
      while (dataarg());
    } else if (!strcmp (arg, "-lloff")) {
      Illevelmult = 1.0;
      Iltleveloff = 0.0;
      do {
         getarg;
         if (arg[0] == '*' || arg[0] == 'x') {
            arg[0] = ' ';
            getfloat (Illevelmult);
            if (Illevelmult <= 0.0)
               barf (BADMULT);
         }
         else
            getfloat (Iltleveloff);
      }
      while (dataarg());
    } else if (!strcmp (arg, "-htoff")) {
      getarg;
      Ihttimeoff = hhmm2seconds (arg);
    } else if (!strcmp (arg, "-ltoff")) {
      getarg;
      Ilttimeoff = hhmm2seconds (arg);
    } else if (!strcmp (arg, "-version")) {
      if (PATCHLEVEL)
        printf ("XTide %s.%d\n", VERSION, PATCHLEVEL);
      else
        printf ("XTide %s\n", VERSION);
      printf("WXTide32 version based on above XTide source\n");
      done=TRUE;
      goto MessageLoop;

/* Switches added by mgh */
    } else if (!strcmp (arg, "-fgmapdot")) {
      getarg;
      if (fgmapdot_color_arg)
        free (fgmapdot_color_arg);
      fgmapdot_color_arg = stradoop(arg);
    } else if (!strcmp (arg, "-keepindex")) {
//      keep_index = 1;
    } else if (!strcmp (arg, "-scrollbar")) {
      usescrollbar = 1;
    } else if (!strcmp (arg, "-showlevels")) {
      showampl = 1;
    } else if (!strcmp (arg, "-moon")) {
      show_moons = 1;
    } else if (!strcmp (arg, "-sunmoon")) {
      sun_moon = 1;
    } else if (!strcmp (arg, "-indexfile")) {
      getarg;
      check_file_path(indexfile_name, arg);
    } else if (!strcmp (arg, "-custname")) {
      getarg;
      if (custom_name) free(custom_name);
      custom_name = stradoop(arg);
    } else if (!strcmp (arg, "-textfile")) {
      getarg;
      output_filename = stradoop(arg);
    } else if (!strcmp (arg, "-autokt")) {
      convert_BOGUS = 1;
    } else if (!strcmp (arg, "-autosave")) {
      auto_save = 1;
    } else if (!strcmp (arg, "-locmap")) {
      had_map = 1;
    } else if (!strcmp (arg, "-numdays")) {
      getarg;
      getint (num_days);
    } else if (!strcmp (arg, "-incrementstep")) {
      getarg;
      getint (increment_step);
    } else if (!strcmp (arg, "-WinDimMain")) {
        save_windows = TRUE;
        getarg;
        getint (WinDimMain.left);
        getarg;
        getint (WinDimMain.right);
        getarg;
        getint (WinDimMain.top);
        getarg;
        getint (WinDimMain.bottom);
    } else if (!strcmp (arg, "-WinDimText")) {
        save_windows = TRUE;
        getarg;
        getint (WinDimText.left);
        getarg;
        getint (WinDimText.right);
        getarg;
        getint (WinDimText.top);
        getarg;
        getint (WinDimText.bottom);
    } else if (!strcmp (arg, "-WinDimMap")) {
        save_windows = TRUE;
        getarg;
        getint (WinDimMap.left);
        getarg;
        getint (WinDimMap.right);
        getarg;
        getint (WinDimMap.top);
        getarg;
        getint (WinDimMap.bottom);
    } else if (!strcmp (arg, "-append")) {
        appendWXTide32 = 1;
    } else if (!strcmp (arg, "-mru")) {
      getarg;
      add_mru( arg );
    } else if (!strcmp (arg, "-maxmru")) {
      getarg;
      getint (max_mru);

/* Back to original code */
    } else {
      list_switches();
      if (strcmp (arg, "-help"))
         printf("\nERROR: I do not know what switch [%s] means.\n",arg);
      done=TRUE;
      goto MessageLoop;
    }
  }
  if (!nowarn) printf (
" WXTide32 (C) 1998-2002 Michael Hopper. Based on XTide (C) 1996 David Flater.\n\
\n\
This program is distributed in the hope that it will be useful, but WITHOUT ANY\n\
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A\n\
PARTICULAR PURPOSE. See the GNU General Public License for more details.\n\
\n\
 *** DO NOT RELY ON THE PREDICTIONS OF THIS PROGRAM FOR DECISIONS THAT CAN ***\n\
 ***      RESULT IN HARM TO ANYONE OR ANYTHING.  REALLY.  I MEAN IT.       ***\n\
  (This warning will disable automatically the first time settings are saved)\n");

  /* Remove glaring inconsistencies in args */
  if (!lines)
    hinc = tinc = 0;
  if (banner || stoptime) {
    graphmode = calendar = ps = 0;
    ppm = gif = NULL;
  }
  if (graphmode && (ppm || gif || ps))
    graphmode = 0;
  if (graphmode || ppm || gif || ps) {
    calendar = 0;
    if (!hinc)
      hincmagic = 1;
  }
  if (calendar) {
    incremental_tides = 0;
    text = 1;
    skinny = 2;
    weekday = 0;   /* Important!  Don't overrun buffer */
  }
  if (incremental_tides) {
    text = 1;
    skinny = 0;
    weekday = 0;   /* Important!  Don't overrun buffer */
  }
  if (graphmode && text)
    skinny = 2;
  if (!graphmode && !ppm && !gif && !hairy && !ps)
    tinc = 0;
  if (hairy && tinc)
    hairycalibrate = 1;

  /* Adjust tstep for ASCII graph mode */
  if (graphmode && text)
    tstep *= AGTSTEP;
  /* Adjust tstep for banner mode */
  if (banner)
    tstep *= BANTSTEP;

  /* Parse geometry for PPM/GIF mode (it's needed here because of -now) */
  if (geometry && (ppm || gif)) {
    if (sscanf (geometry, "%dx%d", &PPMWIDTH, &PPMHEIGHT) != 2)
      barf (BADGEOMETRY);
    if ((PPMWIDTH <= 20)||(PPMHEIGHT <= 20))
      barf (BADGEOMETRY);
    PPMHEIGHT -= 12; /* We take 12 extra, so compensate. */
  }

  /* I can't do this yet for X graph mode because I don't know the geometry
     (oops).  See below. */
  if (!(graphmode && !text)) {
    if (!faketime) {
      faketime = time (NULL);
      /* Center graph around current time if -now */
      if (graphmode && now)
        faketime -= tstep * (TEXTWIDTH>>1);  /* Must be text */
      else if ((ppm || gif) && now)
        faketime -= tstep * (PPMWIDTH>>1);
    }
  }
// We leave XTide processing for a moment to do WXTide32 stuff
  if (!have_index && !list) {
     init_index_file(keep_index, window); // Load all or part based in keep_index flag
     if (!have_index || !load_location_data( location, 0 )) {
        if (output_filename && nowarn) {
           printf("Error: Station \"%s\" is not listed.\r\n", location);
           done = TRUE;
        }
        else {
           if (!have_index) {
//              MessageBox(window,"Please select your station from the following menu",
//                "No index file",MB_ICONINFORMATION);
           }
           else {
              char s[256];
              sprintf(s,"Station \"%s\" is not listed.\r\n"
                  "Please select your station from the following menu",location);
              MessageBox(window,s,"Unknown location",MB_ICONINFORMATION);
           }
           if (!DialogBox(g_hinst,"EnterLocation",window,LocationDlg))
              PostQuitMessage(0);
        }
     }
  }
  else load_data ();
  keep_index = 1; // Any stations loaded after the first will load/keep index */

  done=FALSE;

  if(list) {
    done=TRUE;
    goto MessageLoop;
  }
// Now back to normal XTide stuff
  if (checkyear) {
    check_epoch ();
    done=TRUE;
    goto MessageLoop;
  }

  /* Determine if fudging was sufficient, issue error if not. */
//  if (have_offsets && (ps || java || mark || (stoptime && (!rawstep))))
//    barf (OFFSETSTEXTONLY);
// Corrected mgh

  /* Warn about the interpolation */
  if (have_offsets && (graphmode || ppm || gif || banner || stoptime ||
  (!text)))
  if (!nowarn) printf ("XTide WARNING:  Interpolated tide levels may be \
inaccurate.\n");

  if (loctz)
    change_time_zone (tzfile); /* Moof! */
  if (iscurrent) {
/*    if (have_offsets) {
      printf ("XTide WARNING:  slack water disabled due to \
unequal offsets\n");
    } else { Corrected mgh*/
      mark = 1;
      marklev = 0.0;
//    }
  }
  if (banner) {
    do_banner ();
    done = TRUE;
    goto MessageLoop;
  }
  if (stoptime) {
    if (rawstep)
      do_raw (faketime, stoptime, rawstep);
    else
      do_stats (faketime, stoptime);
    done = TRUE;
    goto MessageLoop;
  }
  if (text) {
    if (graphmode) {
      tide2ascii ();
    }
    if (calendar) {
      do_calendar ();
    } else
    if (incremental_tides) {
      do_incremental ();
    } else
      list_tides ();
    done = TRUE;
    goto MessageLoop;
  }
//  if (ppm) {
//    tide2ppm (ppm);
//    exit (0);
//  }
//  if (gif) {
//    tide2gif (gif);
//    exit (0);
//  }
  if (ps) {
    int hours = tstep * 24 / 180;
    if (hours <= 0)
      barf (STRETCHTOOBIG);
    tide2ps (hours);
    done = TRUE;
    goto MessageLoop;
  }

  fgrise_color=fgrise_color_arg?GetColor(fgrise_color_arg):RGB(0,0,255);//mgh try 255
  fgrise_color_gc=CreatePen(PS_SOLID,0,fgrise_color);
  fgfall_color=fgfall_color_arg?GetColor(fgfall_color_arg):RGB(0,128,0);//mgh was 0,128,0
  fgfall_color_gc=CreatePen(PS_SOLID,0,fgfall_color);
  fgtext_color=fgtext_color_arg?GetColor(fgtext_color_arg):RGB(0,0,0);
  fgtext_color_gc=CreatePen(PS_SOLID,0,fgtext_color);
  fgmark_color=fgmark_color_arg?GetColor(fgmark_color_arg):RGB(255,0,0);
  fgmark_color_gc=CreatePen(PS_SOLID,0,fgmark_color);
  fgmllw_color=fgmllw_color_arg?GetColor(fgmllw_color_arg):RGB(255,255,255);
  fgmllw_color_gc=CreatePen(PS_SOLID,0,fgmllw_color);
  fgmiddle_color=fgmiddle_color_arg?GetColor(fgmiddle_color_arg):RGB(255,255,0);
  fgmiddle_color_gc=CreatePen(PS_SOLID,0,fgmiddle_color);

  bgday_color=bgday_color_arg?GetColor(bgday_color_arg):RGB(0,255,255);
  bgday_color_gc=CreatePen(PS_SOLID,0,bgday_color);
  bgnite_color=bgnite_color_arg?GetColor(bgnite_color_arg):RGB(0,255,255);
  bgnite_color_gc=CreatePen(PS_SOLID,0,bgnite_color);

  fgmapdot_color= fgmapdot_color_arg?GetColor(fgmapdot_color_arg):RGB(255,0,0);
  fgmapdot_color_gc=CreatePen(PS_SOLID,0,fgmark_color);

  if(!hprev)
     Init_Application(hinst);

  Create_Window (hinst);

  /* Draw_graph does not return. */
  if (graphmode) {
    /* See above. */
    if (!faketime) {
      faketime = time (NULL);
      if (now)
        faketime -= tstep * (winW>>1);
    }
//    draw_graph ();
  }
  else {

  /* This clumsy calibration step is made necessary by the new capability
     of using arbitrary offsets in clock mode. */
    time_t this_time;
    if (testmode)
      this_time = faketime;
    else
      this_time = time(NULL);
    assert (this_time > DAYSECONDS*1.5);
    next_ht = this_time - DAYSECONDS*1.5;
    prev_ht_adj = next_ht_adj = 0;
    while (this_time >= prev_ht_adj + 70)
      update_high_tide ();
//  }

    /* Need to do it twice, to get high AND low tides set up */
//    update_high_tide ();
//    update_high_tide ();

    }
    ShowWindow(window,nCmdShow);
    UpdateWindow(window);
    hold_main_window = FALSE;
    had_user_offsets = have_user_offsets;
    new_params = FALSE;
    check_file_path(HelpFileName, "WXTide32.hlp");

MessageLoop:
    if (done == TRUE) {
      DWORD dwThreadId;
      HANDLE hThread;

      hThread=CreateThread(NULL,0,MyMessageLoop,0,0,&dwThreadId);
      SetThreadPriority(hThread,THREAD_PRIORITY_ABOVE_NORMAL);
      if (textLFN) {
         fclose( textLFN );
         return(0);
      }
    }

    hAccel=LoadAccelerators(g_hinst,"wintide");
    while(GetMessage(&msg,NULL,0,0)) {
      if(window!=msg.hwnd || !TranslateAccelerator(window,hAccel,&msg)) {
        TranslateMessage(&msg);
        DispatchMessage(&msg);
      }
    }
    if (textLFN) fclose( textLFN );
    die();
    return msg.wParam;
}


void WinTideCheckMenus(HWND hwnd) {
   char *p;
   int mru, highmru;
   HMENU hmenu=GetMenu(hwnd);

   if(hmenu) {
      CheckMenuItem(hmenu,ID_CLOCK,MF_BYCOMMAND|
                      ((graphmode || hairy)?MF_UNCHECKED:MF_CHECKED));
      CheckMenuItem(hmenu,ID_HAIRY,MF_BYCOMMAND|
                      ((hairy)?MF_CHECKED:MF_UNCHECKED));
      CheckMenuItem(hmenu,ID_GRAPH,MF_BYCOMMAND|
                      ((graphmode)?MF_CHECKED:MF_UNCHECKED));
      EnableMenuItem(hmenu,ID_PRINT,MF_BYCOMMAND|(hwndText!=NULL)?MF_ENABLED:MF_GRAYED);
      EnableMenuItem(hmenu,ID_SAVE,MF_BYCOMMAND|new_params?MF_ENABLED:MF_GRAYED);

      for (highmru=9; highmru>0 && !get_mru_entry(highmru); highmru--)
         DeleteMenu(hmenu, ID_MRU0+highmru, MF_BYCOMMAND);

      for (mru=1; mru<=highmru; mru++) {
        p=get_mru_entry(mru);
        if (!ModifyMenu(hmenu, ID_MRU0+mru, MF_BYCOMMAND|MF_STRING, ID_MRU0+mru, p)) {
             InsertMenu(hmenu, ID_EXIT,     MF_BYCOMMAND|MF_STRING, ID_MRU0+mru, p);
        }
      }
      DrawMenuBar(hwnd);
   }
}
int TZComboIdx;
void LoadTZCombo(HWND hwnd, int ComboBox) {
   char tzname[256];
   int i;

   SendDlgItemMessage(hwnd,ComboBox, CB_RESETCONTENT, 0,0L);
   if (Itadjust) {
      sprintf( tzname, "%s", seconds2hhmm( Itadjust ));
      SendDlgItemMessage(hwnd,ComboBox,CB_INSERTSTRING,-1,(LPARAM)((LPSTR)tzname));
   }
   else if (isalpha(tadjust_last[0])) {
      strcpy( tzname, tadjust_last );
      SendDlgItemMessage(hwnd,ComboBox,CB_INSERTSTRING,-1,(LPARAM)((LPSTR)tzname));
   }
   for (i=0; tz_names[i][0] != NULL; i++)
       SendDlgItemMessage(hwnd,ComboBox,CB_INSERTSTRING,-1,(LPARAM)((LPSTR)tz_names[i][0]));
   if (Itadjust && Usetadjust != 2)
      TZComboIdx = 0;
   else if (isalpha(tadjust_last[0])) {
      SendDlgItemMessage(hwnd,ComboBox,CB_INSERTSTRING,-1,(LPARAM)((LPSTR)tadjust_last));
      TZComboIdx=SendDlgItemMessage(hwnd,ComboBox,CB_FINDSTRING,-1,(LPARAM)((LPSTR)tadjust_last));
   }
   else
      TZComboIdx=SendDlgItemMessage(hwnd,ComboBox,CB_FINDSTRING,-1,(LPARAM)((LPSTR)(tzfile+1)));
   if (TZComboIdx==CB_ERR) TZComboIdx=0;

   SendDlgItemMessage(hwnd,ComboBox, CB_SETCURSEL, TZComboIdx, 0L);
}

int GetTZCombo(HWND hwnd, int ComboBox, int setter) {
   int Changed, tzidx, temptime;
   char tzname[256];
   GetWindowText(GetDlgItem(hwnd, ComboBox), (LPSTR)&tzname, sizeof(tzname));
   Changed=( TZComboIdx !=
         SendDlgItemMessage(hwnd,ComboBox,CB_FINDSTRING,-1,(LPARAM)((LPSTR)&tzname)) );
   if (isalpha(tzname[0])) {
      strcpy(tadjust_last, tzname);
      strcpy(tadjust_tzname, tadjust_last);
      if (IsDlgButtonChecked(hwnd,setter))
           Usetadjust = 2;
      else Usetadjust = 0;
      tadjust = 0;
   }
   else {
      tz_time2sec( tzname, &temptime );
      Itadjust = temptime;
      strcpy(tadjust_last, tzname);
      strcpy(tadjust_tzname, tadjust_last);
      if (IsDlgButtonChecked(hwnd,setter)) {
         Usetadjust = 1;
         tadjust = Itadjust;
      }
      else {
         Usetadjust = 0;
         tadjust = 0;
      }
   }
   return( Changed );
}

void get_flag(HWND hwnd, int *flag, int dlgitem) {
   if (*flag!=IsDlgButtonChecked(hwnd,dlgitem)) {
      *flag = !*flag;
      new_params = TRUE;
   }
}

BOOL CALLBACK FlagDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
int t_keep, i, y, m, d, units_m, pre_BOGUS, new_units, tzidx;
char temp[10], itzname[256];
float t_marklev;
time_t t_now;
static struct tm *tm;
static const char *mon_names[]={
   "January","February","March","April","May","June","July","August","September","October","November","December"
   };

//    t_keep = keep_index;
    switch(msg) {
      case WM_INITDIALOG:
        CheckDlgButton(hwnd,IDF_HINC,hinc);
        CheckDlgButton(hwnd,IDF_TINC,tinc);
        CheckDlgButton(hwnd,IDF_CALIB,hairycalibrate);
        CheckDlgButton(hwnd,IDF_NOW,now);
        CheckDlgButton(hwnd,IDF_24,noampm);
        CheckDlgButton(hwnd,IDF_UTC,utc && !Usetadjust && !loctz);
        CheckDlgButton(hwnd,IDF_REMTZ,loctz);
        CheckDlgButton(hwnd,IDF_FIXEDTZ,Usetadjust);
        CheckDlgButton(hwnd,IDF_LOCALTZ,(!loctz && !utc && !Usetadjust));
        CheckDlgButton(hwnd,IDF_AMPL,showampl);
        CheckDlgButton(hwnd,IDF_MIDDLE,middle);
        CheckDlgButton(hwnd,IDF_MLLW,mllw);
        CheckDlgButton(hwnd,IDF_TOPLINES,toplines);
        CheckDlgButton(hwnd,IDF_MARK,mark);
        CheckDlgButton(hwnd,IDF_SKINNY,skinny == 2);
        CheckDlgButton(hwnd,IDF_THIN,  skinny == 1);
        CheckDlgButton(hwnd,IDF_FULL,  skinny == 0);
        units_m = ((youwant) && tolower(youwant[0] == 'm'));
        CheckDlgButton(hwnd,IDF_MOON,show_moons);
        CheckDlgButton(hwnd,IDF_SUNMOON,sun_moon);
        CheckDlgButton(hwnd,IDF_NOFILL,linegraph);
        CheckDlgButton(hwnd,IDF_SCROLL,usescrollbar);
        CheckDlgButton(hwnd,IDF_ASIS,  (!youwant));
        CheckDlgButton(hwnd,IDF_FEET  ,( youwant && !units_m));
        CheckDlgButton(hwnd,IDF_METERS,( youwant &&  units_m));
        CheckDlgButton(hwnd,IDF_AUTOKT,convert_BOGUS);
        CheckDlgButton(hwnd,IDF_UPDATE,auto_save);
//        CheckDlgButton(hwnd,IDF_MANUAL,auto_save==2);
//        CheckDlgButton(hwnd,IDF_WINDOW,save_windows);
        sprintf(temp,"%d",max_mru);
        SetDlgItemText(hwnd,ID_MAX_MRU, temp);
        sprintf(temp,"%0.2lf",(180.0 / tstep));
        SetDlgItemText(hwnd,IDF_STRETCH, temp);
        sprintf(temp,"%0.2f",marklev);
        SetDlgItemText(hwnd,IDF_MARKLEV, temp);
//        SendDlgItemMessage(hwnd, IDF_MARKLEV, EM_SETREADONLY, !mark, 0L);
//        CheckDlgButton(hwnd,IDF_KEEP,keep_index);
        SetDlgItemInt(hwnd,IDF_NUM_DAYS, num_days, FALSE);
        SetDlgItemInt(hwnd,IDF_INCR, increment_step, FALSE);

        LoadTZCombo(hwnd, IDF_TZNAME);

        if (gstart_time)
             t_now = gstart_time;
        else t_now = time(NULL);
        tm = tzlocaltime( &t_now );
        SendDlgItemMessage(hwnd,IDF_YEAR, CB_RESETCONTENT, 0,0L);
        for (i=first_year; i<(first_year+num_epochs); i++) {
            sprintf(temp,"%d",i);
            SendDlgItemMessage(hwnd,IDF_YEAR, CB_INSERTSTRING, -1, (LPARAM) ((LPSTR) temp));
        }
        SendDlgItemMessage(hwnd,IDF_YEAR, CB_SETCURSEL, tm->tm_year + 1900 - first_year, 0L);

        SendDlgItemMessage(hwnd,IDF_MONTH, CB_RESETCONTENT, 0,0L);
        for (i=0; i<12; i++)
            SendDlgItemMessage(hwnd,IDF_MONTH, CB_INSERTSTRING, -1, (LPARAM) ((LPSTR) mon_names[i]));
        SendDlgItemMessage(hwnd,IDF_MONTH, CB_SETCURSEL, tm->tm_mon, 0L );

        SendDlgItemMessage(hwnd,IDF_DAY, CB_RESETCONTENT, 0,0L);
        for (i=1; i<=31; i++) {
            sprintf(temp,"%d",i);
            SendDlgItemMessage(hwnd,IDF_DAY, CB_INSERTSTRING, -1, (LPARAM) ((LPSTR) temp));
        }
        SendDlgItemMessage(hwnd,IDF_DAY, CB_SETCURSEL, tm->tm_mday-1, 0L );
        PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, IDOK), -1);
        break;

//      case WM_KEYUP:
//           if (VK_F1==wParam)
//              WinHelp(hwnd,HelpFileName,HELP_CONTEXT,IDH_PREF);
//           return FALSE;

      case WM_COMMAND:
        switch(wParam) {
          int scrollsave;
          case IDOK:
            y=SendDlgItemMessage(hwnd,IDF_YEAR,  CB_GETCURSEL, 0, 0L) - 1900 + first_year;
            m=SendDlgItemMessage(hwnd,IDF_MONTH, CB_GETCURSEL, 0, 0L);
            d=SendDlgItemMessage(hwnd,IDF_DAY,   CB_GETCURSEL, 0, 0L) + 1;
            if ((y!=tm->tm_year) || (m!=tm->tm_mon) || (d!=tm->tm_mday)) {
               tm->tm_year = y;
               tm->tm_mon  = m;
               tm->tm_mday = d;
               tm->tm_hour = tm->tm_min = tm->tm_sec = 0;
               gstart_time = mktime( tm );
//               new_params = TRUE;
            }
            new_params |= GetTZCombo(hwnd, IDF_TZNAME, IDF_FIXEDTZ);

            get_flag(hwnd,&hinc,          IDF_HINC);
            if (!hinc) hincmagic = 0;
            get_flag(hwnd,&tinc,          IDF_TINC);
            get_flag(hwnd,&hairycalibrate,IDF_CALIB);
            get_flag(hwnd,&now,           IDF_NOW);
            get_flag(hwnd,&noampm,        IDF_24);

            new_units = loctz;
            get_flag(hwnd,&utc,           IDF_UTC);
            get_flag(hwnd,&loctz,         IDF_REMTZ);
            load_location_data(IDX_station_name, IDX_rec_num);

            get_flag(hwnd,&showampl,      IDF_AMPL);
            get_flag(hwnd,&middle,        IDF_MIDDLE);
            get_flag(hwnd,&mllw,          IDF_MLLW);
            get_flag(hwnd,&toplines,      IDF_TOPLINES);
            get_flag(hwnd,&mark,          IDF_MARK);
            get_flag(hwnd,&show_moons,    IDF_MOON);
            get_flag(hwnd,&sun_moon,      IDF_SUNMOON);
            get_flag(hwnd,&linegraph,     IDF_NOFILL);
            get_flag(hwnd,&auto_save,     IDF_UPDATE);
            scrollsave = usescrollbar;
            get_flag(hwnd,&usescrollbar,  IDF_SCROLL);
            if ((scrollsave != usescrollbar) && graphmode) {
               DoScrollBar(TRUE);
               InvalidateRect(window,NULL,TRUE);
            }
//            save_windows= IsDlgButtonChecked(hwnd,IDF_WINDOW);
//            get_flag(hwnd,&t_keep,        IDF_KEEP);
//            if (!t_keep && keep_index)
//               free_station_index();
//            keep_index = t_keep;

            if  (IsDlgButtonChecked(hwnd,IDF_SKINNY)) {
               if (skinny!=2) new_params = TRUE;
               skinny = 2;
            }
            else if (IsDlgButtonChecked(hwnd,IDF_THIN)) {
               if (skinny!=1) new_params = TRUE;
               skinny = 1;
            }
            else {
               if (skinny) new_params = TRUE;
               skinny = 0;
            }

            GetDlgItemText(hwnd,IDF_MARKLEV, temp, sizeof(temp)-1);
            if ((sscanf(temp,"%f",&t_marklev) == 1) && (marklev!=t_marklev)) {
               new_params = TRUE;
               marklev = t_marklev;
            }

            GetDlgItemText(hwnd,ID_MAX_MRU, temp, sizeof(temp)-1);
            if ((sscanf(temp,"%f",&t_marklev) == 1) &&
                 (t_marklev >= 0) && (t_marklev < 10)) {
              if (t_marklev!=max_mru) new_params = TRUE;
              max_mru = t_marklev;
            }

            GetDlgItemText(hwnd,IDF_STRETCH, temp, sizeof(temp)-1);
            if ((sscanf(temp,"%f",&t_marklev) == 1) &&
                 (t_marklev >= 0.1) && (t_marklev < 10.0)) {
              if ((int)(180.0/t_marklev)!=tstep) new_params = TRUE;
              tstep = 180.0 / t_marklev;
            }

            GetDlgItemText(hwnd,IDF_NUM_DAYS, temp, sizeof(temp)-1);
            if ((sscanf(temp,"%f",&t_marklev) == 1) &&
                 (t_marklev > 0) && (t_marklev < 100)) {
               if (t_marklev!=num_days) new_params = TRUE;
               num_days = t_marklev;
            }

            GetDlgItemText(hwnd,IDF_INCR, temp, sizeof(temp)-1);
            if ((sscanf(temp,"%f",&t_marklev) == 1) &&
                 (t_marklev > 0) && (t_marklev < 1440)) {
               if (t_marklev!=increment_step) new_params = TRUE;
               increment_step = t_marklev;
            }

            units_m   = ( youwant &&  tolower(youwant[0] == 'm'));
            new_units = ( youwant &&  units_m != IsDlgButtonChecked(hwnd,IDF_METERS)) ||
                        ( youwant &&  IsDlgButtonChecked(hwnd,IDF_ASIS)) ||
                        (!youwant && !IsDlgButtonChecked(hwnd,IDF_ASIS));
            if (new_units) {
               if (youwant) free (youwant);
               if (!IsDlgButtonChecked(hwnd,IDF_ASIS)) {
                  if (IsDlgButtonChecked(hwnd,IDF_FEET))
                       youwant = stradoop("ft");
                  else youwant = stradoop("m");
               }
               else youwant = NULL;
            }

            pre_BOGUS = convert_BOGUS;
            convert_BOGUS = IsDlgButtonChecked(hwnd,IDF_AUTOKT);
            if (new_units || (convert_BOGUS != pre_BOGUS)) {
//               convert_level_units(&Ihtleveloff, 0);
//               convert_level_units(&Iltleveloff, 0);
               new_params = TRUE;
               load_location_data(IDX_station_name, IDX_rec_num);
            }
            EndDialog(hwnd,TRUE);
            break;

//          case IDF_UTC:
//            CheckDlgButton(hwnd,IDF_REMTZ, FALSE);
//            break;

//          case IDF_REMTZ:
//            CheckDlgButton(hwnd,IDF_UTC, FALSE);
//            break;

//          case IDF_SKINNY:
//            CheckDlgButton(hwnd,IDF_THIN, FALSE);
//            break;

//          case IDF_THIN:
//            CheckDlgButton(hwnd,IDF_SKINNY, FALSE);
//            break;

//          case IDF_ASIS:
//            CheckDlgButton(hwnd,IDF_ASIS,  TRUE );
//            CheckDlgButton(hwnd,IDF_FEET,  FALSE);
//            CheckDlgButton(hwnd,IDF_METERS,FALSE);
//            break;

//          case IDF_FEET:
//            CheckDlgButton(hwnd,IDF_ASIS,  FALSE);
//            CheckDlgButton(hwnd,IDF_FEET,  TRUE );
//            CheckDlgButton(hwnd,IDF_METERS,FALSE);
//            break;

//          case IDF_METERS:
//            CheckDlgButton(hwnd,IDF_ASIS,  FALSE);
//            CheckDlgButton(hwnd,IDF_FEET,  FALSE);
//            CheckDlgButton(hwnd,IDF_METERS,TRUE );
//            break;

//          case IDF_UPDATE:
//            if (IsDlgButtonChecked(hwnd,IDF_UPDATE))
//               CheckDlgButton(hwnd,IDF_MANUAL, FALSE);
//            break;

//          case IDF_MANUAL:
//            if (IsDlgButtonChecked(hwnd,IDF_MANUAL))
//               CheckDlgButton(hwnd,IDF_UPDATE, FALSE);
//            break;

          case IDCANCEL:
            EndDialog(hwnd,FALSE);
            break;

          case IDHELP:
            WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Preferences Menu"));
            break;

          case IDF_MARK:
            SendDlgItemMessage(hwnd, IDF_MARKLEV, EM_SETREADONLY,
                  !IsDlgButtonChecked(hwnd,IDF_MARK), 0L);
            break;
        }
    }
    return FALSE;
}

void ResetPen(HPEN FAR *lphpen,COLORREF *clr, HWND hwnd, int dlgID) {
    char szBuffer[10];
    GetDlgItemText(hwnd,dlgID,szBuffer,sizeof szBuffer);
    if (*clr != atol(szBuffer)) new_params = TRUE;
    *clr=atol(szBuffer);
    DeletePen(*lphpen);
    *lphpen=CreatePen(PS_SOLID,0,*clr);
}

BOOL CALLBACK ColourDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
    char szBuffer[10];
    COLORREF clr;
    static COLORREF aclrCust[16];
    CHOOSECOLOR cc;
    int i;
    LPDRAWITEMSTRUCT lpdis;
    HBRUSH hbr;
    RECT rc;
    int ptSize;
    static LOGFONT glf, tlf;

    switch(msg) {
      case WM_INITDIALOG:
        for(i=0;i<16;i++)
            aclrCust[i] = RGB(255, 255, 255);

        memset(&glf, 0, sizeof(glf));
        glf.lfHeight = -MulDiv(gFontSize, GetDeviceCaps(GetDC(hwnd), LOGPIXELSY), 72);
        strcpy(glf.lfFaceName, gFontName);

        memset(&tlf, 0, sizeof(tlf));
        tlf.lfHeight = -MulDiv(tFontSize, GetDeviceCaps(GetDC(hwnd), LOGPIXELSY), 72);
        strcpy(tlf.lfFaceName, tFontName);

        wsprintf(szBuffer,"%lu",bgday_color);
        SetDlgItemText(hwnd,IDC_BGDAY,szBuffer);
        wsprintf(szBuffer,"%lu",bgnite_color);
        SetDlgItemText(hwnd,IDC_BGNITE,szBuffer);
        wsprintf(szBuffer,"%lu",fgtext_color);
        SetDlgItemText(hwnd,IDC_FGTEXT,szBuffer);
        wsprintf(szBuffer,"%lu",fgrise_color);
        SetDlgItemText(hwnd,IDC_FGRISE,szBuffer);
        wsprintf(szBuffer,"%lu",fgfall_color);
        SetDlgItemText(hwnd,IDC_FGFALL,szBuffer);
        wsprintf(szBuffer,"%lu",fgmiddle_color);
        SetDlgItemText(hwnd,IDC_FGMIDDLE,szBuffer);
        wsprintf(szBuffer,"%lu",fgmark_color);
        SetDlgItemText(hwnd,IDC_FGMARK,szBuffer);
        wsprintf(szBuffer,"%lu",fgmllw_color);
        SetDlgItemText(hwnd,IDC_FGMLLW,szBuffer);
        wsprintf(szBuffer,"%lu",fgmapdot_color);
        SetDlgItemText(hwnd,IDC_FGMDOT,szBuffer);
        PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, IDOK), -1);
        break;

      case WM_DRAWITEM:
        lpdis=(LPDRAWITEMSTRUCT)lParam;
        GetDlgItemText(hwnd,lpdis->CtlID,szBuffer,sizeof szBuffer);
        clr=atol(szBuffer);
        rc=lpdis->rcItem;
        hbr=CreateSolidBrush(clr);
        hbr=SelectBrush(lpdis->hDC,hbr);
        Rectangle(lpdis->hDC,rc.left,rc.top,rc.right,rc.bottom);
        DeleteBrush(SelectBrush(lpdis->hDC,hbr));
        if(lpdis->itemState&ODS_FOCUS) {
            InflateRect(&rc,-3,-3);
            DrawFocusRect(lpdis->hDC,&rc);
        }
        return TRUE;

//      case WM_KEYUP:
//           if (VK_F1==wParam)
//              WinHelp(hwnd,HelpFileName,HELP_CONTEXT,IDH_COLORS);
//           return FALSE;

      case WM_COMMAND:
        switch(wParam) {
          case IDOK:
            ResetPen(&bgday_color_gc,   &bgday_color,   hwnd,IDC_BGDAY);
            ResetPen(&bgnite_color_gc,  &bgnite_color,  hwnd,IDC_BGNITE);
            ResetPen(&fgtext_color_gc,  &fgtext_color,  hwnd,IDC_FGTEXT);
            ResetPen(&fgrise_color_gc,  &fgrise_color,  hwnd,IDC_FGRISE);
            ResetPen(&fgfall_color_gc,  &fgfall_color,  hwnd,IDC_FGFALL);
            ResetPen(&fgmiddle_color_gc,&fgmiddle_color,hwnd,IDC_FGMIDDLE);
            ResetPen(&fgmark_color_gc,  &fgmark_color,  hwnd,IDC_FGMARK);
            ResetPen(&fgmllw_color_gc,  &fgmllw_color,  hwnd,IDC_FGMLLW);
            ResetPen(&fgmapdot_color_gc,&fgmapdot_color,hwnd,IDC_FGMDOT);
            ptSize = ((float)glf.lfHeight*-72L / (float)GetDeviceCaps(GetDC(hwnd), LOGPIXELSY)+ 0.5);
            if (ptSize != gFontSize || strcmp(gFontName,glf.lfFaceName)) {
               strcpy(gFontName, glf.lfFaceName);
               gFontSize = ptSize;
               new_params = TRUE;
               DeleteObject(gfont);
               gfont = NULL;
            }
            ptSize = ((float)tlf.lfHeight*-72L / (float)GetDeviceCaps(GetDC(hwnd), LOGPIXELSY)+ 0.5);
            if (ptSize != tFontSize || strcmp(tFontName,tlf.lfFaceName)) {
               strcpy(tFontName, tlf.lfFaceName);
               tFontSize = ptSize;
               new_params = TRUE;
               DeleteObject(tfont);
               tfont = NULL;
               if (hwndText!=NULL)
                  printf("");
//                  MessageBox(hwnd, "Text window font change will not take effect\r\n"
//                                   "until the text window has been closed.",
//                                   "Text window font", MB_ICONINFORMATION);
            }
            EndDialog(hwnd,TRUE);
            break;

          case IDCANCEL:
            EndDialog(hwnd,FALSE);
            break;

          case IDHELP:
            WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Colors Menu"));
            break;

          case IDC_BGDAY:
          case IDC_BGNITE:
          case IDC_FGTEXT:
          case IDC_FGRISE:
          case IDC_FGFALL:
          case IDC_FGMIDDLE:
          case IDC_FGMARK:
          case IDC_FGMLLW:
          case IDC_FGMDOT:
            GetDlgItemText(hwnd,wParam,szBuffer,sizeof szBuffer);
            clr=atol(szBuffer);
            memset(&cc, 0, sizeof cc);
            /* Initialize the necessary CHOOSECOLOR members. */
            cc.lStructSize = sizeof cc;
            cc.hwndOwner = hwnd;
            cc.rgbResult = clr;
            cc.lpCustColors = aclrCust;
            cc.Flags = CC_RGBINIT;
            if(ChooseColor(&cc)) {
              wsprintf(szBuffer,"%lu",cc.rgbResult);
              SetDlgItemText(hwnd,wParam,szBuffer);
            }
            break;

          case IDC_GRAF_FONT:
          {
            CHOOSEFONT cf;

            memset(&cf, 0, sizeof(CHOOSEFONT));
            cf.lStructSize = sizeof(CHOOSEFONT);
            cf.hwndOwner = hwnd;
            cf.lpLogFont = &glf;
            cf.iPointSize = gFontSize;
            cf.lpszStyle =  gFontName;
            cf.Flags = CF_ANSIONLY | CF_SCREENFONTS | CF_INITTOLOGFONTSTRUCT;
            cf.rgbColors = fgtext_color;//RGB(0, 255, 255); /* light blue */
            cf.nFontType = SCREEN_FONTTYPE;

            ChooseFont(&cf);
          }
            break;

          case IDC_TEXT_FONT:
          {
            CHOOSEFONT cf;

            memset(&cf, 0, sizeof(CHOOSEFONT));
            cf.lStructSize = sizeof(CHOOSEFONT);
            cf.hwndOwner = hwnd;
            cf.lpLogFont = &tlf;
            cf.iPointSize = tFontSize;
            cf.lpszStyle =  tFontName;
            cf.Flags = CF_FIXEDPITCHONLY | CF_ANSIONLY | CF_SCREENFONTS | CF_INITTOLOGFONTSTRUCT;
            cf.nFontType = SCREEN_FONTTYPE;

            ChooseFont(&cf);
          }
            break;
        }
    }
    return FALSE;
}

char *make_time_offset(long sec) {
int hr, min;
static char str[20], sign;

   hr = abs(sec) / 3600;
   min= (abs(sec) / 60) % 60;
   if (sec < 0)
        sign = '-';
   else sign = '+';
   sprintf(str, "%c%d:%02d", sign, hr, min);
   return(str);
}

char *make_real_str(double n) {
static char str[20];

   sprintf(str, "%0.2lf", n);
   return(str);
}

/* Turn a time displacement of the form [-]HH:MM into the number of seconds. */
int verify_hhmm2sec(long *dst, char *hhmm) {
  int h, m;
  char s;
  if (sscanf (hhmm, "%d:%d", &h, &m) != 2)
    return(1);
  if (sscanf (hhmm, "%c", &s) != 1)
    return(1);
  if (h < 0 || s == '-')
    m = -m;
  *dst = h*HOURSECONDS + m*60;
  return(0);
}

BOOL CALLBACK CustomDlg(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
int i, tzidx, CustomUseRef;
static char t_custom_name[256], t_str[256], tzsave[256]; //itzname[256],
static double ti_hmult, ti_hlevel, ti_lmult, ti_llevel;
static double to_hmult, to_hlevel, to_lmult, to_llevel;
static long ti_htime, ti_ltime, to_htime, to_ltime, err;

    switch(msg) {
      case WM_INITDIALOG:
         if (have_user_offsets)
              strcpy(t_custom_name, custom_name);
         else strcpy(t_custom_name, "");
         ti_hmult = Ihlevelmult;
         ti_hlevel= Ihtleveloff;
//         convert_level_units(&ti_hlevel, 1);

         ti_lmult = Illevelmult;
         ti_llevel= Iltleveloff;
//         convert_level_units(&ti_llevel, 1);

         ti_htime = Ihttimeoff;
         ti_ltime = Ilttimeoff;

         SetDlgItemText(hwnd,IDS_REFERENCE, IDX_reference_name);
         SetDlgItemText(hwnd,IDS_NAME,    t_custom_name);
         SetDlgItemText(hwnd,IDS_HT_TIME, make_time_offset(ti_htime));
         SetDlgItemText(hwnd,IDS_HT_MULT, make_real_str(ti_hmult));
         SetDlgItemText(hwnd,IDS_HT_LEVEL,make_real_str(ti_hlevel));
         SetDlgItemText(hwnd,IDS_LT_TIME, make_time_offset(ti_ltime));
         SetDlgItemText(hwnd,IDS_LT_MULT, make_real_str(ti_lmult));
         SetDlgItemText(hwnd,IDS_LT_LEVEL,make_real_str(ti_llevel));
         if (iscurrent)
            SetDlgItemText(hwnd,IDS_OFF_TEXT,"Offset (kt)");
         else
            SetDlgItemText(hwnd,IDS_OFF_TEXT,"Offset (ft)");

        if (isalpha(tadjust_last[0])) {
           CustomUseRef = !strcmp(tadjust_last, (tzfile+1));
        }
        else
           CustomUseRef = !Itadjust;
        CheckDlgButton(hwnd,IDS_USEREF,CustomUseRef);
        CheckDlgButton(hwnd,IDS_USER, !CustomUseRef);

        LoadTZCombo(hwnd, IDS_TZNAME);
        SendDlgItemMessage(hwnd,IDS_TZNAME,CB_GETLBTEXT, TZComboIdx, (LPARAM)(LPSTR)&tzsave);

        PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, IDOK), -1);
        break;

//      case WM_KEYUP:
//           if (VK_F1==wParam)
//              WinHelp(hwnd,HelpFileName,HELP_CONTEXT,IDH_CUSTOM);
//           return FALSE;

      case WM_COMMAND:
        switch(LOWORD(wParam)) {
          case IDOK:
            err = 0;

            GetTZCombo(hwnd, IDS_TZNAME, IDS_USER);

            GetDlgItemText(hwnd,IDS_HT_TIME,t_str,sizeof(t_str));
            if (verify_hhmm2sec(&to_htime, t_str)) {
               SetDlgItemText(hwnd,IDS_HT_TIME, make_time_offset(ti_htime));
               if (!err) err = IDS_HT_TIME*10 + 1;
            }

            GetDlgItemText(hwnd,IDS_LT_TIME,t_str,sizeof(t_str));
            if (verify_hhmm2sec(&to_ltime, t_str)) {
               SetDlgItemText(hwnd,IDS_LT_TIME, make_time_offset(ti_ltime));
               if (!err) err = IDS_LT_TIME*10 + 1;
            }

            GetDlgItemText(hwnd,IDS_HT_MULT,t_str,sizeof(t_str));
            if ((sscanf(t_str, "%lf", &to_hmult) != 1) ||
                (to_hmult > 5.0) || (to_hmult < 0.2)) {
               SetDlgItemText(hwnd,IDS_HT_MULT, make_real_str(ti_hmult));
               if (!err) err = IDS_HT_MULT*10 + 2;
            }

            GetDlgItemText(hwnd,IDS_LT_MULT,t_str,sizeof(t_str));
            if ((sscanf(t_str, "%lf", &to_lmult) != 1) ||
                (to_lmult > 5.0) || (to_lmult < 0.2)) {
               SetDlgItemText(hwnd,IDS_LT_MULT, make_real_str(ti_lmult));
               if (!err) err = IDS_LT_MULT*10 + 2;
            }

            GetDlgItemText(hwnd,IDS_HT_LEVEL,t_str,sizeof(t_str));
            if ((sscanf(t_str, "%lf", &to_hlevel) != 1) ||
                (to_hlevel > 10.0) || (to_hlevel < -10.0)) {
               SetDlgItemText(hwnd,IDS_HT_LEVEL,make_real_str(ti_hlevel));
               if (!err) err = IDS_HT_LEVEL*10 + 3;
            }

            GetDlgItemText(hwnd,IDS_LT_LEVEL,t_str,sizeof(t_str));
            if ((sscanf(t_str, "%lf", &to_llevel) != 1) ||
                (to_llevel > 10.0) || (to_llevel < -10.0)) {
               SetDlgItemText(hwnd,IDS_LT_LEVEL,make_real_str(ti_llevel));
               if (!err) err = IDS_LT_LEVEL*10 + 3;
            }

//            if ((to_hmult != 1.0) && (to_hlevel != 0.0)) {
//               SetDlgItemText(hwnd,IDS_HT_MULT, make_real_str(ti_hmult));
//               SetDlgItemText(hwnd,IDS_HT_LEVEL,make_real_str(ti_hlevel));
//               if (!err) err = IDS_HT_MULT*10 + 5;
//            }

//            if ((to_lmult != 1.0) && (to_llevel != 0.0)) {
//               SetDlgItemText(hwnd,IDS_LT_MULT, make_real_str(ti_lmult));
//               SetDlgItemText(hwnd,IDS_LT_LEVEL,make_real_str(ti_llevel));
//               if (!err) err = IDS_LT_MULT*10 + 5;
//            }

            GetDlgItemText(hwnd,IDS_NAME,t_custom_name,sizeof(t_custom_name));
            if (!err && !strlen(t_custom_name) &&
                ((to_hmult != 1.0) || (to_hlevel != 0.0) ||
                 (to_lmult != 1.0) || (to_llevel != 0.0) ||
                 (to_htime != 0)   || (to_ltime  != 0)))
               err = IDS_NAME*10 + 4;

            if (err) {
char *msgs[]={
{"Time must be in the form +/-HH:MM"},
{"Multipliers must be in the range 0.2 to 5.0"},
{"Level offsets must be in the range -10.0 to 10.0"},
{"Custom stations should have a name"},
{"You cannot specify both multiplier and offset"}
   };
               MessageBox(hwnd, msgs[(err%10)-1], "Field Error", MB_ICONEXCLAMATION);
               PostMessage( hwnd,WM_NEXTDLGCTL, (WPARAM)GetDlgItem(hwnd, err/10), -1);
               return FALSE;
            }

            if ((to_hmult == ti_hmult) && (to_hlevel == ti_hlevel) &&
                (to_lmult == ti_lmult) && (to_llevel == ti_llevel) &&
                (to_htime == ti_htime) && (to_ltime  == ti_ltime ) &&
//                (!strcmp(tzsave, tadjust_last)) &&
                (( have_user_offsets && !strcmp(t_custom_name, custom_name) && !strcmp(tzsave, tadjust_last))||
                 (!have_user_offsets && !strcmp(t_custom_name, ""))) ) {
               MessageBox(hwnd, "No settings changed, ignoring",
                     "No custom changes", MB_ICONEXCLAMATION);
               EndDialog(hwnd,FALSE);
               return FALSE;
            }

            strcpy(custom_name, t_custom_name);
            Ihlevelmult = to_hmult;
//            convert_level_units(&to_hlevel, 0);
            Ihtleveloff = to_hlevel;

            Illevelmult = to_lmult;
//            convert_level_units(&to_llevel, 0);
            Iltleveloff = to_llevel;

            Ihttimeoff = to_htime;
            Ilttimeoff = to_ltime;

            have_new_custom = TRUE;
            new_params = TRUE;
            EndDialog(hwnd,TRUE);
            break;

          case IDS_USEREF:
            tzidx=SendDlgItemMessage(hwnd,IDS_TZNAME,CB_FINDSTRING,-1,(LPARAM)((LPSTR)(tzfile+1)));
            if (tzidx==CB_ERR) tzidx=0;
            SendDlgItemMessage(hwnd,IDS_TZNAME, CB_SETCURSEL, tzidx, 0L);
            CheckDlgButton(hwnd,IDS_USEREF,CustomUseRef);
            CheckDlgButton(hwnd,IDS_USER, !CustomUseRef);
            break;

          case IDS_TZNAME:
            if ((HIWORD(wParam)) == CBN_SELCHANGE) {
               CustomUseRef = FALSE;
               CheckDlgButton(hwnd,IDS_USEREF,CustomUseRef);
               CheckDlgButton(hwnd,IDS_USER, !CustomUseRef);
            }
            break;

          case IDCANCEL:
            EndDialog(hwnd,FALSE);
            break;

          case IDHELP:
            WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Custom Station Menu"));
            break;
        }
    }
    return FALSE;
}

void DrawDC(HDC hdc) {
HCURSOR hcurSave;
hcurSave = SetCursor(LoadCursor(NULL, IDC_WAIT));
    display = hdc;
    if (gfont == NULL) {
      LOGFONT lf;
      memset(&lf, 0, sizeof(lf));
      lf.lfHeight = -MulDiv(gFontSize, GetDeviceCaps(hdc, LOGPIXELSY), 72);
      strcpy(lf.lfFaceName,gFontName);
      gfont=CreateFontIndirect(&lf);
    }
    SelectFont(hdc,gfont);
    SetTextColor(hdc,fgtext_color);
    SetBkMode(hdc,TRANSPARENT);
    /* Draw_graph does not return. */
    faketime = gstart_time;
    if (graphmode) {
       /* See above. */
       if (!faketime) {
          faketime = time (NULL);
          if (now)
          faketime -= tstep * (winW>>1);
       }
       draw_graph ();
    }
    else {

    /* This clumsy calibration step is made necessary by the new capability
       of using arbitrary offsets in clock mode. */
       time_t this_time;
       if (testmode)
          this_time = faketime;
       else
          this_time = time(NULL);
       assert (this_time > DAYSECONDS*1.5);
       next_ht = this_time - DAYSECONDS*1.5;
       prev_ht_adj = next_ht_adj = 0;
       while (this_time >= prev_ht_adj + 70)
          update_high_tide ();
       if (hairy) set_hairy_water_level ();
       else       set_water_level ();
    }
    DoCrosshair(display);
    SetCursor(hcurSave);
}

LRESULT CALLBACK TidesWndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
     PAINTSTRUCT ps;
     static BOOL bSizeMove, print_graph = 0, ctl_key = 0;
     int skinny_save, graphsave, marksave;
     float marklevelsave;
     char szMessage[256];

     switch(msg) {
       case WM_CREATE:
           if (!have_index) {
              init_index_file(TRUE, window);    // Load full info database
              load_location_info( location, IDX_rec_num );
            }
            WinTideCheckMenus(hwnd);
            bSizeMove=FALSE;
            if(!graphmode)
                 SetTimer(hwnd,1,(UINT)60000,NULL);
            return FALSE;

       case WM_SIZE:
            if (!hold_main_window && save_windows) {
               new_params = TRUE;
               WinTideCheckMenus(hwnd);
            }
            winW=LOWORD(lParam);
            winH=HIWORD(lParam);
            InvalidateRect(hwnd,NULL,TRUE);
            GetWindowRect(hwnd,&WinDimMain);
            DoScrollBar(graphmode&&usescrollbar);
            return FALSE;

       case WM_MOVE:
            if (!hold_main_window && save_windows) {
               new_params = TRUE;
               WinTideCheckMenus(hwnd);
            }
            winX=LOWORD(lParam);
            winY=HIWORD(lParam);
            GetWindowRect(hwnd,&WinDimMain);
            return FALSE;

       case WM_CONTEXTMENU:
           TrackPopupMenu( GetSubMenu(GetMenu(window),0), TPM_RIGHTBUTTON |TPM_TOPALIGN |TPM_LEFTALIGN,
                 LOWORD(lParam), HIWORD(lParam), 0, hwnd, NULL);
           return FALSE;

       case WM_KEYDOWN:
         if (!gstart_time) {
           gstart_time = time (NULL);
           if (now)
              gstart_time -= tstep * (winW>>1);
         }

         switch ((int)wParam) {
           case VK_LEFT:
             if (graphmode || hairy) {
                UndrawCrosshair();
                CrosshairEnable = TRUE;
                if (ctl_key)
                   NextEventAdjusted( &CrosshairTime, -1 );
                else
                   CrosshairTime -= increment_step*60;
                if (CrosshairTime <= (window_left_time + HOURSECONDS)) {
                   if (graphmode) {
                      gstart_time = CrosshairTime - HOURSECONDS;
                      if (usescrollbar) {
                         notnowtime = TRUE;
                         DoScrollBar(FALSE);
                      }
                      InvalidateRect(hwnd,NULL,TRUE);
                      UpdateWindow(hwnd);
                   }
                   else {
                      CrosshairEnable = FALSE;
                      CrosshairTime = window_left_time;
                   }
                }
                else
                   DrawCrosshair(CrosshairTime, NULL);  /* Draw new crosshair */
             }
             return FALSE;

           case VK_RIGHT:
             if (graphmode || hairy) {
                UndrawCrosshair();
                CrosshairEnable = TRUE;
                if (ctl_key)
                   NextEventAdjusted( &CrosshairTime, 1 );
                else
                   CrosshairTime += increment_step*60;
                if (CrosshairTime >= (window_left_time +winW*tstep -HOURSECONDS)) {
                   if (graphmode) {
                      gstart_time = CrosshairTime - winW*tstep + HOURSECONDS;
                      if (usescrollbar) {
                         notnowtime = TRUE;
                         DoScrollBar(FALSE);
                      }
                      InvalidateRect(hwnd,NULL,TRUE);
                      UpdateWindow(hwnd);
                   }
                   else {
                      CrosshairEnable = FALSE;
                      CrosshairTime = window_left_time +winW*tstep;
                   }
                }
                else
                   DrawCrosshair(CrosshairTime, NULL);  /* Draw new crosshair */
             }
             return FALSE;

           case VK_CONTROL:
             ctl_key = TRUE;
             return FALSE;

           case VK_HOME:
             UndrawCrosshair(); /* Undraw current crosshair */
             CrosshairTime = gstart_time = time( NULL );
             if (now)
                gstart_time -= tstep * (winW>>1);
             if ((graphmode && notnowtime) || CrosshairEnable) {
                InvalidateRect(hwnd,NULL,TRUE);
                UpdateWindow(hwnd);
             }
             if (graphmode&&usescrollbar) {
                notnowtime = FALSE;
                DoScrollBar(FALSE);
             }
             return FALSE;

           case VK_ESCAPE:
             UndrawCrosshair(); /* Undraw current crosshair */
             CrosshairEnable = FALSE;
             return FALSE;
         }

       case WM_KEYUP:
         switch((int)wParam) {
           case VK_F1:
             WinHelp(hwnd,HelpFileName,HELP_KEY,(LPARAM)((LPSTR)"Main graphics screen"));
             return FALSE;

           case VK_CONTROL:
             ctl_key = FALSE;
             return FALSE;
          }
          return FALSE;

        case WM_TIMER:
          InvalidateRect(hwnd,NULL,TRUE);
          UpdateWindow(hwnd);
          return FALSE;

        case WM_PAINT:
          if(!bSizeMove) {
              BeginPaint(hwnd,&ps);
              DrawDC(ps.hdc);
              EndPaint(hwnd,&ps);
          }
          return FALSE;

        case WM_CLOSE:
          if (new_params) {
             if ((auto_save) ||
                (IDYES==MessageBox(hwnd,"Parameters have been changed.\r\n"
                                        "Save before exit?",
                  "Save before exiting",MB_ICONEXCLAMATION|MB_YESNO)))
                write_config_file(szConfigPath, TRUE);
          }
          if(!graphmode)
               KillTimer(hwnd,1);
          PostQuitMessage(0);
          return TRUE;

//        case WM_RBUTTONDOWN:
        case WM_LBUTTONDOWN:
        {
           time_t mousetime, moon_time;
           mousetime = window_left_time + (LOWORD(lParam)*tstep);
           UndrawCrosshair();
           UseCrosshairPopup = 0;
           CrosshairTime = mousetime;
           open_popup(lParam, astro_info(mousetime));
           if (CrosshairEnable) {
//             display = GetDC(window);
//             DrawCrosshair(display, CrosshairDrawn); /* Undraw crosshair */
//             DrawCrosshair(display, CrosshairTime);  /* Draw new crosshair */
//             ReleaseDC(display, window);
             InvalidateRect( window, NULL, FALSE );
             UpdateWindow(window);
           }
        }
        return FALSE;

//        case WM_RBUTTONUP:
        case WM_LBUTTONUP:
          close_popup();
          UndrawCrosshair();
          UseCrosshairPopup = 1;
          if (CrosshairEnable) {
             InvalidateRect( window, NULL, FALSE );
             UpdateWindow(window);
//             DrawCrosshair(CrosshairTime, NULL);
          }
          return FALSE;

        case WM_HSCROLL:
        {
          unsigned short int scpos, sccode;

          if (!gstart_time) {
            gstart_time = time (NULL);
            if (now)
               gstart_time -= tstep * (winW>>1);
          }

          notnowtime = TRUE;
          sccode = wParam;
          scpos = HIWORD(wParam);       // Microsoft is not consistent!
               if (sccode == SB_LINELEFT)  gstart_time -= HOURSECONDS;
          else if (sccode == SB_LINERIGHT) gstart_time += HOURSECONDS;
          else if (sccode == SB_PAGELEFT)  gstart_time -= tstep*winW;
          else if (sccode == SB_PAGERIGHT) gstart_time += tstep*winW;
          else if (sccode == SB_ENDSCROLL) {
            InvalidateRect(hwnd,NULL,TRUE);
            DoScrollBar(FALSE);
          }
          else if (sccode == SB_THUMBTRACK) {
            char sbuf[80];
            struct tm *tm;
            POINT FAR mpos;
            gstart_time = scroll_bar_2_time_t(scpos);
            tm = tzlocaltime( &gstart_time );
            strftime( sbuf, sizeof(sbuf), "%Y-%m-%d ", tm );
            GetCursorPos(&mpos);
            ScreenToClient(window,&mpos);
            open_popup(MAKELONG(mpos.x,mpos.y), sbuf);
          }
          else if (sccode == SB_THUMBPOSITION) close_popup();// printf("ThumbPosition %d\n",scpos);
        }
          return FALSE;

        case WM_COMMAND:
          switch(wParam) {
//          case ID_COPY:
//            if(OpenClipboard(hwnd)) {
//                HGLOBAL hData=GlobalAlloc(GMEM_MOVEABLE|GMEM_DDESHARE,
//                                            sizeof(METAFILEPICT));
//                METAFILEPICT FAR * lpmfp=(METAFILEPICT FAR *)GlobalLock(hData);
//                HDC hdc;
//
//                EmptyClipboard();
//                lpmfp->mm=MM_TEXT;
//                lpmfp->xExt=winW;
//                lpmfp->yExt=winH;
//                hdc=CreateMetaFile(NULL);
//                DrawDC(hdc);
//                lpmfp->hMF=CloseMetaFile(hdc);
//                GlobalUnlock(hData);
//                SetClipboardData(CF_METAFILEPICT,hData);
//                CloseClipboard();
//            }
//            else
//                MessageBox(hwnd,"Windows Clipboard is not available","WinTide",MB_ICONINFORMATION);
//            break;

            case ID_PRINT_GRAPH:
              print_graph = 1; // Fall through to print

            case ID_PRINT:
              {
                PRINTDLG pd;

                memset(&pd,0,sizeof pd);

                pd.lStructSize = sizeof pd;
                pd.hwndOwner = hwnd;
                pd.Flags = PD_RETURNDC|PD_NOPAGENUMS|PD_NOSELECTION;

                if(PrintDlg(&pd)) {
                  DOCINFO di;
                  SIZE sz, asz;
                  TEXTMETRIC tm;
                  int LineSpace;
                  int clnPage;
                  unsigned int cln;
                  int ipgMax;
                  int ipg;
                  int copies;
                  UINT ilnLast;
                  UINT ilnBase;
                  UINT iln;
                  HCURSOR hcurSave = SetCursor(LoadCursor(NULL, IDC_WAIT));

                  sz.cx=GetDeviceCaps(pd.hDC,HORZRES);
                  sz.cy=GetDeviceCaps(pd.hDC,VERTRES);

                  di.cbSize=sizeof di;
                  di.lpszDocName=location;
                  di.lpszOutput=NULL;
                  di.lpszDatatype=NULL;
                  di.fwType=0;

                  GetTextMetrics(pd.hDC, &tm);
                  LineSpace = tm.tmHeight + tm.tmExternalLeading;
                  if (print_graph) {
                     time_t tm;
                     char s[128];
                     if (TRUE) {
                        float WinAspect, VpAspect;
                        asz.cx = sz.cx;
                        asz.cy = sz.cy - LineSpace*2;
                        WinAspect = (float)winW / (float)winH;
                        VpAspect  = (float)sz.cx/ (float)sz.cy;
                        if (WinAspect < VpAspect)
                           asz.cx *= (WinAspect / VpAspect);
                        if (WinAspect > VpAspect)
                           asz.cy *= (VpAspect / WinAspect);
                     }
                     tm = time (NULL);
                     strftime(s, sizeof(s), "%m/%d/%y %I:%m%p  ", tzlocaltime( &tm ));
                     sprintf(s+strlen(s),"   %s",custom_name);
                     StartDoc(pd.hDC,&di);
                     for (copies=pd.nCopies; copies > 0; copies--) {
                        StartPage(pd.hDC);
                        SetMapMode(pd.hDC,MM_ANISOTROPIC);
                        SetWindowOrgEx(pd.hDC,0,0,NULL);
                        SetWindowExtEx(pd.hDC,winW,winH,NULL);
                        SetViewportOrgEx(pd.hDC,0,0,NULL);
                        SetViewportExtEx(pd.hDC,sz.cx,sz.cy,NULL);
                        TextOut(pd.hDC, 0, 0,s, strlen(s));
                        SetViewportOrgEx(pd.hDC,0,LineSpace*2,NULL);
                        SetViewportExtEx(pd.hDC,asz.cx,asz.cy,NULL);
                        DrawDC(pd.hDC);
                        EndPage(pd.hDC);
                     }
                  }

                  else {
                     HWND hwndCtl=GetDlgItem(hwndText,ID_TEXT);
                     clnPage = sz.cy / LineSpace - 1;
                     cln = (UINT)SendMessage(hwndCtl, EM_GETLINECOUNT, 0, 0L);
                     ipgMax = cln / clnPage + 1;

                     StartDoc(pd.hDC,&di);
                     for (copies=pd.nCopies; copies > 0; copies--) {
                        for (ipg = 0; ipg < ipgMax ; ipg++) {
                           ilnBase = ipg * clnPage;
                           if (ipg == ipgMax-1) // Special case last page
                              ilnLast = (cln-1) % clnPage;
                           else
                              ilnLast = clnPage - 1;
                           StartPage(pd.hDC);
                           for (iln=0; iln <= ilnLast; iln++) {
                              char szLine[128];
                              int  cchLine;
                              szLine[0] = sizeof(szLine)-1;          // Maximum buffer size
                              szLine[1] = 0;
                              cchLine = (int)SendMessage(hwndCtl,
                                                         EM_GETLINE,
                                                         ilnBase + iln,
                                                         (DWORD)(LPSTR)szLine);
                              TextOut(pd.hDC, 0, iln*LineSpace, (LPSTR)szLine, cchLine);
                           }
                           EndPage(pd.hDC);
                        }
                     }
                  }
                  EndDoc(pd.hDC);
                  if (pd.hDevMode != NULL)
                      GlobalFree(pd.hDevMode);
                  if (pd.hDevNames != NULL)
                      GlobalFree(pd.hDevNames);
                  DeleteDC(pd.hDC);
                  SetCursor(hcurSave);    // Remove the hourglass
                }
              }
              print_graph = 0;
              break;

            case ID_CLOCK:
            {
              int wasgraph = graphmode;
              if(graphmode)
                  SetTimer(hwnd,1,(UINT)60000,NULL);
              graphmode=FALSE;
              hairy=FALSE;
              if (wasgraph || hairy) {
                  new_params = TRUE;
                  DoScrollBar(wasgraph&&usescrollbar);
              }
            }
              WinTideCheckMenus(hwnd);
              break;

            case ID_HAIRY:
            {
              int wasgraph = graphmode;
              if(graphmode)
                  SetTimer(hwnd,1,(UINT)60000,NULL);
              hairy=TRUE;
              graphmode=FALSE;
              if (wasgraph || !hairy) {
                  new_params = TRUE;
                  DoScrollBar(wasgraph&&usescrollbar);
              }
            }
              WinTideCheckMenus(hwnd);
              break;

            case ID_GRAPH:
            {
              int wasgraph = graphmode;
              if(!graphmode)
                  KillTimer(hwnd,1);
              graphmode=TRUE;
              hairy=FALSE;
              if (!wasgraph || hairy) {
                  new_params = TRUE;
                  DoScrollBar(!wasgraph&&usescrollbar);
              }
            }
              WinTideCheckMenus(hwnd);
              break;

            case ID_CALENDAR:
              KillTimer(hwnd,1);
              text = calendar = 1;
              skinny_save = skinny;
              skinny = 2;
              if (gstart_time)
                   faketime = gstart_time;
              else faketime = time( NULL );
              graphsave = graphmode;
              graphmode = 0;
              if (iscurrent) {
                 marksave = mark;
                 marklevelsave = marklev;
                 mark = 1;
                 marklev = 0.0;
              }
              do_calendar();
              if (iscurrent) {
                 mark = marksave;
                 marklev = marklevelsave;
              }
              graphmode = graphsave;
              text = calendar = 0;
              skinny = skinny_save;
              break;

            case ID_TIDES:
              KillTimer(hwnd,1);
              text = num_days;
              skinny_save = skinny;
              skinny = 0;
              if (gstart_time)
                   faketime = gstart_time;
              else faketime = time( NULL );
              graphsave = graphmode;
              graphmode = 0;
              if (iscurrent) {
                 marksave = mark;
                 marklevelsave = marklev;
                 mark = 1;
                 marklev = 0.0;
              }
              list_tides();
              if (iscurrent) {
                 mark = marksave;
                 marklev = marklevelsave;
              }
              graphmode = graphsave;
              text = 0;
              skinny = skinny_save;
              break;

            case ID_INCREMENT:
              KillTimer(hwnd,1);
              if (gstart_time)
                   faketime = gstart_time;
              else faketime = time( NULL );
              graphsave = graphmode;
              graphmode = 0;
              if (iscurrent) {
                 marksave = mark;
                 marklevelsave = marklev;
                 mark = 1;
                 marklev = 0.0;
              }
              do_incremental();
              if (iscurrent) {
                 mark = marksave;
                 marklev = marklevelsave;
              }
              graphmode = graphsave;
              break;

            case ID_LOCATION:
              KillTimer(hwnd,1);
              if (DialogBox(g_hinst,"EnterLocation",hwnd,LocationDlg)) {
                  if (now)
                        faketime -= (long)tstep*(winW>>1);
                  else  faketime  = time(NULL);
                  next_ht = time(NULL);
                  update_high_tide ();
                  update_high_tide();
                  NameWindow(hwnd, IDX_station_name);
                  new_params = TRUE;
              }
              WinTideCheckMenus(hwnd);
              if (hairy || !graphmode)
                  SetTimer(hwnd,1,(UINT)60000,NULL);
              break;

            case ID_FLAGS:
              if(!DialogBox(g_hinst,"Flags",hwnd,FlagDlg))
                return FALSE;
              WinTideCheckMenus(hwnd);
              break;

            case ID_COLOURS:
              if(!DialogBox(g_hinst,"ColourDlg",hwnd,ColourDlg))
                return FALSE;
              WinTideCheckMenus(hwnd);
              break;

//            case ID_SWITCH:
//              list_switches();
//              return FALSE;
//              break;

            case ID_MRU1:
            case ID_MRU2:
            case ID_MRU3:
            case ID_MRU4:
            case ID_MRU5:
            case ID_MRU6:
            case ID_MRU7:
            case ID_MRU8:
            case ID_MRU9:
              load_mru(wParam - ID_MRU0);
              WinTideCheckMenus(hwnd);
              new_params = TRUE;
              break;

            case ID_CUSTOM:
              if (!DialogBox(g_hinst,"Custom",hwnd,CustomDlg))
                 return FALSE; // Cancelled out on us

              WinTideCheckMenus(hwnd);
              strcpy(location, IDX_reference_name);
              load_location_data( location, 0 );
              NameWindow(hwnd, custom_name);
//              MessageBox(hwnd,
//                 "Settings entered.  I will now start \"Save Configuration\"",
//                 "Custom settings ready", MB_ICONINFORMATION);
              // Then fall through to start the save.

            case ID_SAVE_AS:
              init_OPENFILENAME(OFN_OVERWRITEPROMPT,
                  "Save WXTide32 Configuration file");
              if (!GetSaveFileName(&ofn))
                 return FALSE;
              have_new_custom = FALSE;
              write_config_file(ofn.lpstrFile, FALSE);
              strcpy(szConfigPath, ofn.lpstrFile);
              WinTideCheckMenus(hwnd);
              break;

            case ID_SAVE:
              write_config_file(szConfigPath, TRUE);
              WinTideCheckMenus(hwnd);
              break;

            case ID_LOAD:
              if (new_params) {
                 if (IDYES != MessageBox(hwnd,
                    "Some options have been changed and not saved.\r\n"
                    "Load anyway and ignore changes?",
                    "Options changed", MB_ICONEXCLAMATION|MB_YESNO))
                    return FALSE;
              }
              new_params = FALSE;
              init_OPENFILENAME(OFN_FILEMUSTEXIST,
                  "Load WXTide32 Configuration file");
              if (!GetOpenFileName(&ofn))
                 return FALSE; // No file selected
              sprintf(szMessage,"\"%s\"",ofn.lpstrFile);
              ShellExecute(NULL, "open", szProgramName, szMessage,
                 NULL, SW_SHOWNORMAL);  // Start me again with name of file as parameter string
// Fall through to exit this instance

            case ID_EXIT:
              PostMessage(hwnd,WM_CLOSE,0,0L);
              return FALSE;
              break;

            case ID_WEB:
              ShellExecute(NULL, "open", "http://wxtide32.com", NULL,
                 NULL, SW_SHOWNORMAL);  // Start default web browser at base web site
              return FALSE;
              break;

            case ID_ABOUT:
               sprintf(szMessage,"WXTide32 version %s, %s\r\n"
                                 "Copyright  1998-2002 Michael Hopper\r\n"
                                 "\r\n"
                                 "Web site: http://wxtide32.com\r\n"
                                 "Email:       mike@wxtide32.com",
                     WXTIDE32VERSIONNUMBER, WXTIDE32VERSIONDATE);
               MessageBox(hwnd,szMessage,"About WXTide32",MB_OK|MB_ICONINFORMATION);
               return FALSE;

            case ID_HELP:
              WinHelp(hwnd,HelpFileName,HELP_FINDER,0);
              return FALSE;

            case ID_NOW:
              notnowtime = FALSE;
              CrosshairTime = gstart_time = time (NULL);
              if (now)
                 gstart_time -= tstep * (winW>>1);
              InvalidateRect(hwnd,NULL,TRUE);
              DoScrollBar(FALSE);
              return FALSE;
          }
          InvalidateRect(hwnd,NULL,TRUE);
          UpdateWindow(hwnd);
          return FALSE;
     }
     return DefWindowProc(hwnd,msg,wParam,lParam);
}


int win_fprintf(FILE *stream,const char *fstring, ...)
{
     static int nState;
     static char szCaption[256];
     static char szMessage[4096];
     char szOutput[1024];
     int nReturn;
     va_list marker;

     va_start(marker,fstring);
     if(stderr==stream) {
       nReturn=vsprintf(szOutput,fstring,marker);
       switch(nState) {
         case 0:
           if(0==lstrcmp("Tidelib Fatal Error:  ",szOutput)) {
             lstrcpy(szCaption,szOutput);
             nState++;
           }
           else {
             nLenAccum+=strlen(szOutput+2);
             if(szAccum)
               szAccum=realloc(szAccum,nLenAccum);
             else {
               szAccum=malloc(nLenAccum);
               szAccum[0]='\0';
             }
             lstrcat(szAccum,szOutput);
           }
           break;

         case 1:
           lstrcpy(szMessage,szOutput);
           nState++;
           break;

         case 2:
           lstrcat(szMessage,szOutput);
           MessageBox(NULL,szMessage,szCaption,MB_ICONEXCLAMATION);
           PostQuitMessage(0);
           nState=0;
           break;

         default:
           //Oh shit!
           nState=0;
           break;
       }
     }
     else
       nReturn=vfprintf(stream,fstring,marker);
     return nReturn;
}

MSG t_msg;
static int hold_text_window = 1;
int printf(const char *fstring, ...)
{
     char szOutput[4096], szTemp[4096], *pch;
     int nReturn, c;
     va_list marker;
     HWND hwndCtl;

     va_start(marker,fstring);
     nReturn=vsprintf(szTemp,fstring,marker);
     if (output_filename) {
        if (!textLFN) {
           if (!nowarn) {
              sprintf(szOutput, "Redirecting all text output to %s\r\n"
                  "The -nowarn switch disables this message\r\n"
                  "Press OK to continue.", output_filename);
              MessageBox(NULL, szOutput, "Text redirection",
              MB_TASKMODAL | MB_ICONINFORMATION);
           }
           textLFN=fopen( output_filename, "wt" );
        }
        fwrite( szTemp, strlen(szTemp), 1, textLFN );
        return(0);
     }
     pch = szOutput;
     for (c=0; c<nReturn; c++) // Parse the string to change \n to \r\n for windows
        if ((szTemp[c] == '\n') && ((c==0) || ((c>0) && (szTemp[c-1] != '\r')))) {
            *pch++ = '\r';
            *pch++ = '\n';
        }
        else
            *pch++ = szTemp[c];
     *pch = '\0';

     if(NULL==hwndText) {
        int cxscreen, cyscreen;
        hwndText=CreateDialog(t_hinst,"TextTides",NULL,TextTidesDlgProc);
        cxscreen = GetSystemMetrics(SM_CXSCREEN);
        cyscreen = GetSystemMetrics(SM_CYSCREEN);
        if ((WinDimText.left   >= 0) &&
            (WinDimText.right  < cxscreen) &&
            (WinDimText.top    >= 0) &&
            (WinDimText.bottom < cyscreen) &&
            (WinDimText.right > WinDimText.left) &&
            (WinDimText.bottom > WinDimText.top) &&
            (WinDimText.left|WinDimText.right|WinDimText.top|WinDimText.bottom) )
           MoveWindow(hwndText,WinDimText.left,WinDimText.top,
               WinDimText.right-WinDimText.left,WinDimText.bottom-WinDimText.top,TRUE);
        ShowWindow(hwndText,SW_SHOW);
        hold_text_window = 0;
     }

     if (tfont == NULL) {
       LOGFONT lf;
       memset(&lf, 0, sizeof(lf));
       lf.lfHeight = -MulDiv(tFontSize, GetDeviceCaps(GetDC(hwndText), LOGPIXELSY), 72);
       strcpy(lf.lfFaceName,tFontName);
       tfont=CreateFontIndirect(&lf);
       SendDlgItemMessage(hwndText,ID_TEXT,WM_SETFONT,(WPARAM)tfont, MAKELONG(TRUE,0));
       SendDlgItemMessage(hwndText,ID_TEXT,EM_SCROLLCARET,0,0);
     }

     hwndCtl=GetDlgItem(hwndText,ID_TEXT);
     SendMessage(hwndCtl,EM_SETSEL,(WPARAM)-1,(LPARAM)-1);
     SendMessage(hwndCtl,EM_REPLACESEL,0,(LPARAM)(LPSTR)szOutput);
     if (window) SetFocus(window);
     return nReturn;
}

BOOL CALLBACK TextTidesDlgProc(HWND hwndDlg,UINT t_msg,WPARAM wParam,
                                        LPARAM lParam)
{
     RECT rc;
     HWND hwndCtl;
     SIZE sz;
     HMENU smenu, tmenu, hmenu=GetMenu(window);

     switch(t_msg) {
       case WM_INITDIALOG:
         SendDlgItemMessage(hwndDlg,ID_TEXT,WM_SETFONT,
                           (WPARAM)GetStockFont(ANSI_FIXED_FONT),0L);
//         SendDlgItemMessage(hwndDlg,ID_TEXT,EM_SETLIMITTEXT, 65535, 0L);
         WinTideCheckMenus(window);
         if(hmenu)
            EnableMenuItem(hmenu, ID_PRINT, MF_BYCOMMAND|MF_ENABLED);
//         if (window) SetFocus(window);
         smenu = GetSystemMenu(hwndDlg,FALSE);
         AppendMenu(smenu, MF_SEPARATOR, 0, (LPSTR) NULL);
         AppendMenu(smenu, MF_STRING, ID_PRINT, "Print...");
         return FALSE;

       case WM_CONTEXTMENU:
         tmenu = CreateMenu();//GetSystemMenu(hwndDlg,FALSE);
         AppendMenu(tmenu, MF_SEPARATOR, 0, (LPSTR) NULL);
         AppendMenu(tmenu, MF_STRING, ID_PRINT, "Print...");
         TrackPopupMenu( tmenu, TPM_RIGHTBUTTON |TPM_TOPALIGN |TPM_LEFTALIGN,
               LOWORD(lParam), HIWORD(lParam), 0, hwndText, NULL);
         return(FALSE);

       case WM_SIZE:
         hwndCtl=GetDlgItem(hwndDlg,ID_TEXT);
         GetWindowRect(hwndCtl,&rc);
         ScreenToClient(hwndDlg,(LPPOINT)&rc);
         sz.cx=LOWORD(lParam)-rc.left*2;
         sz.cy=HIWORD(lParam)-rc.top*2;
         MoveWindow(hwndCtl,rc.left,rc.top,sz.cx,sz.cy,TRUE);
         if (save_windows && !hold_text_window) {
            new_params = TRUE;
            WinTideCheckMenus(window);
         }
//         if (window) SetFocus(window);
         GetWindowRect(hwndText,&WinDimText);
         return TRUE;

       case WM_MOVE:
         if (save_windows && !hold_text_window) {
            new_params = TRUE;
            WinTideCheckMenus(window);
         }
         GetWindowRect(hwndText,&WinDimText);
         break;

       case WM_CLOSE:
         if(hmenu)
            EnableMenuItem(hmenu, ID_PRINT, MF_BYCOMMAND|MF_GRAYED);
         DestroyWindow(hwndDlg);
         hwndText = NULL;//mgh invalidate the window handle
         WinTideCheckMenus(window);
         if(done)
           PostQuitMessage(0);// Only do this if Started in text mode
         return TRUE;

     }
     return FALSE;
}