#include "block.h"
#include <string.h>
#ifdef WINDOWS
#include <windows.h>
#endif
#include <stdio.h>
#include "utils.hh"
#include "mtx.hh"

class BlockListEntry {
 public:
   enum bl_t { blFull = 0, blHost = 1, blDir = 2, blPath = 3, blAnyPath = 4 };
   bl_t type;
   char *host;
   int port;
   char *path;
   char *body;
   bool bodyblock;
   int count;
   bool match2(const char *_host, int _port, const char *_path);
   BlockListEntry(bl_t _type, const char *_host, int _port, const char *_path, const char *_body, bool _bodyblock, int _count);
   ~BlockListEntry();
   bool match(int _count, const char *_host, int _port, const char *_path, bool &is_bodyblock, const char * (&ret_body));
};

BlockListEntry::BlockListEntry(bl_t _type, const char *_host, int _port,
			       const char *_path, const char *_body, bool _bodyblock, int _count)
: type(_type), host(mystrdup(_host)), port(_port), path(mystrdup(_path)), body(mystrdup(_body)), bodyblock(_bodyblock), count(_count)
{
}

BlockListEntry::~BlockListEntry()
{
   delete host;
   delete path;
   delete body;
}

bool BlockListEntry::match2(const char *_host, int _port, const char *_path)
{
   if (port && port != _port)
     return false;
   if (!_host)
     _host = "";
   if (!_path)
     _path = "";
   switch (type) {
    case blFull:
      return (!strcmp(host, _host)) && (!strcmp(path, _path));
    case blHost:
//      printf("host: .%s. _host: .%s.\n", host, _host);
      // compare tails
	{
	   int hostlen = strlen(host), _hostlen = strlen(_host);
	   if (hostlen > _hostlen)
	     return false;
	   return !strncmp(host, _host+(_hostlen-hostlen), hostlen);
	}
    case blDir:
      return !strcmp(host, _host) && !strncmp(path, _path, strlen(path));
    case blPath:
      return !strncmp(path, _path, strlen(path));
    case blAnyPath:
     return !strstr(path, _path);
   }
   return false;
}

bool BlockListEntry::match(int _count, const char *_host, int _port, const char *_path, bool &is_bodyblock, const char * (&ret_body))
{
   bool ret = match2(_host, _port, _path);
   if (ret) {
     ret_body = body;
     if ((is_bodyblock || !bodyblock) && _count )
       count += _count;
     is_bodyblock = bodyblock;
   }
   return ret;
}

BlockList::BlockList()
: last(0), disabled(false), total(0)
{
   mtx = new Mtx;
   memset(bl, 0, sizeof(bl));
}

BlockList::~BlockList()
{
   Free();
   delete mtx;
}

void BlockList::Free()
{
   int i = last;
   for (BlockListEntry **be = &bl[last-1]; i; i--, be--) {
      delete *be;
   }
   last = 0;
   total = 0;
}

int BlockList::is_blocked(int count, const char *host, int port, const char *path, bool &is_bodyblock, const char * (&ret_body))
{
   if (disabled)
     return -1;
   mtx->Lock();
   int i = last;
   bool bb = is_bodyblock;
   for (BlockListEntry **be = &bl[last-1]; i; i--, be--)
      if ((*be)->match(count, host, port, path, is_bodyblock, ret_body)) {
//	printf("match '%s' '%s' '%s' '%s'\n", host, path, (*be)->host, (*be)->path);
	 if (count && (bb || !is_bodyblock))
	   total += count;
	 mtx->Unlock();
	 return i-1;
      }
   mtx->Unlock();
   return -1;
}

int BlockList::add_e(BlockListEntry *be)
{
   if (last >= MaxEntries) {
      delete be;
      return -1;
   }
   int idx;
   bool bb = false;
   const char *ret_body;
   if ((idx = is_blocked(0, be->host, be->port, be->path, bb, ret_body)) >= 0) {
      delete be;
      return idx;
   }
   bl[last] = be;
   return last++;
}

int BlockList::add_full(const char *host, int port, const char *path, const char *body, bool bodyblock)
{
   return add_e(new BlockListEntry(BlockListEntry::blFull, host, port, path, body, bodyblock, 0));
}

int BlockList::add_host(const char *host, int port, const char *body, bool bodyblock)
{
   return add_e(new BlockListEntry(BlockListEntry::blHost, host, port, 0, body, bodyblock, 0));
}

int BlockList::add_dir(const char *host, int port, const char *dir, const char *body, bool bodyblock)
{
   return add_e(new BlockListEntry(BlockListEntry::blDir, host, port, dir, body, bodyblock, 0));
}

int BlockList::add_path(const char *path, const char *body, bool bodyblock)
{
   return add_e(new BlockListEntry(BlockListEntry::blPath, 0, 0, path, body, bodyblock, 0));
}

int BlockList::add_anypath(const char *path, const char *body, bool bodyblock)
{
   return add_e(new BlockListEntry(BlockListEntry::blAnyPath, 0, 0, path, body, bodyblock, 0));
}

void BlockList::remove_entry(int indx)
{
   mtx->Lock();
   if (indx < 0 || indx >= last) {
      mtx->Unlock();
      return;
   }
   last--;
   for (int i = indx; i < last; i++)
      bl[i] = bl[i+1];
   bl[last] = 0;
   mtx->Unlock();
}

bool BlockList::info_entry(int indx, const char * (&host), int &port, const char * (&path), int &accessed, const char * (&body), bool &bodyblock)
{
   if (indx < 0 || indx >= last)
     return false;
   BlockListEntry *be = bl[indx];
   host = be->host;
   port = be->port;
   path = be->path;
   accessed = be->count;
   body = be->body;
   bodyblock = be->bodyblock;
   return true;
}

void BlockList::Load(const char *filename)
{
   FILE *f = fopen(filename, "r");
   if (!f)
     return;
   int type, port, count;
   const int buf_size = 2000;
   char *buf = new char[buf_size+1];
   char *host = new char[201], *path = new char[201], *body = new char[201];
   int bodyblock;
   host[200] = 0;
   path[200] = 0;
   body[200] = 0;
   int r;
   while (!feof(f)) {
      if (!fgets(buf, buf_size, f))
        break;
      buf[buf_size] = 0;
//      printf("buf: %s\n", buf);
      *body = 0;
      bodyblock = 0;
      r = sscanf(buf, "%d %d %200s %d %200s %d %200s\n", &count, &type, host, &port, path, &bodyblock, body);
      if (r != 7) {
	r = sscanf(buf, "%d %d %200s %d %200s\n", &count, &type, host, &port, path);
	if (r == 5) {
	  *body = 0;
	  bodyblock = 0;
	  r = 7;
	}
      }
      if (r == 7 && type >= BlockListEntry::blFull && type <= BlockListEntry::blAnyPath) {
//	 printf("load: %i \"%s\" %i \"%s\"\n", type, host, port, path);
	 if (*host == '-' && !*(host+1))
	   *host = 0;
	 if (*path == '-' && !*(path+1))
	   *path = 0;
	 if (*body == '-' && !*(body+1))
	   *body = 0;
	 add_e(new BlockListEntry((BlockListEntry::bl_t) type, host, port, path, body, bodyblock, count)); /* !!! FIXME !!! */
	 total += count;
      } else {
	printf("r is %d, count: %d, type: %d, host: %s, port: %d, path: %s\n", r, type, count, host, port, path);
	break;	
      }
   }
   fclose(f);
   delete host;
   delete path;
   delete body;
   delete buf;
}

void BlockList::Save(const char *filename)
{
   FILE *f = fopen(filename, "w");
   if (!f)
     return;
   int i = 0;
   for (BlockListEntry **be = &bl[0]; i < last; i++, be++) {
      BlockListEntry &be2 = **be;
      char *host2 = be2.host, *path2 = be2.path, *body2 = be2.body;
      if (!*host2)
	host2 = "-";
      if (!*path2)
	path2 = "-";
      if (!*body2)
        body2 = "-";
      fprintf(f, "%d %d %s %d %s %d %s\n", be2.count, be2.type, host2, be2.port, path2, be2.bodyblock, body2);
   }
   fclose(f);
}

BlockList blist;
