/* --------------------------------- ils.c ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* paint the Head Up Display: ILS
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"
#include "hud.h"

#undef CP
#undef CW
#undef CS

#define CP	view->viewport
#define CW	view->window
#define CS	view->screen

#define XFULL	(D90/36)		/* 2.5 degrees */
#define YFULL	(D90/120)		/* 0.75 degrees */

extern void FAR
show_ils (OBJECT *p, int sx, int sy, int orgx, int orgy, int ss, int shifty)
{
	int	x, y, d, xref, yref, t, i, dd;
	long	tx, ty, tz;
	ANGLE	a;
	struct ils	*b;

	orgy += shifty;

	b = &ils[EE(p)->ils-1];

/* show navigation data
*/
	dd = num_size (9L, ss);
	x = orgx + sx - dd*9;
	y = orgy + sy - 2;

	t = ANG2DEG(EE(p)->misc[15]);
	if (t < 0)
		t += 360;
	stroke_frac (x, y, t , 3, 0, ss, st.hfg);
	stroke_frac (x+dd*4, y, EE(p)->misc[16], 0, 1, ss, st.hfg);

	y -= ss;
	stroke_str (x, y, b->name, ss, st.hfg);

/* Are we in range?
*/
	tx = (p->R[X] - b->R[X])/VONE;
	ty = (p->R[Y] - b->R[Y])/VONE;
	tz = (p->R[Z] - b->R[Z])/VONE;
	if (labs(tx) > 25000L || labs(ty) > 25000L || labs(tz) > 25000L)
		return;				/* too far out */

	xref = fmul (sx, FONE/8*7);
	yref = fmul (sy, FONE/8*7);

/* localizer deviation.
*/
	a = ATAN ((int)tx-b->l[X], (int)ty-b->l[Y]) - b->localizer;
	if (a > XFULL)
		x = xref;
	else if (a < -XFULL)
		x = -xref;
	else
		x = muldiv (xref, a, XFULL);
	x = orgx + x;
	add_line (x, orgy-yref, T_MOVE);
	add_line (x, orgy+yref, st.hfg);
	t = sx / 64;
	for (i = 1; i < 5; ++i) {
		y = muldiv (yref, i, 5);
		add_line (x-t, orgy+y, T_MOVE);
		add_line (x+t, orgy+y, st.hfg);
		add_line (x-t, orgy-y, T_MOVE);
		add_line (x+t, orgy-y, st.hfg);
	}
	add_line (x-3*t, orgy, T_MOVE);
	add_line (x+3*t, orgy, st.hfg);

/* glide slope deviation.
*/
	d = ihypot2d ((int)tx-b->g[X], (int)ty-b->g[Y]); /* dist. to beacon */
	a = ATAN ((int)tz, d) - b->glidepath;
	a -= EE(p)->misc[14];
	if (iabs((int)a) > 4)
		a = EE(p)->misc[14] + a/4;

	if (a > YFULL)
		y = yref;
	else if (a < -YFULL)
		y = -yref;
	else
		y = muldiv (yref, a, YFULL);
	y = orgy + y;
	add_line (orgx-xref, y, T_MOVE);
	add_line (orgx+xref, y, st.hfg);
	t = sy / 64;
	for (i = 1; i < 5; ++i) {
		x = muldiv (xref, i, 5);
		add_line (orgx+x, y-t, T_MOVE);
		add_line (orgx+x, y+t, st.hfg);
		add_line (orgx-x, y-t, T_MOVE);
		add_line (orgx-x, y+t, st.hfg);
	}
	add_line (orgx, y-3*t, T_MOVE);
	add_line (orgx, y+3*t, st.hfg);
}

struct ils ils[] = {
  {"A18",{    0L*VONE,     0L*VONE, 0L},{   0, -500},{   0, 1500},   0, D90/30},
  {"A27",{    0L*VONE,     0L*VONE, 0L},{-500,    0},{1500,    0}, D90, D90/30},
  {"B18",{10000L*VONE, 10000L*VONE, 0L},{   0, -500},{   0, 1500},   0, D90/30},
  {"B27",{10000L*VONE, 10000L*VONE, 0L},{-500,    0},{1500,    0}, D90, D90/30},

  {    0,{    0L*VONE,     0L*VONE, 0L},{   0,    0},{   0,    0},   0,      0}
};

extern int FAR
menu_ils (void)
{
	MENU	*MenuILS;
	int	sel, i, n, nEntries, EntrySize;

	for (nEntries = 0; ils[nEntries].name; ++nEntries)
		;
	nEntries += 2;		/* "off" and "home" */
	EntrySize = 20;

	n = (nEntries+1) * sizeof (*MenuILS);
	if (!(MenuILS = (MENU *) memory_alloc (n)))
		return (1);

	sel = MENU_FAILED;
	for (i = 0; i < nEntries; ++i)
		if (!(MenuILS[i].text = (char *) memory_alloc (EntrySize)))
			goto ret;

	MenuILS[0].letter = '0';
	strcpy (MenuILS[0].text, "Off");
	for (i = 1; i < nEntries-1; ++i) {
		MenuILS[i].letter = i+'0';
		strcpy (MenuILS[i].text, ils[i-1].name);
	}
	MenuILS[i].letter = '=';
	strcpy (MenuILS[i].text, "Home");

	sel = EE(CV)->ils;
	sel = menu_open (MenuILS, sel);

	switch (sel) {
	case MENU_ABORTED:
	case MENU_FAILED:
		break;
	case 0:
		EE(CV)->hud2 &= ~HUD_ILS;
		break;
	default:
		if (sel == nEntries-1)
			sel = 1;
		if (EE(CV)->ils == sel) {
			EE(CV)->hud2 ^= HUD_ILS;
			break;
		}
		EE(CV)->hud2 |= HUD_ILS;
		EE(CV)->ils = sel;
		EE(CV)->misc[14] = 0;
		break;
	}

ret:
	for (i = 0; i < nEntries; ++i)
		if (MenuILS[i].text)
			memory_free (MenuILS[i].text, EntrySize);

	memory_free (MenuILS, n);

	if (MENU_FAILED == sel)
		return (1);

	menu_close ();
	return (0);
}
