/* --------------------------------- info.c --------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Show text on the screen
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>

#include "fly.h"
#include "hud.h"

#undef CP
#undef CW
#undef CS

#define CP	view->viewport
#define CW	view->window
#define CS	view->screen

static int	x0 = 0, x1 = 0, xs = 0, xm = 0, y0 = 0, y1 = 0, x = 0, y = 0,
		xd = 0, xw = 0, dx = 0, dy = 0;

extern Uchar *FAR
show_l (long n)
{
	int		m, k, u;
	char		*s;
	static Uchar	line[20];

	if (n < 0) {
		n = -n;
		s = "-";
	} else
		s = "";
	u = (int)(n % 1000);		n /= 1000;
	k = (int)(n % 1000);		n /= 1000;
	m = (int)(n % 1000);		n /= 1000;
	if (n)
		sprintf (line, "%s%u,%03u,%03u,%03u",
				s, (int)n, m, k, u);
	else if (m)
		sprintf (line, "%s%u,%03u,%03u",
				s, m, k, u);
	else if (k)
		sprintf (line, "%s%u,%03u",
				s, k, u);
	else
		sprintf (line, "%s%u",
				s, u);
	return (line);
}

extern Uchar *FAR
show_time (char *title, Ulong tt)
{
	int		tn, ss, mm, hh;
	Ulong		t;
	static Uchar	line[40];

	t = tt / 100;
	tn = (int)(t % 10);		t /= 10;
	ss = (int)(t % 60);		t /= 60;
	mm = (int)(t % 60);		t /= 60;
	hh = (int)(t % 24);		t /= 24;
	if (t)
		sprintf (line, "%s%u+%02u:%02u:%02u.%01u",
				title, (int)t, hh, mm, ss, tn);
	else if (hh)
		sprintf (line, "%s%02u:%02u:%02u.%01u",
				title, hh, mm, ss, tn);
	else
		sprintf (line, "%s%02u:%02u.%01u",
				title, mm, ss, tn);
	return (line);
}

static void FAR
ScrNewline (void)
{
	y += dy;
	x = x0;
	if (y > y1) {
		x0 += xs;
		if (xs > 0 && x0 > x1)
			x0 = 2;
		if (xs < 0 && x0 < 0)
			x0 = x1;
		xm = x0 + xw;
		x = x0;
		y = y0 + dy;
	}
}

static int FAR
ScrPrintf (const char *fmt, ...)
{
	va_list	ap;
	int	i, c;
	char	*p;
	static char _Gbuf[256];

	va_start (ap, fmt);
	i = vsprintf (_Gbuf, fmt, ap);
	va_end (ap);

	for (p = _Gbuf; (c = *p); ++p) {
		switch (c) {
		default:
			if (x+xd > xm)
				ScrNewline ();
			x += stroke_char (x, y, c, dx, st.green);
			break;
		case '\n':
			ScrNewline ();
			break;
		case '\r':
			x = x0;
			break;
		case '\f':
			y = y0 + dy;
			x = x0;
			break;
		}
	}
	return (i);
}

#define	NOVALUE	0x8000

static void FAR
ScrOpt (char *title, int state)
{
	if (state)
		ScrPrintf ("%s\n", title);
}

static void FAR
ScrOnOff (char *title, int state, int value)
{
	ScrPrintf ("%-7s %s", title, state ? "On" : "Off");
	if (value != NOVALUE)
		ScrPrintf (" %d\n", value);
	else
		ScrPrintf ("\n");
}

static void FAR
ScrValue (char *title, int value)
{
	ScrPrintf ("%-7s %d\n", title, value);
}

static int FAR
show_ang (char *line, ANGLE a)
{
	int	i, s, f;

	if (a < 0) {
		s = -1;
		a = -a;
	} else
		s = 1;

	i = ANG2DEG (a);
	a -= DEG2ANG (i);
	f = ANG2DEG (10*a);
	sprintf (line, "%4d.%u", s*i, f);
	return (strlen (line));
}

static void FAR
right_margin (VIEW *view, int orgx, int orgy, int maxx, int maxy, int bss)
{
	dx = bss;
	dy = bss;

	xs = -19*xd;
	x0 = orgx + maxx + xs;
	x1 = orgx - maxx + 2;
	xm = x0 + xw;
	y0 = orgy - maxy;
	y1 = orgy + maxy;

	x = x0;
	y = y0 + dy;
}

static void FAR
left_margin (VIEW *view, int orgx, int orgy, int maxx, int maxy, int bss)
{
	dx = bss;
	dy = bss;

	xs = -xw;
	x0 = orgx - maxx + 2;
	x1 = orgx + maxx - xs;
	xm = x0 + xw;
	y0 = orgy - maxy + dy;
	y1 = orgy + maxy;

	x = x0;
	y = y0 + dy;
}

static char *HudTypeNames[] = {
	"Classic", "FA18", "F16", "F15",
	"Classic", "Classic", "Classic", "Classic"
};

static void FAR
screen_hud (void)
{
	int	i;

	if (ET_PLANE != CV->e_type)
		return;

	ScrPrintf ("0 turn off\n");
	ScrPrintf ("1 turn on\n");
	ScrPrintf ("2 toggle\n");
	ScrOnOff ("u hud",     EE(CV)->hud&HUD_ON,      NOVALUE);
	ScrOnOff ("l ladder",  EE(CV)->hud&HUD_LADDER,  NOVALUE);
	ScrOnOff ("b big",     EE(CV)->hud&HUD_BIG,     NOVALUE);
	ScrOnOff ("B border",  EE(CV)->hud1&HUD_BORDER, NOVALUE);
	ScrPrintf ("a area %dDeg\n", (int)EE(CV)->hudarea);
	ScrOnOff ("r reticle", EE(CV)->hud&HUD_RETICLE, NOVALUE);
	ScrPrintf ("z  type %s\n",  EE(CV)->hud&HUD_ROSS ? "Ross" : "Eyal");
	ScrOnOff ("t target",  EE(CV)->hud&HUD_TARGET,  NOVALUE);
	ScrOnOff ("h acc vect", EE(CV)->hud1&HUD_ACCVECT, NOVALUE);
	ScrOnOff ("H thick",   EE(CV)->hud1&HUD_THICK,  NOVALUE);
	ScrOnOff ("d data",    EE(CV)->hud&HUD_DATA,    NOVALUE);
	ScrPrintf("s scale len %d\n", EE(CV)->tapelen);
	ScrOnOff ("f fine",    EE(CV)->hud&HUD_FINE,    NOVALUE);
	ScrOnOff ("x exfine",  EE(CV)->hud&HUD_XFINE,   NOVALUE);
	ScrOnOff ("p pointer", EE(CV)->hud&HUD_CURSOR,  NOVALUE);
	ScrOnOff ("v v.vector", EE(CV)->hud&HUD_VV,     NOVALUE);
	ScrOnOff ("+ cross",   EE(CV)->hud&HUD_PLUS,    NOVALUE);
	ScrPrintf ("c color %lx\n", st.hud_color);
	ScrOnOff ("n name",    EE(CV)->hud1&HUD_INAME,  NOVALUE);
	ScrOnOff ("D dist",    EE(CV)->hud1&HUD_IDIST,  NOVALUE);
	ScrOnOff ("T top",     EE(CV)->hud1&HUD_TOP,    NOVALUE);
	ScrOnOff ("3 heading", EE(CV)->hud&HUD_FULLHEADING, NOVALUE);
	ScrOnOff ("L limit",   EE(CV)->hud1&HUD_LIMIT,  NOVALUE);
	ScrOnOff ("C corner",  EE(CV)->hud1&HUD_CORNER, NOVALUE);
	ScrOnOff ("V visual",  EE(CV)->hud1&HUD_VALARM, NOVALUE);
	ScrOnOff ("A aural",   EE(CV)->hud1&HUD_AALARM, NOVALUE);
	i = (EE(CV)->hud1&HUD_TYPES)/HUD_TYPE;
	ScrPrintf("y type %u %s\n", i, HudTypeNames[i]);
	ScrOnOff ("I ILS",     EE(CV)->hud2&HUD_ILS,  EE(CV)->ils);
	ScrOnOff ("P pendulum", EE(CV)->hud1&HUD_PENDULUM, NOVALUE);
	ScrOnOff ("m misc",    EE(CV)->hud1&HUD_MISC,   NOVALUE);
	ScrOnOff ("k knots",   EE(CV)->hud1&HUD_KNOTS,  NOVALUE);
	ScrPrintf("F font %d\n", (int)st.StFont);
	ScrPrintf("S  size %d/%d\n", (int)st.StFontSize, (int)dx);
}

static struct {
	Uchar	name[4];
	int	index;
} StatsInfo[] = {
	{"sndB",  0},
	{"sndL", 54},
	{"sndP",  1},

	{"rcvB",  2},
	{"rcvL", 53},
	{"rcvP",  3},

	{"sERs",  8},
	{"sERa",  9},

	{"rERd",  5},
	{"rERc",  6},
	{"NoPl",  7},
	{"rERb",  4},
	{"Nois", 55},
	{"rERl", 52},

	{"Fond", 50},
	{"Lost", 51},

	{"----", 56},
	{"----", 57},
	{"----", 58},
	{"----", 59},

	{"----", 60},
	{"----", 61},
	{"----", 62},
	{"----", 63},
	{"----", 64},
	{"----", 65},
	{"----", 66},
	{"----", 67},
	{"----", 68},
	{"----", 69},
{"", 0}};

static void FAR
screen_net (VIEW *view)
{
	int	i, j;
	long	lt;
	PLAYER	*pl;

	ScrPrintf ("Time %s\n", show_time ("", st.present));
	if (st.network & NET_INITED) {
		ScrPrintf ("Ver  %u\n", (int)st.ComVersion);
		ScrPrintf ("Remote players:\n");
		for (pl = 0; (pl = player_next (pl));) {
			ScrPrintf ("%-8.8s", pl->name);
			if (pl->flags & RMT_ACTIVE)
				ScrPrintf (" Active\n");
			else if (pl->flags & RMT_PENDREPLY)
				ScrPrintf (" PendConf\n");
			else if (pl->flags & RMT_PENDCONFIRM)
				ScrPrintf (" PendReply\n");
			else if (pl->flags & RMT_PLAYING)
				ScrPrintf (" Playing\n");
			else
				ScrPrintf ("\n");
		}
	} else
		ScrPrintf ("Net not active\n");
	for (i = 0; StatsInfo[i].name[0]; ++i) {
		if ((lt = st.stats[j = StatsInfo[i].index]))
			ScrPrintf ("%2u) %-4.4s %s\n",
				j, StatsInfo[i].name, show_l (lt));
	}
	if (st.stats[21])
		ScrPrintf ("in  av %lu max %lu\n",
			st.stats[22]/st.stats[20], st.stats[21]);
	if (st.stats[24])
		ScrPrintf ("out av %lu max %lu\n",
			st.stats[25]/st.stats[23], st.stats[24]);
}

static void FAR
screen_modes (VIEW *view)
{
	int	i, n;

	ScrPrintf ("Flags  %4x\n", st.flags);
	ScrOpt   ("Blanker", st.flags&SF_BLANKER);
	ScrOpt   ("Land",    st.flags&SF_LANDSCAPE);
	ScrOpt   ("Sky",     st.flags&SF_SKY);
	ScrOpt   ("Verbose", st.flags&SF_VERBOSE);
	ScrOpt   ("Smoke",   st.flags1&SF_SMOKE);
	ScrOpt   ("G",       st.flags1&SF_USEG);
	ScrOpt   ("Reverse", st.flags1&SF_STEREOREV);
	ScrOpt   ("Dbl Buff",st.flags1&SF_DBUFFERING);
	ScrValue ("Paralax", st.paralax);
	ScrValue ("Drones",  st.drones);
	ScrValue ("Killers", st.killers);
	ScrOnOff ("Zviews",  st.flags1&SF_EXTVIEW, st.extview);
	ScrOnOff ("Zoom",    CP->zoom, CP->zoom);
	ScrOnOff ("Gaze",    CP->rotz, ANG2DEG(CP->rotz));
	ScrOnOff ("Stereo",  st.stereo, st.stereo);
	ScrOnOff ("Info",    st.flags1&SF_INFO, st.info);
	ScrOnOff ("Sound", st.quiet, st.quiet);
	ScrPrintf ("%s\n", show_time ("Time  ", st.present));
	if (st.ShutdownTime)
		ScrPrintf ("%s\n", show_time ("Limit ", st.ShutdownTime));
	ScrPrintf ("last obj %s\n", show_l (st.object_id));
	if (ET_PLANE == CV->e_type) {
		ScrOnOff ("Instr", EE(CV)->flags & PF_INSTRUMENTS, NOVALUE);
		ScrOnOff ("OnGround", EE(CV)->flags & PF_ONGROUND, NOVALUE);
		ScrPrintf ("Radar");
		ScrPrintf ("%s", EE(CV)->radar&R_ON ? " On" : "");
		ScrPrintf ("%s", EE(CV)->radar&R_LOCK ? " Lck" : "");
		ScrPrintf ("%s", EE(CV)->radar&R_INTELCC ? " P" : "");
		ScrPrintf ("\n");
		n = 0;
		if (EE(CV)->radar & R_SELECT3)
			ScrPrintf (" Sbr"), ++n;
		else if (EE(CV)->radar & R_SELECT20)
			ScrPrintf (" Shd"), ++n;
		else if (EE(CV)->radar & R_SELECT5)
			ScrPrintf (" Svt"), ++n;
		if (EE(CV)->radar&R_INTEL)
			ScrPrintf (" Itl"), ++n;
		i = (EE(CV)->radar&R_MODES) / R_MODE;
		ScrPrintf (" M%d", i), ++n;
		if (EE(CV)->flags&PF_AUTO)
			ScrPrintf (" Aut"), ++n;
		if (EE(CV)->flags&PF_CHASE)
			ScrPrintf (" Chs"), ++n;
		if (EE(CV)->flags&PF_KILL)
			ScrPrintf (" Kil"), ++n;
		if (n)
			ScrPrintf ("\n"), n = 0;
	}
}

static void FAR
screen_help (void)
{
	ScrPrintf ("A   reticle mode   \n");
	ScrPrintf ("b   wheel brakes   \n");
	ScrPrintf ("c   clear text area\n");
	ScrPrintf ("C   chase target   \n");
	ScrPrintf ("D   descend chute  \n");
	ScrPrintf ("E   eject          \n");
	ScrPrintf ("f   tgt acq mode   \n");
	ScrPrintf ("g   landing gear   \n");
	ScrPrintf ("h   help (also ?)  \n");
	ScrPrintf ("i   intel mode     \n");
	ScrPrintf ("j   radar on pilots\n");
	ScrPrintf ("k   auto shoot     \n");
	ScrPrintf ("l   lock radar     \n");
	ScrPrintf ("m   show modes     \n");
	ScrPrintf ("n   net stats      \n");
	ScrPrintf ("o   observer select\n");
	ScrPrintf ("O    +show all objs\n");
	ScrPrintf ("p   pause          \n");
	ScrPrintf ("q   quiet modes    \n");
	ScrPrintf ("r   activate radar \n");
	ScrPrintf ("s   show obj stats \n");
	ScrPrintf ("S   reSupply plane \n");
	ScrPrintf ("u   hud modes      \n");
	ScrPrintf ("v   alternate view \n");
	ScrPrintf ("w   weapon select  \n");
	ScrPrintf ("W   remove weapons \n");
	ScrPrintf ("x   calib. pointer \n");
	ScrPrintf ("/*- view L/C/R     \n");
	ScrPrintf ("]/[ flaps +/-      \n");
	ScrPrintf ("}/{ spoilers +/-   \n");
	ScrPrintf ("+   air brakes     \n");
	ScrPrintf ("!   shell          \n");
	ScrPrintf ("F1  shoot          \n");
	ScrPrintf ("F2/3/4 rudder L/C/R\n");
	ScrPrintf ("F7/8 macro rec/play\n");
	ScrPrintf ("ESC exit           \n");
}

static void FAR
screen_debug (void)
{
	OBJECT	*target;

	ScrPrintf ("Debug info:\n");
	if (CO)
		ScrPrintf ("CO %04lu %04x %-6.6s\n",
			CO->id%10000, (int)CO->flags, TITLE(CO));
	else
		ScrPrintf ("CO (null)\n");
	ScrPrintf ("CC %04lu %04x %-6.6s\n",
		CC->id%10000, (int)CC->flags, TITLE(CC));
	ScrPrintf ("CV %04lu %04x %-6.6s\n",
		CV->id%10000, (int)CV->flags, TITLE(CV));
	if (ET_PLANE == CV->e_type && (target = EE(CV)->target)) {
		if (target->id == EE(CV)->tid)
			ScrPrintf ("TG %04lu %04x %-6.6s\n",
				target->id%10000, (int)target->flags,
				TITLE(target));
		else
			ScrPrintf ("TG %lu?%lu\n",
				target->id, EE(CV)->tid);
	}
#ifdef PCDOS
	i = get_8254 ();	/* PCDOS specific!!! */
	ScrPrintf ("T0 %02x\n", i);
	if (3*2 != (i & 0x0e) && st.quiet)
		Snd->Beep (4000, 1000);
#endif
	if (st.stats[42])
		ScrPrintf ("ERRS %lu\n", st.stats[42]);
	ScrPrintf ("___________________\n");
}

static void FAR
screen_font (int orgx, int orgy, int dx, int dy)
{
	int	x, y;

	x = orgx-dx*6;				/* test: show font */
	y = orgy-dy*3;

	y += dy;
	stroke_str (x, y, " !\"#$%&'()*+,-./", dx, st.tfg);
	y += dy;
	stroke_str (x, y, "0123456789:;<=>?", dx, st.tfg);
	y += dy;
	stroke_str (x, y, "@ABCDEFGHIJKLMNO", dx, st.tfg);
	y += dy;
	stroke_str (x, y, "PQRSTUVWXYZ[\\]^_", dx, st.tfg);
	y += dy;
	stroke_str (x, y, "`abcdefghijklmno", dx, st.tfg);
	y += dy;
	stroke_str (x, y, "pqrstuvwxyz{|}~\x7f", dx, st.tfg);
}

static void FAR
screen_alarm (int orgx, int orgy, int ss, int color)
{
	int	blink, dx;
/*
 * Put high priority alarms last to activate the most urgent aural warning.
*/
	blink = ((int)st.present)&0x0080;
	dx = stroke_size ("A", ss);

	if (O_CHUTE == CV->name && CV->R[Z] <= 0L) {
		if ((st.flags & SF_MAIN) && blink)
			stroke_str (orgx-dx*8, orgy, "WAIT", ss*4, color);
	}
}

extern void FAR
screen_info (VIEW *view, int orgx, int orgy, int maxx, int maxy, int bss,
	int mode)
{
	int	dx, dy, x, y, mg, info, i;
	Ulong	tt, tscore;
	OBJECT	*p;
	char	line[80], *l;

	if (!(p = CV))
		return;

	dx = bss;
	dy = bss;

	if (st.flags & SF_MAIN) {
		xd = stroke_size ("x", dx);
		xw = 19*xd;

		right_margin (view, orgx, orgy, maxx, maxy, bss);

		if (st.flags & SF_MODES)
			screen_modes (view);

		if (st.flags & SF_NET)
			screen_net (view);

		if (st.flags & SF_HUD)
			screen_hud ();

		if (st.flags & SF_DEBUG)
			screen_debug ();

		left_margin (view, orgx, orgy, maxx, maxy, bss);

		if (st.flags & SF_HELP)
			screen_help ();

		if (st.flags & SF_FONT)
			screen_font (orgx, orgy, dx, dy);

		for (x = 0; x < NHDD; ++x) {
			if ((st.hdd[x].flags & HDD_ON) &&
			    st.hdd[x].type == HDD_INFO)
				break;
		}
		if (x >= NHDD)
			show_menu (view, orgx, orgy, maxx, maxy, bss);

		msg_show (orgx, orgy, maxx, maxy, bss);

		show_prompt (view, orgx, orgy, maxx, maxy, bss);
	}

	y = orgy - maxy;
	if (HDD_INFO == mode) {
		mg = orgx - maxx + dx;
		info = st.info;
	} else {
		mg = orgx - maxx/2;
		if (HDD_FRONT == mode && st.flags1&SF_INFO)
			info = st.info;
		else
			info = 0;
	}

	if (info) {
		i = st.misc[0] - st.misc[1] - st.misc[2]
			- st.misc[3] - st.misc[4] - st.misc[5];

		sprintf (line, "%3u %3u %3u %3u %3u %3u %3d", /* times */
			st.misc[0], st.misc[1], st.misc[2], st.misc[5],
			st.misc[3], st.misc[4], i);
		y += dy;
		stroke_str (mg, y, line, dx, st.tfg);
	}

	if (2 == info) {
		y += dy;
		stroke_str (mg, y, show_time ("", st.present), dx, st.wfg);

		sprintf (line, "S%4u %4lu %4lu",
			(int)st.nobjects, st.stats[15], st.stats[26]);
		y += dy;
		stroke_str (mg, y, line, dx, st.tfg);

		sprintf (line, "C%4lu%4lu%4lu%4lu",
			st.stats[10], st.stats[11], st.stats[12], st.stats[13]);
		y += dy;
		stroke_str (mg, y, line, dx, st.tfg);

		sprintf (line, "P%6ld%6ld%6ld",
			vuscale (CV->R[X]), vuscale (CV->R[Y]),
			vuscale (CV->R[Z]));
		y += dy;
		stroke_str (mg, y, line, dx, st.tfg);

		sprintf (line, "V%6d%6d%6d",
			(int)vuscale (CV->V[X]), (int)vuscale (CV->V[Y]),
			(int)vuscale (CV->V[Z]));
		y += dy;
		stroke_str (mg, y, line, dx, st.tfg);

		if (ET_PLANE == CV->e_type) {
			sprintf (line, "v%6d%6d%6d",
				(int)vuscale (EE(CV)->v[X]),
				(int)vuscale (EE(CV)->v[Y]),
				(int)vuscale (EE(CV)->v[Z]));
			y += dy;
			stroke_str (mg, y, line, dx, st.tfg);
		}

		l = line;
		sprintf (l, "A");
		l += 1;
		l += show_ang (l, CV->a[X]);
		l += show_ang (l, CV->a[Y]);
		l += show_ang (l, CV->a[Z]);
		y += dy;
		stroke_str (mg, y, line, dx, st.tfg);

		l = line;
		sprintf (l, "D");
		l += 1;
		l += show_ang (l, CV->da[X]*VONE);
		l += show_ang (l, CV->da[Y]*VONE);
		l += show_ang (l, CV->da[Z]*VONE);
		y += dy;
		stroke_str (mg, y, line, dx, st.tfg);
	} else if (3 == info) {
		if (st.flags1 & SF_TESTING)
			tt = st.present - st.test_start;
		else
			tt = st.present;
		sprintf (line, "%s %u %lu %u", show_time ("", tt),
			(int)st.ntargets, st.nbullets, p->score);
		y += dy;
		stroke_str (mg, y, line, dx, st.tfg);
		y += dy;
		x = mg;
		if (st.flags1 & SF_TESTING) {
			tscore = st.mscore - (tt/1000)*10L - st.nbullets*2L;
			sprintf (line, "%ld ", tscore);
			x += stroke_str (x, y, line, dx, st.wfg);
		}
		sprintf (line, "%d", (int)(p->speed/VONE));
		x += stroke_str (x, y, line, dx, st.wfg);
	}

	if (!(st.flags & SF_BLANKER) && scenery (mode)) {
		if (CP && CP->rotz) {
			y += dy;
			x = orgx-dx;
			x += stroke_char (x, y, 'v', dx, st.hfg);
			if (CP->rotz > 0)
				x += stroke_char (x, y, ' ', dx, st.hfg);
			x += stroke_num (x, y, ANG2DEG(CP->rotz), dx, st.hfg);
		}

		if (CP && CP->rotx) {
			y += dy;
			x = orgx-dx;
			x += stroke_char (x, y, 'x', dx, st.hfg);
			if (CP->rotx > 0)
				x += stroke_char (x, y, ' ', dx, st.hfg);
			x += stroke_num (x, y, ANG2DEG(CP->rotx), dx, st.hfg);
		}

		if (CP && CP->zoom) {
			y += dy;
			stroke_char (orgx-dx, y, 'z', dx, st.hfg);
			stroke_num (orgx, y, CP->zoom, dx, st.hfg);
		}
	}

	screen_alarm (orgx, orgy, bss, st.hfgi);
}
