/* -------------------------------- land.c ---------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Look after the landscape. These objects do not move or get simulated!
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"

extern int	FAR paddoc_init (BODY *b);

#define	RANGE	20

extern int FAR
land_init (void)
{
	BODY	*b;
	OBJECT	*p;
	int	i, n;

	CL = CLT = 0;

	i  = !create_land (O_GROUND);		/* must be first! */
	i |= !create_land (O_LOW);		/* must be second! */

	if (create_land (O_RUNWAY)) {		/* main runway */
		if ((p = create_land (O_TOWER))) {
			p->R[X] = 200L*VONE;
			p->R[Y] = 200L*VONE;
		} else
			i = 1;
	} else
		i = 1;

	for (n = 0; n < 4; ++n) {		/* the barracks */
		if ((p = create_land (O_HOUSE))) {
			p->R[X] = (100L+20L*n)*VONE;
			p->R[Y] = (100L+40L*n)*VONE;
		} else {
			i = 1;
			break;
		}
	}

	if ((p = create_land (O_RUNWAY))) {	/* secondary runway */
		p->R[X] = 10000L*VONE;
		p->R[Y] = 10000L*VONE;
		if ((p = create_land (O_TOWER))) {
			p->R[X] = (10000L+200L)*VONE;
			p->R[Y] = (10000L+200L)*VONE;
		} else
			i = 1;
	} else
		i = 1;

	if (i)
		return (1);

	if (!(st.flags&SF_LANDSCAPE))
		return (0);

	b = new_body ();
	if (!b)
		return (1);
	b->init = paddoc_init;
	if ((*b->init) (b))
		return (1);

	st.landx = 0x8000;		/* force landscaping */
	st.landy = 0x8000;

	return (0);
}

extern void FAR
land_term (void)
{
	list_clear (&CL);
}

static int FAR
land_add (ONAME lname, int xx, int yy)
{
	OBJECT	*p;
	int	cloud;

	srand ((xx^(yy<<4))^yy);
	if (rand() % 128)
		return (0);

	cloud = (rand()%10)/5;

	p = create_land (lname);

	if (p) {
		p->flags |= F_LAND;
		p->R[X] = xx * 1000L * VONE;
		p->R[Y] = yy * 1000L * VONE;
		if (cloud) {
			p->R[Z] = (rand()%2000 + 2000)*(long)VONE;
			p->color = st.white;
		} else
			p->R[Z] = 0;
	} else
		return (0);		/* ignore lost paddoc */
	return (0);
}

extern int FAR
land_update (OBJECT *pov)
{
	long	minx, miny, maxx, maxy;
	int	x, y, xl, xh, yl, yh, xxl, xxh, xx, yy;
	OBJECT	*p, *prev;
	ONAME	lname;

	if (!(p = CL) || O_GROUND != p->name)
		return (1);
	st.bodies[p->name]->dynamics (p, st.interval);

	if (!(p = p->next) || O_LOW != p->name)
		return (1);
	st.bodies[p->name]->dynamics (p, st.interval);

	if (!(st.flags&SF_LANDSCAPE))
		return (0);

	x = (int)(pov->R[X] / VONE / 1000);		/* get square */
	y = (int)(pov->R[Y] / VONE / 1000);

	if (x == st.landx && y == st.landy)
		return (0);

	minx = (x-RANGE)*1000L*VONE;
	maxx = (x+RANGE)*1000L*VONE;
	miny = (y-RANGE)*1000L*VONE;
	maxy = (y+RANGE)*1000L*VONE;
	for (prev = 0, p = CL; p;) {			/* delete old */
		if ((p->flags & F_LAND) &&
		    (p->R[X] < minx || p->R[X] > maxx || 
		     p->R[Y] < miny || p->R[Y] > maxy))
		    	p = delete_land (p);
		else
			p = p->next;
	}

	if (st.landx < x) {
		xxl = x-RANGE;
		xh  = x+RANGE;
		xxh = st.landx+RANGE;
		xl  = xxh+1;
		if (xl < xxl)
			xl = xxl;
	} else {
		xl  = x-RANGE;
		xxh = x+RANGE;
		xxl = st.landx-RANGE;
		xh  = xxl-1;
		if (xh > xxh)
			xh = xxh;
	}

	if (st.landy < y) {
		yh = y+RANGE;
		yl = st.landy+RANGE+1;
		if (yl < y-RANGE)
			yl = y-RANGE;
	} else {
		yl = y-RANGE;
		yh = st.landy-RANGE-1;
		if (yh > y+RANGE)
			yh = y+RANGE;
	}

	st.landx = x;
	st.landy = y;

	if (-1 == (lname = body_name ("PADDOC")))
		return (1);

	for (xx = xl; xx <= xh; ++xx) {
		for (yy = y-RANGE; yy <= y+RANGE; ++yy) {
			if (land_add (lname, xx, yy))
				return (0);
		}
	}

	for (xx = xxl; xx <= xxh; ++xx) {
		for (yy = yl; yy <= yh; ++yy) {
			if (land_add (lname, xx, yy))
				return (0);
		}
	}

	return (0);
}
