/* ------------------------------------ mat.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Matrix and vector manipulation.
*/

#include <stdlib.h>
#include <string.h>

#include "fly.h"

extern void FAR
Mident (MAT m)
{
	m[0][1] = m[0][2] = m[1][0] = 
	m[1][2] = m[2][0] = m[2][1] = 0;
	m[0][0] = m[1][1] = m[2][2] = FONE;
}

extern void FAR
Mrotx (MAT m, ANGLE d)
{
	register int	c, s, t;

	c = COS(d);
	s = SIN(d);

	t = m[0][1];
	m[0][1] = fmul(t,c) - fmul(m[0][2],s);
	m[0][2] = fmul(t,s) + fmul(m[0][2],c);

	t = m[1][1];
	m[1][1] = fmul(t,c) - fmul(m[1][2],s);
	m[1][2] = fmul(t,s) + fmul(m[1][2],c);

	t = m[2][1];
	m[2][1] = fmul(t,c) - fmul(m[2][2],s);
	m[2][2] = fmul(t,s) + fmul(m[2][2],c);
}

extern void FAR
Mroty (MAT m, ANGLE d)
{
	register int	c, s, t;

	c = COS(d);
	s = SIN(d);

	t = m[0][0];
	m[0][0] =  fmul(t,c) + fmul(m[0][2],s);
	m[0][2] = -fmul(t,s) + fmul(m[0][2],c);

	t = m[1][0];
	m[1][0] =  fmul(t,c) + fmul(m[1][2],s);
	m[1][2] = -fmul(t,s) + fmul(m[1][2],c);

	t = m[2][0];
	m[2][0] =  fmul(t,c) + fmul(m[2][2],s);
	m[2][2] = -fmul(t,s) + fmul(m[2][2],c);
}

extern void FAR
Mrotz (MAT m, ANGLE d)
{
	register int	c, s, t;

	c = COS(d);
	s = SIN(d);

	t = m[0][0];
	m[0][0] = fmul(t,c) - fmul(m[0][1],s);
	m[0][1] = fmul(t,s) + fmul(m[0][1],c);

	t = m[1][0];
	m[1][0] = fmul(t,c) - fmul(m[1][1],s);
	m[1][1] = fmul(t,s) + fmul(m[1][1],c);

	t = m[2][0];
	m[2][0] = fmul(t,c) - fmul(m[2][1],s);
	m[2][1] = fmul(t,s) + fmul(m[2][1],c);
}

extern void FAR
Mobj (register OBJECT *p)
{
	build_mat (p->T, SIN(p->a[X]), COS(p->a[X]),
		SIN(p->a[Y]), COS(p->a[Y]), SIN(p->a[Z]), COS(p->a[Z]));
}

extern void FAR
Myxz (MAT m, AVECT a)
{
	build_mat (m, SIN(a[X]), COS(a[X]), SIN(a[Y]), COS(a[Y]),
			SIN(a[Z]), COS(a[Z]));
}

extern void FAR
Mxpose (MAT m)
{
	register int	t;

	t = m[0][1];	m[0][1] = m[1][0];	m[1][0] = t;
	t = m[0][2];	m[0][2] = m[2][0];	m[2][0] = t;
	t = m[1][2];	m[1][2] = m[2][1];	m[2][1] = t;
}

extern void FAR
Mmul (MAT m, MAT t)
{
	VECT	temp;

	VMmul (temp, m[0], t);	Vcopy (m[0], temp);
	VMmul (temp, m[1], t);	Vcopy (m[1], temp);
	VMmul (temp, m[2], t);	Vcopy (m[2], temp);
}

/*
 * rot(y)*rot(x)*rot(z):
 *
 *	.[0]		.[1]		.[2]
 *
 * [0].	cy*cz-sy*sz*sx	cy*sz+sy*cz*sx	-sy*cx	[0].
 * [1].	-cx*sz		cx*cz		sx	[1].
 * [2].	sy*cz+cy*sz*sx	sy*sz-cy*cz*sx	cy*cx	[2].
 *
 * If cx == 0 we have:
 *
 * [0].	cy*cz-sy*sz*sx	cy*sz+sy*cz*sx	0	[0].
 * [1].	0		0		sx	[1].
 * [2].	sy*cz+cy*sz*sx	sy*sz-cy*cz*sx	0	[2].
 *
 * which when sx == 1 is:
 *
 * [0].	c(y+z)          s(y+z)		0	[0].
 * [1].	0		0		1	[1].
 * [2].	s(y+z)		-c(y+z)		0	[2].
 *
 * and when sx == -1 is:
 *
 * [0].	c(y-z)		-s(y-z)		0	[0].
 * [1].	0		0		-1	[1].
 * [2].	s(y-z)		c(y-z)		0	[2].
 *
 *	.[0]		.[1]		.[2]
*/

extern void FAR
Mangles1 (MAT m, AVECT a)		/* obsolete */
{
	ANGLE	t1;
	int	t2;

	t1 = ASIN (m[1][2]);				/* pitch +-pi/2 */
	t2 = COS (t1);					/* always +ve	*/
	if (iabs(t1) > (D90-D90/16)) {			/* 6 degrees	*/
		t2 = ihypot2d (m[1][0], m[1][1]);
		a[X] = ATAN (m[1][2], t2);

		a[Y] = 0;
		if (m[2][2] < 0)
			a[Y] = D180 - a[Y];

		t1 = ATAN (m[2][0], m[0][0]);		/* roll+-dir	*/
		if (m[1][2] >= 0)			/* roll+dir	*/
			a[Z] = t1 - a[Y];		/* dir		*/
		else					/* roll-dir	*/
			a[Z] = a[Y] - t1;		/* dir		*/
	} else {
		a[X] = t1;				/* pitch	*/
		a[Z] = -ATAN (m[1][0], m[1][1]);	/* dir		*/
		a[Y] = -ATAN (m[0][2], m[2][2]);	/* roll		*/
	}
}

extern void FAR
Mangles (MAT m, AVECT a, ANGLE dy)
{
	ANGLE	t1;
	int	t2;

	t1 = ASIN (m[1][2]);				/* pitch +-pi/2 */
	if (iabs(t1) > (D90-D90/16)) {			/* 6 degrees	*/
		t2 = ihypot2d (m[1][0], m[1][1]);
		a[X] = ATAN (m[1][2], t2);

		a[Y] += dy;				/* roll		*/
		if ((m[2][2] < 0) != (iabs(a[Y]) > D90))
			a[Y] = D180 - a[Y];

		t1 = ATAN (m[2][0], m[0][0]);		/* roll+-dir	*/
		if (m[1][2] >= 0)			/* roll+dir	*/
			a[Z] = t1 - a[Y];		/* dir		*/
		else					/* roll-dir	*/
			a[Z] = a[Y] - t1;		/* dir		*/
	} else {
		a[X] = t1;				/* pitch	*/
		a[Z] = -ATAN (m[1][0], m[1][1]);	/* dir		*/
		a[Y] = -ATAN (m[0][2], m[2][2]);	/* roll		*/
	}
}

#if 0			/* old one */
extern void FAR
Mangles (MAT m, AVECT a)
/*
 * Extract angles from orientation matrix. The pitch is always in the range
 * -90...+90 (cos is +ve). When pointing along the z coord the direction
 * and roll are mixed, so we assume old roll stays and we extract new
 * direction only. There is some problem with the resolution of the arcsin
 * near this vertical direction (actualy, whenever sin is close to 1).
*/
{
	ANGLE	t1;
	int	t2;

	t1 = ASIN (m[1][2]);				/* pitch +-pi/2 */
	a[X] = t1;
#if 1
	t2 = COS (t1);					/* always +ve	*/
#else
	t2 = fmul(m[1][0],m[1][0])+fmul(m[1][1],m[1][1]); /* cx**2 */
#endif
	if (t2 < 64) {
#if 0
		if (m[0][2]) {
			t2 = SIN (a[Y]);
			t2 = fmul (m[1][2], iabs (t2));
			a[X] = ATAN (t2, iabs (m[0][2]));
		} else if (m[2][2]) {
			t2 = COS (a[Y]);
			t2 = fmul (m[1][2], iabs (t2));
			a[X] = ATAN (t2, iabs (m[2][2]));
		}
#endif
#if 0
		if (m[1][0]) {
			t2 = SIN (a[Z]);
			t2 = fmul (m[1][2], iabs (t2));
			a[X] = ATAN (t2, iabs (m[1][0]));
		} else if (m[1][1]) {
			t2 = COS (a[Z]);
			t2 = fmul (m[1][2], iabs (t2));
			a[X] = ATAN (t2, iabs (m[1][1]));
		}
#endif
		t1 = ATAN (m[2][0], m[0][0]);		/* roll+-dir	*/
#if 0
						/* keep old roll a[Y]	*/
		if (a[X] >= 0)				/* roll+dir	*/
			a[Z] = t1 - a[Y];		/* dir		*/
		else					/* roll-dir	*/
			a[Z] = a[Y] - t1;		/* dir		*/
#endif
#if 0
						/* keep old dir a[Z]	*/
		if (a[X] >= 0)				/* roll+dir	*/
			a[Y] = t1 - a[Z];		/* roll		*/
		else					/* roll-dir	*/
			a[Y] = t1 + a[Z] ;		/* roll		*/
#endif
#if 1
		a[Y] = 0;				/* zero roll	*/
		if (a[X] >= 0)				/* roll+dir	*/
			a[Z] = t1;			/* dir		*/
		else					/* roll-dir	*/
			a[Z] = -t1;			/* dir		*/
#endif
	} else {
		a[Z] = -ATAN (m[1][0], m[1][1]);	/* dir		*/
		a[Y] = -ATAN (m[0][2], m[2][2]);	/* roll		*/
	}
}
#endif

#if 0
}
	ANGLE	t1;
	int	t2, t3;

	t1 = ASIN(m[1][2]);				/* pitch +-pi/2*/
	a[X] = t1;
	t2 = COS (t1);					/* always +ve	*/
	if (t2 < 1) {
		t1 = ASIN (m[2][0]);			/* roll+-dir	*/
		if (m[0][0] < 0)
			t1 = D180 - t1;
#if 0
		/* keep old roll a[Y]					*/
		if (a[X] >= 0)				/* roll+dir	*/
			a[Z] = t1 - a[Y];		/* dir		*/
		else					/* roll-dir	*/
			a[Z] = a[Y] - t1;		/* dir		*/
#else
		/* keep old dir a[Z]					*/
		if (a[X] >= 0)				/* roll+dir	*/
			a[Y] = t1 - a[Z];		/* toll		*/
		else					/* roll-dir	*/
			a[Y] = t1 + a[Z] ;		/* roll		*/
#endif
	} else {
		if (m[1][0] > t2)
			t3 = FONE;
		else if (-m[1][0] > t2)
			t3 = -FONE;
		else
			t3 = muldiv(m[1][0],FONE,t2);
		a[Z] = -ASIN (t3);			/* dir		*/
		if (m[1][1] < 0)
			a[Z] = D180 - a[Z];

		if (m[0][2] > t2)
			t3 = FONE;
		else if (-m[0][2] > t2)
			t3 = -FONE;
		else
			t3 = muldiv(m[0][2],FONE,t2);
		a[Y] = -ASIN (t3);			/* roll		*/
		if (m[2][2] < 0)
			a[Y] = D180 - a[Y];
	}
}
#endif

extern void FAR
Vscale (VECT a, VECT b, int scalar)
{
	a[X] = fmul (b[X], scalar);
	a[Y] = fmul (b[Y], scalar);
	a[Z] = fmul (b[Z], scalar);
}

extern void FAR
Vmuldiv (VECT a, VECT b, int m, int d)
{
	a[X] = muldiv (b[X], m, d);
	a[Y] = muldiv (b[Y], m, d);
	a[Z] = muldiv (b[Z], m, d);
}
