/* --------------------------------- speaker.c ------------------------------ */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Sound device driver for the PC speaker.
*/

#include <conio.h>
#include <string.h>

#include "fly.h"

#define	QSIZE	10

#define	COMMAND_REG	0x43
#define	CHANNEL_2	0x42
#define	PORT_B		0x61
#define	XTAL		1193000L

struct beep {
	int	freq;
	int	milli;
	int	*list;
	int	id;
};

static struct beep beeps[QSIZE] = {0};
static Ulong	last = 0L;
static int	playing = 0, nbeeps = 0;


static void FAR
sound_start (int n)
{
	int	i;
	short	flags;

	flags = Sys->Disable ();

	i = inp (PORT_B);	/* get 8255 port B */
	i |= 3;			/* turn on speaker */
	outp (PORT_B, i);	/* set 8255 port B */

	n = (int)(XTAL / n);

	outp (COMMAND_REG, 0xb6);
	outp (CHANNEL_2, n & 0x00ff);		/* LSB */
	outp (CHANNEL_2, (n >> 8) & 0x00ff);	/* MSB */

	Sys->Enable (flags);
}

static void FAR
sound_stop (void)
{
	int	i;
	short	flags;

	flags = Sys->Disable ();

	i = inp (PORT_B);	/* get 8255 port B */
	i &= ~3;		/* turn off speaker */
	outp (PORT_B, i);	/* set 8255 port B */
	Sys->Enable (flags);
}

static int FAR
sound_command (struct beep *b)
{
	int	l, ret;

again:
	for (ret = 1; ret && (l = b->list[0]) < 0;) {
		switch (l) {
		default:
		case -1:			/* END */
			ret = 0;
			break;
		case -2:			/* GOTO,address */
			b->list += 2*b->list[1];
			break;
		case -3:			/* REPEAT,address,n,cnt */
			if (b->list[3]) {
				if (--b->list[3])	/* Nth time */
					b->list += 2*b->list[1];
				else			/* done */
					b->list += 2*2;
			} else {			/* start */
				b->list[3] = b->list[2];
				b->list += 2*b->list[1];
			}
			break;
		}
	}
	if (ret) {
		b->freq = *b->list++;
		b->milli = *b->list++/4;
		if (!b->milli)
			goto again;
	}

	return (ret);
}

static int	force = 0;

static void FAR
SndPoll (void)
{
	Ulong		t;
	int		i, diff, highest;
	struct beep	*b;

	t = Tm->Milli ();
	if (!nbeeps) {
		last = t;
		return;
	}
	if (!force && t == last)
		return;

	diff = (int)(t - last);
	last = t;
	
	highest = 0;
	for (i = 0; i < QSIZE; ++i) {
		b = beeps+i;
		if (!b->milli)
			continue;
		if (b->milli <= diff) {
			if (b->list && sound_command (b))
				goto use_it;
			b->milli = 0;
			--nbeeps;
			continue;
		}
		b->milli -= diff;
use_it:
		if (b->freq > highest)
			highest = b->freq;
	}
	if (highest) {
		if (highest != playing) {
			if (playing)
				sound_stop ();
			playing = highest;
			sound_start (playing);
		}
	} else if (playing) {
		sound_stop ();
		playing = 0;
	}
}

static int FAR
AllocBeep (int f, int milli, int *list, int id)
{
	int		i;
	struct beep	*b;

	if (f <= 0)
		f = 440;

	SndPoll ();
	for (i = QSIZE; --i >= 0;) {
		b = beeps+i;
		if (b->milli)
			continue;
		b->freq = f;
		b->milli = milli;
		b->list = list;
		b->id = id;
		++nbeeps;
		break;
	}
	if (i >= 0) {
		force = 1;
		SndPoll ();
		force = 0;
	}
	return (i);
}

static int FAR
SndBeep (int f, int milli)
{
	return (AllocBeep (f, milli, 0, -1));
}

static int FAR
SndList (int *list, int id)
{
	if (!list || *list < 0 || !(list[1]/4))
		return (-1);
	return (AllocBeep (list[0], list[1]/4, list+2, id));
}

static void FAR
SndEffect (int eff)
{
	return;
}

static int FAR
SndInit (void)
{
	memset (beeps, 0, sizeof (beeps));
	playing = 0;
	nbeeps = 0;
	last = Tm->Milli ();

	return (0);
}

static void FAR
SndTerm (void)
{
	memset (beeps, 0, sizeof (beeps));
	sound_stop ();
};

extern struct SndDriver SndSpeaker = {
	"SPEAKER",
	0,
	SndInit,
	SndTerm,
	SndPoll,
	SndBeep,
	SndEffect,
	SndList
};
