/* --------------------------------- obasic.c ------------------------------- */

/* This is part of the flight simulator 'fly8'.
 * Author: Eyal Lebedinsky (eyal@ise.canberra.edu.au).
*/

/* Dynamics of the Basic plane.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "fly.h"
#include "plane.h"

#define	GPLIMIT		(9*GACC)	/* +ve acceleration limit */
#define	GNLIMIT		(3*GACC)	/* -ve acceleration limit */
#define	GLIMIT		(10*GACC)	/* total G limit */

static int CCl = 0;

static void FAR
CCshow (OBJECT *p, int frac, char *title, long value)
{
	if (CC == p && (st.flags & SF_DEBUG)) {
		stroke_str  (1*8, CCl*8, title, 8, st.hfg);
		stroke_frac (6*8, CCl*8, value, 0, frac, 8, st.hfg);
		++CCl;
	}
}

static int FAR
check_land (OBJECT *p)
{
	int	errs;

	if (p->R[Z] > 0L)
		return (0);
	
	EX->flags |= PF_ONGROUND;

	errs = 0;

	if (!(EX->equip & EQ_GEAR)) {
		CCnote (p, "belly");
		++errs;
	}

	if (p->a[Y] > D90/9) {
		CCnote (p, "chinese");
		++errs;
	}
	p->a[Y] = 0;
	p->da[Y] = 0;
	if (p->a[X] < EP->gpitch) {
		CCnote (p, "nose down");
		++errs;
	}

	if (EX->v[X] < -10*VONE || EX->v[X] > 10*VONE) {
		CCnote (p, "sideways");
		++errs;
	}
	EX->v[X] = 0;

	if (p->V[Z] < -7*VONE) {
		CCnote (p, "crash");
		++errs;
	}
	p->V[Z] = 0;
	if (!on_runway (p)) {
		CCnote (p, "grass");
		++errs;
	}
	p->R[Z] = 0;
	if (!errs)
		CCland (p);
	return (errs);
}

static int FAR
check_takeoff (OBJECT *p)
{
	int	errs;

	errs = 0;
	if (!on_runway (p)) {
		CCnote (p, "grass");
		++errs;
	}
	if (!(EX->equip & EQ_GEAR)) {
		CCnote (p, "belly");
		++errs;
	}
	if (p->a[Y] > D90/9) {
		CCnote (p, "chinese");
		++errs;
	}
	if (p->a[X] < EP->gpitch)
		p->a[X] = EP->gpitch;
	if (!errs && p->R[Z] > 0L) {
		EX->flags &= ~PF_ONGROUND;
		CCfly (p);
	}
	return (errs);
}

extern void FAR
dynamics_basic (OBJECT *p, int interval)
{
	POINTER	*ptr;
	int	glimit, onground, taxiing;
	int	t, v, vmag, weight, force, drag, acc, lift, q, qf;
	int	rho, sos, Cd, Cl, Cdp, Cds, Cdg, Cdi, FEff, sina;
	long	tt;
	ANGLE	alpha, beta, stall, a;
	VECT	AA;
	AVECT	da;
CCl = 4;
	if (dynamics_input (p, interval))
		return;

	ptr = p->pointer;

	onground = EX->flags & PF_ONGROUND;
	glimit = 0;

	AA[X] = -fmul (GACC, p->T[X][Z]);	/* gravity */
	AA[Y] = -fmul (GACC, p->T[Y][Z]);
	AA[Z] = -fmul (GACC, p->T[Z][Z]);

	EX->misc[5] = -AA[Z];		/* pilots gravity */
	v = EX->v[Y];

	t = muldiv (EP->pitch_rate, EX->elevators, 100);
	t = muldiv (t, v, 150*VONE);
	dampen (&p->da[X], t, EP->opt[1]);

	t = -muldiv (EP->roll_rate,  EX->ailerons, 100);
	t = muldiv (t, v, 150*VONE);
	dampen (&p->da[Y], t, EP->opt[1]);

	if (onground ? check_takeoff (p) : check_land (p)) {
		p->flags |= F_HIT;
		return;
	}

	taxiing = onground && p->speed < 40*VONE;	/* NWS */

	if (100 == EX->throttle) {
		t = EP->ab_thrust - EP->mil_thrust;
		t = EP->mil_thrust + muldiv (EX->afterburner, t, 100);
	} else
		t = muldiv (EX->throttle, EP->mil_thrust, 100);
	if (t < 0) {
		if (onground)
			t /= 2;		/* reverse thrust is 50% efficient? */
		else
			t = 0;
	}
	dampen (&EX->thrust, t, 8);

	t = EX->afterburner ? EP->ab_sfc : EP->mil_sfc;
	t = muldiv (iabs(EX->thrust), t, 60*60/10);
	EX->misc[12] = t;
	EX->fuel -= TADJ(t);
	if (EX->fuel < 0) {
		EX->fuel = 0;
		EX->thrust = 0;
	}
	if (onground && 0 == p->speed && 0 == EX->thrust) {	/* refuel */
		if (EX->fuel < EP->fuel_capacity*100L) {
			EX->fuel += TADJ(500)*100L;		/* 500lb/sec */
			if (EX->fuel >= EP->fuel_capacity*100L)
				supply (p);
		}
	}

	force = fmul (EX->thrust*2, FCON(0.453*9.8/VONE*5));	/* N*VONE */
	tt = EP->weight + EX->fuel/100;				/* lb */
	weight = fmul ((int)(tt/VONE), FCON (0.453));		/* Kg*VONE */
	if ((t = EX->stores[WE_M61-1]) > 0)
		weight += (int)(t * st.bodies[O_M61]->shape->weight/1000/VONE);
	if ((t = EX->stores[WE_MK82-1]) > 0)
		weight += (int)(t * st.bodies[O_MK82]->shape->weight/1000/VONE);
	acc = muldiv (force, VONE, weight);
CCshow (p, 0, "thrust", (long)force);
CCshow (p, 0, "weight", (long)weight);
CCshow (p, 0, "thrA", (long)acc);

#define	BRAKE_FORCE	(5*VONE)
#define	WHEEL_DRAG	(VONE/8)

	if (taxiing) {
CCshow (p, 0, "vold", (long)p->speed);
		p->speed += TADJ(acc);				/* v */
CCshow (p, 0, "vnew", (long)p->speed);
		drag = TADJ((EX->equip & EQ_GNDBRAKE) ? BRAKE_FORCE : WHEEL_DRAG);
CCshow (p, 0, "drag", (long)drag);
		if (p->speed > drag)
			p->speed -= drag;
		else if (p->speed < -drag)
			p->speed += drag;
		else
			p->speed = 0;				/* stop! */
CCshow (p, 0, "v", (long)p->speed);

		p->a[X] = EP->gpitch;
		p->a[Y] = 0;

		EX->v[X] = 0;
		EX->v[Y] = fmul (p->speed, COS (-p->a[X]));
		EX->v[Z] = fmul (p->speed, SIN (-p->a[X]));
CCshow (p, 0, "vy", (long)EX->v[Y]);
		p->da[X] = p->da[Y] = 0;
		p->da[Z] = muldiv (p->speed, EX->rudder, VONE);
		sos = 366*VONE;
	} else {
		if (v) {
			alpha = -ATAN (EX->v[Z], v);
			if (alpha > D90)			/* temp */
				alpha = D90;
			else if (alpha < -D90)
				alpha = -D90;
		} else
			alpha = 0;
		EX->misc[7] = alpha;
		p->da[X] -= alpha/64;

		t = ihypot2d (EX->v[Y], EX->v[Z]);
		beta = t ? ATAN (EX->v[X], t) : 0;
#if 1
		p->da[Z] = muldiv (p->speed, EX->rudder, 100*VONE);
		p->da[Z] -= beta/16;
		AA[X] -= EX->v[X]/2;
#endif
#if 0
		p->da[Z] = muldiv (p->speed, EX->rudder, 100*VONE);
		p->da[Z] -= beta/16;
#endif
#if 0
		AA[X] = -muldiv (p->speed, EX->rudder, 16*VONE);
		AA[X] -= EX->v[X] / (4*VONE);
		p->da[Z] = -beta/8;
#endif

		EX->wing_stall = 0;
#if 0
		stall = D90/4;
		stall -= muldiv (stall, EX->flaps, 100);
		if (alpha > D90/4) {
			t = (stall-alpha)/VONE/4;
			if (p->da[X] > t)
				p->da[X] = t;
			EX->wing_stall = 1;
			if (EP->opt[4]) {
				if (alpha > stall/2*3)
					alpha = 0;
				else
					alpha = (stall/2*3-alpha)*2;
			} else
				alpha = stall;
		} else if (alpha < -stall) {
			t = (-stall-alpha)/VONE/4;
			if (p->da[X] < t)
				p->da[X] = t;
			EX->wing_stall = 1;
			if (EP->opt[4]) {
				if (alpha < -stall/2*3)
					alpha = 0;
				else
					alpha = (-stall/2*3-alpha)*2;
			} else
				alpha = -stall;
		}
#else
		stall = 0;	/* avoid compiler warning */
#endif
		if (p->R[Z] < 14000L*VONE)
			rho = FONE - muldiv (FCON(.12),
				(int)(p->R[Z]/VONE), 3000);
		else
			rho = FCON(.4) - muldiv (FCON(.1),
				(int)(p->R[Z]/VONE)-14000, 3000);
		rho = fmul (rho, rho);
CCshow (p, 3, "rho", (long)fmul(rho, 1000));

		if (p->R[Z] <= 12000L*VONE)
			sos = 366*VONE + muldiv ((366-317)*VONE,
						(((int)p->R[Z])/VONE), 12000);
		else
			sos = 317*VONE;
CCshow (p, 0, "sos", (long)sos/VONE);

#if 0
		a = alpha + muldiv (EX->flaps, D90, 90*4);
#else
		FEff = FCON(0.5);
		a = alpha + fmul (FEff, DEG2ANG (EX->flaps));
#endif
		sina = SIN (a);
CCshow (p, 2, "aoa", (long)ANG2DEG00 (alpha));
CCshow (p, 2, "aeff", (long)ANG2DEG00 (a));
		t = fdiv (EP->wing_area, EP->wing_span * EP->wing_span);
		t = fmul (t, 5215);		/* 1/pi */
CCshow (p, 3, "t", (long)fmul(t, 1000));
/*
 * the '/4' and (later) '*4' are to evoid overflow. Fractionc are limited
 * to the range [-2...+2].
*/
		Cl  = muldiv (sina, FONE/4, FCON(.17) + t);
		Cdi = fmul (Cl, t*4);
		Cdi = fmul (Cdi, Cl)*4;
		Cdi = muldiv (Cdi, 100, EP->efficiency_factor);
		Cdp = FCON(0.05);		/* parasitic */
		Cds = FCON(0.50);		/* speed brakes */
		Cdg = FCON(0.10);		/* gear */
CCshow (p, 3, "Cl", (long)fmul(Cl, 1000));
CCshow (p, 3, "Cdi", (long)fmul(Cdi, 1000));
CCshow (p, 3, "Cdp", (long)fmul(Cdp, 1000));
		vmag = iabs(p->speed/VONE);
		if (vmag >= 180) {
			q = muldiv (vmag, vmag, 128);
			qf = 128;
		} else {
			q = vmag * vmag;
			qf = 1;
		}
		q = fmul (rho, q);
/*
 * VONE is internal force factor
 * 10 = ft^2 -> m^2
*/
		if (EP->wing_area > 10*2*VONE*2) {
			q = muldiv (q, EP->wing_area, 10*2*VONE*32);
			qf *= 32;
		} else {
			q = muldiv (q, EP->wing_area, 10*2*VONE*2);
			qf *= 2;
		}
CCshow (p, 0, "q", (long)q);
CCshow (p, 0, "qf", (long)qf);
		drag = fmul (q, Cdi);
CCshow (p, 0, "Di", (long)drag);
		Cd = Cdp;
CCshow (p, 3, "Cdp", (long)fmul(Cd, 1000));
		if (EX->airbrake)
			Cd += muldiv (Cds, EX->airbrake, 100);
CCshow (p, 3, "+Cds", (long)fmul(Cd, 1000));
		if (EX->equip & EQ_GEAR)
			Cd += Cdg;
CCshow (p, 3, "+Cdg", (long)fmul(Cd, 1000));
		drag += fmul (q, Cd);
CCshow (p, 0, "dragF", (long)drag);
		drag = muldiv (drag, VONE*qf, weight);
CCshow (p, 0, "dragA", (long)drag);
		if (v >= 0)
			acc -= drag;
		else
			acc += drag;
CCshow (p, 0, "acc", (long)acc);

		lift = muldiv (fmul (q, Cl), EP->opt[3], 4);
CCshow (p, 0, "liftF", (long)lift);
		lift = muldiv (lift, VONE*qf*4, weight);
CCshow (p, 0, "liftA", (long)lift);

#if 1
		if (onground) {
			t = p->speed - EP->liftoff_speed*VONE/2; /*nm->meter*/
			if (t < 0 && p->a[X] <= EP->gpitch)
				t = 0;
			if (p->da[X] > t)
				p->da[X] = t;
			if (p->da[X] < 0 && p->a[X] <= EP->gpitch)
				p->da[X] = 0;
			p->da[Y] = 0;
			p->a[Y] = 0;
		}
#else
		if (onground) {
			/* say torque is 1/4 of lift against 1/10 of the weight */
			t = lift/2 + AA[Z]/5;
			if (t < 0 && p->a[X] <= EP->gpitch)
				t = 0;
			if (p->da[X] > t)
				p->da[X] = t;
			if (p->da[X] < 0 && p->a[X] <= EP->gpitch)
				p->da[X] = 0;
			p->da[Y] = 0;
			p->a[Y] = 0;
		}
#endif
		if (onground && (EX->equip & EQ_GNDBRAKE)) {
			t = BRAKE_FORCE*VONE;
			if (abs (v) <= t) {		/* stop! */
				acc = 0;
				lift = 0;
				EX->v[X] = EX->v[Y] = EX->v[Z] = 0;
			} else if (p->speed > 0)
				acc -= t;
			else
				acc += t;
		}

CCshow (p, 0, "G[Y]", (long)AA[Y]);
		AA[Y] += acc;
CCshow (p, 0, "AA[Y]", (long)AA[Y]);
		AA[Z] += lift;

		if (onground) {
			EX->wing_stall = 0;
		} else {
			EX->misc[5] += AA[Z];
			if (EX->misc[5] > GPLIMIT) {
				glimit = 1;
			} else if (EX->misc[5] < -GNLIMIT) {
				glimit = 1;
			}
			if (EX->equip & EQ_GEAR) {
				t = ~1 & fmul (p->speed, FCON(0.02));
				t = t*t;
				AA[X] += TADJ(rand()%(1+t) - t/2);
				AA[Y] -= TADJ(rand()%(  t));
				AA[Z] += TADJ(rand()%(1+t) - t/2);
			}
		}

		EX->v[X] += TADJ(AA[X]);
CCshow (p, 0, "v[Y]", (long)EX->v[Y]);
		EX->v[Y] += (t = TADJ(AA[Y]));
CCshow (p, 0, "dv", (long)t);
CCshow (p, 0, "v[Y]", (long)EX->v[Y]);
		EX->v[Z] += TADJ(AA[Z]);
		VMmul (p->V, EX->v, p->T);
#if 0
		if (onground && p->V[Z] < 0) {
			p->V[Z] = 0;
			Mxpose (p->T);
			VMmul (EX->v, p->V, p->T);
			Mxpose (p->T);
		}
#endif
		p->speed = ihypot3d (p->V);
	}

	da[X] = TADJ(p->da[X])*VONE;
	da[Y] = TADJ(p->da[Y])*VONE;
	da[Z] = TADJ(p->da[Z])*VONE;

	Myxz (p->T, da);		/* rebuild from scratch */

	if (!taxiing) {
		Mxpose (p->T);
		Vcopy (AA, EX->v);
		VMmul (EX->v, AA, p->T);
		Mxpose (p->T);
	}

	Mroty (p->T, p->a[Y]);
	Mrotx (p->T, p->a[X]);
	Mrotz (p->T, p->a[Z]);

	Mangles1 (p->T, p->a);

	if (onground && p->a[X] < EP->gpitch) {
		p->a[X] = EP->gpitch;
		Mobj (p);
	}

	if (taxiing) {
		VMmul (p->V, EX->v, p->T);
		p->V[Z] = 0;
	}

	if (onground && p->V[Z] < 0) {
		p->V[Z] = 0;
		Mxpose (p->T);
		VMmul (EX->v, p->V, p->T);
		Mxpose (p->T);
		p->speed = ihypot3d (p->V);
	}

#define MAX_SPEED	1000
	if (p->speed > MAX_SPEED*VONE) {		/* temp */
		t = muldiv (FONE, (int)(MAX_SPEED*VONE), p->speed);
		EX->v[X] = fmul (t, EX->v[X]);
		EX->v[Y] = fmul (t, EX->v[Y]);
		EX->v[Z] = fmul (t, EX->v[Z]);
		p->V[X]  = fmul (t, p->V[X]);
		p->V[Y]  = fmul (t, p->V[Y]);
		p->V[Z]  = fmul (t, p->V[Z]);
		p->speed = ihypot3d (p->V);
	}
	EX->misc[10] = muldiv (p->speed, 1000, sos); /* good enough */

	if (p == CC) {
		st.indicators[0] = (p->R[Z] < 0) ? st.red : 0;
		st.indicators[1] = (p->R[Z]+p->V[Z] < 0) ? st.brown : 0;
		st.indicators[2] = (p->a[Y] > D90 || p->a[Y] < -D90)
					? st.green : 0;
		st.indicators[3] = EX->wing_stall ? st.red : 0;
		st.indicators[4] = glimit ? st.red : 0;
	}
}
