/* ----------------------- circbuff.c --------------------------------	*/
/* 									*/
/* Xin:  use XON/XOFF to control incoming data. (send them)		*/
/* Xout: use XON/XOFF to control outgoing data.	(read them)		*/
/* 									*/
/* port numbers are 1-4 for COM1-COM4					*/
/* 									*/
/* 									*/

#include <stdlib.h>
#include <string.h>
#include <conio.h>

#include "portio.h"
#include "com.h"

#define  BSIZE	   4096

int	near portbase[] = {0, COM1BASE, COM2BASE, COM3BASE, COM4BASE};
int	near ccbidle[]  = {0, 0,        0,        0,        0       };
CCB	near ccb[4] = {
	{0, COM1IRQ, com1_int, 115200L, 'n', 8, 1, 0, BSIZE, 0},
	{0, COM2IRQ, com2_int, 115200L, 'n', 8, 1, 0, BSIZE, 0},
	{0, COM3IRQ, com3_int, 115200L, 'n', 8, 1, 0, BSIZE, 0},
	{0, COM4IRQ, com4_int, 115200L, 'n', 8, 1, 0, BSIZE, 0}
};

extern int far
port_set (int port, int isize, int osize, int irq, int base)
{
	register CCB	near *ccp;

	if (port < 1 || port > 4)
		return (1);
	ccp = &ccb[port-1];
	if (ccp->flags & On)
		return (1);

	if (-1 != isize) {
		ccp->isize = isize;
		ccp->himark = (isize/5)*4;
		ccp->lowmark = isize/4;
	}

	if (-1 != osize) {
		ccp->osize = osize;
	}

	if (-1 != irq) {
		ccp->irq = irq;
	}

	if (-1 != base) {
		portbase[port] = base;
	}
	return (0);
}

extern int far
initccb (int port)
{
	register CCB	near *ccp;

	if (port < 1 || port > 4)
		return (1);
	ccp = &ccb[port-1];
	if (ccp->flags & On)
		return (1);

	if (ccp->idata) {
		free (ccp->idata);
		ccp->idata = 0;
	}
	if (ccp->odata) {
		free (ccp->odata);
		ccp->odata = 0;
	}

	if (!(ccp->idata = (char *)malloc(ccp->isize)))
		return (1);
	if (ccp->osize) {
		if (!(ccp->odata = (char *)malloc(ccp->osize))) {
			free (ccp->idata);
			ccp->idata = 0;
			return (1);
		}
		ccp->flags = On|Idle;
		ccbidle[port-1] = 1;
	} else {
		ccp->flags = On;
		ccbidle[port-1] = 0;
		ccp->odata = 0;
	}

	ccp->itail = ccp->ihead = ccp->idata;
	ccp->iend = ccp->idata+ccp->isize;
	ccp->icnt = 0;

	ccp->otail = ccp->ohead = ccp->odata;
	ccp->oend = ccp->odata+ccp->osize;
	ccp->ocnt = 0;

	return (0);
}

extern void far
termccb (int port)
{
	register CCB	near *ccp;

	if (port < 1 || port > 4)
		return;
	ccp = &ccb[port-1];
	if (!(ccp->flags & On))
		return;

	if (ccp->idata) {
		free (ccp->idata);
		ccp->idata = 0;
	}
	if (ccp->odata) {
		free (ccp->odata);
		ccp->odata = 0;
	}
	ccp->flags &= ~On;
}

extern void far
flushccb (int port)
{
	register CCB	near *ccp;
	short		iflags;

	if (port < 1 || port > 4)
		return;
	ccp = &ccb[port-1];
	if (!(ccp->flags & On))
		return;

	iflags = comcli ();
	ccp->flags &= On;
	if (ccp->osize) {
		ccp->flags |= Idle;
		ccbidle[port-1] = 1;
	}
	ccp->itail = ccp->ihead = ccp->idata;
	ccp->icnt = 0;
	ccp->otail = ccp->ohead = ccp->odata;
	ccp->ocnt = 0;
	comsti (iflags);

	while (!(inp (LSR) & XMTRDY))	/* get XMTR ready */
		;
}

extern int far
drainccb (int port)
{
	register CCB	near *ccp;
	int		i, cnt;

	if (port < 1 || port > 4)
		return (1);
	ccp = &ccb[port-1];
	if (!(ccp->flags & On))
		return (1);

	for (cnt = ccp->ocnt, i = 0; i < 10000; ++i) {
		if (ccp->flags & Idle)		/* drained */
			return (0);
		if (ccp->ocnt < cnt) {
			cnt = ccp->ocnt;
			i = 0;
		}
	}
	return (1);				/* timed out */
}

static int far
xmthard (int port, int c)
/*
 * As this function can be called from comout() as well as from the receive
 * interrupt rcvint() some carefull exclusion must be done.
*/
{
	long	tout;
	short	flags;

	tout = 10000L;
retry:
	while (!(inp (LSR) & XMTRDY))	/* wait for XMTR ready */
		if (--tout <= 0)
			return (-1);

	flags = comcli ();
	if (!(inp (LSR) & XMTRDY)) {
		comsti (flags);
		goto retry;
	}

	outp (TX, c);

	comsti (flags);
	return (0);
}

static int far
xmt (int port, int c)			/* send a char syncronousely */
{
	register CCB	near *ccp;

	ccp = &ccb[port-1];

	if (ccp->flags & XoffPend) {
		if (!xmthard (port, XOFF)) {		/* urgent */
			ccp->flags &= ~XoffPend;
			ccp->flags ^= XinOff;
		}
	} else if (ccp->flags & XonPend) {
		if (!xmthard (port, XON)) {		/* urgent */
			ccp->flags &= ~XonPend;
			ccp->flags &= ~XinOff;
		}
	}

	if (-1 == c)
		return (0);

	if ((ccp->xmode & Xout) && (ccp->flags & XoutOff))	/* blocked */
		return (-1);

	return (xmthard (port, c));
}

extern int far				/* was ogetccb () */
xmtint (int port)			/* get the next char to send */
{					/* called from write interrupt */
	register CCB	near *ccp;	/* called internally if xmtr idle */
	int		c;		/* Must have interrupts disabled! */

	ccp = &ccb[port-1];

	if (ccp->flags & XoffPend) {
		ccp->flags &= ~XoffPend;
		ccp->flags ^= XinOff;
		c = XOFF;
	} else if (ccp->flags & XonPend) {
		ccp->flags &= ~XonPend;
		ccp->flags &= ~XinOff;
		c = XON;
	} else if ((ccp->xmode & Xout) && (ccp->flags & XoutOff) || /* blocked */
		   !ccp->ocnt) {	/* nothing to send: Xoff or empty */
		ccp->flags |= Idle;
		ccbidle[port-1] = 1;
#ifdef PLAY_TX_INT
		f = inp (IER) & ~TX_INT;
		outp (IER, f);
#endif
		return (-1);
	} else {
		c = *ccp->ohead++;
		if (ccp->ohead == ccp->oend)
			ccp->ohead = ccp->odata;
		ccp->ocnt--;
	}

	outp (TX, c);
	if (ccp->flags & Idle) {
		ccp->flags &= ~Idle;		/* now active */
		ccbidle[port-1] = 0;
#ifdef PLAY_TX_INT
		f = inp (IER) | TX_INT;
		outp (IER, f);
#endif
	}
	return (c);
}

extern int far			/* was putccb() */
rcvint (int port, int c)	/* called from read interrupt */
{				/* return -1 if buffer full, c otherwise */
	register CCB	near *ccp;	/* Must have interrupts disabled! */

	ccp = &ccb[port-1];

	if (ccp->xmode & Xout) {
		if (c == XON) {
			ccp->flags &= ~XoutOff;
			goto xret;
		} else if (c == XOFF) {
			ccp->flags ^= XoutOff;
			goto xret;
		}
	}

	if (ccp->icnt == ccp->isize)		/* full */
		return (-1);

	*ccp->itail++ = (char)c;
	if (ccp->itail == ccp->iend)
		ccp->itail = ccp->idata;

	if (++ccp->icnt == ccp->himark && (ccp->xmode & Xin)) {
		ccp->flags &= ~(XoffPend|XonPend);
		if (!(ccp->flags & XinOff)) {
			ccp->flags |= XoffPend;
			goto xret;
		}
	}
	return (c);

xret:
	if (!ccp->osize)			/* unbuffered output */
		xmt (port, -1);
	else if (ccp->flags & Idle)
		xmtint (port);

	return (c);
}

extern int far			/* was getccb () */
comin (int port)		/* user calls to read an incoming char */
{				/* return -1 if none */
	register CCB	near *ccp;
	int		c;
	short		flags;

	if (port < 1 || port > 4)
		return (-1);
	ccp = &ccb[port-1];
	if (!(ccp->flags & On))
		return (-1);

	if (!ccp->icnt)		/* buffer empty */
		return (-1);

	flags = comcli ();
	if ((ccp->xmode & Xin) && ccp->icnt < ccp->lowmark) {
		ccp->flags &= ~(XoffPend|XonPend);
		if (ccp->flags & XinOff)
			ccp->flags |= XonPend;
	}

	c = *ccp->ihead++;
	if (ccp->ihead == ccp->iend)
		ccp->ihead = ccp->idata;
	ccp->icnt--;

	comsti (flags);
	return (c);
}

extern int far			/* was oputccb() */
comout (int port, int c)	/* user calls to send a char */
{
	register CCB	near *ccp;
	short		iflags;

	if (port < 1 || port > 4)
		return (-1);
	ccp = &ccb[port-1];
	if (!(ccp->flags & On))
		return (-1);

	if ((ccp->xmode & Xin) && (c == XON || c == XOFF))
		return (-1);		/* control chars not allowed */

	if (!ccp->osize)		/* unbuffered */
		return (xmt (port, c));

	if (ccp->ocnt == ccp->osize)	/* buffer full */
		return (-1);

	iflags = comcli ();
	*ccp->otail++ = (char)c;	/* add 'c' to buffer */
	if (ccp->otail == ccp->oend)
		ccp->otail = ccp->odata;
	++ccp->ocnt;

	if (ccp->flags & Idle)
		xmtint (port);

	comsti (iflags);

	return (0);
}

extern int far
comoutb (int port, unsigned char *b, int len)	/* user call to send a block */
/*
 * Interrupts are not immediately blocked, the free space can only grow while
 * we are deliberating.
*/
{
	register CCB	near *ccp;
	short		iflags;
	int		i, t;

	if (port < 1 || port > 4)
		return (-1);
	ccp = &ccb[port-1];
	if (!(ccp->flags & On))
		return (-1);

	if (ccp->osize) {				/* buffered */
		if (ccp->osize - ccp->ocnt < len)	/* free space */
			goto byte_by_byte;
		iflags = comcli ();
		i = ccp->oend - ccp->otail;		/* room till end */
		if (i > len)
			i = len;
		memcpy (ccp->otail, b, i);
		ccp->otail += i;
		ccp->ocnt += i;
		b += i;
		len -= i;
		if (ccp->otail == ccp->oend) {
			ccp->otail = ccp->odata;
			if (len) {
				memcpy (ccp->otail, b, len);
				ccp->otail += len;
				ccp->ocnt += len;
			}
		}
		if (ccp->flags & Idle)
			xmtint (port);
		comsti (iflags);
		return (0);
	}

byte_by_byte:
	for (; len-- > 0; ++b)
		for (t = 0; comout (port, *b);)
			if (++t > 100)
				return (-1);
	return (0);
}

extern void far
get_ccbinfo (int port, int info[])
{
	if (port < 1 || port > 4)
		port = 1;
	info[0] = ccb[port-1].icnt;
	info[1] = ccb[port-1].ocnt;
	info[2] = ccb[port-1].flags;
}
