/*
 *  This file is part of mortgage, which is Copyright (C) 1999 Jean-Jacques Moreau
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software Foundation,
 *  Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include "pal.h"
#include "palsmsvc.h"
#include "sampledl.h"

#define KEYDEPTH   9

typedef enum {
   CMD_ABOUT = 1, CMD_EXIT
} MENUCOMMANDS;

typedef struct _MORTGAGE {
   double Loan;
   double Rate;
   double Years;
   double Period;
   double Payment;
   double Interests;
   int LoanKnown;
   int IsValid;
   int Number;
} MORTGAGE;

PALWIN *pWin[5];
MORTGAGE Mortgages[5];

/* the 'File' pulldown: Items and Menu */
MENUITEM FileItems[] = {
   { "E&xit",              MA_VAL,        MENUVAL(CMD_EXIT)   },
};
MENU FileMenu = { NULL, MS_PULLDN, 0,0,1,1, FileItems };

/* the 'About' pulldown: Items and Menu */
MENUITEM AboutItems[] = {
   { "&About...", MA_VAL, MENUVAL(CMD_ABOUT)},
};
MENU AboutMenu = { NULL, MS_PULLDN, 0,0,1,1, AboutItems };

/* the menu bar  */
MENUITEM BarItems[] = {
   { "&File",         MA_MNU, &FileMenu      },
   { "&Quit",         MA_VAL, MENUVAL(CMD_EXIT) },
   { "&Help",         MA_MNU, &AboutMenu      },
};
MENU BarMenu = { NULL, MS_HORIZ|MS_TOPLVL, 0,0,3,3, BarItems };


void CalcInterests(MORTGAGE *pMortgage) {
   pMortgage->Interests = pMortgage->Payment*pMortgage->Years*pMortgage->Period - pMortgage->Loan;
}

void CalcPayment(MORTGAGE *pMortgage) {
   double t = pMortgage->Rate / pMortgage->Period / 100;
   double payment = (pMortgage->Loan*t) / (1 - (1/pow(1+t, pMortgage->Period*pMortgage->Years)));

   pMortgage->Payment = ceil(payment);

   pMortgage->IsValid = 1;
}

void CalcLoan(MORTGAGE *pMortgage) {
   double t = pMortgage->Rate / pMortgage->Period / 100;
   double loan = pMortgage->Payment * (1 - 1/pow(1+t, pMortgage->Period*pMortgage->Years)) / t;

   pMortgage->Loan = floor(loan);

   pMortgage->IsValid = 1;
}

void GetMortgage(MORTGAGE *pMortgage, DIALOG *pSampleDlg) {
   char LoanStr[40];
   char RateStr[40];
   char YearsStr[40];
   char PeriodStr[40];
   char PaymentStr[40];
   char InterestsStr[40];
   int  CalculatePayment;

   GetDlgItem(pSampleDlg, EDLOAN,    EDGI_TXT,   LoanStr);
   pMortgage->Loan = atof(LoanStr);
   GetDlgItem(pSampleDlg, EDRATE,    EDGI_TXT,   RateStr);
   pMortgage->Rate = atof(RateStr);
   GetDlgItem(pSampleDlg, EDYEARS,   EDGI_TXT,   YearsStr);
   pMortgage->Years = atof(YearsStr);
   GetDlgItem(pSampleDlg, EDPERIOD,  EDGI_TXT,   PeriodStr);
   pMortgage->Period = atof(PeriodStr);
   GetDlgItem(pSampleDlg, EDPAYMENT, EDGI_TXT,   PaymentStr);
   pMortgage->Payment = atof(PaymentStr);

   GetDlgItem(pSampleDlg, RAPAYMENT, RAGI_STATE, &CalculatePayment);
   pMortgage->LoanKnown = CalculatePayment ? 1 : 0;

   if (pMortgage->Rate == 0 || pMortgage->Years == 0 || pMortgage->Period == 0 || (pMortgage->Loan == 0 && pMortgage->Payment == 0)) {
      pMortgage->IsValid = 0;
      return;
   }

   if (pMortgage->LoanKnown)
      CalcPayment(pMortgage);
   else
      CalcLoan(pMortgage);

   CalcInterests(pMortgage);
}

void UpdateMortgage(MORTGAGE *pMortgage, DIALOG *pSampleDlg) {
   char Buffer[40];

   if (!pMortgage->IsValid) {
      SetDlgItem(pSampleDlg, EDLOAN, EDSI_TXT, "");
      SetDlgItem(pSampleDlg, EDRATE, EDSI_TXT, "");
      SetDlgItem(pSampleDlg, EDYEARS, EDSI_TXT, "");
      SetDlgItem(pSampleDlg, EDPERIOD, EDSI_TXT, "");
      SetDlgItem(pSampleDlg, EDPAYMENT, EDSI_TXT, "");
      SetDlgItem(pSampleDlg, EDINTERESTS, EDSI_TXT, "");
      SetDlgItem(pSampleDlg, RAPAYMENT, RASI_STATE, (void *)1);
      SetDlgItem(pSampleDlg, RALOAN, RASI_STATE, (void *)0);

      return;
   }

   sprintf(Buffer, "%.0f", pMortgage->Loan);
   SetDlgItem(pSampleDlg, EDLOAN, EDSI_TXT, Buffer);
   sprintf(Buffer, "%.2f", pMortgage->Rate);
   SetDlgItem(pSampleDlg, EDRATE, EDSI_TXT, Buffer);
   sprintf(Buffer, "%.0f", pMortgage->Years);
   SetDlgItem(pSampleDlg, EDYEARS, EDSI_TXT, Buffer);
   sprintf(Buffer, "%.0f", pMortgage->Period);
   SetDlgItem(pSampleDlg, EDPERIOD, EDSI_TXT, Buffer);
   sprintf(Buffer, "%.0f", pMortgage->Payment);
   SetDlgItem(pSampleDlg, EDPAYMENT, EDSI_TXT, Buffer);
   sprintf(Buffer, "%.0f", pMortgage->Interests);
   SetDlgItem(pSampleDlg, EDINTERESTS, EDSI_TXT, Buffer);
   SetDlgItem(pSampleDlg, RAPAYMENT, RASI_STATE, (void *)pMortgage->LoanKnown);
   SetDlgItem(pSampleDlg, RALOAN, RASI_STATE, (void *)!pMortgage->LoanKnown);
}

void ShowScenario(MORTGAGE *pMortgage, PALWIN *pWin) {
   char Buffer[128];

   if (! pMortgage->IsValid)
      return;

   sprintf(Buffer, "%-4s: %7.0f", "LOAN", pMortgage->Loan);
   WinText(pWin, 5, 5, Buffer);
   sprintf(Buffer, "%-4s: %7.0f", "PMT", pMortgage->Payment);
   WinText(pWin, 5, 15, Buffer);
   sprintf(Buffer, "%-4s: %7.0f", "I", pMortgage->Interests);
   WinText(pWin, 5, 25, Buffer);

   WinLine(pWin, 5, 37, MAX_X/4-5, 37);

   sprintf(Buffer, "%-4s: %7.0f", "CST", pMortgage->Loan+pMortgage->Interests);
   WinText(pWin, 5, 40, Buffer);

   sprintf(Buffer, "%-4s: %7.2f", "I%YR", pMortgage->Rate);
   WinText(pWin, 5, 135, Buffer);
   sprintf(Buffer, "%-4s: %7.0f", "YRs", pMortgage->Years);
   WinText(pWin, 5, 145, Buffer);
   sprintf(Buffer, "%-4s: %7.0f", "P/YR", pMortgage->Period);
   WinText(pWin, 5, 155, Buffer);
}

void Input(int Scenario) {
   int n;
   int  Result;
   char pBuffer[128];

   InitDialog(&SampleDlg);                       /* initialize dialog */
   ShowDialog(&SampleDlg, 50, 20, "Mortgage Interest Calculator"); /* display it */
   UpdateMortgage(&Mortgages[Scenario], &SampleDlg);

   Result = HandleDialog(&SampleDlg, 0);         /* handle keys */
   while (Result == BNCALC) {
      GetMortgage(&Mortgages[Scenario], &SampleDlg);
      UpdateMortgage(&Mortgages[Scenario], &SampleDlg);

      Result = HandleDialog(&SampleDlg, 0);
   }
   GetMortgage(&Mortgages[Scenario], &SampleDlg);
   UpdateMortgage(&Mortgages[Scenario], &SampleDlg);

   CloseDialog(&SampleDlg);

   ShowScenario(&Mortgages[Scenario], pWin[Scenario]);
}

void About(void) {
   PALWIN *pWin;
   char *KeyLabels[10] = {
      NULL, NULL, NULL, NULL, NULL,
      NULL, NULL, NULL, NULL, "OK"
   };
   WORD Key;

   pWin  = OpenWin(WS_HP200, 80, 35, 585, 145, "About Mortgage Calculator");
   WinTextAlignH(pWin, 10, "HP200LX Mortgage Calculator 1.0", MEDIUM_FONT, TAS_CENTER);
   WinTextAlignH(pWin, 25, "This program is Freeware.", MEDIUM_FONT, TAS_CENTER);
   WinTextAlignH(pWin, 50, "(c) 1999-2000 Jean-Jacques Moreau.", MEDIUM_FONT, TAS_CENTER);
   WinTextAlignH(pWin, 65, "jean-jacques.moreau@wanadoo.fr", MEDIUM_FONT, TAS_CENTER);
   WinTextAlignH(pWin, 80, "Contact me for praise or bug reports", MEDIUM_FONT, TAS_CENTER);

   SaveFKeys();
   ShowFKeys(KeyLabels);
   do {
      Key = GetKey();
      if(Key & 0xff)
         Key &= 0xff;
      if (Key != ESC_KEY && Key != F10_KEY)
         m_beep();
   } while (Key != ESC_KEY && Key != F10_KEY);
   RestoreFKeys();

   CloseWin(pWin);
}

void Interact(void) {
   WORD Key;

   do {
      Key = GetKey();
      if(Key & 0xff)
         Key &= 0xff;

      if(Key == MENU_KEY || Key == AF10_KEY || Key == F8_KEY)
         Key = HandleMenu(&BarMenu, 0, 10);

      switch(Key) {
         case F2_KEY:
            Input(1);
            break;
         case F3_KEY:
            Input(2);
            break;
         case F4_KEY:
            Input(3);
            break;
         case F5_KEY:
            Input(4);
            break;
         case F1_KEY:
         case CMD_ABOUT:
            About();
            break;
         case CMD_EXIT:
            return;
      }

   } while(Key != ESC_KEY && Key != F10_KEY);
}

void Init(void) {
   int i;
   char *KeyLabels[10] = {
      "About", "Mort1", "Mort2", "Mort3", "Mort4",
      NULL, NULL, NULL, NULL, "Quit"
   };

   if(!PalInit(1))
      FatalExit("Init failed - CGAGRAPH not loaded ?", 1);
   ShowTopTime("Mortgage Calculator", FALSE);
   ShowFKeys(KeyLabels);

   pWin[1] = OpenWin(WS_BORDER | WS_SAVEBG | WS_SMLFONT, 1,           3+FNTD(SMALL_FONT), MAX_X/4,   MAX_Y-KEYDEPTH-3, "Mortgage1");
   pWin[2] = OpenWin(WS_BORDER | WS_SAVEBG | WS_SMLFONT, MAX_X/4+1,   3+FNTD(SMALL_FONT), MAX_X/2,   MAX_Y-KEYDEPTH-3, "Mortgage2");
   pWin[3] = OpenWin(WS_BORDER | WS_SAVEBG | WS_SMLFONT, MAX_X/2+1,   3+FNTD(SMALL_FONT), 3*MAX_X/4, MAX_Y-KEYDEPTH-3, "Mortgage3");
   pWin[4] = OpenWin(WS_BORDER | WS_SAVEBG | WS_SMLFONT, 3*MAX_X/4+1, 3+FNTD(SMALL_FONT), MAX_X-1,   MAX_Y-KEYDEPTH-3, "Mortgage4");

   for(i = 1; i <= 4; i++) {
      Mortgages[i].IsValid = 0;
      Mortgages[i].Number = i;
   }
}

void Terminate(void) {
   int i;

   for(i = 1; i <= 4; i++) {
      CloseWin(pWin[i]);
   }
   PalDeInit(1);
}

void main(void) {
   Init();
   Interact();
   Terminate();
}
