/* serial port routines */

#include "currentcost.h"

#define BAUDRATE 57600

/********************************************************************
 * open_weatherstation, Linux version
 *
 * Input:   devicename (/dev/tty0, /dev/tty1 etc)
 * 
 * Returns: Handle to the weatherstation (type CCHANDLE)
 *
 ********************************************************************/
CCHANDLE open_weatherstation(char *device)
{
	CCHANDLE ws2300;
	struct termios adtio;
	int portstatus, fdflags;

	//Setup serial port

	if ((ws2300 = open(device, O_RDWR | O_NONBLOCK)) < 0)
	{
		printf("\nUnable to open serial device %s\n", device);
		exit(EXIT_FAILURE);
	}
	
	if ( flock(ws2300, LOCK_EX|LOCK_NB) < 0 ) {
		perror("\nSerial device is locked by other program\n");
		exit(EXIT_FAILURE);
	}
	
	if ((fdflags = fcntl(ws2300, F_GETFL)) == -1 ||
	     fcntl(ws2300, F_SETFL, fdflags & ~O_NONBLOCK) < 0)
	{
		perror("couldn't reset non-blocking mode");
		exit(EXIT_FAILURE);
	}
	
	//We want full control of what is set and simply reset the entire adtio struct
	memset(&adtio, 0, sizeof(adtio));
	
	//tcgetattr(ws2300, &adtio);   // Commented out and replaced by the memset above
	
	// Serial control options
	adtio.c_cflag &= ~PARENB;      // No parity
	adtio.c_cflag &= ~CSTOPB;      // One stop bit
	adtio.c_cflag &= ~CSIZE;       // Character size mask
	adtio.c_cflag |= CS8;          // Character size 8 bits
	adtio.c_cflag |= CREAD;        // Enable Receiver
	adtio.c_cflag &= ~HUPCL;       // No "hangup"
	adtio.c_cflag &= ~CRTSCTS;     // No flowcontrol
	adtio.c_cflag |= CLOCAL;       // Ignore modem control lines

	// Baudrate, for newer systems
	cfsetispeed(&adtio, BAUDRATE);
	cfsetospeed(&adtio, BAUDRATE);	
	
	// Serial local options: adtio.c_lflag
	// Raw input = clear ICANON, ECHO, ECHOE, and ISIG
	// Disable misc other local features = clear FLUSHO, NOFLSH, TOSTOP, PENDIN, and IEXTEN
	// So we actually clear all flags in adtio.c_lflag
	adtio.c_lflag = 0;

	// Serial input options: adtio.c_iflag
	// Disable parity check = clear INPCK, PARMRK, and ISTRIP 
	// Disable software flow control = clear IXON, IXOFF, and IXANY
	// Disable any translation of CR and LF = clear INLCR, IGNCR, and ICRNL	
	// Ignore break condition on input = set IGNBRK
	// Ignore parity errors just in case = set IGNPAR;
	// So we can clear all flags except IGNBRK and IGNPAR
	adtio.c_iflag = IGNBRK|IGNPAR;
	
	// Serial output options
	// Raw output should disable all other output options
	adtio.c_oflag &= ~OPOST;

	adtio.c_cc[VTIME] = 10;		// timer 1s
	adtio.c_cc[VMIN] = 0;		// blocking read until 1 char
	
	if (tcsetattr(ws2300, TCSANOW, &adtio) < 0)
	{
		printf("Unable to initialize serial device");
		exit(EXIT_FAILURE);
	}

	tcflush(ws2300, TCIOFLUSH);

	// Set DTR low and RTS high and leave other ctrl lines untouched

	ioctl(ws2300, TIOCMGET, &portstatus);	// get current port status
	portstatus &= ~TIOCM_DTR;
	portstatus |= TIOCM_RTS;
	ioctl(ws2300, TIOCMSET, &portstatus);	// set current port status

	return ws2300;
}

/********************************************************************
 * close_weatherstation, Linux version
 *
 * Input: Handle to the weatherstation (type CCHANDLE)
 *
 * Returns nothing
 *
 ********************************************************************/
void close_weatherstation(CCHANDLE ws)
{
	close(ws);
	return;
}


/********************************************************************
 * read_device in the Linux version is identical
 * to the standard Linux read() 
 *
 * Inputs:  serdevice - opened file handle
 *          buffer - pointer to the buffer to read into (unsigned char)
 *          size - number of bytes to read
 *
 * Output:  *buffer - modified on success (pointer to unsigned char)
 * 
 * Returns: number of bytes read
 *
 ********************************************************************/
int read_device(CCHANDLE serdevice, unsigned char *buffer, int size)
{
	int ret;

	for (;;) {
		ret = read(serdevice, buffer, size);
		if (ret == 0 && errno == EINTR)
			continue;
		return ret;
	}
}

