// $Id: password_strength.js,v 1.4.2.2.2.1 2009/08/13 21:51:49 jrbeeman Exp $

/**
 * Attach handlers to evaluate the strength of any password fields and to check
 * that its confirmation is correct.
 */
Drupal.passwordAttach = function() {
  var translate = Drupal.settings.password;
  delete Drupal.behaviors.password;
  $("input.password-field:not(.password-processed)").each(function() {
    var passwordInput = $(this).addClass('password-processed');
    var parent = $(this).parent();
    // Wait this number of milliseconds before checking password.
    var monitorDelay = 700;

    // Add the password strength layers.
    $(this).after('<span class="password-strength"><span class="password-title">'+ translate.strengthTitle +'</span> <span class="password-result"></span></span>').parent();
    var passwordStrength = $("span.password-strength", parent);
    var passwordResult = $("span.password-result", passwordStrength);
    parent.addClass("password-parent");

    // Add the password confirmation layer.
    var outerItem  = $(this).parent().parent();
    $("input.password-confirm", outerItem).after('<span class="password-confirm">'+ translate["confirmTitle"] +' <span></span></span>').parent().addClass("confirm-parent");
    var confirmInput = $("input.password-confirm", outerItem);
    var confirmResult = $("span.password-confirm", outerItem);
    var confirmChild = $("span", confirmResult);

    // Add the description box at the end.
    $(confirmInput).parent().after('<div class="password-description"></div>');
    var passwordDescription = $("div.password-description", $(this).parent().parent()).hide();

    // Check the password fields.
    var passwordCheck = function () {
      // Remove timers for a delayed check if they exist.
      if (this.timer) {
        clearTimeout(this.timer);
      }

      // Verify that there is a password to check.
      if (!passwordInput.val()) {
        passwordStrength.css({ visibility: "hidden" });
        passwordDescription.hide();
        return;
      }

      // Evaluate password strength.

      var result = Drupal.evaluatePasswordStrength(passwordInput.val());
      passwordResult.html(result.strength == "" ? "" : translate[result.strength +"Strength"]);

      // Map the password strength to the relevant drupal CSS class.
      var classMap = { low: "error", medium: "warning", high: "ok" };
      var newClass = classMap[result.strength] || "";

      // Remove the previous styling if any exists; add the new class.
      if (this.passwordClass) {
        passwordResult.removeClass(this.passwordClass);
        passwordDescription.removeClass(this.passwordClass);
      }
      passwordDescription.html(result.message);
      passwordResult.addClass(newClass);
      if (result.strength == "high") {
        passwordDescription.hide();
      }
      else {
        passwordDescription.addClass(newClass);
      }
      this.passwordClass = newClass;

      // Check that password and confirmation match.

      // Hide the result layer if confirmation is empty, otherwise show the layer.
      confirmResult.css({ visibility: (confirmInput.val() == "" ? "hidden" : "visible") });

      var success = passwordInput.val() == confirmInput.val();

      // Remove the previous styling if any exists.
      if (this.confirmClass) {
        confirmChild.removeClass(this.confirmClass);
      }

      // Fill in the correct message and set the class accordingly.
      var confirmClass = success ? "ok" : "error";
      confirmChild.html(translate["confirm"+ (success ? "Success" : "Failure")]).addClass(confirmClass);
      this.confirmClass = confirmClass;

      // Show the indicator and tips.
      passwordStrength.css({ visibility: "visible" });
      passwordDescription.show();
    };

    // Do a delayed check on the password fields.
    var passwordDelayedCheck = function() {
      // Postpone the check since the user is most likely still typing.
      if (this.timer) {
        clearTimeout(this.timer);
      }

      // When the user clears the field, hide the tips immediately.
      if (!passwordInput.val()) {
        passwordStrength.css({ visibility: "hidden" });
        passwordDescription.hide();
        return;
      }

      // Schedule the actual check.
      this.timer = setTimeout(passwordCheck, monitorDelay);
    };
    // Monitor keyup and blur events.
    // Blur must be used because a mouse paste does not trigger keyup.
    passwordInput.keyup(passwordDelayedCheck).blur(passwordCheck);
    confirmInput.keyup(passwordDelayedCheck).blur(passwordCheck);
  });
};


/**
 * Evaluate the strength of a user's password.
 *
 * Returns the estimated strength and the relevant output message.
 */
Drupal.evaluatePasswordStrength = function(value) {
  var strength = "", msg = [], out = "", translate = Drupal.settings.passwordStrength;

  var hasLetters = value.match(/[a-zA-Z]+/);
  var hasNumbers = value.match(/[0-9]+/);
  var hasPunctuation = value.match(/[^a-zA-Z0-9]+/);
  var hasCasing = value.match(/[a-z]+.*[A-Z]+|[A-Z]+.*[a-z]+/);
  var userNameForm = document.getElementById('edit-name');
  var userName =  translate.username;
  if((typeof(userNameForm)  != "undefined") && ((userNameForm  != null))) {
     userName =  userNameForm.value;
   } 
  // Check if the password is blank.
  if (!value.length) {
    strength = "";
    msg = "";
  }
  // Check if length is less than setting characters.
  else if (value.length < translate.minLength) {
    strength = "low";
    msg.push(translate.tooShort);
  }
  // Check if password is the same as the username (convert both to lowercase).
  else if (value.toLowerCase() == userName.toLowerCase()) {
    strength  = "low";
    msg.push(translate.sameAsUsername);
  }
  // Check if it contains letters, numbers, punctuation, and upper/lower case.
  else if (hasLetters && hasNumbers && hasPunctuation && hasCasing) {
    strength = "high";
  }

  if (strength != "high") {
    // Password is not secure enough so construct the medium-strength message.
    // Extremely bad passwords still count as low.
    var count = (hasLetters ? 1 : 0) + (hasNumbers ? 1 : 0) + (hasPunctuation ? 1 : 0) + (hasCasing ? 1 : 0);
    strength = count > 2 ? "medium" : "low";

    if (!hasLetters || !hasCasing) {
      msg.push(translate.addLetters);
    }
    if (!hasNumbers) {
      msg.push(translate.addNumbers);
    }
    if (!hasPunctuation) {
      msg.push(translate.addPunctuation);
    }
    out = (count >= translate.requiredStrength) ? translate.recommendVariation : translate.needsMoreVariation;
    out += "<ul><li>"+ msg.join("</li><li>") +"</li></ul>";
  }

  return { strength: strength, message: out };
};


if (Drupal.jsEnabled) {
  $(document).ready(Drupal.passwordAttach);
}